import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { ProgramRegistration } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CreateProgramRegistrationDto } from './dto/create-program-registration.dto';
import { UpdateProgramRegistrationDto } from './dto/update-program-registration.dto';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';

@Injectable()
export class ProgramRegistrationRepository {
  constructor(
    @InjectRepository(ProgramRegistration)
    private readonly repo: Repository<ProgramRegistration>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async createEntity(dto: CreateProgramRegistrationDto) {
    try {
      const entity = this.repo.create({
        ...dto,
        programSession: { id: dto.programSessionId } as any,
        user: dto.userId ? ({ id: dto.userId } as any) : undefined,
        cancelledByUser: dto.cancelledBy ? ({ id: dto.cancelledBy } as any) : undefined,
        rmContactUser: dto.rmContact ? ({ id: dto.rmContact } as any) : undefined,
        createdBy: { id: dto.createdBy } as any,
        updatedBy: { id: dto.updatedBy } as any,
      });
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  async findAll(limit: number, offset: number) {
    try {
      const data = await this.commonDataService.get(
        this.repo,
        undefined,
        {},
        limit,
        offset,
        { id: 'ASC' },
        undefined,
        ['programSession', 'user'],
      );
      const total = await this.repo.count();
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }
/**
 * Fetches all ProgramRegistration entities based on the provided where conditions and relations.
 * @param where - Partial object containing conditions to filter the ProgramRegistration entities.
 * @param relations - Array of relation names to be included in the query.
 * @returns An array of ProgramRegistration entities that match the conditions.
 */
   async getAll(where?: any, relations?: string[]) {
    try {
      return await this.commonDataService.get(
        this.repo,
        undefined,
        where || {},
        undefined,
        undefined,
        { id: 'ASC' },
        undefined,
        relations || ['programSession', 'user', 'program', 'allocatedProgram'],
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    try {
      const data = await this.commonDataService.findOneById(this.repo, id, false, ['programSession', 'user', 'program', 'allocatedProgram']);
      if (!data) {
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND, null, null, id.toString());
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  async updateEntity(id: number, dto: UpdateProgramRegistrationDto) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND, null, null, id.toString());
      }
      Object.assign(entity, dto);
      if (dto.programSessionId !== undefined) entity.programSession = { id: dto.programSessionId } as any;
      if (dto.userId !== undefined) entity.user = dto.userId ? ({ id: dto.userId } as any) : undefined;
      if (dto.cancelledBy !== undefined) entity.cancelledByUser = dto.cancelledBy ? ({ id: dto.cancelledBy } as any) : undefined;
      if (dto.rmContact !== undefined) entity.rmContactUser = dto.rmContact ? ({ id: dto.rmContact } as any) : undefined;
      entity.updatedBy = { id: dto.updatedBy } as any;
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Update a program registration entity
   * @param id - ID of the program registration to update
   * @param data - Partial data to update the program registration
   * @returns The updated program registration entity
   */
  async update(id: number, data: Partial<ProgramRegistration>) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
       handleKnownErrors(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
            null,
            null,
            id.toString()
          )
        );
      }
      Object.assign(entity, data);
      return await this.commonDataService.save(this.repo, entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  async remove(id: number) {
    try {
      const entity = await this.repo.findOne({ where: { id } });
      if (!entity) {
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND, null, null, id.toString());
      }
      await this.repo.remove(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
    }
  }

  /**
   * Fetches all ProgramRegistration entities based on the provided where conditions and relations.
   * @param where - Partial object containing conditions to filter the ProgramRegistration entities.
   * @param relations - Array of relation names to be included in the query.
   * @returns An array of ProgramRegistration entities that match the conditions.
   */
  async findBy(where?: Partial<ProgramRegistration>, relations?: string[], limit?: number, offset?: number) {
    try {
      const filteredWhere = Object.fromEntries(
        Object.entries(where ?? {}).filter(([_, v]) => v !== null && v !== undefined)
      ) as any;
      return await this.repo.find({
        where: filteredWhere,
        relations,
        take: limit,
        skip: offset,
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }
}
