import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  ParseIntPipe,
  Post,
  Put,
  Query,
  Res,
  UseGuards,
  ValidationPipe,
  Req,
} from '@nestjs/common';
import { Response } from 'express';
import { ApiBearerAuth, ApiBody, ApiOperation, ApiParam, ApiQuery, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { programRegistrationMessages, paginationConstants, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { ProgramRegistrationService } from './program-registration.service';
import { CreateProgramRegistrationDto } from './dto/create-program-registration.dto';
import { UpdateProgramRegistrationDto } from './dto/update-program-registration.dto';
import { ProgramRegistrationRmRatingService } from './services/program-registration-rm-rating.service';
import { ProgramRegistrationRMRatingDto, UpdateProgramRegistrationRMRatingDto } from './dto/program-registration-rm-ratings.dto';
import { CreateProgramRegistrationSwapDto, GetSwapRequestsDto, MakeSwapRequestDto } from './dto/program-registration-swap.dto';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('program-registration')
@Controller('program-registration')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin', 'mahatria', 'rm','finance_manager', 'relational_manager', 'shoba', 'operational_manger')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class ProgramRegistrationController {
  constructor(
    private readonly service: ProgramRegistrationService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
    private readonly rmRatingService: ProgramRegistrationRmRatingService,
  ) { }

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: programRegistrationMessages.CREATE_NEW })
  @ApiBody({ type: CreateProgramRegistrationDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: CreateProgramRegistrationDto,
    @Res() res: Response,
  ) {
    this.logger.log(programRegistrationMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      const data = await this.service.create(dto);
      await this.responseService.success(res, programRegistrationMessages.REGISTRATION_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.GET_ALL })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Res() res: Response,
  ) {
    this.logger.log(programRegistrationMessages.FIND_ALL_REQUEST_RECEIVED);
    try {
      const data = await this.service.findAll(+limit, +offset);
      await this.responseService.success(res, programRegistrationMessages.REGISTRATIONS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':programId/sub-program/:subProgramId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.GET_SUB_PROGRAM_REGISTRATIONS })
  @ApiParam({ name: 'programId', type: Number, description: programRegistrationMessages.PROGRAM_ID })
  @ApiParam({ name: 'subProgramId', type: Number, description: programRegistrationMessages.SUB_PROGRAM_ID })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiResponse({ status: HttpStatus.OK, description: programRegistrationMessages.SUB_PROGRAM_REGISTRATIONS_RETRIEVED })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findSubProgramRegistrations(
    @Param('programId', ParseIntPipe) programId: number,
    @Param('subProgramId', ParseIntPipe) subProgramId: number,
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Res() res: Response,
    @Req() req: any,
    @Query('status') status?: string,
    @Query('search') search?: string,
  ) {
    this.logger.log(programRegistrationMessages.FIND_SUB_PROGRAM_REQUEST_RECEIVED, { programId, subProgramId });
    try {
      const data = await this.service.findBy(programId, limit, offset, subProgramId, search);
      await this.responseService.success(res, programRegistrationMessages.SUB_PROGRAM_REGISTRATIONS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.GET_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: programRegistrationMessages.REGISTRATION_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(programRegistrationMessages.FIND_ONE_REQUEST_RECEIVED, { id });
    try {
      const data = await this.service.findOne(id);
      await this.responseService.success(res, programRegistrationMessages.REGISTRATION_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.UPDATE_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: programRegistrationMessages.REGISTRATION_ID })
  @ApiBody({ type: UpdateProgramRegistrationDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body(new ValidationPipe({ transform: true, whitelist: true })) dto: UpdateProgramRegistrationDto,
    @Res() res: Response,
  ) {
    this.logger.log(programRegistrationMessages.UPDATE_REQUEST_RECEIVED, { id, dto });
    try {
      const data = await this.service.update(id, dto);
      await this.responseService.success(res, programRegistrationMessages.REGISTRATION_UPDATED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.DELETE_BY_ID })
  @ApiParam({ name: 'id', type: Number, description: programRegistrationMessages.REGISTRATION_ID })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(programRegistrationMessages.DELETE_REQUEST_RECEIVED, { id });
    try {
      await this.service.remove(id);
      await this.responseService.success(res, programRegistrationMessages.REGISTRATION_DELETED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post(':programRegistrationId/rm-rating')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.POST_RM_RATINGS })
  @ApiBody({ type: ProgramRegistrationRMRatingDto })
  @ApiResponse({ status: HttpStatus.OK, description: programRegistrationMessages.RATINGS_SUBMITTED })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  async postRmRatingsAndReview(
    @Param('programRegistrationId', ParseIntPipe) programRegistrationId: number,
    @Res() res: Response,
    @Body() body: ProgramRegistrationRMRatingDto,
    @Req() req: any,
  ) {
    this.logger.log(programRegistrationMessages.POST_RM_RATINGS, { programRegistrationId, rmId: req.user.id, body });
    try {
      const rmId = req.user.id;
      await this.rmRatingService.postRmRatingsAndReview(programRegistrationId, rmId, body);
      await this.responseService.success(res, programRegistrationMessages.RATINGS_SUBMITTED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put(':programRegistrationId/rm-rating')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.UPDATE_RM_RATINGS })
  @ApiParam({ name: 'programRegistrationId', type: Number, description: 'RM Rating ID' })
  @ApiBody({ type: UpdateProgramRegistrationRMRatingDto })
  @ApiResponse({ status: HttpStatus.OK, description: programRegistrationMessages.RATINGS_UPDATED })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  async updateRmRatingAndReview(
    @Param('programRegistrationId', ParseIntPipe) programRegistrationId: number,
    @Res() res: Response,
    @Body() body: UpdateProgramRegistrationRMRatingDto,
    @Req() req: any,
  ) {
    this.logger.log(programRegistrationMessages.UPDATE_RM_RATINGS, { programRegistrationId, rmId: req.user.id, body });
    try {
      const rmId = req.user.id;
      await this.rmRatingService.updateRmRatingAndReview(programRegistrationId, rmId, body);
      await this.responseService.success(res, programRegistrationMessages.RATINGS_UPDATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post('swap')
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: programRegistrationMessages.REQUEST_SWAP })
  @ApiBody({ type: CreateProgramRegistrationSwapDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: programRegistrationMessages.SWAP_REQUEST_CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  async requestSwap(
    @Body(new ValidationPipe({ transform: true, whitelist: true }))
    body: CreateProgramRegistrationSwapDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(programRegistrationMessages.SWAP_REQUEST_RECEIVED, body);
    try {
      const userId = req.user.id;
      const data = await this.service.requestSwap(body, userId);
      await this.responseService.success(res, programRegistrationMessages.SWAP_REQUEST_CREATED, data, HttpStatus.CREATED);
      this.logger.log(programRegistrationMessages.SWAP_REQUEST_CREATED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Post('swap/direct')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.DIRECT_SWAP })
  @ApiBody({ type: MakeSwapRequestDto })
  @ApiResponse({ status: HttpStatus.OK, description: programRegistrationMessages.DIRECT_SWAP_SUCCESS })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  async directSwap(
    @Body() body: MakeSwapRequestDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(programRegistrationMessages.DIRECT_SWAP, body);
    try {
      const userId = req.user.id;
      let existingSwapRequest = await this.service.getSwapRequestByField({ programRegistrationId: body.movingSeekerRegistrationId, status: SwapRequestStatus.ACTIVE });
      if (!existingSwapRequest) {
        // 1. Create swap request
        let movingToSubProgramId;
        // if move
        if (body.swappingType === 'move') {
          if (!body.movingSeekerRegistrationId || !body.movingToSubProgramId) {
            throw new Error(programRegistrationMessages.INVALID_SWAP_REQUEST);
          } else {
            movingToSubProgramId = body.movingToSubProgramId;
          }
        } // else if swap
        else if (body.swappingType === 'swap') {
          if (!body.movingSeekerRegistrationId || !body.outgoingSeekerRegistrationId) {
            throw new Error(programRegistrationMessages.INVALID_SWAP_REQUEST);
          } else {
            const outGoingSeekerRegistration = await this.service.findOne(
              body.outgoingSeekerRegistrationId,
            );
            movingToSubProgramId = outGoingSeekerRegistration.allocatedProgram?.id;
          }
        }
        existingSwapRequest = await this.service.requestSwap({
          programRegistrationId: body.movingSeekerRegistrationId,
          targetPrograms: [{ id: movingToSubProgramId }],
          type: SwapType.DirectSwap,
          comment: body.comment || '',
        }, userId);
      };
      // 2. Perform the swap using the created swap request's ID
      await this.service.updateSwapRequestById(existingSwapRequest.id, userId, {
        ...body,
        status: SwapRequestStatus.ACCEPTED,
      },
        body.swappingType
      );
      await this.responseService.success(
        res,
        programRegistrationMessages.DIRECT_SWAP_SUCCESS,
        existingSwapRequest,
      );
      this.logger.log(programRegistrationMessages.DIRECT_SWAP_SUCCESS, {
        swapRequestId: existingSwapRequest.id,
      });
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Put('swap/:id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.UPDATE_SWAP_REQUEST })
  @ApiParam({ name: 'id', type: Number })
  @ApiBody({ type: MakeSwapRequestDto })
  @ApiResponse({ status: HttpStatus.OK, description: programRegistrationMessages.SWAP_REQUEST_UPDATED })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  async updateSwapRequest(
    @Param('id', ParseIntPipe) id: number,
    @Body() body: MakeSwapRequestDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(programRegistrationMessages.UPDATE_SWAP_REQUEST, { id, body });
    try {
      const userId = req.user.id;
      await this.service.updateSwapRequestById(id, userId, body, body.swappingType);
      await this.responseService.success(res, programRegistrationMessages.SWAP_REQUEST_UPDATED);
      this.logger.log(programRegistrationMessages.SWAP_REQUEST_UPDATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('swap/:programId')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programRegistrationMessages.GET_SWAP_REQUESTS })
  @ApiParam({ name: 'programId', type: Number, description: programRegistrationMessages.PROGRAM_ID })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'filters', type: GetSwapRequestsDto, required: false, description: programRegistrationMessages.SWAP_REQUEST_FILTERS })
  @ApiResponse({ status: HttpStatus.OK, description: programRegistrationMessages.SWAP_REQUESTS_RETRIEVED })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async getSwapRequests(
    @Query('limit') limit = 10,
    @Query('offset') offset = 0,
    @Param('programId', ParseIntPipe) programId: number,
    @Res() res: Response,
    @Query('filters') filters?: GetSwapRequestsDto,
  ) {
    this.logger.log(programRegistrationMessages.GET_SWAP_REQUESTS);
    try {
      const data = await this.service.getSwapRequestKPI({ programId, filters: filters, limit, offset });
      await this.responseService.success(res, programRegistrationMessages.SWAP_REQUESTS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
