import { Type } from 'class-transformer';
import {
  ArrayMinSize,
  IsIn,
  IsNotEmpty,
  IsNumber,
  IsString,
  Max,
  Min,
  ValidateNested,
} from 'class-validator';
import { ApiProperty, OmitType } from '@nestjs/swagger';

const RATING_KEYS = ['passion', 'growth', 'infinitheist', 'continuity'];

export class RmRatingDto {
  @ApiProperty({
    description: 'The key for the rating',
    enum: RATING_KEYS,
    example: 'passion',
  })
  @IsNotEmpty()
  @IsString()
  @IsIn(RATING_KEYS)
  ratingKey: string;

  @ApiProperty({
    description: 'The rating value',
    minimum: 0,
    maximum: 5,
    example: 4,
  })
  @IsNotEmpty()
  @IsNumber()
  @Min(0)
  @Max(5)
  rating: number;
}

export class UpdateRmRatingDto extends RmRatingDto {
  @ApiProperty({
    description: 'Unique identifier for the rating',
    example: 1,
  })
  @IsNotEmpty()
  @IsNumber()
  id: number;
}

export class ProgramRegistrationRMRatingDto {
  @ApiProperty({
    description: 'List of ratings',
    type: [RmRatingDto],
    minItems: RATING_KEYS.length,
    example: [
      { ratingKey: 'passion', rating: 5 },
      { ratingKey: 'growth', rating: 4 },
      { ratingKey: 'infinitheist', rating: 3 },
      { ratingKey: 'continuity', rating: 4 },
    ],
  })
  @IsNotEmpty()
  @ValidateNested({ each: true })
  @Type(() => RmRatingDto)
  @ArrayMinSize(RATING_KEYS.length)
  ratings: RmRatingDto[];

  @ApiProperty({
    description: 'Review text',
    example: 'Great program, learned a lot!',
  })
  @IsNotEmpty()
  @IsString()
  review: string;
}

export class UpdateProgramRegistrationRMRatingDto  {
  
  @ApiProperty({
    description: 'List of ratings with IDs for updates',
    type: [UpdateRmRatingDto],
    minItems: RATING_KEYS.length,
    example: [
      { id: 1, ratingKey: 'passion', rating: 5 },
      { id: 2, ratingKey: 'growth', rating: 4 },
      { id: 3, ratingKey: 'infinitheist', rating: 3 },
      { id: 4, ratingKey: 'continuity', rating: 4 },
    ],
  })
  @IsNotEmpty()
  @ValidateNested({ each: true })
  @Type(() => UpdateRmRatingDto)
  @ArrayMinSize(RATING_KEYS.length)
  ratings: UpdateRmRatingDto[];

  @ApiProperty({
    description: 'Review text',
    example: 'Great program, learned a lot!',
  })
  @IsNotEmpty()
  @IsString()
  review: string;
}
