import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { UpdateProgramQuestionDto } from './dto/update-program-question.dto';
import { In, IsNull, Repository } from 'typeorm';
import { InjectRepository } from '@nestjs/typeorm';
import { Injectable } from '@nestjs/common';
import {
  Program,
  ProgramQuestion,
  Question,
  User,
  ProgramSession,
  Option,
  QuestionOptionMap,
  FormSection,
} from 'src/common/entities';
import { AppLoggerService } from 'src/common/services/logger.service';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { programQuestionConstMessages } from 'src/common/constants/strings-constants';
import { CommonDataService } from 'src/common/services/commonData.service';
import { RegistrationLevelEnum } from 'src/common/enum/program.enums';
import { DeleteProgramQuestionDto } from './dto/delete-program-question.dto';

@Injectable()
export class ProgramQuestionRepository {
  constructor(
    @InjectRepository(ProgramQuestion)
    private readonly programQuestionRepo: Repository<ProgramQuestion>,
    @InjectRepository(Program)
    private readonly programRepo: Repository<Program>,
    @InjectRepository(Question)
    private readonly questionRepo: Repository<Question>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    @InjectRepository(ProgramSession)
    private readonly programSessionRepo: Repository<ProgramSession>,
    @InjectRepository(Option)
    private readonly optionRepo: Repository<Option>,
    @InjectRepository(QuestionOptionMap)
    private readonly questionOptionMapRepo: Repository<QuestionOptionMap>,
    private readonly logger: AppLoggerService,
    private readonly commonDataService: CommonDataService,
  ) {}

  // ==================== CREATE OPERATIONS ====================

  async createProgramQuestionMappings(mappings: ProgramQuestion[]): Promise<ProgramQuestion[]> {
    this.logger.log(programQuestionConstMessages.REPO_CREATE_CALLED);
    try {
      const result = await this.programQuestionRepo.save(mappings);
      this.logger.log(programQuestionConstMessages.MAPPINGS_SAVED(result.length));
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_CREATION, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_SAVE_FAILED, error);
    }
  }

  async createFormSection(sectionData: Partial<FormSection>): Promise<FormSection> {
    try {
      const section = this.programQuestionRepo.manager.create(FormSection, sectionData);
      const result = await this.programQuestionRepo.manager.save(FormSection, section);
      this.logger.log(programQuestionConstMessages.SECTION_CREATED(result.id));
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_CREATION, error);
      handleKnownErrors(ERROR_CODES.FORM_SECTION_SAVE_FAILED, error);
    }
  }

  async createCustomQuestion(questionData: Partial<Question>): Promise<Question> {
    try {
      const question = this.programQuestionRepo.manager.create(this.questionRepo.target, questionData);
      const result = await this.programQuestionRepo.manager.save(this.questionRepo.target, question);
      this.logger.log(programQuestionConstMessages.CUSTOM_QUESTION_CREATED(result.id));
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_CREATION, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_CUSTOM_QUESTION_CREATION_FAILED, error);
    }
  }

  async createOption(optionData: Partial<Option>): Promise<Option> {
    try {
      const option = this.programQuestionRepo.manager.create(this.optionRepo.target, optionData);
      const result = await this.programQuestionRepo.manager.save(this.optionRepo.target, option);
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_CREATION, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_OPTION_MAPPING_FAILED, error);
    }
  }

  async createQuestionOptionMapping(mappingData: Partial<QuestionOptionMap>): Promise<QuestionOptionMap> {
    try {
      const mapping = this.programQuestionRepo.manager.create(this.questionOptionMapRepo.target, mappingData);
      const result = await this.programQuestionRepo.manager.save(this.questionOptionMapRepo.target, mapping);
      this.logger.log(programQuestionConstMessages.OPTION_MAPPING_CREATED);
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_CREATION, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_OPTION_MAPPING_FAILED, error);
    }
  }

  // ==================== READ OPERATIONS ====================

  async findProgramById(programId: number): Promise<Program | null> {
    this.logger.log(programQuestionConstMessages.VALIDATING_PROGRAM(programId));
    try {
      return await this.commonDataService.findOneById(this.programRepo, programId);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  async findProgramSessionById(sessionId: number, includeProgram: boolean = true): Promise<ProgramSession | null> {
    this.logger.log(programQuestionConstMessages.VALIDATING_PROGRAM_SESSION(sessionId));
    try {
      const relations = includeProgram ? ['program'] : undefined;
      return await this.commonDataService.findOneById(this.programSessionRepo, sessionId, true, relations);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_FIND_BY_ID_FAILED, error);
    }
  }

  async findUserById(userId: number): Promise<User | null> {
    this.logger.log(programQuestionConstMessages.VALIDATING_USER(userId));
    try {
      return await this.commonDataService.findOneById(this.userRepo, userId, false);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.USER_NOTFOUND, error);
    }
  }

  async findFormSectionById(sectionId: number): Promise<FormSection | null> {
    this.logger.log(programQuestionConstMessages.VALIDATING_FORM_SECTION(sectionId));
    try {
      return await this.programQuestionRepo.manager.findOne(FormSection, {
        where: { id: sectionId, deletedAt: IsNull() },
      });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.FORM_SECTION_FIND_FAILED, error);
    }
  }

  async findPublishedQuestionsByIds(questionIds: number[]): Promise<Question[]> {
    this.logger.log(programQuestionConstMessages.VALIDATING_QUESTIONS(questionIds));
    try {
      return await this.programQuestionRepo.manager.find(this.questionRepo.target, {
        where: {
          id: In(questionIds),
          status: 'published',
          deletedAt: IsNull(),
        },
        take: questionIds.length,
      });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.QUESTION_FIND_BY_ID_FAILED, error);
    }
  }

  async findExistingMappingsInScope(
    programId: number,
    programSessionId: number | null,
    questionIds: number[],
    entityManager?: any,
  ): Promise<ProgramQuestion[]> {
    this.logger.log(programQuestionConstMessages.CHECKING_EXISTING_MAPPINGS);
    try {
      const manager = entityManager || this.programQuestionRepo.manager;
      const where: any = {
        program: { id: programId },
        question: { id: In(questionIds) },
        deletedAt: IsNull(),
      };

      if (programSessionId) {
        where.programSession = { id: programSessionId };
      } else {
        where.programSession = IsNull();
      }

      const existingMappings = await manager.find(ProgramQuestion, {
        where,
        relations: ['question'],
      });

      this.logger.log(programQuestionConstMessages.EXISTING_MAPPINGS_FOUND(existingMappings.length));
      return existingMappings;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_EXISTING_MAPPING_CHECK_FAILED, error);
    }
  }

  async findAllProgramQuestions(
    limit: number,
    offset: number,
    programId: number | null = null,
    programSessionId: number | null = null,
    registrationLevel: RegistrationLevelEnum | null = null,
  ): Promise<{ data: ProgramQuestion[]; total: number }> {
    this.logger.log(programQuestionConstMessages.REPO_FIND_ALL_CALLED);
    try {
      this.logger.log(programQuestionConstMessages.BUILDING_QUERY_CONDITIONS);
      const where: any = { deletedAt: IsNull() };
      
      if (programId) {
        where.program = { id: programId };
      }
      if (programSessionId) {
        where.programSession = { id: programSessionId };
      }
      if (registrationLevel) {
        where.registrationLevel = registrationLevel;
      }

      this.logger.log(programQuestionConstMessages.EXECUTING_FIND_QUERY);
      const data = await this.commonDataService.get(
        this.programQuestionRepo,
        undefined,
        where,
        limit,
        offset,
        { displayOrder: 'ASC', id: 'ASC' },
        undefined,
        ['program', 'question', 'programSession', 'createdBy', 'updatedBy'],
      );

      const total = await this.programQuestionRepo.count({ where });
      this.logger.log(programQuestionConstMessages.QUERY_EXECUTED(data.length));
      
      return { data, total };
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_FIND_ALL_FAILED, error);
    }
  }

  async findProgramQuestionById(id: number): Promise<ProgramQuestion | null> {
    this.logger.log(programQuestionConstMessages.REPO_FIND_BY_ID_CALLED(id));
    try {
      return await this.commonDataService.findOneById(this.programQuestionRepo, id, true, [
        'program',
        'question',
        'programSession',
        'createdBy',
        'updatedBy',
      ]);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_FIND_BY_ID_FAILED, error);
    }
  }

  async findExistingMappingForUpdate(
    programId: number,
    programSessionId: number | null,
    questionId: number,
    excludeId?: number,
  ): Promise<ProgramQuestion | null> {
    try {
      const where: any = {
        program: { id: programId },
        programSession: programSessionId ? { id: programSessionId } : IsNull(),
        question: { id: questionId },
        deletedAt: IsNull(),
      };

      if (excludeId) {
        // Add condition to exclude current mapping being updated
        const mappings = await this.programQuestionRepo.find({ where });
        return mappings.find(m => m.id !== excludeId) || null;
      }

      return await this.programQuestionRepo.findOne({ where });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_EXISTING_MAPPING_CHECK_FAILED, error);
    }
  }

  async findProgramQuestionsInSection(
    programId: number,
    programSessionId: number | null,
    sectionId: number,
  ): Promise<ProgramQuestion[]> {
    try {
      return await this.programQuestionRepo.find({
        where: {
          program: { id: programId },
          programSession: programSessionId ? { id: programSessionId } : IsNull(),
          programQuestionFormSection: { id: sectionId },
          deletedAt: IsNull(),
        },
      });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_FIND_ALL_FAILED, error);
    }
  }

  async findQuestionsInSection(sectionId: number): Promise<Question[]> {
    try {
      return await this.programQuestionRepo.manager.find(this.questionRepo.target, {
        where: {
          formSection: { id: sectionId },
          deletedAt: IsNull(),
        },
      });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.QUESTION_FIND_BY_ID_FAILED, error);
    }
  }

  async findProgramQuestionsToDelete(
    programId: number,
    programSessionId: number | null,
    questionIds: number[],
  ): Promise<ProgramQuestion[]> {
    try {
      return await this.programQuestionRepo.find({
        where: {
          program: { id: programId },
          programSession: programSessionId ? { id: programSessionId } : IsNull(),
          question: { id: In(questionIds) },
          deletedAt: IsNull(),
        },
      });
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_FETCH, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_FIND_ALL_FAILED, error);
    }
  }

  // ==================== UPDATE OPERATIONS ====================

  async updateFormSection(section: FormSection): Promise<FormSection> {
    try {
      const result = await this.programQuestionRepo.manager.save(FormSection, section);
      this.logger.log(programQuestionConstMessages.SECTION_UPDATED(result.id));
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_UPDATE, error);
      handleKnownErrors(ERROR_CODES.FORM_SECTION_UPDATE_FAILED, error);
    }
  }

  async updateProgramQuestion(mapping: ProgramQuestion): Promise<ProgramQuestion> {
    this.logger.log(programQuestionConstMessages.REPO_UPDATE_CALLED(mapping.id));
    try {
      return await this.commonDataService.save(this.programQuestionRepo, mapping);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_UPDATE, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_UPDATE_FAILED, error);
    }
  }

  async updateQuestion(question: Question): Promise<Question> {
    try {
      return await this.programQuestionRepo.manager.save(this.questionRepo.target, question);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_UPDATE, error);
      handleKnownErrors(ERROR_CODES.QUESTION_SAVE_FAILED, error);
    }
  }

  async updateProgramQuestions(mappings: ProgramQuestion[]): Promise<ProgramQuestion[]> {
    try {
      return await this.programQuestionRepo.save(mappings);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_UPDATE, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_UPDATE_FAILED, error);
    }
  }

  // ==================== DELETE OPERATIONS ====================

  async softDeleteProgramQuestion(mapping: ProgramQuestion): Promise<ProgramQuestion> {
    this.logger.log(programQuestionConstMessages.REPO_DELETE_CALLED(mapping.id));
    try {
      mapping.deletedAt = new Date();
      return await this.commonDataService.save(this.programQuestionRepo, mapping);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_DELETE, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_DELETE_FAILED, error);
    }
  }

  async softDeleteFormSection(section: FormSection): Promise<FormSection> {
    try {
      section.deletedAt = new Date();
      const result = await this.programQuestionRepo.manager.save(FormSection, section);
      this.logger.log(programQuestionConstMessages.SECTION_DELETED(result.id, 0));
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_DELETE, error);
      handleKnownErrors(ERROR_CODES.FORM_SECTION_DELETE_FAILED, error);
    }
  }

  async softDeleteQuestions(questions: Question[]): Promise<Question[]> {
    try {
      const updatedQuestions = questions.map(q => {
        q.deletedAt = new Date();
        return q;
      });
      return await this.programQuestionRepo.manager.save(this.questionRepo.target, updatedQuestions);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_DELETE, error);
      handleKnownErrors(ERROR_CODES.QUESTION_DELETE_FAILED, error);
    }
  }

  async softDeleteProgramQuestions(mappings: ProgramQuestion[]): Promise<ProgramQuestion[]> {
    try {
      const updatedMappings = mappings.map(m => {
        m.deletedAt = new Date();
        return m;
      });
      return await this.programQuestionRepo.save(updatedMappings);
    } catch (error) {
      this.logger.error(programQuestionConstMessages.ERROR_CONTEXT_DELETE, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_DELETE_FAILED, error);
    }
  }

  // ==================== TRANSACTION OPERATIONS ====================

  async executeInTransaction<T>(operation: (entityManager: any) => Promise<T>): Promise<T> {
    this.logger.log(programQuestionConstMessages.TRANSACTION_STARTED);
    try {
      const result = await this.programQuestionRepo.manager.transaction(operation);
      this.logger.log(programQuestionConstMessages.TRANSACTION_COMMITTED);
      return result;
    } catch (error) {
      this.logger.error(programQuestionConstMessages.TRANSACTION_FAILED, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_QUESTION_TRANSACTION_FAILED, error);
    }
  }

  // ==================== UTILITY OPERATIONS ====================

  async calculatePagination(total: number, limit: number, offset: number): Promise<{
    totalPages: number;
    pageNumber: number;
    pageSize: number;
    totalRecords: number;
    numberOfRecords: number;
  }> {
    this.logger.log(programQuestionConstMessages.CALCULATING_PAGINATION);
    const totalPages = Math.ceil(total / limit);
    const pageNumber = Math.floor(offset / limit) + 1;
    
    this.logger.log(programQuestionConstMessages.PAGINATION_CALCULATED(totalPages, pageNumber));
    
    return {
      totalPages,
      pageNumber,
      pageSize: +limit,
      totalRecords: total,
      numberOfRecords: Math.min(limit, total - offset),
    };
  }
}