import { Injectable } from '@nestjs/common';
import { InitiatePaymentDto } from './dto/create-payment.dto';
import { UpdatePaymentDto } from './dto/update-payment.dto';
import { PaymentRepository } from './payment.repository';
import { InvoiceService } from 'src/invoice/invoice.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { RegistrationService } from 'src/registration/registration.service';
import { CommunicationService } from 'src/communication/communication.service';
import { zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { areGSTsFromSameState, formatDate } from 'src/common/utils/common.util';
import { CommonDataService } from 'src/common/services/commonData.service';
import { Program, ProgramRegistration, ProgramType } from 'src/common/entities';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import { TdsApplicabilityEnum } from 'src/common/enum/tds-applicability.enum';
import { UserRepository } from 'src/user/user.repository';

@Injectable()
export class PaymentService {
  constructor(
    private readonly paymentRepository: PaymentRepository,
    private readonly invoiceService: InvoiceService,
    private readonly logger: AppLoggerService,
    private readonly registrationService: RegistrationService,
    private readonly communicationService: CommunicationService,
    private readonly commonDataService: CommonDataService,
    private readonly registrationRepository: ProgramRegistrationRepository,
    private readonly userRepository: UserRepository,
  ) {}

  async initiatePayment(registrationId: number, dto: InitiatePaymentDto, userId: number) {
    this.logger.log('Initiating payment for registration', {
      registrationId,
      paymentMode: dto.paymentMode,
    });

    try {
      // Check if payment already exists for this registration
      const existingPayment = await this.paymentRepository.findByRegistrationId(registrationId);
      // if (existingPayment && existingPayment.paymentStatus !== PaymentStatusEnum.DRAFT) {
      //   throw new InifniBadRequestException(
      //     ERROR_CODES.PAYMENT_ALREADY_EXISTS,
      //     null,
      //     null,
      //     `Payment already exists for registration ${registrationId} with status ${existingPayment.paymentStatus}`
      //   );
      // }

      // Get registration details and program/session information
      const registrationDetails =
        await this.paymentRepository.getRegistrationWithProgramDetails(registrationId);
      if (!registrationDetails) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          `Registration ${registrationId} not found`,
        );
      }

      // Calculate payment amounts from program/session
      const paymentCalculation = await this.calculatePaymentAmounts(registrationDetails, dto.gstNumber ?? null, dto.tanNumber ? dto.tdsAmount ?? 0 : 0);

      // Check seat availability and waitlist status
      let seatAvailability;
      if (!registrationDetails.program.requiresApproval) {
        seatAvailability = await this.registrationService.checkSeatAvailability(
          registrationDetails.program,
          registrationDetails.programSession,
          registrationDetails.program.type.registrationLevel,
        );
        
        console.log('Seat availability', seatAvailability);
      } else {
        // For programs requiring approval, set default seat availability
        // This ensures waitlisting logic doesn't interfere with approval process
        seatAvailability = {
          totalSeats: registrationDetails.program.totalSeats,
          waitlistTriggerCount: registrationDetails.program.waitlistTriggerCount,
          shouldWaitlist: false,
          isWaitlistTriggered: false,
          canRegister: true,
        };
        console.log('Program requires approval - skipping seat availability check');
      }

      let result;
      console.log('Payment mode', dto.paymentMode);
      console.log('Payment Mode Enum', PaymentModeEnum.ONLINE);
      if (dto.paymentMode === PaymentModeEnum.ONLINE) {
        console.log('Initiating online payment');
        result = await this.initiateOnlinePayment(
          registrationId,
          dto,
          userId,
          paymentCalculation,
          seatAvailability,
          registrationDetails,
        );
      } else {
        result = await this.initiateOfflinePayment(
          registrationId,
          dto,
          userId,
          paymentCalculation,
          seatAvailability,
          registrationDetails,
        );
      }

      this.logger.log('Payment initiated successfully', {
        registrationId,
        paymentId: result.paymentId,
      });
      return result;
    } catch (error) {
      this.logger.error('Payment initiation failed', 'failed payment', {
        registrationId,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.PAYMENT_INITIATION_FAILED, error);
    }
  }

  async updatePayment(registrationId: number, dto: UpdatePaymentDto, userId: number) {
    this.logger.log('Updating payment for registration', {
      registrationId,
      newStatus: dto.paymentStatus,
    });

    try {
      // Get current payment details first
      const currentPayment = await this.paymentRepository.findPaymentDetailsByRegistrationId(registrationId);
      console.log('Current payment details', currentPayment);
      if (!currentPayment) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_NOTFOUND,
          null,
          null,
          `No payment found for registration ${registrationId}`
        );
      }

      // Check if payment is already completed
      if (currentPayment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || 
          currentPayment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_ALREADY_COMPLETED,
          null,
          null,
          registrationId.toString(),
          currentPayment.paymentStatus
        );
      }

      const result = await this.paymentRepository.updatePaymentStatus(registrationId, dto, userId);

      // Handle registration status updates based on payment status
      if (dto.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || 
          dto.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {
        await this.handlePaymentConfirmation(registrationId, userId);
      }

      this.logger.log('Payment updated successfully', {
        registrationId,
        paymentId: result.paymentId,
      });
      // if (result) {
      //   const paymentDetails = await this.getPaymentDetails(registrationId);
      //   const registrationDetails = await this.registrationRepository.findOne(registrationId);
      //   const invoiceDetails = paymentDetails.invoice;
       
      //   const email = paymentDetails.invoice?.invoiceEmail;
      //   if (email) {
      //      // Prepare merge info for email
      //     const mergeInfo = {
      //       hdb_msd_no: "HDB",
      //       hdb_msd_date: formatDate(registrationDetails.program.checkinAt.toISOString()),
      //       hdb_msd_amount: paymentDetails.payment?.subTotal,
      //       reg_name: paymentDetails.invoice?.invoiceName || 'Infinitheist',
      //       hdb_msd: "HDB",
      //       last_date:  formatDate(registrationDetails.program.registrationEndsAt.toISOString())
      //     }
      //     const emailData = {
      //       templateKey: process.env.ZEPTO_INVOICE_EMAIL_TEMPLATE_ID,
      //       from: {
      //         address: zeptoEmailCreadentials.ZEPTO_EMAIL,
      //         name: zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
      //       },
      //       to: {
      //         emailAddress: email,
      //         name: invoiceDetails?.invoiceName || 'Infinitheist',
      //       },
      //       mergeInfo: mergeInfo,
      //       attachments: [],
      //       subject: 'Payment Confirmation - Infinitheism',
      //     }
      //     await this.communicationService.sendSingleEmail(emailData);
      //   } else {
      //     this.logger.warn('No invoice email provided for payment confirmation', {
      //       registrationId,
      //       invoiceEmail: invoiceDetails?.invoiceEmail,
      //     });

      //   }
        
      // }

      return result;
    } catch (error) {
      this.logger.error('Payment update failed', 'failed payment', {
        registrationId,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.PAYMENT_UPDATE_FAILED, error);
    }
  }

  async getPaymentDetails(registrationId: number) {
    this.logger.log('Retrieving payment details', { registrationId });

    try {
      const paymentDetails =
        await this.paymentRepository.getPaymentDetailsWithInvoice(registrationId);
      if (!paymentDetails) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_NOTFOUND,
          null,
          null,
          `No payment found for registration ${registrationId}`,
        );
      }

      return paymentDetails;
    } catch (error) {
      this.logger.error('Failed to retrieve payment details', '', {
        registrationId,
        error: error.message,
      });
      handleKnownErrors(ERROR_CODES.PAYMENT_GET_FAILED, error);
    }
  }

  async handleWebhook(webhookData: any, headers: any) {
    this.logger.log('Processing payment webhook', { event: webhookData.event });

    try {
      // Verify webhook signature (implement based on your payment gateway)
      // const isValidSignature = await this.verifyWebhookSignature(webhookData, headers);
      // if (!isValidSignature) {
      //   throw new InifniBadRequestException(
      //     ERROR_CODES.INVALID_WEBHOOK_SIGNATURE,
      //     null,
      //     null,
      //     'Invalid webhook signature'
      //   );
      // }

      // Process payment status update
      const result = await this.paymentRepository.processWebhookUpdate(webhookData);

      // Handle additional processing if payment is successful
      if (
        (webhookData.event === 'payment.captured' || webhookData.event === 'payment.authorized') &&
        result.registrationId
      ) {
        await this.handlePaymentConfirmation(result.registrationId, null);
      }

      return result;
    } catch (error) {
      this.logger.error('Webhook processing failed', '', { error: error.message });
      handleKnownErrors(ERROR_CODES.WEBHOOK_PROCESSING_FAILED, error);
    }
  }

  private async initiateOnlinePayment(
    registrationId: number,
    dto: InitiatePaymentDto,
    userId: number,
    paymentCalculation: any,
    seatAvailability: any,
    registrationDetails: any,
  ) {
    // Create Razorpay order

    console.log('Creating Razorpay order for registration', {
      registrationId,
      amount: paymentCalculation.finalAmount,
    });
    const finalAmount: number = parseFloat(paymentCalculation.finalAmount.toFixed(2)); // Ensure only 2 floating numbers
    console.log('Final amount for Razorpay order', finalAmount);
    const razorpayOrder = await this.createRazorpayOrder(finalAmount, registrationId);
    console.log('Razorpay order created', razorpayOrder);
    // Determine if should be waitlisted
    const shouldWaitlist = this.shouldWaitlist(seatAvailability, registrationDetails.program);
    console.log('Should waitlist:', shouldWaitlist);
    const program = registrationDetails.program;
    console.log('Program details', program);
    // Create payment and invoice records
    const result = await this.paymentRepository.createPaymentWithInvoice({
      registrationId,
      paymentMeta: dto.paymentMeta,
      paymentMode: PaymentModeEnum.ONLINE,
      paymentStatus: PaymentStatusEnum.ONLINE_PENDING,
      razorpayOrderId: razorpayOrder.id,
      amounts: paymentCalculation,
      billingDetails: {
        billingName: dto.invoiceName,
        billingAddress: dto.invoiceAddress,
        panNumber: dto.panNumber,
        tanNumber: dto.tanNumber,
        isGstRegistered: dto.gstNumber ? true : false,
        invoiceEmail: dto.invoiceEmail,
        gstNumber: dto.gstNumber,
        zip: dto.zip,
        tdsAmount: dto.tdsAmount,
      },
      userId,
      shouldWaitlist,
      seatAvailability,
      program,
    });
    console.log('Payment and invoice created', result);
    return {
      ...result,
      razorpayOrderId: razorpayOrder.id,
      amount: razorpayOrder.amount,
      currency: razorpayOrder.currency,
      waitlisted: shouldWaitlist,
    };
  }

  private async initiateOfflinePayment(
    registrationId: number,
    dto: InitiatePaymentDto,
    userId: number,
    paymentCalculation: any,
    seatAvailability: any,
    registrationDetails: any,
  ) {
    console.log('Initiating offline payment for registration')
    // Determine if should be waitlisted
    const shouldWaitlist = this.shouldWaitlist(seatAvailability, registrationDetails.program);
    console.log('Should waitlist:', shouldWaitlist);
    const program = registrationDetails.program;
    console.log('Program details', program);
    // Create payment and invoice records
    const result = await this.paymentRepository.createPaymentWithInvoice({
      registrationId,
      paymentMeta: dto.paymentMeta,
      paymentMode: PaymentModeEnum.OFFLINE,
      paymentStatus: PaymentStatusEnum.OFFLINE_PENDING,
      offlinePaymentMeta: dto.offlinePaymentMeta,
      amounts: paymentCalculation,
      billingDetails: {
        billingName: dto.invoiceName,
        billingAddress: dto.invoiceAddress,
        panNumber: dto.panNumber,
        tanNumber: dto.tanNumber,
        isGstRegistered: dto.gstNumber ? true : false,
        invoiceEmail: dto.invoiceEmail,
        gstNumber: dto.gstNumber,
        zip: dto.zip,
        tdsAmount: dto.tdsAmount,
      },
      userId,
      shouldWaitlist,
      seatAvailability,
      program,

    });
    console.log('Payment and invoice created', result);
    if (result) {
      const mergeInfo = {
          pay_date: formatDate(new Date().toISOString()),
          pay_amount: paymentCalculation.totalAmount,
          reg_name: dto.invoiceName,
          pay_method: dto.paymentMode === PaymentModeEnum.ONLINE ? "Online" : dto.offlinePaymentMeta?.paymentMethod || "Offline",
          hdb_msd: "HDB"
      }
      console.log('Payment details for email', dto);
      // let bccEmails: { emailAddress: string; name: string }[] = [];
      // if (process.env.HDB_BCC_EMAIL) {
      //   bccEmails.push({ emailAddress: process.env.HDB_BCC_EMAIL, name: process.env.HDB_BCC_EMAIL_NAME || 'HDB' });
      // }
      // try {
      //   const financeUsers = await this.userRepository.getUsersByRoleKey('ROLE_FINANCE_MANAGER');
      //   if (financeUsers && financeUsers.length > 0) {
      //     bccEmails = bccEmails.concat(
      //       financeUsers.map((user) => ({
      //         emailAddress: user.email,
      //         name: user.fullName || 'Finance Manager',
      //       }))
      //     );
      //   }
      // } catch (err) {
      //   this.logger.error('Failed to fetch finance managers for BCC', err);
      // }
      const emailData: SendSingleEmailDto = {
        templateKey: process.env.ZEPTO_PAYMENT_AKNOWLEDGEMENT_OFFLINE_EMAIL_TEMPLATE_ID,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name: registrationDetails.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        to: {
          emailAddress: dto.invoiceEmail,
          name: dto.invoiceName,
        },
        // bcc: bccEmails,
        mergeInfo: mergeInfo,
        attachments: [],
        subject: 'Payment Confirmation - Infinitheism',
        trackinfo: {
          registrationId: registrationId,
          createdBy: userId,
          updatedBy: userId,
        },
      };
      this.logger.log('Sending payment acknowledgement email', emailData);

      await this.communicationService.sendSingleEmail(emailData);
      this.logger.log('Payment acknowledgement email sent successfully', {
        email: dto.invoiceEmail,
        name: dto.invoiceName,
      });

      // Send WhatsApp message for payment acknowledgement
      const messageData: SendTemplateMessageDto = {
        whatsappNumber: registrationDetails.mobileNumber, // Replace with the appropriate value
        templateName: process.env.WATI_PAYMENT_AKNOWLEDGEMENT_OFFLINE_TEMPLATE_ID || '', // Replace with the correct template name
        broadcastName: process.env.WATI_PAYMENT_AKNOWLEDGEMENT_OFFLINE_TEMPLATE_ID || '', // Replace with the correct broadcast name
        parameters: [
          { name: 'reg_name', value: dto.invoiceName },
        ],
        trackinfo: {
          registrationId: registrationId,
          // createdBy: userId,
          // updatedBy: userId,
        },
      };
    
      await this.communicationService.sendTemplateMessage(messageData);
      this.logger.log('WhatsApp payment acknowledgement message sent', {
        whatsappNumber: registrationDetails.mobileNumber,
      });
    }
    console.log('Payment and invoice created', result);
    return {
      ...result,
      waitlisted: shouldWaitlist,
    };
  }

  private async calculatePaymentAmounts(registrationDetails: any, gstNumber: string | null, totalTdsAmount: number) {
    const program = registrationDetails.program;
    const session = registrationDetails.programSession;
    const allocatedProgram = registrationDetails.allocatedProgram;
    const allocatedProgramSession = registrationDetails.allocatedProgramSession;
    // const registrationLevel = program.type.registrationLevel;

    // Get base amount from program or session
    const baseAmount: number =
      // registrationLevel === 'session'
      //   ? Number(session?.fee) || Number(program.basePrice)
      //   : 
        Number(allocatedProgram.basePrice);

    if (!baseAmount) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_FEE_NOTFOUND,
        null,
        null,
        'Program fee is not defined',
      );
    }

    let totalGstRate = 0;
    let cgstRate = 0;
    let sgstRate = 0;
    let igstRate = 0;

    // Calculate GST
    if (gstNumber) {
      if (areGSTsFromSameState(
        gstNumber,
        program.gstNumber,
      )) {
        // If GST numbers are from the same state, apply CGST and SGST
        cgstRate = Number(program.cgst) ?? 0;
        sgstRate = Number(program.sgst) ?? 0;
        totalGstRate = cgstRate + sgstRate;
      } else {
        // If GST numbers are from different states, apply IGST
        igstRate = Number(program.igst) ?? 0;
        totalGstRate = igstRate;
      }
    } else {
      cgstRate = Number(program.cgst) ?? 0;
      sgstRate = Number(program.sgst) ?? 0;
      totalGstRate = cgstRate + sgstRate;
    }

    const tdsPercent = Number(program.tdsPercent) ?? 0;
    const gstAmount = (baseAmount * totalGstRate) / 100;
    const tdsAmountOnBaseAmount = (baseAmount * tdsPercent) / 100;
    const tdsAmountOnGst = (gstAmount * tdsPercent) / 100;

    if (totalTdsAmount < 0) {
      throw new InifniBadRequestException(
        ERROR_CODES.TDS_AMOUNT_INVALID,
        null,
        null,
        'TDS amount cannot be negative',
      );
    } else {
      if (program.tdsApplicability === TdsApplicabilityEnum.BASE_ONLY) {
        console.log('TDS Applicability: BASE_ONLY',  program.tdsApplicability);
        if (totalTdsAmount > tdsAmountOnBaseAmount) {
          throw new InifniBadRequestException(
            ERROR_CODES.TDS_AMOUNT_EXCEEDS,
            null,
            null,
            `TDS amount cannot exceed ${tdsAmountOnBaseAmount}`,
          );
        } else {
          totalTdsAmount = totalTdsAmount;
        }
      } else if (program.tdsApplicability === TdsApplicabilityEnum.BASE_PLUS_TAX) {
        if (totalTdsAmount > (tdsAmountOnGst + tdsAmountOnBaseAmount)) {
          throw new InifniBadRequestException(
            ERROR_CODES.TDS_AMOUNT_EXCEEDS,
            null,
            null,
            `TDS amount cannot exceed ${tdsAmountOnGst}`,
          );
        } else {
          totalTdsAmount = totalTdsAmount;
        }
      }
    }

    // Calculate total
    const totalAmount = baseAmount + gstAmount - totalTdsAmount;

    console.log('Payment calculation', {
      baseAmount,
      gstAmount,
      totalAmount,
      cgstRate,
      sgstRate,
      igstRate,
      tdsPercent,
      totalTdsAmount,
      gstNumber,
      programGstNumber: program.gstNumber,
    });

    return {
      baseAmount,
      gstAmount,
      totalAmount,
      finalAmount: totalAmount,
      cgstRate,
      sgstRate,
      igstRate,
      tdsPercent,
      totalTdsAmount,
    };
  }

  private shouldWaitlist(seatAvailability: any, program: any): boolean {
    // Logic to determine if registration should be waitlisted
    if (program.requiresApproval) {
      return false; // No waitlisting for programs requiring approval
    }
    const { filledSeats, waitlistTriggerCount, isWaitlistTriggered } = seatAvailability;

    return filledSeats >= waitlistTriggerCount || isWaitlistTriggered;
  }

  private async createRazorpayOrder(amount: number, registrationId: number) {
    // Implement Razorpay order creation
    // This is a placeholder - implement based on your Razorpay integration
    try{
    console.log('Creating Razorpay order', { amount, registrationId });
    const integerAmount = Math.round(amount * 100) / 100;
    const Razorpay = require('razorpay');
    const razorpay = new Razorpay({
      key_id: process.env.RAZORPAY_KEY_ID,
      key_secret: process.env.RAZORPAY_SECRET_KEY,
    });
    const receiptId = `receipt_${registrationId}_${Date.now()}`;
    const options = {
      amount: integerAmount * 100, // Convert to paise
      currency: 'INR',
      receipt: receiptId,
    };
    const order = await razorpay.orders.create(options);
    console.log('Razorpay order created', order);
    return {
      id: order.id,
      amount: integerAmount * 100, // Convert to paise
      currency: 'INR',
      receipt: receiptId,
    };
  }
    catch (error) {
      console.log('Razorpay order creation failed', error);
      handleKnownErrors('ERROR_CODES.RAZORPAY_ORDER_CREATION_FAILED', error);
    }
  }

  private async verifyWebhookSignature(webhookData: any, headers: any): Promise<boolean> {
    // Implement webhook signature verification
    // This is a placeholder - implement based on your payment gateway
    return true;
  }

  private async handlePaymentConfirmation(registrationId: number, userId: number | null) {
    // Handle post-payment processing
    try {
      // Update registration status
      await this.paymentRepository.updateRegistrationAfterPayment(registrationId, userId);

      // Generate and send invoice
      await this.invoiceService.generateAndSendInvoice(registrationId);


      // Handle seat count updates
      // await this.paymentRepository.updateSeatCountsAfterPayment(registrationId);

      this.logger.log('Payment confirmation processed', { registrationId });
    } catch (error) {
      this.logger.error('Payment confirmation processing failed', '', {
        registrationId,
        error: error.message,
      });
      // Don't throw here - payment was successful, just log the issue
      handleKnownErrors(ERROR_CODES.PAYMENT_CONFIRMATION_FAILED, error);
    }
  }
}
