import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource, IsNull, Not } from 'typeorm';
import { 
  RegistrationPaymentDetail, 
  RegistrationPaymentDetailsHistory,
  RegistrationInvoiceDetail,
  ProgramRegistration,
  Program,
  ProgramSession,
} from 'src/common/entities';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { UpdatePaymentDto } from './dto/update-payment.dto';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { InvoiceTypeEnum } from 'src/common/enum/invoice-type.enum';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { incrementSeatCounts } from '../common/utils/seat-count.util';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { RegistrationService } from 'src/registration/registration.service';
import { generateNextSequence } from 'src/common/utils/common.util';

@Injectable()
export class PaymentRepository {
  constructor(
    @InjectRepository(RegistrationPaymentDetail)
    private readonly paymentRepo: Repository<RegistrationPaymentDetail>,
    @InjectRepository(RegistrationPaymentDetailsHistory)
    private readonly paymentHistoryRepo: Repository<RegistrationPaymentDetailsHistory>,
    @InjectRepository(RegistrationInvoiceDetail)
    private readonly invoiceRepo: Repository<RegistrationInvoiceDetail>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    @InjectRepository(Program)
    private readonly programRepo: Repository<Program>,
    @InjectRepository(ProgramSession)
    private readonly sessionRepo: Repository<ProgramSession>,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
    private readonly registartionService: RegistrationService,
  ) {}

  async findByRegistrationId(registrationId: number): Promise<RegistrationPaymentDetail | null> {
    try {
      return await this.paymentRepo.findOne({
        where: { registrationId },
        relations: ['registration'],
      });
    } catch (error) {
      this.logger.error('Error finding payment by registration ID', '', { registrationId, error });
      return null;
    }
  }

  async findPaymentDetailsByRegistrationId(registrationId: number): Promise<RegistrationPaymentDetail | null> {
    try {
      return await this.paymentRepo.findOne({
        where: { registrationId },
      });
    } catch (error) {
      this.logger.error('Error finding payment details by registration ID', '', { registrationId, error });
      return null;
    }
  }

  async getRegistrationWithProgramDetails(registrationId: number) {
    try {
      return await this.registrationRepo.findOne({
        where: { id: registrationId },
        relations: [
          'program', 
          'program.type', 
          'programSession',
          'user',
          'allocatedProgram',
          'allocatedSession',
        ],
      });
    } catch (error) {
      this.logger.error('Error getting registration details', '', { registrationId, error });
      throw error;
    }
  }

  /**
   * Create payment history record
   * This method captures the state of payment for audit trail
   */
  private async createPaymentHistory(
    queryRunner: any,
    paymentId: number,
    payment: RegistrationPaymentDetail,
    userId: number | null,
    action: string = 'UPDATE'
  ) {
    try {
      const historyRecord = queryRunner.manager.create(RegistrationPaymentDetailsHistory, {
        paymentDetailsId: paymentId,
        paymentMode: payment.paymentMode,
        razorpayId: payment.razorpayId,
        offlineMeta: payment.offlineMeta,
        paymentStatus: payment.paymentStatus,
        createdBy: userId,
        updatedBy: userId,
      });

      await queryRunner.manager.save(RegistrationPaymentDetailsHistory, historyRecord);

      this.logger.log('Payment history record created', {
        paymentId,
        paymentStatus: payment.paymentStatus,
        action,
        historyId: historyRecord.id
      });

      return historyRecord;
    } catch (error) {
      queryRunner.rollbackTransaction();
      this.logger.error('Failed to create payment history record', error);
      handleKnownErrors(ERROR_CODES.PAYMENT_INITIATION_FAILED, error);
      // Don't throw here - history is important but shouldn't break the main flow
    }
  }

  /**
   * Get payment history for a specific payment
   */
  async getPaymentHistory(paymentId: number): Promise<RegistrationPaymentDetailsHistory[]> {
    try {
      return await this.paymentHistoryRepo.find({
        where: { paymentDetailsId: paymentId },
        order: { createdAt: 'DESC' },
        relations: ['createdByUser', 'updatedByUser'],
      });
    } catch (error) {
      this.logger.error('Error getting payment history', '', { paymentId, error });
      return [];
    }
  }

  /**
   * Get payment history for a registration
   */
  async getPaymentHistoryByRegistration(registrationId: number): Promise<RegistrationPaymentDetailsHistory[]> {
    try {
      const payment = await this.findByRegistrationId(registrationId);
      if (!payment) {
        return [];
      }
      return await this.getPaymentHistory(payment.id);
    } catch (error) {
      this.logger.error('Error getting payment history by registration', '', { registrationId, error });
      return [];
    }
  }

  async createPaymentWithInvoice(paymentData: any) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      // Start transaction following payment module pattern
      await queryRunner.startTransaction();
      transactionStarted = true;
      
      this.logger.log('Creating payment and invoice', { 
        registrationId: paymentData.registrationId,
        paymentMode: paymentData.paymentMode 
      });

      // Store question answers using registration service logic
      if (paymentData.paymentMeta && paymentData.paymentMeta.length) {
        await this.registartionService.storeQuestionAnswers(
          queryRunner,
          paymentData.registrationId,
          paymentData.paymentMeta,
          paymentData.userId,
        );
      }
      // Find existing payment
      const existedPayment = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
        where: { registrationId: paymentData.registrationId }
        // relations: ['registration'],
      });

      // Create payment record
      const payment = queryRunner.manager.create(RegistrationPaymentDetail, {
        id: existedPayment ? existedPayment.id : undefined,
        registrationId: paymentData.registrationId,
        programSessionId: paymentData.programSessionId || null,
        paymentMode: paymentData.paymentMode,
        paymentStatus: paymentData.paymentStatus,
        originalAmount: paymentData.amounts.baseAmount,
        gstAmount: paymentData.amounts.gstAmount,
        tds: paymentData.billingDetails.tanNumber ? paymentData.billingDetails.tdsAmount || 0 : 0,
        taxAmount: paymentData.amounts.gstAmount,
        subTotal: paymentData.amounts.totalAmount,
        razorpayId: paymentData.razorpayOrderId || null,
        offlineMeta: paymentData.offlinePaymentMeta || null,
        createdBy: { id: paymentData.userId } as any,
        updatedBy: { id: paymentData.userId } as any,
      });

      const savedPayment = await queryRunner.manager.save(RegistrationPaymentDetail, payment);
      // create payment history record for this payment
      await this.createPaymentHistory(
        queryRunner,
        savedPayment.id,
        savedPayment,
        paymentData.userId,
        'CREATE'
      );

      // Check if invoice already exists for this registration
      const existedInvoice = await queryRunner.manager.findOne(RegistrationInvoiceDetail, {
        where: { registrationId: paymentData.registrationId }
        // relations: ['registration'],
      });

      // Create invoice record
      const invoice = await queryRunner.manager.create(RegistrationInvoiceDetail, {
        id: existedInvoice?.id ? existedInvoice.id : undefined,
        registrationId: paymentData.registrationId,
        invoiceType: paymentData.paymentMode === PaymentModeEnum.ONLINE 
          ? InvoiceTypeEnum.ONLINE 
          : InvoiceTypeEnum.OFFLINE,
        invoiceName: paymentData.billingDetails.billingName,
        invoiceEmail: paymentData.billingDetails.invoiceEmail,
        invoiceAddress: paymentData.billingDetails.billingAddress,
        panNumber: paymentData.billingDetails.panNumber || null,
        tdsApplicable: paymentData.billingDetails.tanNumber ? true : false,
        tanNumber: paymentData.billingDetails.tanNumber || null,
        tdsAmount: paymentData.billingDetails.tanNumber ? paymentData.billingDetails.tdsAmount || 0 : 0,
        isGstRegistered: paymentData.billingDetails.gstNumber ? true : false,
        taxAmount: paymentData.amounts.gstAmount,
        invoiceStatus: InvoiceStatusEnum.DRAFT,
        gstNumber: paymentData.billingDetails.gstNumber || null,
        zip: paymentData.billingDetails.zip || null,
        razorpayId: paymentData.razorpayOrderId || null,
        createdBy: { id: paymentData.userId } as any,
        updatedBy: { id: paymentData.userId } as any,
      });

      const savedInvoice = await queryRunner.manager.save(RegistrationInvoiceDetail, invoice);

      // Update registration status based on waitlist and payment requirements
      let newRegistrationStatus = RegistrationStatusEnum.PENDING;
      
      if (paymentData.shouldWaitlist) {
        newRegistrationStatus = RegistrationStatusEnum.WAITLISTED;
        
        // Generate waitlist sequence number
        const waitlistQuery: any = { waitingListSeqNumber: Not(IsNull()) };
        if (paymentData.registrationLevel === 'program') {
          waitlistQuery.program = { id: paymentData.program.programId };
        } else {
          waitlistQuery.programSession = { id: paymentData.programSessionId };
        }
        
        const lastWaitlistRegistration = await queryRunner.manager.findOne(ProgramRegistration, {
          where: waitlistQuery,
          order: { waitingListSeqNumber: 'DESC' },
        });
        
        const nextWaitlistSeqNumber = lastWaitlistRegistration?.waitingListSeqNumber
          ? lastWaitlistRegistration.waitingListSeqNumber + 1
          : 1;

        await queryRunner.manager.update(ProgramRegistration, paymentData.registrationId, {
          registrationStatus: newRegistrationStatus,
          waitingListSeqNumber: nextWaitlistSeqNumber,
        });
      } else {
        // await queryRunner.manager.update(ProgramRegistration, paymentData.registrationId, {
        //   registrationStatus: newRegistrationStatus,
        // });
        if (paymentData.paymentMode === PaymentModeEnum.OFFLINE) {
          await this.updateRegistrationAfterPayment(paymentData.registrationId, paymentData.userId);
        }
      }

      await queryRunner.commitTransaction();
      transactionCommitted = true;

      this.logger.log('Payment and invoice created successfully', {
        paymentId: savedPayment.id,
        invoiceId: invoice.id,
        registrationId: paymentData.registrationId
      });

      return {
        paymentId: savedPayment.id,
        invoiceId: invoice.id,
        registrationId: paymentData.registrationId,
        paymentStatus: savedPayment.paymentStatus,
        invoiceStatus: invoice.invoiceStatus,
        remainingSeats: paymentData.seatAvailability.waitlistTriggerCount - paymentData.seatAvailability.filledSeats,
        participantRemainingSeats: paymentData.seatAvailability.totalSeats - paymentData.seatAvailability.filledSeats,
      };

    } catch (error) {
      // Simple rollback like payment module
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Payment creation transaction failed', error);
      handleKnownErrors(ERROR_CODES.PAYMENT_CREATION_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  async updatePaymentStatus(registrationId: number, dto: UpdatePaymentDto, userId: number) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      // Start transaction
      await queryRunner.startTransaction();
      transactionStarted = true;

      // Find existing payment
      const payment = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
        where: { registrationId },
        relations: ['registration','registration.program', 'registration.programSession'],
      });

      if (!payment) {
        throw new InifniBadRequestException(
          ERROR_CODES.PAYMENT_NOTFOUND,
          null,
          null,
          `No payment found for registration ${registrationId}`
        );
      }

      // Store the current state before update for history
      const currentPaymentState = { ...payment };

      // Update payment details
      const updateData: any = {
        paymentStatus: dto.paymentStatus,
        updatedBy: { id: userId } as any,
      };

      if (dto.paymentDate) {
        updateData.paymentDate = new Date(dto.paymentDate);
      }

      if (dto.paymentMeta) {
        updateData.offlineMeta = { ...payment.offlineMeta, ...dto.paymentMeta };
      }

      if(dto.markAsReceivedDate){
        updateData.markAsReceivedDate = dto.markAsReceivedDate ?? null;
      }

      await queryRunner.manager.update(RegistrationPaymentDetail, payment.id, updateData);

      // Get updated payment for history
      const updatedPayment = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
        where: { id: payment.id },
      });

      // Create payment history record for this update
      if (updatedPayment) {
        await this.createPaymentHistory(
          queryRunner,
          payment.id,
          updatedPayment,
          userId,
          'UPDATE'
        );
      }

      // Update invoice status if payment is completed
      if (dto.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
          dto.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {

            console.log('Payment completed, updating invoice status',payment,payment.programSession);
        
          
          const existedInvoice = await queryRunner.manager.findOne(RegistrationInvoiceDetail, {
            where: { registrationId },
          });

          let latestInvoiceSequenceNumber: string;
          if (existedInvoice?.invoiceSequenceNumber) {
            latestInvoiceSequenceNumber = existedInvoice.invoiceSequenceNumber;
          } else {
            const invoiceType = payment.paymentMode === PaymentModeEnum.ONLINE ? 'L' : 'F';
            latestInvoiceSequenceNumber = await this.generateInvoiceSequenceNumber(queryRunner, payment, invoiceType);
            console.log('Invoice number:', latestInvoiceSequenceNumber);
          }
        
          // const latestInvoiceSequenceNumber = latestInvoice 
          //   ? generateNextSequence(payment.registration.program.code, latestInvoice.invoiceSequenceNumber) 
          //   : `${payment.registration.program.code}001`;


        await queryRunner.manager.update(RegistrationInvoiceDetail, 
          { registrationId },
          { 
            invoiceStatus: InvoiceStatusEnum.INVOICE_COMPLETED,
            invoiceIssuedDate: new Date(),
            updatedBy: { id: userId } as any,
            invoiceSequenceNumber: existedInvoice?.invoiceSequenceNumber ? existedInvoice?.invoiceSequenceNumber : latestInvoiceSequenceNumber,
          }
        );
      }

      await queryRunner.commitTransaction();
      transactionCommitted = true;

      this.logger.log('Payment status updated successfully', {
        paymentId: payment.id,
        registrationId,
        oldStatus: currentPaymentState.paymentStatus,
        newStatus: dto.paymentStatus
      });

      return {
        paymentId: payment.id,
        registrationId,
        paymentStatus: dto.paymentStatus,
        registrationConfirmed: dto.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED ||
                              dto.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED,
      };

    } catch (error) {
      // Simple rollback like payment module
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Payment update transaction failed', error);
      throw error;
    } finally {
      await queryRunner.release();
    }
  }
  private async generateInvoiceSequenceNumber(
    queryRunner: any,
    payment: RegistrationPaymentDetail,
    invoiceType: string
  ): Promise<string> {
    const startsAt = payment.registration.program?.startsAt;
    const startYear = startsAt ? startsAt.getFullYear().toString() : '';
    const nextYear = startsAt ? (startsAt.getFullYear() + 1).toString().slice(-2) : '';
    const programCode = payment.registration.allocatedProgram?.code || payment.registration.program.code;
  
    const pattern = `${programCode}/${startYear}-${nextYear}/${invoiceType}`;
  
    // Fetch the latest invoice matching the pattern
    const latestInvoice = await queryRunner.manager
      .createQueryBuilder(RegistrationInvoiceDetail, 'invoice')
      .andWhere('invoice.invoiceSequenceNumber IS NOT NULL')
      .andWhere('invoice.invoiceSequenceNumber LIKE :pattern', { pattern: `${pattern}%` })
      .orderBy('invoice.invoiceSequenceNumber', 'DESC')
      .getOne();
  
    if (latestInvoice) {
      // Generate the next sequence number based on the latest invoice
      return generateNextSequence(pattern, latestInvoice.invoiceSequenceNumber);
    } else {
      // If no previous invoice, start with 001
      return `${pattern}001`;
    }
  }

  async getPaymentDetailsWithInvoice(registrationId: number) {
    try {
      const payment = await this.paymentRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program', 'registration.programSession'],
      });

      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
      });

      // Get payment history if payment exists
      let paymentHistory: RegistrationPaymentDetailsHistory[] = [];
      if (payment) {
        paymentHistory = await this.getPaymentHistory(payment.id);
      }

      return {
        payment,
        invoice,
        paymentHistory,
        registrationId,
      };
    } catch (error) {
      this.logger.error('Error getting payment details with invoice', '', { registrationId, error });
      throw error;
    }
  }

  async processWebhookUpdate(webhookData: any) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      // Start transaction
      await queryRunner.startTransaction();
      transactionStarted = true;

      // Extract payment details from webhook
      const razorpayOrderId = webhookData.payload?.payment?.entity?.order_id;
      const paymentId = webhookData.payload?.payment?.entity?.id;
      const status = webhookData.payload?.payment?.entity?.status;

      if (!razorpayOrderId) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVALID_WEBHOOK_DATA,
          null,
          null,
          'Razorpay order ID not found in webhook'
        );
      }

      // Find payment by Razorpay order ID
      const payment = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
        where: { razorpayId: razorpayOrderId },
        relations: ['registration', 'registration.program', 'registration.programSession'],
      });

      if (!payment) {
        this.logger.warn('Payment not found for webhook', { razorpayOrderId });
        return { success: false, message: 'Payment not found' };
      }

      // Check if payment is already completed
      if (payment.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || 
        payment.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED) {
        this.logger.warn('Payment already completed for webhook', { paymentId, status });
        return { success: false, message: 'Payment already completed'  };
      }

      // Store current state for history
      const currentPaymentState = { ...payment };

      // Update payment status based on webhook event
      let newPaymentStatus: PaymentStatusEnum;
      if (status === 'captured' || status === 'authorized') {
        newPaymentStatus = PaymentStatusEnum.ONLINE_COMPLETED;
        // Update registration status based on payment
        await this.updateRegistrationAfterPayment(payment.registrationId, null);
      } else if (status === 'failed') {
        newPaymentStatus = PaymentStatusEnum.ONLINE_PENDING; // Keep pending for retry
      } else {
        newPaymentStatus = payment.paymentStatus; // No change
      }

      // Update payment
      await queryRunner.manager.update(RegistrationPaymentDetail, payment.id, {
        paymentStatus: newPaymentStatus,
        paymentDate: status === 'captured' || status === 'authorized' ? new Date() : undefined,
        offlineMeta: {
          ...payment.offlineMeta,
          razorpayPaymentId: paymentId,
          webhookEvent: webhookData.event,
          webhookProcessedAt: Date.now().toString,
        },
      });

      // Get updated payment for history
      // const updatedPayment = await queryRunner.manager.findOne(RegistrationPaymentDetail, {
      //   where: { id: payment.id },
      // });

      // // Create payment history record for webhook update
      // if (updatedPayment) {
      //   await this.createPaymentHistory(
      //     queryRunner,
      //     payment.id,
      //     updatedPayment,
      //     null, // Webhook updates don't have a specific user
      //     'WEBHOOK'
      //   );
      // }


      const invoiceType = payment.paymentMode === PaymentModeEnum.ONLINE ? 'L' : 'F';
      const latestInvoiceSequenceNumber = await this.generateInvoiceSequenceNumber(queryRunner, payment, invoiceType);


      // Update invoice if payment is successful
      if (newPaymentStatus === PaymentStatusEnum.ONLINE_COMPLETED) {
        await queryRunner.manager.update(RegistrationInvoiceDetail, 
          { registrationId: payment.registrationId },
          { 
            invoiceStatus: InvoiceStatusEnum.INVOICE_COMPLETED,
            invoiceIssuedDate: new Date(),
            invoiceSequenceNumber: latestInvoiceSequenceNumber,
          }
        );
      }

      await queryRunner.commitTransaction();
      transactionCommitted = true;

      this.logger.log('Webhook processed successfully', {
        paymentId: payment.id,
        registrationId: payment.registrationId,
        razorpayOrderId,
        oldStatus: currentPaymentState.paymentStatus,
        newStatus: newPaymentStatus
      });

      return {
        success: true,
        paymentId: payment.id,
        registrationId: payment.registrationId,
        paymentStatus: newPaymentStatus,
      };

    } catch (error) {
      // Simple rollback like payment module
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Webhook processing transaction failed', error);
      throw error;
    } finally {
      await queryRunner.release();
    }
  }

  async updateRegistrationAfterPayment(registrationId: number, userId: number | null) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      // Start transaction
      await queryRunner.startTransaction();
      transactionStarted = true;

      // Get registration details
      const registration = await queryRunner.manager.findOne(ProgramRegistration, {
        where: { id: registrationId },
        relations: ['program', 'programSession', 'program.type'],
      });

      if (!registration) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          `Registration ${registrationId} not found`
        );
      }
      if (!registration.program.requiresApproval) {
        // Check if registration is waitlisted
        if (registration.registrationStatus === RegistrationStatusEnum.WAITLISTED) {
          // Check if seat is now available
          const seatAvailability = await this.registartionService.checkSeatAvailability(
            registration.program,
            registration.programSession,
            registration.program.type.registrationLevel
          );

          if (!seatAvailability.shouldWaitlist) {
            // Move from waitlist to confirmed
            await queryRunner.manager.update(ProgramRegistration, registrationId, {
              registrationStatus: RegistrationStatusEnum.COMPLETED,
              waitingListSeqNumber: undefined,
            });

            // Update seat counts
            await incrementSeatCounts(queryRunner.manager, registration.program.id, registration.programSession?.id);
          }
          // If still should waitlist, keep status as is
        } else {
          // Direct confirmation
          await queryRunner.manager.update(ProgramRegistration, registrationId, {
            registrationStatus: RegistrationStatusEnum.COMPLETED,
          });

          // Update seat counts
          await incrementSeatCounts(queryRunner.manager, registration.program.id, registration.programSession?.id);
        }
      }

      await queryRunner.commitTransaction();
      transactionCommitted = true;

      this.logger.log('Registration updated after payment', { registrationId });

    } catch (error) {
      // Simple rollback like payment module
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Registration update after payment failed', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    } finally {
      await queryRunner.release();
    }
  }

  async updateSeatCountsAfterPayment(registrationId: number) {
    // This method is called from service layer after payment confirmation
    // Seat count updates are handled in updateRegistrationAfterPayment
    await this.updateRegistrationAfterPayment(registrationId, null);
    this.logger.log('Seat count update triggered for registration', { registrationId });
  }
}