import { Injectable } from '@nestjs/common';
import { User } from 'src/common/entities';
import { CreateOptionDto } from './dto/create-option.dto';
import { UpdateOptionDto } from './dto/update-option.dto';
import { OptionRepository } from './option.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { optionConstMessages } from 'src/common/constants/strings-constants';

@Injectable()
export class OptionService {
  constructor(
    private readonly optionRepository: OptionRepository,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Creates a new option.
   * Validates the existence of the category and creator user.
   * Ensures the option is unique (name, type, and category).
   * @param createDto - Data transfer object containing option details.
   * @returns The created option.
   * @throws NotFoundException if the category or user does not exist.
   * @throws BadRequestException if the option already exists or data is invalid.
   */
  async create(createDto: CreateOptionDto) {
      this.logger.log(`${optionConstMessages.CREATING_OPTION}: ${JSON.stringify(createDto)}`);
      try {
        return await this.optionRepository.createOption(createDto);
      } catch (error) {
        handleKnownErrors(ERROR_CODES.OPTION_SAVE_FAILED, error);
      }
    }

  /**
   * Retrieves all options with optional pagination and search functionality.
   * @param limit - Number of records per page.
   * @param offset - Offset for pagination.
   * @param searchText - Optional search text to filter options by name.
   * @param categoryId - Optional category ID to filter options by category.
   * @returns A paginated list of options and metadata.
   */
   async findAll(limit: number, offset: number, searchText: string, categoryId: number[], parsedFilters: Record<string, any>) {
    this.logger.log(`${optionConstMessages.FINDING_ALL_OPTIONS} ${JSON.stringify({ limit, offset, searchText, categoryId, parsedFilters })}`);
    try {
      return await this.optionRepository.findAllOptions(limit, offset, searchText, categoryId, parsedFilters);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_GET_FAILED, error);
    }
  }

  /**
   * Retrieves a single option by its ID.
   * @param id - ID of the option to retrieve.
   * @returns The option data if found.
   * @throws NotFoundException if the option does not exist.
   */
  async findOne(id: number) {
    this.logger.log(`${optionConstMessages.FINDING_OPTION_BY_ID}: ${id}`);
    try {
      const option = await this.optionRepository.findOneById(id);
      if (!option) {
        this.logger.error(optionConstMessages.OPTION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(ERROR_CODES.OPTION_NOTFOUND, null, null, id.toString());
      }
      return option;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Updates an existing option by its ID.
   * Validates the existence of the category and updater user.
   * Ensures the updated option is unique (name, type, and category).
   * @param id - ID of the option to update.
   * @param updateDto - Data transfer object containing updated option details.
   * @returns The updated option.
   * @throws NotFoundException if the option, category, or user does not exist.
   * @throws BadRequestException if a duplicate option exists or data is invalid.
   */
  async update(id: number, updateDto: UpdateOptionDto) {
    this.logger.log(`${optionConstMessages.UPDATING_OPTION}: ${id}`);
    try {
      return await this.optionRepository.updateOption(id, updateDto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_SAVE_FAILED, error);
    }
  }

  /**
   * Deletes an option by its ID.
   * @param id - ID of the option to delete.
   * @returns The deleted option.
   * @throws NotFoundException if the option does not exist.
   * @throws BadRequestException if the deletion fails.
   */
  async remove(id: number, user: User) {
    this.logger.log(`${optionConstMessages.DELETING_OPTION_BY_ID}: ${id}`);
    try {
      return await this.optionRepository.softDeleteOption(id, user);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_DELETE_FAILED, error);
    }
  }
}