import { Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { Repository, IsNull, In, ILike } from "typeorm";
import { Option, OptionCategory, User } from "src/common/entities";
import { CreateOptionDto } from "./dto/create-option.dto";
import { UpdateOptionDto } from "./dto/update-option.dto";
import { CommonDataService } from "src/common/services/commonData.service";
import { AppLoggerService } from "src/common/services/logger.service";
import { ERROR_CODES } from "src/common/constants/error-string-constants";
import { handleKnownErrors } from "src/common/utils/handle-error.util";
import { InifniNotFoundException } from "src/common/exceptions/infini-notfound-exception";
import InifniBadRequestException from "src/common/exceptions/infini-badrequest-exception";
import { optionCategoryMessages, optionConstMessages, userConstMessages } from "src/common/constants/strings-constants";

@Injectable()
export class OptionRepository {
  constructor(
    @InjectRepository(Option)
    private readonly optionRepo: Repository<Option>,
    @InjectRepository(OptionCategory)
    private readonly optionCategoryRepo: Repository<OptionCategory>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService
  ) {}

  async createOption(createDto: CreateOptionDto): Promise<Option> {
    try {
      const { categoryId, name, type, createdBy, updatedBy } = createDto;
      console.log(categoryId,name,type,createdBy,updatedBy);

      // Validate category
      if (categoryId) {
        const category = await this.optionCategoryRepo.findOne({
          where: { id: categoryId },
        });
        if (!category) {
          this.logger.error(optionCategoryMessages.CATEGORY_NOT_FOUND_ID(categoryId));
          throw new InifniNotFoundException(
            ERROR_CODES.OPTION_CATEGORY_NOTFOUND,
            null,
            null,
            categoryId.toString()
          );
        }
      }

      // Validate creator and updater
      const [creator, updater] = await Promise.all([
        this.userRepo.findOne({ where: { id: createdBy } }),
        this.userRepo.findOne({ where: { id: updatedBy } }),
      ]);

      if (!creator) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(createdBy));
        throw new InifniNotFoundException(
          ERROR_CODES.OPTION_CREATOR_NOTFOUND,
          null,
          null,
          createdBy.toString()
        );
      }
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.OPTION_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }

      // Check for duplicate option
      const whereClause: any = { name, type, deletedAt: IsNull() };
      if (categoryId !== null && categoryId !== undefined) {
        whereClause.categoryId = categoryId;
      }

      const existingOption = await this.optionRepo.findOne({
        where: whereClause,
      });
      if (existingOption) {
        this.logger.error(optionConstMessages.DUPLICATE_OPTION);
        throw new InifniBadRequestException(
          ERROR_CODES.OPTION_DUPLICATE_BADREQUEST,
          null,
          null,
          name
        );
      }

      const option = new Option({ ...createDto,
          createdBy: creator,
          updatedBy: updater,
       });
      return this.commonDataService.save(this.optionRepo, option);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_SAVE_FAILED, error);
    }
  }

  async findAllOptions(
    limit: number,
    offset: number,
    searchText: string,
    categoryId: number[],
    parsedFilters: Record<string, any>
  ) {
    try {
      const whereClause: any = { deletedAt: IsNull() };
      if (categoryId?.length) whereClause.categoryId = In(categoryId);
      if (parsedFilters?.status) whereClause.status = parsedFilters.status;
      if (searchText) whereClause.name = ILike(`%${searchText}%`);

      const data = await this.commonDataService.get(
        this.optionRepo,
        ["id", "name", "type", "categoryId", "status"],
        whereClause,
        limit,
        offset,
        { id: "ASC" },
        searchText ? { name: searchText } : undefined,
        ["category"],
        { category: ["id", "name"] }
      );

      const total = await this.optionRepo.count({ where: whereClause });
      return {
        data,
        pagination: {
          totalPages: Math.ceil(total / limit),
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: limit,
          totalRecords: total,
          numberOfRecords: data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_GET_FAILED, error);
    }
  }
  async findOneById(id: number): Promise<Option | null> {
    try {
      return await this.commonDataService.findOneById(this.optionRepo, id, true, [
        "category",
      ]);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_FIND_BY_ID_FAILED, error);
    }
  }

  async updateOption(id: number, updateDto: UpdateOptionDto): Promise<Option> {
    try {
      const option = await this.findOneById(id);
      if (!option) {
        this.logger.error(optionConstMessages.OPTION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.OPTION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      const { categoryId, name, type, updatedBy } = updateDto;

      // Validate category
      if (categoryId) {
        const category = await this.optionCategoryRepo.findOne({
          where: { id: categoryId },
        });
        if (!category) {
          this.logger.error(optionCategoryMessages.CATEGORY_NOT_FOUND_ID(categoryId));
          throw new InifniNotFoundException(
            ERROR_CODES.OPTION_CATEGORY_NOTFOUND,
            null,
            null,
            categoryId.toString()
          );
        }
        option.category = category;
      }

      // Validate updater
      const updater = await this.userRepo.findOne({ where: { id: updatedBy } });
      if (!updater) {
        this.logger.error(userConstMessages.USER_NOT_FOUND_ID(updatedBy));
        throw new InifniNotFoundException(
          ERROR_CODES.OPTION_UPDATOR_NOTFOUND,
          null,
          null,
          updatedBy.toString()
        );
      }

      // Check for duplicate option
      const whereClause: any = { name, type, deletedAt: IsNull() };
      if (categoryId !== null && categoryId !== undefined) {
        whereClause.categoryId = categoryId;
      }

      const existingOption = await this.optionRepo.findOne({
        where: whereClause,
      });
      if (existingOption && existingOption.id !== id) {
        this.logger.error(optionConstMessages.DUPLICATE_OPTION);
        throw new InifniBadRequestException(
          ERROR_CODES.OPTION_DUPLICATE_BADREQUEST,
          null,
          null,
          name || ""
        );
      }

      Object.assign(option, updateDto);
      return this.commonDataService.save(this.optionRepo, option);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_SAVE_FAILED, error);
    }
  }

  async softDeleteOption(id: number, user: User): Promise<void> {
    try {
      const option = await this.findOneById(id);
      if (!option) {
        this.logger.error(optionConstMessages.OPTION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(
          ERROR_CODES.OPTION_NOTFOUND,
          null,
          null,
          id.toString()
        );
      }

      option.deletedAt = new Date();
      option.updatedBy = user;
      await this.commonDataService.save(this.optionRepo, option);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_DELETE_FAILED, error);
    }
  }
}
