import { Injectable } from '@nestjs/common';
import { OptionCategoryRepository } from './option-category.repository';
import { CreateOptionCategoryDto } from './dto/create-option-category.dto';
import { UpdateOptionCategoryDto } from './dto/update-option-category.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { optionCategoryMessages } from 'src/common/constants/strings-constants';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { User } from 'src/common/entities';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';

@Injectable()
export class OptionCategoryService {
  constructor(
    private readonly repository: OptionCategoryRepository,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Creates a new option category.
   * @param dto - Data transfer object containing category details.
   * @returns The created option category.
   */
  async create(dto: CreateOptionCategoryDto) {
    this.logger.log(optionCategoryMessages.CREATING_CATEGORY(dto));
    try {
      return await this.repository.create(dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_SAVE_FAILED, error);
    }
  }

  /**
   * Retrieves all option categories with pagination and optional search functionality.
   * @param limit - Number of records per page.
   * @param offset - Offset for pagination.
   * @param searchText - Optional search text to filter categories by name.
   * @returns Paginated list of option categories.
   */
  async findAll(limit: number, offset: number, searchText: string) {
    this.logger.log(optionCategoryMessages.FINDING_ALL_CATEGORIES(limit, offset, searchText));
    try {
      return await this.repository.findAll(limit, offset, searchText);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_GET_FAILED, error);
    }
  }

  /**
   * Retrieves a single option category by its ID.
   * @param id - ID of the category to retrieve.
   * @returns The option category if found.
   * @throws NotFoundException if the category does not exist.
   */
  async findOne(id: number) {
    this.logger.log(optionCategoryMessages.FINDING_CATEGORY_BY_ID(id));
    try {
      const category = await this.repository.findOneById(id);
      if (!category) {
        this.logger.warn(optionCategoryMessages.CATEGORY_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(ERROR_CODES.OPTION_CATEGORY_NOTFOUND, null, null, id.toString());
      }
      return category;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Updates an existing option category.
   * @param id - ID of the category to update.
   * @param dto - Data transfer object containing updated category details.
   * @returns The updated option category.
   */
  async update(id: number, dto: UpdateOptionCategoryDto) {
    this.logger.log(optionCategoryMessages.UPDATING_CATEGORY(id));
    try {
      return await this.repository.update(id, dto);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_SAVE_FAILED, error);
    }
  }

  /**
   * Deletes an option category by its ID.
   * @param id - ID of the category to delete.
   * @param user - User performing the deletion.
   * @returns The deleted option category.
   * @throws NotFoundException if the category does not exist.
   */
  async remove(id: number, user: User) {
    this.logger.log(optionCategoryMessages.REMOVING_CATEGORY(id));
    try {
      return await this.repository.remove(id, user);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.OPTION_CATEGORY_DELETE_FAILED, error);
    }
  }
}