import {
  Controller, Get, Post, Body, Param, Put, Delete, HttpStatus, HttpCode, ParseIntPipe, UsePipes, ValidationPipe,
  Query, Res,
  UseGuards,
  Req,
} from '@nestjs/common';
import { Response } from 'express';
import { OptionCategoryService } from './option-category.service';
import { CreateOptionCategoryDto } from './dto/create-option-category.dto';
import { UpdateOptionCategoryDto } from './dto/update-option-category.dto';
import {
  ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiQuery,
  ApiBearerAuth,
  ApiSecurity
} from '@nestjs/swagger';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { optionCategoryMessages, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants'; 
import { Roles } from 'src/common/decorators/roles.decorator';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('option-category')
@Controller('option-category')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class OptionCategoryController {
  constructor(
    private readonly service: OptionCategoryService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Creates a new option category.
   * Validates the input using a validation pipe and ensures the category is created successfully.
   * @param dto - Data transfer object containing category details.
   * @returns A success message and the created category data.
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: optionCategoryMessages.CREATE_NEW_CATEGORY_OPTION })
  @ApiBody({ type: CreateOptionCategoryDto })
   @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(@Body() dto: CreateOptionCategoryDto, @Req() req: any,@Res() res: Response) {
    this.logger.log(optionCategoryMessages.CREATE_CATEGORY_REQUEST_RECEIVED, dto);
    try {
      const userId = req.user?.id || dto.createdBy;
      dto.createdBy = userId;
      dto.updatedBy = userId;
      const data = await this.service.create(dto);
      const respData = {
        id: data.id,
        name: data.name,
      };
      this.logger.log(optionCategoryMessages.CATEGORY_CREATED);
      return this.responseService.success(
        res, 
        optionCategoryMessages.CATEGORY_CREATED, 
        respData, 
        HttpStatus.CREATED
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves all option categories with optional pagination and search functionality.
   * @param limit - Number of records per page (default: 10).
   * @param offset - Offset for pagination (default: 0).
   * @param searchText - Optional search text to filter categories by name.
   * @returns A paginated list of categories and metadata.
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: optionCategoryMessages.GET_ALL_OPTION_CATEGORIES_SEARCH })
  @ApiQuery({ name: 'limit', required: false, type: Number, example: 10 })
  @ApiQuery({ name: 'offset', required: false, type: Number, example: 0 })
  @ApiQuery({ name: 'searchText', required: false, type: String, example: 'discount' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Res() res: Response
  ) {
    this.logger.log(optionCategoryMessages.FIND_ALL_CATEGORIES_REQUEST_RECEIVED, { limit, offset, searchText });
    try {
      const result = await this.service.findAll(limit, offset, searchText);
      this.logger.log(optionCategoryMessages.CATEGORIES_RETRIEVED);
      return this.responseService.success(
        res, 
        optionCategoryMessages.CATEGORIES_RETRIEVED,
        result
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves a single option category by its ID.
   * @param id - ID of the category to retrieve.
   * @returns The category data if found.
   */
  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: optionCategoryMessages.GET_OPTION_CATEGORY_BY_ID })
  @ApiParam({ name: 'id', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(optionCategoryMessages.FIND_ONE_CATEGORY_REQUEST_RECEIVED, { id });
    try {
      const data = await this.service.findOne(id);
      const respData = {
        id: data.id,
        name: data.name,
      };
      
      this.logger.log(optionCategoryMessages.CATEGORY_RETRIEVED, respData);
      return this.responseService.success(
        res, 
        optionCategoryMessages.CATEGORY_RETRIEVED,
        respData
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Updates an existing option category by its ID.
   * Validates the input using a validation pipe and ensures the category is updated successfully.
   * @param id - ID of the category to update.
   * @param dto - Data transfer object containing updated category details.
   * @returns A success message and the updated category data.
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: optionCategoryMessages.UPDATE_CATEGORY_ID })
  @ApiParam({ name: 'id', type: Number })
  @ApiBody({ type: UpdateOptionCategoryDto })
   @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateOptionCategoryDto,
    @Req() req: any,
    @Res() res: Response
  ) {
    this.logger.log(optionCategoryMessages.UPDATE_CATEGORY_REQUEST_RECEIVED, { id, dto });
    try {
      const userId = req.user?.id || dto.updatedBy;
      dto.updatedBy = userId;
      const data = await this.service.update(id, dto);
      const respData = {
        id: data.id,
        name: data.name,
      };
      this.logger.log(optionCategoryMessages.CATEGORY_UPDATED, respData);
      
      return this.responseService.success(
        res, 
        optionCategoryMessages.CATEGORY_UPDATED,
        respData
      );
    } catch (error) {
      handleControllerError(res, error)
    }
  }

  /**
   * Deletes an option category by its ID.
   * @param id - ID of the category to delete.
   * @returns A success message indicating the category was deleted.
   */
  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: optionCategoryMessages.DELETE_CATEGORY_ID })
  @ApiParam({ name: 'id', type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number,@Req() req: any ,@Res() res: Response) {
    this.logger.log(optionCategoryMessages.DELETE_CATEGORY_REQUEST_RECEIVED, { id });
    const user = req.user;
    if (!user) {
      return this.errorHandler.unauthorized(res);
    }
    try {
      await this.service.remove(id, user);
      this.logger.log(optionCategoryMessages.CATEGORY_DELETED);
      return this.responseService.success(
        res, 
        optionCategoryMessages.CATEGORY_DELETED,
        undefined,
        HttpStatus.OK
      );
    } catch (error) {
       handleControllerError(res, error);
    }
  }
}