import { NestFactory } from '@nestjs/core';
import { AppModule } from './app.module';
import { ConfigService } from '@nestjs/config';
import { appConfig } from './common/config/config';
import { DocumentBuilder, SwaggerModule } from '@nestjs/swagger';
import { SWAGGER } from './common/constants/strings-constants';
import { SecuritySchemeType } from '@nestjs/swagger/dist/interfaces/open-api-spec.interface';
import { HttpExceptionFilter } from './common/filters/http-exception.filter';
import { ValidationPipe } from '@nestjs/common';

async function bootstrap() {
  const app = await NestFactory.create(AppModule);

  // Apply the global exception filter
   app.useGlobalFilters(new HttpExceptionFilter());
  
   // Apply the global validation pipe
   app.useGlobalPipes(new ValidationPipe({ transform: true, whitelist: true }));

  //Get environment variables
  const configService = app.get(ConfigService);
  const { port, corsOrigin, backendUrl } = appConfig(configService);

  //Set Swagger Documentation
  const swaggerConfig = new DocumentBuilder()
    .setTitle(SWAGGER.TITLE)
    .setDescription(SWAGGER.DESCRIPTION)
    .setVersion(SWAGGER.VERSION)
    .addServer(backendUrl, SWAGGER.ENVIRONMENT)
    .addBearerAuth(
      {
        type: SWAGGER.BEARERAUTH.TYPE as SecuritySchemeType,
        name: SWAGGER.BEARERAUTH.NAME,
        description: SWAGGER.BEARERAUTH.DESCRIPTION,
        in: SWAGGER.BEARERAUTH.IN,
      },
      'Authorization',
    )
    .addApiKey(
    {
      type: SWAGGER.USER_ID_AUTH.TYPE as SecuritySchemeType,
      name: SWAGGER.USER_ID_AUTH.NAME,
      in: SWAGGER.USER_ID_AUTH.IN,
      description: SWAGGER.USER_ID_AUTH.DESCRIPTION,
    },
    'userIdAuth',
  )

    .build();
  const document = SwaggerModule.createDocument(app, swaggerConfig);
  SwaggerModule.setup(SWAGGER.PATH, app, document, {
    swaggerOptions: {
      persistAuthorization: true,
    },
  });

  

  //Enable CORS
  app.enableCors({
    origin: '*',
    credentials: true,
  });
  await app.listen(port);
}
bootstrap();
