import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { LookupData } from 'src/common/entities/lookup-data.entity';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { lookupDataConstMessages } from 'src/common/constants/strings-constants';
import { CommonStatus } from 'src/common/enum/common-status.enum';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

@Injectable()
export class LookupDataRepository {
  constructor(
    @InjectRepository(LookupData)
    private readonly repo: Repository<LookupData>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async findActiveByCategory(category: string) {
    this.logger.log(lookupDataConstMessages.FINDING_BY_CATEGORY(category));
    try {
      const lookups = await this.commonDataService.findByData(this.repo, {
        lookupCategory: category,
        lookupStatus: CommonStatus.ACTIVE,
      });
      const data = lookups.map((item) => ({ key: item.lookupKey, label: item.lookupLabel }));
      const total = data.length;
      return {
        data,
        pagination: {
          totalPages: 1,
          pageNumber: 1,
          pageSize: total + 1,
          totalRecords: total,
          numberOfRecords: total,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }

  async findActiveCategories() {
    this.logger.log(lookupDataConstMessages.FINDING_CATEGORIES);
    try {
      const cats = await this.repo
        .createQueryBuilder('lookup')
        .select(['DISTINCT lookup.lookupCategory AS key', 'lookup.lookupCategoryName AS label'])
        .where('lookup.lookupStatus = :status', { status: CommonStatus.ACTIVE })
        .orderBy('lookup.lookupCategoryName', 'ASC')
        .getRawMany();
      const data = cats.map((item) => ({
        key: item.key,
        label: item.label,
      }));

      const total = data.length;
      return {
        data,
        pagination: {
          totalPages: 1,
          pageNumber: 1,
          pageSize: total + 1,
          totalRecords: total,
          numberOfRecords: total,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.LOOKUP_DATA_GET_FAILED, error);
    }
  }
}
