import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource } from 'typeorm';
import { RegistrationInvoiceDetail, RegistrationPaymentDetail, ProgramRegistration, Program } from 'src/common/entities';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { InvoiceTypeEnum } from 'src/common/enum/invoice-type.enum';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { amountInWordsIndian, areGSTsFromSameState, deductDaysFromDate, formatDate } from 'src/common/utils/common.util';
import { eInvoiceConstants, stateGstCodes, zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { CommunicationService } from 'src/communication/communication.service';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import { INVOICE } from 'src/common/templates/templates';
import { InvoiceTemplateDto } from 'src/common/templates/dtos/templates.dto';
import axios from 'axios';
import { v4 as uuidv4 } from 'uuid';
import { EInvoiceResponse } from './dto/create-invoice.dto';
import { oAuth } from 'src/common/utils/common.util';
import { normalizeBoolean } from 'razorpay/dist/utils/razorpay-utils';
import { AwsS3Service } from 'src/common/services/awsS3.service';
import { UserRepository } from 'src/user/user.repository';
import chromium from "@sparticuz/chromium-min";

const puppeteer = require('puppeteer-core');
// const chromium = require('@sparticuz/chromium-min');


@Injectable()
export class InvoiceService {
  constructor(
    @InjectRepository(RegistrationInvoiceDetail)
    private readonly invoiceRepo: Repository<RegistrationInvoiceDetail>,
    @InjectRepository(RegistrationPaymentDetail)
    private readonly paymentRepo: Repository<RegistrationPaymentDetail>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
    private readonly communicationService: CommunicationService,
    private readonly awsS3Service: AwsS3Service, 
    private readonly userRepository: UserRepository,
  ) {}

  async generateAndSendInvoice(registrationId: number) {
    this.logger.log('Generating and sending invoice', { registrationId });

    try {
      // Get invoice and related data
      
      // Generate invoice sequence number if not exists
      // if (!invoiceData.invoice.invoiceSequenceNumber) {
      //   const sequenceNumber = await this.generateInvoiceSequenceNumber(invoiceData.invoice.invoiceType);
      //   await this.updateInvoiceSequenceNumber(invoiceData.invoice.id, sequenceNumber);
      //   invoiceData.invoice.invoiceSequenceNumber = sequenceNumber;
      // }
      const invoiceData = await this.getInvoiceGenerationData(registrationId);
      // Generate PDF (implement based on your requirements)
      // console.log('invoiceData', JSON.stringify(invoiceData, null, 2));
      let invoicePdfUrl
      if (invoiceData.invoice.invoicePdfUrl) {
        invoicePdfUrl = invoiceData.invoice.invoicePdfUrl;
      } else {
        var updatedInvoiceData = invoiceData
        if (invoiceData.invoice.gstNumber && invoiceData.invoice.isGstRegistered && !invoiceData.invoice.einvoiceAckNumber) {
          // const eInvoice = await this.generateEInvoice(registrationId);
          // this.logger.log('E-invoice generated successfully', {eInvoice});
          // updatedInvoiceData = await this.getInvoiceGenerationData(registrationId);
        }
        const pdfBuffer = await this.generateInvoicePDF(updatedInvoiceData);
        invoicePdfUrl = pdfBuffer.url;
      }
      
      this.logger.log('PDF generated successfully',  invoicePdfUrl );
      const response = await axios.get(invoicePdfUrl, { responseType: 'arraybuffer' });
      const fileData = Buffer.from(response.data, 'binary').toString('base64');

      // console.log('E-invoice generated successfully', eInvoice);

      // Send email with invoice (implement based on your email service)
      await this.sendInvoiceEmail(invoiceData, fileData);

      // Update invoice status
      await this.updateInvoiceStatus(invoiceData.invoice.id, InvoiceStatusEnum.INVOICE_COMPLETED, invoicePdfUrl);

      this.logger.log('Invoice generated and sent successfully', { 
        registrationId, 
        invoiceId: invoiceData.invoice.id,
        sequenceNumber: invoiceData.invoice.invoiceSequenceNumber
      });

      return {
        invoiceId: invoiceData.invoice.id,
        sequenceNumber: invoiceData.invoice.invoiceSequenceNumber,
        status: InvoiceStatusEnum.INVOICE_COMPLETED,
      };

    } catch (error) {
      this.logger.error('Invoice generation failed', '', { registrationId, error: error.message });
      handleKnownErrors(ERROR_CODES.INVOICE_GENERATION_FAILED, error);
    }
  }

  async getInvoiceDetails(registrationId: number) {
    try {
      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program', 'registration.allocatedProgram', 'registration.allocatedSession', 'registration.user' ],
      });

      if (!invoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      return invoice;
    } catch (error) {
      this.logger.error('Error getting invoice details', '', { registrationId, error });
      throw error;
    }
  }
  async getRegistrationDetails(registrationId: number) {
    try {
      const registration = await this.registrationRepo.findOne({
        where: { id: registrationId },
        relations: [
          'program', 
          'program.type', 
          'program.venueAddress',
          'programSession', 
          'user',
          'createdBy',
          'allocatedProgram',
          'allocatedSession',
        ],
      });

      if (!registration) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          `Registration ${registrationId} not found`
        );
      }

      return registration;
    } catch (error) {
      this.logger.error('Error getting registration details', '', { registrationId, error });
      throw error;
    }
  }

  async updateInvoiceStatus(invoiceId: number, status: InvoiceStatusEnum, invoicePdfUrl?: string, userId?: number) {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    let transactionStarted = false;
    let transactionCommitted = false;

    try {
      // Start transaction following payment module pattern
      await queryRunner.startTransaction();
      transactionStarted = true;

      const updateData: any = {
        invoiceStatus: status,
      };

      if (userId) {
        updateData.updatedBy = { id: userId } as any;
      }

      if (status === InvoiceStatusEnum.INVOICE_COMPLETED) {
        updateData.invoiceIssuedDate = new Date();
      }

      if (invoicePdfUrl) {
        updateData.invoicePdfUrl = invoicePdfUrl;
      }

      await queryRunner.manager.update(RegistrationInvoiceDetail, invoiceId, updateData);

      await queryRunner.commitTransaction();
      transactionCommitted = true;

      this.logger.log('Invoice status updated', { invoiceId, status });

    } catch (error) {
      // Simple rollback like payment module
      if (transactionStarted && !transactionCommitted) {
        try {
          await queryRunner.rollbackTransaction();
        } catch (rollbackError) {
          this.logger.error('Failed to rollback transaction', rollbackError);
        }
      }
      this.logger.error('Invoice status update failed', error);
      throw error;
    } finally {
      await queryRunner.release();
    }
  }

  async regenerateInvoice(registrationId: number, userId: number) {
    this.logger.log('Regenerating invoice', { registrationId });

    try {
      // Check if invoice exists
      const existingInvoice = await this.invoiceRepo.findOne({
        where: { registrationId },
      });

      if (!existingInvoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      // Reset invoice status and regenerate
      await this.updateInvoiceStatus(existingInvoice.id, InvoiceStatusEnum.DRAFT, undefined, userId);
      
      // Generate new invoice
      const result = await this.generateAndSendInvoice(registrationId);

      return result;

    } catch (error) {
      this.logger.error('Invoice regeneration failed', '', { registrationId, error: error.message });
      throw error;
    }
  }

  private async getInvoiceGenerationData(registrationId: number) {
    try {
      // Get invoice details
      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program', 'registration.allocatedProgram', 'registration.allocatedSession'],
      });

      if (!invoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      // Get payment details
      const payment = await this.paymentRepo.findOne({
        where: { registrationId },
      });

      // Get registration details
      const registration = await this.registrationRepo.findOne({
        where: { id: registrationId },
        relations: [
          'program', 
          'program.type', 
          'program.venueAddress',
          'programSession', 
          'user',
          'createdBy',
          'allocatedProgram',
          'allocatedSession',
        ],
      });

      if (!registration) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          `Registration ${registrationId} not found`
        );
      }

      return {
        invoice,
        payment,
        registration,
        program: registration.program,
        session: registration.programSession,
        allocatedProgram: registration.allocatedProgram,
        allocatedSession: registration.allocatedSession,
        user: registration.user,
      };
    } catch (error) {
      this.logger.error('Error getting invoice generation data', '', { registrationId, error });
      throw error;
    }
  }

  private async generateInvoiceSequenceNumber(invoiceType: InvoiceTypeEnum): Promise<string> {
    try {
      // Get the latest invoice sequence number for this type
      const latestInvoice = await this.invoiceRepo.findOne({
        where: { 
          invoiceType,
          invoiceSequenceNumber: { $ne: null } as any
        },
        order: { id: 'DESC' },
      });

      let nextNumber = 1;
      let prefix = invoiceType === InvoiceTypeEnum.ONLINE ? 'ONL' : 'OFF';

      if (latestInvoice?.invoiceSequenceNumber) {
        // Extract number from existing sequence (e.g., "ONL-2024-001" -> 1)
        const matches = latestInvoice.invoiceSequenceNumber.match(/(\d+)$/);
        if (matches) {
          nextNumber = parseInt(matches[1]) + 1;
        }
      }

      // Generate sequence number: PREFIX-YYYY-NNN
      const year = new Date().getFullYear();
      const paddedNumber = nextNumber.toString().padStart(3, '0');
      
      return `${prefix}-${year}-${paddedNumber}`;

    } catch (error) {
      this.logger.error('Error generating invoice sequence number', '', { invoiceType, error });
      throw error;
    }
  }

  private async updateInvoiceSequenceNumber(invoiceId: number, sequenceNumber: string) {
    try {
      await this.invoiceRepo.update(invoiceId, {
        invoiceSequenceNumber: sequenceNumber,
      });
    } catch (error) {
      this.logger.error('Error updating invoice sequence number', '', { invoiceId, sequenceNumber, error });
      throw error;
    }
  }

  private async generateInvoicePDF(invoiceData: any): Promise<{ url: string; buffer: Buffer }> {
    // Implement PDF generation based on your requirements
    // This is a placeholder implementation
    
    this.logger.log('Generating invoice PDF', { 
      invoiceId: invoiceData.invoice.id,
      registrationId: invoiceData.registration.id
    });

    try {
      // You can use libraries like:
      // - puppeteer for HTML to PDF
      // const puppeteer = require('puppeteer');

      // Launch Puppeteer
      // const browser = await puppeteer.launch()
      const browser = await puppeteer.launch({
        args: [
          ...(Array.isArray(chromium.args) ? chromium.args : []),
          '--no-sandbox',
          '--disable-setuid-sandbox',
          '--disable-features=AudioServiceOutOfProcess',
          '--disable-gpu',
          '--disable-software-rasterize',
          '--disable-features=AudioServiceOutOfProcessKillAtHang',
          '--single-process',
          '--disable-software-rasterizer',
        ],
        defaultViewport: null,
        executablePath: await chromium.executablePath('https://github.com/Sparticuz/chromium/releases/download/v119.0.2/chromium-v119.0.2-pack.tar'),
        headless: true,
      });
      const page = await browser.newPage();
      // Set the HTML content
      this.logger.log('Generating invoice template data',invoiceData );
      const invoiceTemplateData: InvoiceTemplateDto = {
        companyAddress: invoiceData.program?.invoiceSenderAddress || 'Infinitheism',
        companyGST: invoiceData.program?.gstNumber,
        companyCIN: invoiceData.program?.invoiceSenderCin || '',
        invoiceNumber: invoiceData.invoice.invoiceSequenceNumber || '',
        currentDate: formatDate(new Date().toISOString()),
        placeOfSupply: invoiceData?.invoice?.gstNumber ? stateGstCodes.find(code => code.code === invoiceData.invoice?.gstNumber.substring(0, 2))?.stateName || null : null,
        seekerName: invoiceData.invoice.invoiceName,
        seekerAddress: invoiceData.invoice.invoiceAddress,
        seekerGstin: invoiceData.invoice?.gstNumber || '',
        programName: invoiceData.program?.name || '',
        reportingTime: formatDate(String(invoiceData.allocatedProgram.checkinAt)),
        checkOutTime: formatDate(String(invoiceData.allocatedProgram.checkoutAt)),
        amount: invoiceData.payment?.originalAmount || 0,
        igstCheck: (invoiceData.invoice?.gstNumber && invoiceData.program?.gstNumber) ? !areGSTsFromSameState(invoiceData.program?.gstNumber, invoiceData.invoice?.gstNumber) : false,
        igst: `${(Number(invoiceData.registration.allocatedProgram?.basePrice) * (Number(invoiceData.program.igst) / 100)).toFixed(2) || 0}`,
        cgst: `${(Number(invoiceData.registration.allocatedProgram?.basePrice) * (Number(invoiceData.program.cgst) / 100)).toFixed(2) || 0}`,
        sgst: `${(Number(invoiceData.registration.allocatedProgram?.basePrice) * (Number(invoiceData.program.sgst) / 100)).toFixed(2) || 0}`,
        igstPercentage: (Number(invoiceData.program?.igst)) || 0,
        cgstPercentage: (Number(invoiceData.program?.cgst)) || 0,
        sgstPercentage: (Number(invoiceData.program?.sgst)) || 0,
        totalAmount: invoiceData.payment?.subTotal || 0,
        amountInWords: amountInWordsIndian(Number(invoiceData.payment?.subTotal) || 0),
        qr: invoiceData?.invoice?.einvoiceQrLink || '', // 'https://delta-node-test.s3.ap-south-1.amazonaws.com/qr.png', // sample file, // Placeholder QR code URL
        irpDate: formatDate(new Date().toISOString()) || '',
        acknowledgeNumber: invoiceData?.invoice?.einvoiceAckNumber || '', // '1234567890123456789',
        irn: invoiceData?.invoice?.einvoiceInvRefNum || '',
        companyPAN: invoiceData.program?.invoiceSenderPan || '', 
      };
      // console.log('INVOICE(invoiceTemplateData)',INVOICE(invoiceTemplateData))
      await page.setContent(INVOICE(invoiceTemplateData));


      
      // For now, return a placeholder buffer
      const pdfBuffer = await page.pdf({
        format: 'A4',
        printBackground: true,
      });      
      await browser.close();

      // const bucketName = process.env.AWS_S3_BUCKET_NAME || '';
      const key = `invoices/${invoiceData.program.code}/${invoiceData.invoice.invoiceSequenceNumber || uuidv4()}.pdf`;
      const uploadedInvoice = await this.awsS3Service.uploadToS3(key, pdfBuffer, 'application/pdf');
      // const s3Url = this.awsS3Service.getS3Url(key);
      // const s3Url = await this.awsS3Service.getSignedUrl(key);
      return {url: uploadedInvoice, buffer: pdfBuffer};
      

    } catch (error) {
      this.logger.error('PDF generation failed', '', { invoiceId: invoiceData.invoice.id, error });
      throw error;
    }
  }

  private async sendInvoiceEmail(invoiceData: any, pdfBuffer: string) {
    // Implement email sending based on your email service
    
    this.logger.log('Sending invoice email', { 
      invoiceId: invoiceData.invoice.id,
      email: invoiceData.invoice.invoiceEmail
    });

    try {
      // You can use your existing email service here
      // Example:
      /*
      await this.emailService.sendEmail({
        to: invoiceData.invoice.invoiceEmail,
        subject: `Invoice ${invoiceData.invoice.invoiceSequenceNumber} - ${invoiceData.program.name}`,
        template: 'invoice',
        context: {
          invoiceNumber: invoiceData.invoice.invoiceSequenceNumber,
          customerName: invoiceData.invoice.invoiceName,
          programName: invoiceData.program.name,
          amount: invoiceData.payment.subTotal,
        },
        attachments: [
          {
            filename: `invoice-${invoiceData.invoice.invoiceSequenceNumber}.pdf`,
            content: pdfBuffer,
            contentType: 'application/pdf',
          }
        ]
      });
      */
     const getFinanceUsers = await this.userRepository.getUsersByRoleKey('ROLE_FINANCE_MANAGER');
     let ccEmails: { emailAddress: string; name: string }[] = [];
     let bccEmails: { emailAddress: string; name: string }[] = [];
    //  if (process.env.HDB_BCC_EMAIL) {
    //    bccEmails.push({ emailAddress: process.env.HDB_BCC_EMAIL, name: process.env.HDB_BCC_EMAIL_NAME || 'HDB' });
    //  }
     console.log('getFinanceUsers', getFinanceUsers);
     if (getFinanceUsers && getFinanceUsers.length > 0) {
      //  ccEmails = getFinanceUsers.map((user) => ({
      //    emailAddress: user.email,
      //    name: user.fullName || 'Finance Manager',
      //  }));
       bccEmails = bccEmails.concat(
         getFinanceUsers.map((user) => ({
           emailAddress: user.email,
           name: user.fullName || 'Finance Manager',
         }))
       );
     }
      const email = invoiceData.invoice.invoiceEmail;
      if (email) {
        // Prepare merge info for email
        const lastDate = formatDate(deductDaysFromDate(new Date(invoiceData?.allocatedProgram?.startsAt?.toISOString() ?? ""), 10).toISOString());
        const mergeInfo = {
          hdb_msd_no: invoiceData?.allocatedProgram?.name || '',
          hdb_msd_date: formatDate(invoiceData?.allocatedProgram?.startsAt?.toISOString()),
          hdb_msd_amount: invoiceData?.payment?.subTotal,
          reg_name: invoiceData?.invoice?.invoiceName || 'Infinitheist',
          hdb_msd: invoiceData?.allocatedProgram?.name || '',
          last_date: lastDate // formatDate(invoiceData?.allocatedProgram?.registrationEndsAt?.toISOString())
        }
        const attachments = {
          filename: `${invoiceData?.invoice?.invoiceSequenceNumber ?? "invoice"}.pdf`, //`invoice-${invoiceData.invoice.invoiceSequenceNumber || invoiceData.invoice.id}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf',
        }
        const emailData: SendSingleEmailDto = {
          templateKey: process.env.ZEPTO_INVOICE_EMAIL_TEMPLATE_ID,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_EMAIL,
            name: invoiceData.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: email,
            name: email || 'Infinitheist',
          },
          cc: ccEmails,
          bcc: bccEmails,
          mergeInfo: mergeInfo,
          attachments: [{
            name: attachments.filename,
            content: attachments.content,
            mime_type: attachments.contentType,
          }],
          subject: 'Payment Confirmation - Infinitheism',
          trackinfo: {
            registrationId: invoiceData.registrationId,
            // createdBy: invoiceData.user.id,
            // updatedBy: invoiceData.user.id,
          }
        }
        await this.communicationService.sendSingleEmail(emailData);

        const messageData: SendTemplateMessageDto = {
          whatsappNumber: invoiceData.registration.mobileNumber.slice(0) || '',
          templateName: process.env.WATI_INVOICE_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_INVOICE_TEMPLATE_ID || '',
          parameters: [
            { name: 'reg_name', value: invoiceData.invoice.invoiceName },
            { name: 'hbd_msd_variable', value: invoiceData.program.name }
          ],
          trackinfo: {
            registrationId: invoiceData.invoice.registrationId,
            // createdBy: invoiceData.user.id,
            // updatedBy: invoiceData.user.id,
          }
        };

        await this.communicationService.sendTemplateMessage(messageData);

        this.logger.log('WhatsApp message sent successfully', {messageData});



      } else {
        this.logger.warn('No invoice email provided for payment confirmation', {
          invoiceData,
          invoiceEmail: invoiceData.invoice?.invoiceEmail,
        });

      }

    } catch (error) {
      this.logger.error('Invoice email sending failed', '', { 
        invoiceId: invoiceData.invoice.id, 
        email: invoiceData.invoice.invoiceEmail,
        error 
      });
      throw error;
    }
  }

  async getInvoicesByRegistration(registrationId: number) {
    try {
      const invoices = await this.invoiceRepo.find({
        where: { registrationId },
        order: { createdAt: 'DESC' },
      });

      return invoices;
    } catch (error) {
      this.logger.error('Error getting invoices by registration', '', { registrationId, error });
      throw error;
    }
  }

  async downloadInvoice(invoiceId: number): Promise<{ buffer: Buffer; filename: string }> {
    try {
      const invoice = await this.invoiceRepo.findOne({
        where: { id: invoiceId },
        relations: ['registration'],
      });

      if (!invoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `Invoice ${invoiceId} not found`
        );
      }

      // Get full invoice data for PDF generation
      const invoiceData = await this.getInvoiceGenerationData(invoice.registrationId);
      
      // Generate PDF
      const pdfBuffer = await this.generateInvoicePDF(invoiceData);
      
      const filename = `invoice-${invoice.invoiceSequenceNumber || invoice.id}.pdf`;
      
      return { buffer: pdfBuffer.buffer, filename };

    } catch (error) {
      this.logger.error('Invoice download failed', '', { invoiceId, error });
      throw error;
    }
  }
  async generateEInvoice(registrationId: number): Promise<EInvoiceResponse | null> {
    this.logger.log('Generating e-invoice', { registrationId });

    try {
      // Generate e-invoice data
      const eInvoiceData = await this.generateEInvoiceData(registrationId);
      
      if (!eInvoiceData) {
        throw new InifniBadRequestException(
          ERROR_CODES.EINVOICE_GENERATION_FAILED,
          null,
          null,
          `Failed to generate e-invoice for registration ${registrationId}`
        );
      }
      const zohoEinvoiceAuth = await oAuth();

      const eInvoiceInfo = {
        method: 'post',
        url: process.env.ZOHO_E_INVOICE_URL,
        headers: {
          'Content-Type': 'application/json',
          Authorization: `Zoho-oauthtoken ${zohoEinvoiceAuth.access_token}`,
          'X-com-zoho-invoice-organizationid': 782126836,
        },
        data: JSON.stringify(eInvoiceData),
      };
      this.logger.log('E-invoice data prepared for API', { 
        registrationId, 
        eInvoiceData, 
        eInvoiceInfo
      });
      // const response = await axios(eInvoiceInfo);
      // console.log('auth------- ', response);
      // const einvoiceBody = {
      //   einvoiceQrLink: response.data.invoice.einvoice_details.qr_link,
      //   einvoiceAckNumber: response.data.invoice.einvoice_details.ack_number,
      //   einvoiceAckDate: response.data.invoice.einvoice_details.ack_date,
      //   einvoiceInvRefNum: response.data.invoice.einvoice_details.inv_ref_num,
      //   einvoiceIsCancellable: response.data.invoice.einvoice_details.is_cancellable,
      //   einvoiceStatusFormatted: response.data.invoice.einvoice_details.status_formatted,
      //   einvoiceStatus: response.data.invoice.einvoice_details.status,
      //   einvoiceFormattedStatus: response.data.invoice.einvoice_details.formatted_status,
      //   updatedAt: new Date(),
      // };
      // console.log('einvoiceBody', einvoiceBody);

      // Update the invoice with e-invoice details
      // await this.invoiceRepo.update(
      //   { registrationId },
      //   {
      //     einvoiceQrLink: response.data.invoice.einvoice_details.qr_link || null,
      //     einvoiceAckNumber: response.data.invoice.einvoice_details.ack_number || null,
      //     einvoiceAckDate: response.data.invoice.einvoice_details.ack_date || null,
      //     einvoiceInvRefNum: response.data.invoice.einvoice_details.inv_ref_num || null,
      //     einvoiceIsCancellable: response.data.invoice.einvoice_details.is_cancellable || null,
      //     einvoiceStatusFormatted: response.data.invoice.einvoice_details.status_formatted || null,
      //     einvoiceStatus: response.data.invoice.einvoice_details.status || null,
      //     einvoiceFormattedStatus: response.data.invoice.einvoice_details.formatted_status || null,
      //     updatedAt: new Date(),
      //   }
      // );
      // await this.invoiceRepo.update(
      //   { registrationId },
      //   {
      //     einvoiceQrLink: 'https://delta-node-test.s3.ap-south-1.amazonaws.com/qr.png',
      //     einvoiceAckNumber: '1234567890123456789',
      //     einvoiceAckDate: new Date().toISOString(),
      //     einvoiceInvRefNum: '23f498ee41441ecad30f72ba5b9907506c3df70a17b0e0dff46b76a78640',
      //     updatedAt: new Date(),
      //   }
      // );
      // Here you can implement the logic to send the e-invoice to IRP or save it as needed
      // For now, we will just return the generated data
      return eInvoiceData;

    } catch (error) {
      console.log('Error generating e-invoice', error);
      this.logger.error('E-invoice generation failed', '', { registrationId, error: error.message });
      return null;
    }
  }
  async generateEInvoiceData(registrationId: number): Promise<EInvoiceResponse | null> {
    try {
      this.logger.log('Generating e-invoice data', { registrationId });

      // Get invoice data with all related information
      const invoiceData = await this.getInvoiceDetails(registrationId);

      // console.log('invoiceData', invoiceData);
      
      if (!invoiceData) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }
      // if (invoiceData.gstNumber && invoiceData.isGstRegistered) {
      //   return null; // No e-invoice for GST registered users
      // }

      // Determine tax name based on GST comparison
      const taxName = this.determineTaxName(
        invoiceData.gstNumber,
        invoiceData?.registration?.program?.gstNumber || ''
      );

      this.logger.log('Tax name determined', { 
        taxName, 
        billingGst: invoiceData?.gstNumber,
        programGst: invoiceData?.registration?.program?.gstNumber 
      });

      // console.log('invoiceData einvoice', invoiceData);
      // Process address information
      const addressInfo = this.processAddressInformation(invoiceData);

      // console.log('Address information processed', addressInfo);

      // Build e-invoice response
      const eInvoiceData: EInvoiceResponse = {
        contact: {
          contact_name: invoiceData?.invoiceName,
          company_name: eInvoiceConstants?.COMPANY_NAME,
          currency_code: eInvoiceConstants?.CURRENCY.INDIA,
          billing_address: {
            attention: eInvoiceConstants?.COMPANY_NAME,
            address: addressInfo?.address_1,
            street2: addressInfo?.address_1,
            state_code: addressInfo?.stateCode,
            city: addressInfo?.city,
            state: addressInfo?.state,
            zip: Number(addressInfo.zip ?? ''),
            country: addressInfo?.country,
            phone: Number(addressInfo?.phoneNumber),
          },
          gst_no: invoiceData?.gstNumber ?? '',
          gst_treatment: eInvoiceConstants.GST_TREATMENT,
        },
        invoice: {
          place_of_supply: addressInfo?.placeOfSupplyCode,
          invoice_number: invoiceData?.invoiceSequenceNumber,
          date: invoiceData?.updatedAt 
            ? new Date(invoiceData?.updatedAt).toISOString().split('T')[0] 
            : new Date().toISOString().split('T')[0],
          discount: 0,
          is_discount_before_tax: true,
          discount_type: 'item_level',
          shipping_charge: 0,
          line_items: [
            {
              description: `Fee for training program '${invoiceData?.registration?.allocatedProgram?.name}'`,
              rate: invoiceData?.registration?.allocatedProgram?.basePrice || 0,
              quantity: 1,
              unit: eInvoiceConstants?.UNITS,
              product_type: eInvoiceConstants?.SERVICE,
              hsn_or_sac: eInvoiceConstants?.HSN_CODE,//Number(invoiceData.hsn_or_sac),
              tax_name: taxName,
            },
          ],
          adjustment: 0,
          adjustment_description: '',
          notes: '',
          terms: '',
          subject_content: '',
          seller_gstin: invoiceData?.registration?.program?.gstNumber || '',
          is_inclusive_tax: false,
          tax_rounding: '',
          shipping_charge_tax_name: taxName,
          shipping_charge_sac_code: eInvoiceConstants.SHIPPING_CHARGE_SEC_CODE,
          is_reverse_charge_applied: false,
          is_customer_liable_for_tax: false,
          is_export_with_payment: false,
        },
      };
      console.log('E-invoice data generated', eInvoiceData, eInvoiceData.invoice.line_items);

      // this.logger.log('E-invoice data generated successfully', { 
      //   registrationId,
      //   invoiceNumber: eInvoiceData.invoice.invoice_number,
      //   contactName: eInvoiceData.contact.contact_name
      // });

      return eInvoiceData;

    } catch (error) {
      this.logger.error('Failed to generate e-invoice data', '', { 
        registrationId, 
        error: error.message 
      });
      return null;
    }
  }
  private determineTaxName(billingGst: string, programGst: string): string {
    if (!billingGst || !programGst) {
      return eInvoiceConstants.SHIPPING_CHANRGE_TAX; // Default to intrastate
    }

    const billingStateCode = billingGst.substring(0, 2);
    const programStateCode = programGst.substring(0, 2);

    this.logger.log('GST code comparison', { 
      billingStateCode, 
      programStateCode,
      isInterstate: billingStateCode !== programStateCode
    });

    // If state codes are the same, it's intrastate (CGST + SGST)
    // If state codes are different, it's interstate (IGST)
    return billingStateCode === programStateCode 
      ? eInvoiceConstants.SHIPPING_CHANRGE_TAX 
      : eInvoiceConstants.SHIPPING_INTERSTATE_CHARGE_TAX;
  }
 /**
   * Processes and formats address information
   */
 private processAddressInformation(invoiceData: RegistrationInvoiceDetail) {
  let address_1 = 'No billing address found';
  let address_2 = '';
  let city = '';
  let stateCode = '';
  let state = '';
  let country = 'India';
  let phoneNumber = 0;
  let zip = '';
  let placeOfSupplyCode = '';


  // try {
  //   const registration = await this.paymentRepo.findOne({
  //     where: { registrationId: invoiceData.registrationId }
  //   });
    
    if (invoiceData) {
      address_1 = invoiceData?.invoiceAddress;
      address_2 = invoiceData?.invoiceAddress || '';
      city = invoiceData?.invoiceAddress || '';
      
      // Find state information from GST code
      if (invoiceData?.gstNumber) {
        const gstStateCode = invoiceData?.gstNumber?.substring(0, 2);
        const stateInfo = stateGstCodes.find(code => code.code === gstStateCode);
        
        if (stateInfo) {
          stateCode = stateInfo.state;
          state = stateInfo.stateName;
          placeOfSupplyCode = stateInfo.state;
        }
      }
  
      // Process phone number
      if (invoiceData?.registration?.mobileNumber) {
        const cleanPhone = invoiceData?.registration?.mobileNumber.replace('+91', '').trim();
        phoneNumber = cleanPhone ? Number(cleanPhone) : 0;
      }
  
      // Process ZIP code
      if (invoiceData.zip) {
        zip = invoiceData.zip ?? '';
      }
    }
  
    const addressInfo = {
      address_1,
      city,
      stateCode,
      state,
      country,
      phoneNumber,
      zip,
      placeOfSupplyCode
    };
  
    this.logger.log('Address information processed', { addressInfo });
  
    return addressInfo;
  // }catch (error) {
  //   this.logger.error('Error fetching registration data for address processing', error);
  //  return {
  //     address_1: '',
  //     city: '',
  //     stateCode: '',
  //     state: '',
  //     country: 'India',
  //     phoneNumber: 0,
  //     zip: 0,
  //     placeOfSupplyCode: '',
  //   };
  // }
  

  
}
 async getInvoiceByRegistrationId(registrationId: number): Promise<object | null> {
    try {
      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program',],
      });

      if (!invoice?.invoicePdfUrl) {
        this.logger.log('No invoice found for registration', { registrationId });
        handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, null);
      }

      return { invoicePdfUrl: invoice.invoicePdfUrl };
    } catch (error) {
      this.logger.error('Error getting invoice by registration ID', '', { registrationId, error });
      handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, error);
    }
  }
  async downloadProformaInvoice(registrationId: number): Promise<object | null> {
    this.logger.log('Sending proforma invoice', { registrationId });

    try {
      // Get invoice data
      const invoiceData = await this.getRegistrationDetails(registrationId) // this.getInvoiceDetails(registrationId);
      
      if (!invoiceData) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      this.logger.log('Proforma invoice download successfully', { registrationId });

      return {proFormaInvoice: invoiceData.proFormaInvoicePdfUrl};

    } catch (error) {
      this.logger.error('Proforma invoice download failed', '', { registrationId, error: error.message });
      handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, error);
    }
  }

}