import { Controller, Get, Post, Body, Patch, Param, Delete, UseGuards, Put, HttpStatus, Req, Res } from '@nestjs/common';
import { Response } from 'express';
import { InvoiceService } from './invoice.service';
import { CreateInvoiceDto } from './dto/create-invoice.dto';
import { UpdateInvoiceDto } from './dto/update-invoice.dto';
import { ApiBearerAuth, ApiResponse, ApiSecurity, ApiTags } from '@nestjs/swagger';
import { Roles } from 'src/common/decorators/roles.decorator';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('invoice')
@Controller('invoice')
@UseGuards(CombinedAuthGuard, RolesGuard)
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class InvoiceController {
  constructor(
    private readonly invoiceService: InvoiceService,
    private readonly responseService: ResponseService,
  ) {}
  @Roles('admin', 'relational_manager', 'finance_manager', 'mahatria', 'shoba',)
  @Get('sendto/:registrationId')
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  async create(@Req() req: any, @Res() res: Response, @Param('registrationId') registrationId: number) {
    try {
    
      await this.invoiceService.generateAndSendInvoice(registrationId);
      
      await this.responseService.success(res, "Invoice sent successfully", undefined, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
  @Get('proforma/download/:registrationId')
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  async createWithEmail(@Req() req: any, @Res() res: Response, @Param('registrationId') registrationId: number) {
    try {
      const data = await this.invoiceService.downloadProformaInvoice(registrationId);
      await this.responseService.success(res, "Invoice sent successfully", {data}, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('download/:registrationId')
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.UPDATED })
  async generatePdf(@Req() req: any, @Res() res: Response, @Param('registrationId') registrationId: number) {
    try {
      const invoice = await this.invoiceService.getInvoiceByRegistrationId(registrationId);
      

      return await this.responseService.success(res, "invoice generated successfully", invoice || undefined, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  // @Get()
  // findAll() {
  //   return this.invoiceService.findAll();
  // }

  // @Get(':id')
  // findOne(@Param('id') id: string) {
  //   return this.invoiceService.findOne(+id);
  // }

  // @Patch(':id')
  // update(@Param('id') id: string, @Body() updateInvoiceDto: UpdateInvoiceDto) {
  //   return this.invoiceService.update(+id, updateInvoiceDto);
  // }

  // @Delete(':id')
  // remove(@Param('id') id: string) {
  //   return this.invoiceService.remove(+id);
  // }
}
