import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, IsNull, ILike } from 'typeorm';
import { FormSection } from 'src/common/entities/form-section.entity';
import { CommonDataService } from 'src/common/services/commonData.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

@Injectable()
export class FormSectionRepository {
  constructor(
    @InjectRepository(FormSection)
    private readonly formSectionRepo: Repository<FormSection>,
    private readonly commonDataService: CommonDataService,
  ) {}

  async findAll(limit: number, offset: number, searchText: string) {
    try {
      const selectFields: (keyof FormSection)[] = [
        'id',
        'name',
        'description',
        'displayOrder',
        'createdAt',
        'updatedAt',
      ];
      let whereClause: any = { deletedAt: IsNull() };
       if (searchText) {
        whereClause.name = ILike(`%${searchText}%`);
       }

      const data = await this.commonDataService.get(
        this.formSectionRepo,
        selectFields,
        whereClause,
        limit,
        offset,
        { displayOrder: 'ASC', id: 'ASC' },
      );

      const total = await this.formSectionRepo.count({ where: whereClause });
      const totalPages = Math.ceil(total / limit);

      return {
        data,
        pagination: {
          totalPages,
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: +limit,
          totalRecords: total,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.FORM_SECTION_FIND_FAILED, error);
    }
  }

  async findAllNoLimit() {
    try {
      const selectFields: (keyof FormSection)[] = [
        'id',
        'name',
        'description',
        'displayOrder',
        'createdAt',
        'updatedAt',
      ];
      const whereClause: any = { deletedAt: IsNull() };
      return await this.formSectionRepo.find({
        select: selectFields,
        where: whereClause,
        order: { displayOrder: 'ASC', id: 'ASC' }
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.FORM_SECTION_FIND_FAILED, error);
    }
  }
}
