import {
  Controller, Get, Query, Res, HttpStatus, HttpCode, UseGuards
} from '@nestjs/common';
import { Response } from 'express';
import { FormSectionService } from './form-section.service';
import { ApiTags, ApiOperation, ApiResponse, ApiQuery, ApiBearerAuth, ApiSecurity } from '@nestjs/swagger';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { Roles } from 'src/common/decorators/roles.decorator';
import { FirebaseAuthGuard } from 'src/auth/firebase-auth.guard';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { AppLoggerService } from 'src/common/services/logger.service';
import { FORM_SECTION_MESSAGES } from 'src/common/constants/strings-constants';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';
@ApiTags('form-section')
@Controller('form-section')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class FormSectionController {
  constructor(
    private readonly formSectionService: FormSectionService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Retrieves all form sections with optional pagination and search functionality.
   * @param limit - Number of records per page (default: 10).
   * @param offset - Offset for pagination (default: 0).
   * @param searchText - Optional search text to filter sections by name.
   * @returns A paginated list of form sections and metadata.
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: 'Get all form sections with optional search and pagination' })
  @ApiQuery({ name: 'limit', required: false, type: Number })
  @ApiQuery({ name: 'offset', required: false, type: Number })
  @ApiQuery({ name: 'searchText', required: false, type: String })
  @ApiResponse({ status: HttpStatus.OK, description: 'Form sections retrieved successfully' })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: 'Internal server error' })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Res() res: Response
  ) {
    this.logger.log(FORM_SECTION_MESSAGES.FIND_ALL_REQUEST_RECEIVED, { limit, offset, searchText });
    try {
      const result = await this.formSectionService.findAll(limit, offset, searchText);
      this.logger.log(FORM_SECTION_MESSAGES.FOUND);
      return this.responseService.success(
        res,
        FORM_SECTION_MESSAGES.FOUND,
        result
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}