import { Injectable, Logger } from '@nestjs/common';
import { SendSingleEmailDto, SendBulkEmailDto } from './dto/email-communication.dto';
import {
  SendTemplateMessageDto,
  SendBulkTemplateMessageDto,
} from './dto/whatsapp-communication.dto';
import { SendMailClient } from 'zeptomail';
import axios from 'axios';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { delay, chunkArray } from 'src/common/utils/common.util';
import { CommunicationRepository } from './communication.repository';
import { CommunicationTypeEnum } from 'src/common/enum/communication-type.enum';
import { CommunicationStatusEnum } from 'src/common/enum/communication-status.enum';
import { CreateCommunicationTrackDto } from './dto/communication-track.dto';

@Injectable()
export class CommunicationService {
  private readonly logger = new Logger(CommunicationService.name);
  private readonly emailApiKey = process.env.ZEPTO_EMAIL_TOKEN;
  private readonly emailApiUrl = process.env.ZEPTO_EMAIL_URL;
  private readonly emailWebhookSecret = process.env.ZEPTO_EMAIL_WEBHOOK_SECRET;
  private readonly whatsappApiUrl = process.env.WATI_BASE_URL;
  private readonly whatsappAccessToken = process.env.WATI_TOKEN;
  private readonly whatsappWebhookSecret = process.env.WATI_WEBHOOK_SECRET;

  private readonly client: SendMailClient;

  constructor(
    // private readonly configService: ConfigService,
    // private readonly loggerService: Logger,
    private readonly trackRepo: CommunicationRepository,
    // private readonly httpService: HttpService,
  ) {
    this.client = new SendMailClient({
      url: process.env.ZEPTO_EMAIL_URL,
      token: process.env.ZEPTO_EMAIL_TOKEN,
    });
    // Validate required configurations
    if (!this.emailApiKey) {
      this.logger.warn('ZeptoMail API key is not configured');
    }
    if (!this.whatsappApiUrl || !this.whatsappAccessToken) {
      this.logger.warn('WATI API URL or access token is not configured');
    }
  }

  // ==================== EMAIL METHODS ====================

  /**
   * Send a single email using ZeptoMail API
   */
  async sendSingleEmail(emailData: SendSingleEmailDto): Promise<any> {
    // if (!this.emailApiKey) {
    //   throw new BadRequestException('Email service is not configured');
    // }

    try {
      this.logger.log(
        `Sending single email to: ${emailData.templateKey} , ${emailData.to.emailAddress}, ${emailData.from.address}`,
      );

      const mailPayload: any = {
        template_key: emailData.templateKey,
        from: {
          address: emailData.from.address,
          name: emailData.from.name || 'Infinitheism',
        },
        to: [
          {
            email_address: {
              address: emailData.to.emailAddress,
              name: emailData.to.name || '',
            },
            // merge_info: JSON.stringify(merge_info),
          },
        ],
        merge_info: JSON.stringify(emailData.mergeInfo),
        subject: 'Test Email',
        attachments: emailData.attachments || [],
      };

      // Add cc if provided
      if (emailData.cc && emailData.cc.length > 0) {
        mailPayload.cc = emailData.cc.map((ccRecipient) => ({
          email_address: {
            address: ccRecipient.emailAddress,
            name: ccRecipient.name || '',
          },
        }));
      }

      // Add bcc if provided
      if (emailData.bcc && emailData.bcc.length > 0) {
        mailPayload.bcc = emailData.bcc.map((bccRecipient) => ({
          email_address: {
            address: bccRecipient.emailAddress,
            name: bccRecipient.name || '',
          },
        }));
      }

      const response = await this.client.sendMailWithTemplate(mailPayload);
      //this.logger.log(`Email sent successfully to: ${emailData.to.emailAddress}`, { response });

      if (emailData.trackinfo && emailData.trackinfo.registrationId) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: emailData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.EMAIL,
          stats: CommunicationStatusEnum.SUCCESS,
          failedReason: response || '',
        };

        if (emailData.trackinfo.createdBy) {
          trackData.createdBy = emailData.trackinfo.createdBy;
        }

        if (emailData.trackinfo.updatedBy) {
          trackData.updatedBy = emailData.trackinfo.updatedBy;
        }
        await this.updateCommunicationTrack(trackData);
      }
      return response.data;
    } catch (error) {
      this.logger.error(`Failed to send email: ${error}`, error.stack);
      if (emailData.trackinfo) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: emailData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.EMAIL,
          stats: CommunicationStatusEnum.FAILED,
          failedReason: error.message || 'Unknown error',
        };

        if (emailData.trackinfo.createdBy) {
          trackData.createdBy = emailData.trackinfo.createdBy;
        }

        if (emailData.trackinfo.updatedBy) {
          trackData.updatedBy = emailData.trackinfo.updatedBy;
        }
        await this.updateCommunicationTrack(trackData);
        // await this.trackRepo.createEntity(trackData)
      }
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_EMAIL, error);
    }
  }

  /**
   * Send bulk emails using ZeptoMail API
   */
  async sendBulkEmail(emailData: SendBulkEmailDto): Promise<any> {
    // if (!this.emailApiKey) {
    //   throw new BadRequestException('Email service is not configured');
    // }

    try {
      this.logger.log(`Sending bulk email to ${emailData.to} recipients`);

      // const merge_info = {
      //   course_name: 'Infinitheism',
      //   prg_venue: 'Hyderabad telangan.', // This should be a single string, not an object. It should be like `venue_name, venue_address, city, state, country` and not `venue_name: ${}, venue_address: ${}, city: ${}, state: ${}, country:
      //   Email_Data_Checkin_and_Checkout: `Check-in Time: From ${Date.now()} to ${Date.now()}.
      //   \n Check-out Time: Latest by ${Date.now()}.`,
      //   reg_mobile: '+91 7659950042',
      //   pay_method: 'ONLINE PAYMENT',
      //   Email_Data_Program_Date: `${Date.now()}` + ' to ' + `${Date.now()}`,
      //   reg_number: 'INF-0001',
      //   reg_fullname: 'Harsha',
      //   reg_email: 'harsha@divami.com',
      //   firstPara: `Please find enclosed the invoice in acknowledgement of your payment for 'Infinitheism' program.`,
      // };
      const toRecipients = emailData.to.map((recipient) => ({
        email_address: {
          address: recipient.emailAddress,
          name: recipient.name || '',
        },
        // merge_info: recipient.mergeInfo || {},
      }));
      this.logger.log(`Preparing to send bulk email to ${JSON.stringify(toRecipients)} recipients`);

      const response = await this.client.sendMailWithTemplate({
        template_key: emailData.templateKey,
        from: {
          address: emailData.from.address,
          name: emailData.from.name || 'Infinitheism',
        },
        to: toRecipients,
        merge_info: JSON.stringify(emailData.mergeInfo),
        subject: 'Test Email',
        attachments: emailData.attachments || [],
      });

      this.logger.log(`Bulk email sent successfully to ${emailData.to.length} recipients`);
      return response.data;
    } catch (error) {
      this.logger.error(`Failed to send bulk email: ${error}`, error.stack);
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_BULK_EMAIL, error);
    }
  }

  async updateCommunicationTrack(trackInfo: CreateCommunicationTrackDto): Promise<void> {
    try {
      this.logger.log(
        `Updating communication track info for registration ID: ${trackInfo.registrationId}`,
      );

      await this.trackRepo.createEntity(trackInfo);
      this.logger.log(
        `communication track info updated successfully for registration ID: ${trackInfo.registrationId}`,
      );
    } catch (error) {
      this.logger.error(`Failed to update communication track info: ${error.message}`, error.stack);
      // handleKnownErrors(ERROR_CODES.FAILED_TO_UPDATE_EMAIL_TRACK, error);
    }
  }

  /**
   * Validate ZeptoMail webhook signature
   */
  validateEmailWebhookSignature(payload: string, signature: string): boolean {
    try {
      if (!this.emailWebhookSecret) {
        this.logger.warn('Email webhook secret not configured, skipping validation');
        return true; // Allow if no secret configured
      }

      const crypto = require('crypto');
      const expectedSignature = crypto
        .createHmac('sha256', this.emailWebhookSecret)
        .update(payload)
        .digest('hex');

      const providedSignature = signature.replace('sha256=', '');

      return crypto.timingSafeEqual(
        Buffer.from(providedSignature, 'hex'),
        Buffer.from(expectedSignature, 'hex'),
      );
    } catch (error) {
      this.logger.error(`Email webhook signature validation failed: ${error.message}`);
      return false;
    }
  }

  // ==================== WHATSAPP METHODS ====================

  /**
   * Send WhatsApp template message using WATI API
   */
  async sendTemplateMessage(messageData: SendTemplateMessageDto): Promise<any> {

    try {
      this.logger.log(`Sending WhatsApp template message to: ${messageData.whatsappNumber}`);
      const url = `${process.env.WATI_BASE_URL}/sendTemplateMessage?whatsappNumber=${messageData.whatsappNumber}`;
      const payload = {
        template_name: messageData.templateName,
        broadcast_name: messageData.broadcastName,
        parameters:
          messageData.parameters?.map((param) => ({
            name: param.name,
            value: param.value,
          })) || [],
      };
      const headers = {
        Authorization: `Bearer ${process.env.WATI_TOKEN}`,
        'Content-Type': 'application/json',
      };

      const response = await axios.post(url, payload, { headers });

      this.logger.log(
        `WhatsApp template message sent successfully to: ${messageData.whatsappNumber} with response: ${JSON.stringify(response.data)}`,
      );
      if (messageData.trackinfo) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: messageData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.WHATSAPP,
          stats: CommunicationStatusEnum.SUCCESS,
          failedReason: JSON.stringify(response.data) || '',
        };

        if (messageData.trackinfo.createdBy) {
          trackData.createdBy = messageData.trackinfo.createdBy;
        }

        if (messageData.trackinfo.updatedBy) {
          trackData.updatedBy = messageData.trackinfo.updatedBy;
        }
        this.updateCommunicationTrack(trackData);
        // await this.trackRepo.createEntity(trackData)
      }
      return response.data;
    } catch (error) {
      this.logger.error(`Failed to send WhatsApp template message: ${error.message}`, error.stack);
      if (messageData.trackinfo) {
        const trackData: CreateCommunicationTrackDto = {
          registrationId: messageData.trackinfo.registrationId,
          typ: CommunicationTypeEnum.WHATSAPP,
          stats: CommunicationStatusEnum.FAILED,
          failedReason: error.message || 'Unknown error',
        };

        if (messageData.trackinfo.createdBy) {
          trackData.createdBy = messageData.trackinfo.createdBy;
        }

        if (messageData.trackinfo.updatedBy) {
          trackData.updatedBy = messageData.trackinfo.updatedBy;
        }
        this.updateCommunicationTrack(trackData);
        // await this.trackRepo.createEntity(trackData)
      }
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_WHATSAPP_MESSAGE, error);
    }
  }

  /**
   * Send bulk WhatsApp template messages using WATI API
   */
  async sendBulkTemplateMessage(messageData: SendBulkTemplateMessageDto): Promise<any> {

    try {
      this.logger.log(
        `Sending bulk WhatsApp template messages to ${messageData.recipients.length} recipients`,
      );

      const results: { status: string; [key: string]: any }[] = [];
      const failedRecipients: { whatsappNumber: string; name?: string; error: any }[] = [];

      // Process recipients in batches to avoid rate limiting
      const batchSize = 10; // Adjust based on WATI rate limits
      const batches = chunkArray(messageData.recipients, batchSize);
      const watiBaseUrl = process.env.WATI_BASE_URL;
      const watiApiKey = process.env.WATI_TOKEN;
      const url = `${watiBaseUrl}/sendTemplateMessages`;
      for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
        const batch = batches[batchIndex];
        this.logger.log(
          `Processing batch ${batchIndex + 1}/${batches.length} with ${batch.length} recipients`,
        );

        const batchPromises = batch.map(async (recipient) => {
          try {
            const payload = {
              template_name: messageData.templateName,
              broadcast_name: messageData.broadcastName,
              receivers: 
              [
                recipient,
                ...(messageData.globalParameters || []),
              ]
            };

            const headers = {
              Authorization: `Bearer ${watiApiKey}`,
              'Content-Type': 'application/json',
            };
            this.logger.log(
              `Sending template message to ${recipient.whatsappNumber} with payload: ${JSON.stringify(payload)}`,
            );


            const response = await axios.post(url, payload, { headers });
            this.logger.log(
              `WhatsApp template message sent successfully to: ${recipient.whatsappNumber} and ${JSON.stringify(response.data)}`,
            );

            return {
              whatsappNumber: recipient.whatsappNumber,
              name: recipient,
              status: 'success',
              response: response.data,
            };
          } catch (error) {
            this.logger.error(
              `Failed to send template message to ${recipient.whatsappNumber}: ${error.message}`,
            );
            failedRecipients.push({
              whatsappNumber: recipient.whatsappNumber,
              // name: recipient,
              error: error.response?.data?.message || error.message,
            });
            return {
              whatsappNumber: recipient.whatsappNumber,
              name: recipient,
              status: 'failed',
              error: error.response?.data?.message || error.message,
            };
          }
        });

        const batchResults = await Promise.allSettled(batchPromises);
        results.push(
          ...batchResults
            .map((result) => (result.status === 'fulfilled' ? result.value : null))
            .filter((item) => item !== null),
        );

        // Add delay between batches to respect rate limits
        if (batchIndex < batches.length - 1) {
          await delay(1000); // 1 second delay between batches
        }
      }

      const successCount = results.filter((r) => r && r.status === 'success').length;
      const failedCount = failedRecipients.length;

      this.logger.log(
        `Bulk WhatsApp template messages completed: ${successCount} successful, ${failedCount} failed`,
      );

      return {
        total: messageData.recipients.length,
        successful: successCount,
        failed: failedCount,
        results,
        failedRecipients,
      };
    } catch (error) {
      this.logger.error(
        `Failed to send bulk WhatsApp template messages: ${error.message}`,
        error.stack,
      );
      handleKnownErrors(ERROR_CODES.FAILED_TO_SEND_BULK_WHATSAPP_MESSAGE, error);
    }
  }
  // /**
  //  * Split array into chunks of specified size
  //  */
  // private chunkArray<T>(array: T[], chunkSize: number): T[][] {
  //   const chunks: T[][] = [];
  //   for (let i = 0; i < array.length; i += chunkSize) {
  //     chunks.push(array.slice(i, i + chunkSize));
  //   }
  //   return chunks;
  // }

  //  /**
  //  * Add delay between operations
  //  */
  //  private delay(ms: number): Promise<void> {
  //   return new Promise(resolve => setTimeout(resolve, ms));
  // }
  

  /**
   * Validate WATI webhook token
   */
  validateWhatsAppWebhookToken(providedToken: string): boolean {
    if (!this.whatsappWebhookSecret) {
      this.logger.warn('WhatsApp webhook secret not configured, skipping validation');
      return true; // Allow if no secret configured
    }

    return providedToken === this.whatsappWebhookSecret;
  }
}
