import {
  Controller,
  Post,
  Body,
  Headers,
  RawBody,
  Query,
  Logger,
  BadRequestException,
  HttpCode,
  HttpStatus,
} from '@nestjs/common';
import { CommunicationService } from './communication.service';
import { SendSingleEmailDto, SendBulkEmailDto, EmailWebhookDto } from './dto/email-communication.dto';
import { SendTemplateMessageDto, SendSessionMessageDto, WhatsAppWebhookDto, SendBulkTemplateMessageDto } from './dto/whatsapp-communication.dto';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';

@Controller('communication')
export class CommunicationController {
  private readonly logger = new Logger(CommunicationController.name);

  constructor(private readonly communicationService: CommunicationService) {}

  // ==================== EMAIL ENDPOINTS ====================

  /**
   * Send a single email
   */
  @Post('email/send')
  async sendSingleEmail(@Body() emailData: SendSingleEmailDto) {
    this.logger.log(`Received request to send single email to:`);
    
    
    const templateKey = process.env.ZEPTO_EMAIL_TEMPLATE_ID // '2d6f.2a60732b5ed84938.k1.2b4f1a40-49b4-11ef-a499-5254000b1a0e.190e49c41e4'; // Replace with the actual template key

    const emailDataWithDefaults: SendSingleEmailDto = {
      to: {
        emailAddress: 'harsha@divami.com',
        name: 'Harsha',
        mergeInfo: {
          reg_fullname: "Harsha",
        } ,
      },
      from: {
        address: 'noreply@infinitheism.com',
        name: 'Infinitheism',
      },
      subject: emailData.subject || 'Default Subject',
      htmlbody: emailData.htmlbody || '<p>This is a default email body</p>',
      textbody: emailData.textbody || 'This is a default email body',
      templateKey: templateKey,
      mergeInfo: {
        reg_fullname: "Harsha",
      },
      attachments:  [],
      trackinfo: {
        registrationId: emailData.trackinfo.registrationId,
        createdBy: emailData.trackinfo.createdBy,
        updatedBy: emailData.trackinfo.updatedBy,
      }
    };
    return await this.communicationService.sendSingleEmail(emailDataWithDefaults);
  }

  /**
   * Send bulk emails
   */
  @Post('email/send-bulk')
  async sendBulkEmail(@Body() emailData: SendBulkEmailDto) {
    this.logger.log(`Received request to send bulk email to recipients`);
    const templateKey = process.env.ZEPTO_EMAIL_TEMPLATE_ID // '2d6f.2a60732b5ed84938.k1.2b4f1a40-49b4-11ef-a499-5254000b1a0e.190e49c41e4'; // Replace with the actual template key
    const emailDataWithDefaults: SendBulkEmailDto = {
      to: [
        {
          emailAddress: 'harsha@divami.com',
          name: 'Harsha',
          mergeInfo: {
            reg_fullname: "Harsha",
          },
        },
        {
          emailAddress: 'harshavardhanreddya1@gmail.com',
          name: 'Harsha',
          mergeInfo: {
            reg_fullname: "Harsha",
          },
        }
      ],
      from: {
        address: 'noreply@infinitheism.com',
        name: 'Infinitheism',
      },
      subject: emailData.subject || 'Default Subject',
      htmlbody: emailData.htmlbody || '<p>This is a default email body</p>',
      textbody: emailData.textbody || 'This is a default email body',
      templateKey: templateKey,
      mergeInfo: {
        reg_fullname: "Harsha",
      },
    };
    
    return await this.communicationService.sendBulkEmail(emailDataWithDefaults);
  }

  /**
   * Handle ZeptoMail webhook events
   */
  @Post('email/webhook')
  @HttpCode(HttpStatus.OK)
  async handleEmailWebhook(
    @RawBody() rawBody: Buffer,
    @Headers('x-zepto-signature') signature: string,
    @Body() webhookData: EmailWebhookDto,
  ) {
    try {
      this.logger.log(`Received email webhook: ${webhookData.event} for ${webhookData.email}`);

      // Validate webhook signature
      const payload = rawBody.toString();
      const isValid = this.communicationService.validateEmailWebhookSignature(payload, signature);

      if (!isValid) {
        this.logger.error('Invalid email webhook signature received');
        throw new BadRequestException('Invalid webhook signature');
      }

      // Process webhook based on event type
      await this.processEmailWebhook(webhookData);

      return { 
        status: 'success', 
        message: 'Email webhook processed successfully',
        event: webhookData.event,
        email: webhookData.email 
      };
    } catch (error) {
      this.logger.error(`Failed to process email webhook: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.FAILED_TO_HANDLE_EMAIL_WEBHOOK , error);
    }
  }

  // ==================== WHATSAPP ENDPOINTS ====================

  /**
   * Send WhatsApp template message
   */
  @Post('whatsapp/send-template')
  async sendTemplateMessage(@Body() messageData: SendTemplateMessageDto) {
    this.logger.log(`Received request to send WhatsApp template message to: `);
    const tempMessageData: SendTemplateMessageDto = {
      whatsappNumber: '+917659950042',
      templateName: 'ip_registration_open_dev_v3',
      broadcastName: 'ip_registration_open_dev_v3',
      parameters: [{
        name: 'first_name',
        value: 'Harsha',
      },
      { 
        name: 'last_name', 
        value: 'Welcome to Infinitheism'
      },
      {
        name: 'infinipath_date',
        value: "formatShortDate(webinar.startAt)",
      },]
    };

    const tempBulkMessageData: SendBulkTemplateMessageDto = 
      {
        recipients: [
          {
            whatsappNumber: '917659950042',
            // name: 'Harsha',
            customParams: [{
              name: 'first_name',
              value: 'Harsha',
            },
            { 
              name: 'last_name', 
              value: 'Welcome to Infinitheism'
            },
            {
              name: 'infinipath_date',
              value: "formatShortDate(webinar.startAt)",
            },]
          },
          {
            whatsappNumber: '919876543210',
            // name: 'John Doe',
            customParams: [{
              name: 'first_name',
              value: 'John',
            },
            { 
              name: 'last_name', 
              value: 'Doe'
            },
            {
              name: 'infinipath_date',
              value: "formatShortDate(webinar.startAt)",
            },]
          }
        ],
        templateName: 'ip_registration_open_dev_v3',
        broadcastName: 'ip_registration_open_dev_v3',
        globalParameters: []
      }
    return await this.communicationService.sendBulkTemplateMessage(tempBulkMessageData);

    // return await this.communicationService.sendTemplateMessage(tempMessageData);
  }



  /**
   * Handle WATI WhatsApp webhook events
   */
  @Post('whatsapp/webhook')
  @HttpCode(HttpStatus.OK)
  async handleWhatsAppWebhook(
    @Body() webhookData: WhatsAppWebhookDto,
    @Query('token') token?: string,
  ) {
    try {
      this.logger.log(`Received WhatsApp webhook: ${webhookData.eventType} for ${webhookData.whatsappNumber}`);

      // Validate webhook token if provided
      if (token && !this.communicationService.validateWhatsAppWebhookToken(token)) {
        this.logger.error('Invalid WhatsApp webhook token received');
        throw new BadRequestException('Invalid webhook token');
      }

      // Process webhook based on event type
      await this.processWhatsAppWebhook(webhookData);

      return { 
        status: 'success', 
        message: 'WhatsApp webhook processed successfully',
        eventType: webhookData.eventType,
        whatsappNumber: webhookData.whatsappNumber 
      };
    } catch (error) {
      this.logger.error(`Failed to process WhatsApp webhook: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.FAILED_TO_HANDLE_WHATSAPP_WEBHOOK , error);
    }
  }


  /**
   * Process email webhook events
   */
  private async processEmailWebhook(webhookData: EmailWebhookDto): Promise<void> {
    switch (webhookData.event) {
      case 'delivered':
        this.logger.log(`Email delivered to: ${webhookData.email}, MessageID: ${webhookData.message_id}`);
        // Add your delivery processing logic here
        // Example: Update database, trigger analytics, etc.
        break;

      case 'open':
        this.logger.log(`Email opened by: ${webhookData.email}, MessageID: ${webhookData.message_id}`);
        // Add your open tracking logic here
        // Example: Track engagement metrics, trigger follow-up campaigns
        break;

      case 'click':
        this.logger.log(`Email clicked by: ${webhookData.email}, URL: ${webhookData.click_url}, MessageID: ${webhookData.message_id}`);
        // Add your click tracking logic here
        // Example: Track link performance, user behavior analytics
        break;

      case 'bounce':
        this.logger.warn(`Email bounced for: ${webhookData.email}, Type: ${webhookData.bounce_type}, Reason: ${webhookData.bounce_reason}`);
        // Add your bounce handling logic here
        // Example: Mark email as invalid, remove from mailing list, alert admins
        break;

      case 'spam':
        this.logger.warn(`Email marked as spam by: ${webhookData.email}, MessageID: ${webhookData.message_id}`);
        // Add your spam handling logic here
        // Example: Update sender reputation, review email content
        break;

      case 'unsubscribed':
        this.logger.log(`User unsubscribed: ${webhookData.email}, MessageID: ${webhookData.message_id}`);
        // Add your unsubscribe handling logic here
        // Example: Update user preferences, remove from active lists
        break;

      default:
        this.logger.log(`Unhandled email event: ${webhookData.event} for ${webhookData.email}`);
    }
  }

  /**
   * Process WhatsApp webhook events
   */
  private async processWhatsAppWebhook(webhookData: WhatsAppWebhookDto): Promise<void> {
    switch (webhookData.eventType) {
      case 'sent':
        this.logger.log(`WhatsApp message sent to: ${webhookData.whatsappNumber}, MessageID: ${webhookData.messageId}`);
        // Add your sent message processing logic here
        // Example: Update message status in database
        break;

      case 'delivered':
        this.logger.log(`WhatsApp message delivered to: ${webhookData.whatsappNumber}, MessageID: ${webhookData.messageId}`);
        // Add your delivery processing logic here
        // Example: Update delivery status, trigger next action in sequence
        break;

      case 'read':
        this.logger.log(`WhatsApp message read by: ${webhookData.whatsappNumber}, MessageID: ${webhookData.messageId}`);
        // Add your read receipt processing logic here
        // Example: Track engagement, update customer interaction history
        break;

      case 'failed':
        this.logger.error(`WhatsApp message failed for: ${webhookData.whatsappNumber}, Status: ${webhookData.status}, MessageID: ${webhookData.messageId}`);
        // Add your failure handling logic here
        // Example: Retry logic, alert admins, update user status
        break;

      case 'message':
        this.logger.log(`Incoming WhatsApp message from: ${webhookData.whatsappNumber}, Message: "${webhookData.message}", Group: ${webhookData.isGroup}`);
        // Add your incoming message processing logic here
        // Example: Auto-reply logic, customer service routing, chatbot integration
        break;

      default:
        this.logger.log(`Unhandled WhatsApp event: ${webhookData.eventType} for ${webhookData.whatsappNumber}`);
    }
  }

}
