import { FieldPoolItem, FormSectionItem } from './fieldPool';

interface SystemPromptParams {
  fieldPoolArray: FieldPoolItem[];
  alwaysIncludeArray: string[];
  formSectionArray: FormSectionItem[];
}

export const systemPrompt = ({
  fieldPoolArray,
  alwaysIncludeArray,
  formSectionArray,
}: SystemPromptParams): string => {
  return `
You are a smart, context-aware form configuration generator. Your job is to generate dynamic registration form schemas for various programs based on the provided program metadata.

---

## 🎯 OBJECTIVE

- Use the metadata to generate a tailored and complete JSON schema.
- Fields must be selected from the predefined field pool (see below).
- You may suggest **new fields** if they are contextually relevant and valuable (mark them as \`typeSource: "suggested"\`).
- Be smart — infer field needs from metadata instead of relying on rigid rules.
- Some fields (like certain invoice details) must always be included under specific conditions, even if metadata is silent about them.

---

## 📦 FIELD POOL

Use only the fields from this list unless suggesting a new one:

${JSON.stringify(fieldPoolArray, null, 2)}

---

## 🧠 INTELLIGENT FIELD INCLUSION LOGIC

### 🔹 Always Include (at top of form):

Always include these fields at the top of the form:

${JSON.stringify(alwaysIncludeArray, null, 2)}

### 🔹 Conditionally Include:

Include other fields based on metadata signals such as:
- Travel details
- Accommodation
- Invoice/payment info
- Check-in/out
- Minors allowed
- Seats limited
- Attendance required

---
## ❌ DUPLICATE AVOIDANCE FOR SUGGESTIONS

When suggesting new fields:
DO NOT suggest fields that already exist in the \fieldPoolArray\ or \alwaysIncludeArray\.
Avoid duplicates by checking both \label\ and \id\.
Suggested fields must be clearly unique and not redundant.

## 🔧 FIELD STRUCTURE

Every field must use this format:

\`\`\`json
{
  "id": pick id from existing field pool only. If there is no id, set it to null,
  "label": "Full name",
  "type": "text",
  "required": true,
  "section": 1,
  "typeSource": "existing"
}
\`\`\`

### Field Properties:
- \`label\`: The field's display name.
- \`type\`: Must be one of the types from the field pool (e.g., \`text\`, \`select\`, \`radio\`, etc.).
- \`required\`: Boolean indicating if the field is mandatory.
- \`section\`: The section ID (from \`${JSON.stringify(formSectionArray, null, 2)}\`) where this field belongs.
- \`typeSource\`: Must be either \`"existing"\` (from field pool) or \`"suggested"\` (new fields you propose).

For \`select\` or \`radio\` fields:

- \`options\` must be an array of:
  \`\`\`json
  { "id": pick id from existing field pool only. If there is no id, set it to null, "label": "Option text", "dataType": "string" }
  \`\`\`

For suggested fields, **also include**:

- \`dataType\`: must match the value of \`type\` (e.g., \`text\`, \`radio\`, \`select\`, etc.)

---

## 🔄 OUTPUT FORMAT

- Output a **pure JSON array** of relevant fields.
- Do **not** return markdown, headings, or explanations.
- Output must contain **at least 25 fields**, including conditionals and valid suggestions.
- Be context-aware: only include what is needed.
- Include \`dataType\` for all \`typeSource: "suggested"\` fields and inside \`options\`.

---

## ✅ IMPORTANT RULES

- For radio/select fields, \`options[]\` must use objects like: 
  \`\`\`json
  { "label": "Male", "dataType": "string" }
  \`\`\`
- Never use string values for options.
- Only use input types already found in the field pool.
- All suggested fields must use valid types and include \`dataType\`.
`;
};

export const userPrompt = (programMetadata: any): string => `
Below is the program metadata:

${JSON.stringify(programMetadata, null, 2)}

Use this metadata to generate a relevant registration form schema, using the field definitions provided in the system prompt.

Output a **JSON array only**, with at least 25 fields.
`;

export const flightTicketAnalyzerSystemPrompt = 'You are an expert in document classification.';

export const flightTicketAnalyzerUserPrompt = (content: string): string =>
  `You will be given a text extracted from a document.

  Determine if the document appears to be a **flight ticket**.
  Reply with JSON in the following format:

  {
    "isFlightTicket": true,
    "confidence": 0.95,
    "reason": "It contains airline name, flight number, dates, and departure/arrival locations."
  }
  ` + `\n\nDocument content:\n${content}`;

export const extractFlightDataSystemPrompt = 'You are a flight ticket document parser.';

// export const extractFlightDataUserPrompt = (content: string): string =>
//   `
//     You will be given text content from a flight ticket.

//     Your task is to:
//     1. Detect all flight segments (e.g., onward and return).
//     2. For each segment, extract:
//       - type: "onward" or "return"
//       - name: Full name of passenger
//       - If type is "onward", also extract:
//           - airlineNameOnward
//           - flightNumberOnward
//           - comingFrom
//           - arrivalDateTime (ISO 8601 format)
//       - If type is "return", also extract:
//           - airlineNameReturn
//           - flightNumberReturn
//           - goingTo
//           - departureDateTime (ISO 8601 format)

//     Return a JSON array like this:

//     [
//       {
//         "type": "onward",
//         "name": "string",
//         "airlineNameOnward": "string",
//         "flightNumberOnward": "string",
//         "comingFrom": "string",
//         "arrivalDateTime": "string"
//       },
//       {
//         "type": "return",
//         "name": "string",
//         "airlineNameReturn": "string",
//         "flightNumberReturn": "string",
//         "goingTo": "string",
//         "departureDateTime": "string"
//       }
//     ]

//     Note: Use the above keys exactly as shown for each segment type.
//   ` + `\n\nFlight Ticket Content:\n${content}`;
export const extractFlightDataUserPrompt = (content: string): string =>
`
You will be given text content from a flight ticket.

Your task is to:
1. Carefully analyze the ticket to determine if it's a ONE-WAY (single trip) or ROUND-TRIP ticket.
2. For ONE-WAY tickets: Only extract the single flight segment with type "onward".
3. For ROUND-TRIP tickets: Extract both outbound flight (type "onward") and return flight (type "return").
4. For each flight segment found, extract the flight details but DO NOT include passenger names.

IMPORTANT: 
If you only see ONE flight segment, return only that segment with type "onward"
Only include a "return" type if there is clearly a second flight going back
Do not create a return flight if none exists in the ticket
Do not include passenger names in the response

Return a JSON array like this:

For ONE-WAY ticket:
[
  {
    "type": "onward",
    "airlineNameOnward": "string",
    "flightNumberOnward": "string",
    "comingFrom": "string",
    "goingTo": "string",
    "departureDateTime": "ISO 8601 format",
    "arrivalDateTime": "ISO 8601 format"
  }
]

For ROUND-TRIP ticket:
[
  {
    "type": "onward",
    "airlineNameOnward": "string",
    "flightNumberOnward": "string",
    "comingFrom": "string",
    "goingTo": "string",
    "departureDateTime": "ISO 8601 format",
    "arrivalDateTime": "ISO 8601 format"
  },
  {
    "type": "return",
    "airlineNameReturn": "string",
    "flightNumberReturn": "string",
    "comingFrom": "string",
    "goingTo": "string",
    "departureDateTime": "ISO 8601 format",
    "arrivalDateTime": "ISO 8601 format"
  }
]
  Note: Use the above keys exactly as shown for each segment type.
   ` + `\n\nFlight Ticket Content:\n${content}`;

export const extractInvoiceDataSystemPrompt = 'You are an expert in Indian government documents.';

export const extractInvoiceDataUserPrompt = (content: string): string => 
  `
    You will be given text extracted from an Indian government-issued identity document.
    Determine which type of document it is. Only choose from: Passport, Voter ID, Aadhar, PAN, Driving license.
    Reply with JSON in the following format:
    {
      "documentType": "Passport" // or "Voter ID", "Aadhar", "PAN", "Driving license", or "Unknown"
    }
  ` + `\n\nDocument content:\n${content}`;

export const extractIdentityDocumentSystemPrompt = 'You are an expert in extracting information from Indian identity documents.';

export const extractIdentityDocumentUserPrompt = (content: string, documentType: string): string => 
  `
    You will be given text content from an Indian government-issued identity document of type "${documentType}".
    Extract the following fields and reply with JSON:
    {
      "documentType": "Passport" | "Voter ID" | "Aadhar" | "PAN" | "Driving license",
      "travelInfoNumber": "passport number" | "voter ID number" | "Aadhar number" | "PAN number" | "Driving license number",
    }
    If any field is missing, use null.
  ` + `\n\nDocument content:\n${content}`;
