import axios from 'axios';
import { ONES, TENS } from '../constants/constants';

const qs = require('qs');

/**
 * Split array into chunks of specified size
 */
export function chunkArray<T>(array: T[], chunkSize: number): T[][] {
  const chunks: T[][] = [];
  for (let i = 0; i < array.length; i += chunkSize) {
    chunks.push(array.slice(i, i + chunkSize));
  }
  return chunks;
}

/**
 * Add delay between operations
 */
export function delay(ms: number): Promise<void> {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

/**
 * Get the week name based on the date
 */
export function getWeekName(date: Date): string {
  const weekNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
  return weekNames[date.getDay()];
}

/**
 * Convert date format from ISO string to DD-Month-YYYY
 */
export function formatDate(dateString: string): string {
  const date = new Date(dateString);
  const day = String(date.getDate()).padStart(2, '0');
  const monthNames = [
    'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'
  ];
  const month = monthNames[date.getMonth()];
  const year = date.getFullYear();
  return `${day}-${month}-${year}`;
}

/**
 * Determines whether two GST (Goods and Services Tax) numbers belong to the same state.
 *
 * GST numbers are assumed to have their state codes as the first two characters.
 *
 * @param gst1 - The first GST number as a string.
 * @param gst2 - The second GST number as a string.
 * @returns `true` if both GST numbers have the same state code, otherwise `false`.
 */
export function areGSTsFromSameState(gst1: string, gst2: string): boolean {
  if (!gst1 || !gst2) return false;

  const stateCode1 = gst1.slice(0, 2);
  const stateCode2 = gst2.slice(0, 2);

  return stateCode1 === stateCode2;
}

export function convertRupeesToWords(amount: number): string {
  const units = [
    '', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine',
    'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen', 'Seventeen', 'Eighteen', 'Nineteen'
  ];
  const tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];
  const scales = ['', 'Thousand', 'Lakh', 'Crore'];

  if (amount === 0) return 'Zero Rupees';

  const getWordsForThreeDigits = (num: number): string => {
    let result = '';
    if (num > 99) {
      result += `${units[Math.floor(num / 100)]} Hundred `;
      num %= 100;
    }
    if (num > 19) {
      result += `${tens[Math.floor(num / 10)]} `;
      num %= 10;
    }
    if (num > 0) {
      result += `${units[num]} `;
    }
    return result.trim();
  };

  let rupees = Math.floor(amount); // Extract Rupees
  const paise = Math.round((amount - rupees) * 100); // Extract Paise

  let words = '';
  let scaleIndex = 0;

  while (rupees > 0) {
    const chunk = rupees % 1000;
    if (chunk > 0) {
      const chunkWords = getWordsForThreeDigits(chunk);
      words = `${chunkWords} ${scales[scaleIndex]} ${words}`.trim();
    }
    rupees = Math.floor(rupees / 1000);
    scaleIndex++;
  }

  const rupeesInWords = `${words.trim()} Rupees`;
  const paiseInWords = paise > 0 ? `${getWordsForThreeDigits(paise)} Paise` : '';

  return `${rupeesInWords}${paiseInWords ? ` and ${paiseInWords}` : ''}`;
}

/**
 * Convert a number to words in Indian numbering system.
 * @param num - The number to convert.
 * @returns The number in words.
 */
function convertNumberToWords(num: number): string {
  if (num === 0) return 'Zero';

  const units = [
    { value: 1000000000000, str: 'Lakh Crore' },
    { value: 10000000, str: 'Crore' },
    { value: 100000, str: 'Lakh' },
    { value: 1000, str: 'Thousand' },
    { value: 100, str: 'Hundred' }
  ];

  let result = '';

  for (const unit of units) {
    const quotient = Math.floor(num / unit.value);
    if (quotient > 0) {
      result += `${convertNumberToWords(quotient)} ${unit.str} `;
      num %= unit.value;
    }
  }

  if (num > 0) {
    if (num < 20) {
      result += ONES[num];
    } else {
      result += `${TENS[Math.floor(num / 10)]} ${ONES[num % 10]}`.trim();
    }
  }

  return result.trim();
}

/**
 * Convert an amount in rupees and paise to words in Indian numbering system.
 *
 * @param amount - The amount in rupees (can be a number or string).
 * @returns The amount in words.
 */
export function amountInWordsIndian(amount: number): string {
  if (isNaN(amount) || amount < 0) {
    throw Error('Invalid amount');
  }
  const [rupees, paise] = Number(amount).toFixed(2).split('.').map(Number);

  const rupeeWords = rupees > 0
    ? `${convertNumberToWords(rupees)} Rupees`
    : '';

  const paiseWords = paise > 0
    ? `${convertNumberToWords(paise)} Paise`
    : '';

  let result = '';
  if (rupeeWords && paiseWords) {
    result = `${rupeeWords} and ${paiseWords}`;
  } else if (rupeeWords) {
    result = rupeeWords;
  } else if (paiseWords) {
    result = paiseWords;
  } else {
    result = 'Zero Rupees';
  }

  return result.charAt(0).toUpperCase() + result.slice(1);
}


/**
 * Generate the next sequence number based on a prefix and the previous sequence number.
 *
 * @param prefix - The prefix for the sequence.
 * @param previousSequence - The previous sequence number as a string.
 * @returns The next sequence number as a string.
 */
export function generateNextSequence(prefix: string, previousSequence: string): string {
  const prefixLength = prefix.length;

  if (!previousSequence.startsWith(prefix)) {
    throw new Error('Previous sequence does not match the provided prefix.');
  }

  const sequenceNumberPart = previousSequence.slice(prefixLength);
  const nextSequenceNumber = (parseInt(sequenceNumberPart, 10) + 1).toString().padStart(sequenceNumberPart.length, '0');

  return `${prefix}${nextSequenceNumber}`;
}

export const oAuth = async () => {
  try {
    const data = qs.stringify({
      refresh_token: process.env.ZOHO_REFERESH_TOKEN,
      client_id: process.env.ZOHO_CLIENT_ID,
      client_secret: process.env.ZOHO_CLIENT_SECRET,
      redirect_uri: process.env.ZOHO_REDIRECT_URL,
      grant_type: process.env.ZOHO_GRANT_TYPE,
    });

    const config = {
      method: 'post',
      url: process.env.ZOHO_TOKEN_URL,
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
      },
      data: data,
    };

    try {
      const response = await axios(config);
      console.log('auth ', response.data);
      return response.data;
    } catch (error) {
      console.error(error);
    }
  } catch (error) {
    console.log('------', error);
    return error;
  }
};

export function toSnakeCase(str: string): string {
  return str
    .trim()                         // Remove leading/trailing whitespace
    .replace(/\s+/g, '_')          // Replace one or more spaces with underscore
    .replace(/[^a-zA-Z0-9_]/g, '_') // Replace non-alphanumeric chars with underscore
    .replace(/_+/g, '_')           // Replace multiple consecutive underscores with single
    .replace(/^_+|_+$/g, '')       // Remove leading and trailing underscores
    .toLowerCase();                // Convert to lowercase
}

export function generateFormSectionKey(name: string): string {
  const prefix = name.replace(/\s+/g, '').substring(0, 6).toUpperCase();
  const random = Math.floor(Math.random() * 10000)
    .toString()
    .padStart(4, '0');
  return `FS_${prefix}_${random}`;
}
/**
 * Deduct a specified number of days from a given date and return the resulting date.
 *
 * @param date - The initial date.
 * @param days - The number of days to deduct.
 * @returns The date after deducting the specified number of days.
 */
export function deductDaysFromDate(date: Date, days: number): Date {
  const resultDate = new Date(date);
  resultDate.setDate(resultDate.getDate() - days);
  return resultDate;
}

