import { Injectable, HttpException, HttpStatus } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';

interface Recipient {
  mobiles: string;
  [key: string]: string;
}

interface SendSmsResponse {
  status: string;
  type: string;
  message: string;
}

@Injectable()
export class Msg91Service {
  private readonly authKey: string;
  private readonly senderId: string;
  private readonly baseUrl: string;

  constructor(private readonly configService: ConfigService) {
    this.authKey = this.configService.get<string>('MSG91_AUTH_KEY') ?? '';
    // this.senderId = this.configService.get<string>('MSG91_SENDER_ID') ?? '';
    this.baseUrl =
      this.configService.get<string>('MSG91_BASE_URL') ?? 'https://api.msg91.com/api/v5';
  }
  /**
   * Sends SMS using MSG91 Flow API
   * @param mobiles - Single mobile number or array of mobile numbers
   * @param variables - Key-value pairs of template variables
   * @param templateId - MSG91 template ID
   * @returns Promise containing the SMS send response
   */
  async sendSms(
    mobiles: string | string[],
    variables: Record<string, string>,
    templateId: string,
  ): Promise<SendSmsResponse> {
    try {
      const recipients: Recipient[] = (Array.isArray(mobiles) ? mobiles : [mobiles]).map(
        (mobile) => ({
          mobiles: mobile.toString(),
          ...variables,
        }),
      );
      const data = {
        template_id: templateId,
        recipients,
      };

      const response = await axios.post(`${this.baseUrl}/flow`, data, {
        headers: {
          'Content-Type': 'application/json',
          authkey: this.authKey,
        },
      });
      //console.log('SMS sent to:', recipients, response.data);
      return response.data as SendSmsResponse;
    } catch (error: any) {
      console.error('MSG91 sendSms error:', error?.response?.data || error.message);
      throw new HttpException('Failed to send SMS', HttpStatus.INTERNAL_SERVER_ERROR);
    }
  }
}
