import { Injectable } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { S3Client, PutObjectCommand, GetObjectCommand } from '@aws-sdk/client-s3';
import { getSignedUrl } from '@aws-sdk/s3-request-presigner';
import { isProd } from '../config/config';

@Injectable()
export class AwsS3Service {
  private s3Client: S3Client;

  constructor(private configService: ConfigService) {
    const isProdvar = isProd(this.configService);
    const s3Config: any = {
      region: this.configService.get<string>('AWS_REGION') || '',
    };
    if (!isProdvar) {
      s3Config.credentials = {
        accessKeyId: this.configService.get<string>('AWS_ACCESS_KEY_ID') || '',
        secretAccessKey: this.configService.get<string>('AWS_SECRET_ACCESS_KEY') || '',
      };
    }
    this.s3Client = new S3Client(s3Config);
  }

  /**
   * Upload data to S3
   * @param key - S3 key
   * @param data - data to upload
   * @param contentType - content type
   * @returns S3 upload result
   */
  async uploadToS3(key: string, data: Buffer | string, contentType: string) {
    const params = {
      Bucket: this.configService.get('AWS_S3_BUCKET_NAME'),
      Key: key,
      Body: Buffer.isBuffer(data) ? data : Buffer.from(data, 'base64'),
      ContentType: contentType,
    };
    
    try {
      const command = new PutObjectCommand(params);
      const result = await this.s3Client.send(command);
      console.log('Upload successful:', result);
      return this.getS3Url(key); // Return the public URL of the uploaded file
    } catch (error) {
      console.error('Upload failed:', error);
      throw error;
    }
  }

  /**
   * Generate pre-signed URL for file upload to S3
   * @param fileName - Name of the file to be uploaded
   * @param contentType - Content type of the file
   * @returns Object containing upload URL and access URL
   */
  async generateUploadUrl(
    fileName: string,
    contentType: string,
    prefix = 'assets',
    userLevelDetails = false,
    defaultContent = false
  ): Promise<{ putUrl: string; accessUrl: string }> {
    try {
      const timestamp = Date.now();
      let key = `${prefix}`;
      if (!userLevelDetails) {
        if (defaultContent) {
         key += `/${timestamp}_${fileName}`;
        } else {
        key += `_${timestamp}_${fileName}`;
        }
      } else {
        if (defaultContent) {
          key += `/${timestamp}_${fileName}`;
        }
      }
      
      const params = {
        Bucket: this.configService.get('AWS_S3_BUCKET_NAME'),
        Key: key,
        ContentType: contentType,
      };

      const command = new PutObjectCommand(params);
      const putUrl = await getSignedUrl(this.s3Client, command, { expiresIn: 600 });
      const accessUrl = `https://${this.configService.get('AWS_S3_BUCKET_NAME')}.s3.${this.configService.get('AWS_REGION')}.amazonaws.com/${key}`;

      return { putUrl, accessUrl };
    } catch (error) {
      console.error('Error generating upload URL:', error);
      throw error;
    }
  }

  /**
   * Generate a signed URL for accessing an S3 object
   * @param key - S3 object key
   * @param expiresIn - URL expiration time in seconds (default: 3600)
   * @returns Promise<string> Signed URL for accessing the S3 object
   */
  async getSignedUrl(key: string, expiresIn: number = 3600): Promise<string> {
    const command = new GetObjectCommand({
      Bucket: this.configService.get('AWS_S3_BUCKET_NAME'),
      Key: key,
    });

    try {
      return await getSignedUrl(this.s3Client, command, { expiresIn });
    } catch (error) {
      console.error('Error generating signed URL:', error);
      throw error;
    }
  }
  /**
   * Generate a public URL for accessing an S3 object
   * @param key - S3 object key
   * @returns string Public URL for accessing the S3 object
   */
  getS3Url(key: string): string {
    const bucketName =  process.env.AWS_S3_BUCKET_NAME
    const region = process.env.AWS_REGION
    return `https://${bucketName}.s3.${region}.amazonaws.com/${key}`;
  }
}