import { Injectable, NestMiddleware } from '@nestjs/common';
import { Response, NextFunction } from 'express';
import { AppLoggerService } from '../services/logger.service'; 
import { UserService } from 'src/user/user.service';
import { userConstMessages } from '../constants/strings-constants';

@Injectable()
export class LoggerContextMiddleware implements NestMiddleware {
  constructor(private readonly loggerService: AppLoggerService,
    private readonly userService: UserService
  ) {}

  async use(req: any, res: Response, next: NextFunction) {
    // Extract user ID from request (adjust according to your auth implementation)
    let userId = req.headers?.userid|| 'anonymous';
    const requestId = req.headers['x-request-id'] || `req-${Date.now()}`;

     // If userId exists in headers, fetch the normal user ID
    if (req.headers?.userid) {
      try {
        const user = await this.userService.getUserByFirebaseId(userId);
        if (user) {
          userId = user.id; // Replace Firebase user ID with normal user ID
        }
      } catch (error) {
        // Log the error if fetching the user fails
        this.loggerService.warn(userConstMessages.FAILED_TO_FETCH_USER('firebaseId', userId));
      }
    }
    
    // Create a new context map with the user ID
    const context: Record<string, any> = { userId, requestId };
    if (req.method) {
      context.method = req.method;
    }
    
    // Run the next middleware in the context

    if (req.originalUrl) {
      context.originalUrl = req.originalUrl;
    }

    // Run the next middleware in the context with full request details
    this.loggerService.runWithContext(context, () => {
      next();
    });
  }
}