import {
  ExceptionFilter,
  Catch,
  ArgumentsHost,
  HttpException,
  HttpStatus,
  Injectable,
} from '@nestjs/common';
import { Response } from 'express';
import { ErrorHandler } from '../response-handling/error-handling';

@Injectable()
@Catch()
export class HttpExceptionFilter implements ExceptionFilter {
  private readonly errorHandler = new ErrorHandler();

  catch(exception: unknown, host: ArgumentsHost) {
    const ctx = host.switchToHttp();
    const response = ctx.getResponse<Response>();
    const status =
      exception instanceof HttpException
        ? exception.getStatus()
        : HttpStatus.INTERNAL_SERVER_ERROR;

    const message =
      exception instanceof HttpException
        ? exception.getResponse()
        : 'Internal server error';

    console.error('Exception caught:', exception);

    if (status === HttpStatus.BAD_REQUEST) {
      this.errorHandler.badRequest(response, message as string);
    } else if (status === HttpStatus.NOT_FOUND) {
      this.errorHandler.notFound(response, typeof message === 'string' ? message : 'Not Found');
    } else if (status === HttpStatus.FORBIDDEN) {
      this.errorHandler.forbidden(response, typeof message === 'string' ? message : 'Forbidden');
    } else if (status === HttpStatus.UNAUTHORIZED) {
      this.errorHandler.unauthorized(response);
    } else {
      this.errorHandler.serverError(response, exception instanceof Error ? exception : undefined);
    }
  }
}
