import {
    Injectable,
    CanActivate,
    ExecutionContext,
    UnauthorizedException,
  } from '@nestjs/common';
  import { FirebaseAuthService } from './firebase-auth.service';
import { UserService } from 'src/user/user.service';
  
  @Injectable()
  export class FirebaseAuthGuard implements CanActivate {
    constructor(private readonly firebaseAuthService: FirebaseAuthService,
                private readonly userService: UserService,
    ) {}
  
    async canActivate(context: ExecutionContext): Promise<boolean> {
      const request = context.switchToHttp().getRequest();
      const firebaseToken =
        request.headers.authorization?.split('Bearer ')[1]?.trim();
      const userId = request.headers.userid;
      
      if (!firebaseToken || !userId) {
        throw new UnauthorizedException('Missing authentication headers');
      }
  
      const decodedToken = await this.firebaseAuthService.verifyToken(
        firebaseToken,
        userId,
      );
      const user = await this.userService.getUserByCombinedPhoneNumber(decodedToken.phone_number);
      if (!user) {
        throw new UnauthorizedException('User not found');
      }
    if (user != null) {
      // Ensure userRoleMaps is an array
      if (!Array.isArray(user.userRoleMaps)) {
        user.userRoleMaps = [];
      }
    
      // If userRoleMaps is empty or has no valid roles, append a default viewer role
      const hasValidRole = user.userRoleMaps.some(
        (urm) => urm.role && typeof urm.role.name === 'string' && urm.role.name.length > 0
      );
      if (!hasValidRole) {
        user.userRoleMaps.push({
          // Only the fields needed for role mapping; you can expand if needed
          role: {
            name: user.role || 'viewer', // Default to 'viewer' if no roleKey
            id: 0,
            roleKey: '',
            createdAt: new Date(),
            updatedAt: new Date(),
            userRoleMaps: []
          },
          id: 0,
          user: user,
          createdAt: new Date(),
          updatedAt: new Date()
        });
      }
    
      request.user = {
        ...user,
        roles: user.userRoleMaps.map((urm) => urm.role?.name).filter(Boolean),
      };
    }    
    return true;
    }
  }