import { CanActivate, ExecutionContext, Injectable, UnauthorizedException } from '@nestjs/common';
import { AuthService } from './auth.service';
import { FirebaseAuthService } from './firebase-auth.service';
import { UserService } from 'src/user/user.service';

@Injectable()
export class CombinedAuthGuard implements CanActivate {
  constructor(
    private readonly authService: AuthService,
    private readonly firebaseAuthService: FirebaseAuthService,
    private readonly userService: UserService,
  ) {}

  async canActivate(context: ExecutionContext): Promise<boolean> {
    const request = context.switchToHttp().getRequest();
    const token = request.headers.authorization?.split('Bearer ')[1]?.trim();

    if (!token) {
      throw new UnauthorizedException('Missing authentication token');
    }

    let user;

    // Check if token ends with 'custom'
    if (token.endsWith('custom')) {
       console.log('Using OTP Authentication');
      // OTP Authentication
      let clearedToken = token.replace(' custom', '');
      const session = await this.authService.validateJwt(clearedToken);
      console.log('Session:', session);
      if (!session || !session.user) {
        throw new UnauthorizedException('Invalid token');
      }
      user = session.user;
    } else {
      // Firebase Authentication
      const userId = request.headers.userid;
      if (!userId) {
        throw new UnauthorizedException('Missing userId header');
      }

      const decodedToken = await this.firebaseAuthService.verifyToken(token, userId);
      user = await this.userService.getUserByCombinedPhoneNumber(decodedToken.phone_number);
      
      if (!user) {
        throw new UnauthorizedException('User not found');
      }
    }

    // Common user role mapping logic
    if (!Array.isArray(user.userRoleMaps)) {
      user.userRoleMaps = [];
    }

    const hasValidRole = user.userRoleMaps.some(
      (urm) => urm.role && typeof urm.role.name === 'string' && urm.role.name.length > 0,
    );

    if (!hasValidRole) {
      user.userRoleMaps.push({
        role: {
          name: user.role || 'viewer',
          id: 0,
          roleKey: '',
          createdAt: new Date(),
          updatedAt: new Date(),
          userRoleMaps: [],
        },
        id: 0,
        user: user as any,
        createdAt: new Date(),
        updatedAt: new Date(),
      });
    }

    request.user = {
      ...user,
      roles: user.userRoleMaps.map((u) => u.role?.name).filter(Boolean),
    };

    return true;
  }
}