import { Injectable, InternalServerErrorException } from '@nestjs/common';
import DocumentIntelligence, {
  getLongRunningPoller,
  isUnexpected,
} from '@azure-rest/ai-document-intelligence';
import { AzureOpenAI } from 'openai';
import axios from 'axios';
import { ExtractionInfo } from './dto/create-ai-communication.dto';
import { DocumentTypeEnum } from 'src/common/enum/document-type.enum';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { AppLoggerService } from 'src/common/services/logger.service';
import {
  extractFlightDataSystemPrompt,
  extractFlightDataUserPrompt,
  extractIdentityDocumentSystemPrompt,
  extractIdentityDocumentUserPrompt,
  extractInvoiceDataSystemPrompt,
  extractInvoiceDataUserPrompt,
  flightTicketAnalyzerSystemPrompt,
  flightTicketAnalyzerUserPrompt,
} from 'src/common/utils/prompts';
import { azureOpenAIConfig } from 'src/common/config/config';
import { ConfigService } from '@nestjs/config';

@Injectable()
export class AiCommunicationService {
  private openAiClient: AzureOpenAI;

  constructor(
    private configService: ConfigService,
    private readonly logger: AppLoggerService,
  ) {
    const config = azureOpenAIConfig(this.configService);
    this.openAiClient = new AzureOpenAI({
      endpoint: config.endpoint,
      apiKey: config.key,
      apiVersion: config.apiVersion,
    });
  }

  async processDocumentWithDocumentIntelligence(base64Source: string): Promise<string> {
    // console.log('[INFO] Processing document with Azure Document Intelligence:', filePath);
    const endpoint = this.configService.get<string>('AZURE_DOCUMENT_INTELLIGENCE_ENDPOINT') || '';
    const key = this.configService.get<string>('AZURE_DOCUMENT_INTELLIGENCE_KEY') || '';

    if (!endpoint || !key) {
      this.logger.log('[ERROR] Azure Document Intelligence credentials are missing');
      throw new InternalServerErrorException('Azure Document Intelligence credentials are missing');
    }

    this.logger.log('[INFO] Initializing Azure Document Intelligence client', { endpoint, key });

    const client = DocumentIntelligence(endpoint, { key });

    this.logger.log('[INFO] Sending document to Azure Document Intelligence for analysis:', {
      base64Source,
    });
    const initialResponse = await client
      .path('/documentModels/{modelId}:analyze', 'prebuilt-layout')
      .post({ contentType: 'application/json', body: { base64Source } });

    if (isUnexpected(initialResponse)) {
      throw new InternalServerErrorException(
        `Azure returned unexpected response: ${initialResponse.status}`,
      );
    }

    const poller = getLongRunningPoller(client, initialResponse);
    const result: any = await poller.pollUntilDone();
    this.logger.log('[INFO] Document analysis completed:', result.status);
    return result.body.analyzeResult.content;
  }

  private async callOpenAIChatCompletion(
    systemPrompt: string,
    userPrompt: string,
    temperature = 0,
    max_tokens = 500,
  ): Promise<string> {
    const response = await this.openAiClient.chat.completions.create({
      model: this.configService.get<string>('AZURE_OPENAI_MODEL_NAME') || '',
      messages: [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: userPrompt },
      ],
      temperature,
      max_tokens,
    });
    return response.choices[0].message.content || '';
  }

  private parseOpenAIJsonResponse(raw: string): any {
    try {
      const markdownMatch = raw.match(/```(?:json)?\s*([\s\S]*?)\s*```/);
      if (markdownMatch) {
        return JSON.parse(markdownMatch[1].trim());
      }
      return this.extractJsonFromString(raw);
    } catch (err){
      console.error('[ERROR] Failed to parse OpenAI response:', err);
      throw new InternalServerErrorException('Failed to parse JSON from OpenAI response');
    }
  }

  private async isFlightTicket(content: string): Promise<any> {
    const systemPrompt = flightTicketAnalyzerSystemPrompt;
    const userPrompt = flightTicketAnalyzerUserPrompt(content);

    const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0, 500);
    try {
      return this.parseOpenAIJsonResponse(raw);
    } catch (err) {
      this.logger.error('[ERROR] Failed to parse isFlightTicket response:', raw, err);
      handleKnownErrors(ERROR_CODES.FAILED_TO_VERIFY_FLIGHT_TICKET, err);
    }
  }

  private async extractFlightData(content: string): Promise<any> {
    const systemPrompt = extractFlightDataSystemPrompt;
    const userPrompt = extractFlightDataUserPrompt(content);

    const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0.2, 1200);
    let structuredData;
    try {
      structuredData = this.parseOpenAIJsonResponse(raw);
    } catch (err) {
      this.logger.error('[ERROR] Failed to parse extractFlightData response:', err);
      handleKnownErrors(ERROR_CODES.FAILED_TO_EXTRACT_FLIGHT_DATA, err);
    }

    const onward = structuredData.find((seg) => seg.type === 'onward');
    const ret = structuredData.find((seg) => seg.type === 'return');

    let tripType = 'one-way';
    if (onward?.comingFrom?.toLowerCase() === ret?.goingTo?.toLowerCase()) {
      tripType = 'round-trip';
    }

    return { tripType, segments: structuredData };
  }

  private extractJsonFromString(str: string): any {
    console.log('[INFO] Extracting JSON from string:', str);
    str = JSON.stringify(str); // Convert to string if not already
    const jsonString = str
      .replace(/(\w+):/g, '"$1":') // Add quotes around keys
      .replace(/'/g, '"'); // Replace single quotes with double quotes

    return JSON.parse(jsonString);
  }

  private async fetchFileAsBase64(s3Url: string): Promise<string> {
    try {
      const response = await axios.get(s3Url, { responseType: 'arraybuffer' });
      return Buffer.from(response.data, 'binary').toString('base64');
    } catch (err: any) {
      this.logger.error('[ERROR] Failed to fetch file from S3:', err.message);
      handleKnownErrors(ERROR_CODES.FAILED_TO_EXTRACT_FLIGHT_DATA, err);
      // throw new InternalServerErrorException('Failed to fetch file from S3');
    }
  }

  async analyzeFlightTicket(s3Url: string): Promise<any> {
    try {
      const base64 = await this.fetchFileAsBase64(s3Url);
      const content = await this.processDocumentWithDocumentIntelligence(base64);
      const verification = await this.isFlightTicket(content);

      if (!verification.isFlightTicket) {
        return {
          statusCode: 200,
          message: 'The uploaded file is not a Flight ticket',
          data: null,
        };
      }

      const structuredResult = await this.extractFlightData(content);

      return structuredResult;
    } catch (err) {
      this.logger.error('[ERROR] Flight ticket analysis failed:', err.message);
      handleKnownErrors(ERROR_CODES.FAILED_TO_ANALYZE, err);
    }
  }
  async analyzeTheDocument(info: ExtractionInfo): Promise<any> {
    try {
      let result;
      this.logger.log('[INFO] Analyzing document:', info);
      if (info.type === DocumentTypeEnum.FLIGHT_TICKET) {
        result = await this.analyzeFlightTicket(info.url);
      } else if (info.type === DocumentTypeEnum.IDENTITY_DOCUMENT) {
        result = await this.analyzeIdentityDocumentFromS3(info.url);
      }

      this.logger.log('[INFO] Document analysis result:', result);
      return result;
    } catch (err) {
      this.logger.error('[ERROR] Document analysis failed:', err.message);
      handleKnownErrors(ERROR_CODES.FAILED_TO_ANALYZE, err);
    }
  }

  async analyzeIdentityDocumentFromS3(s3Url: string): Promise<any> {
    const allowedTypes = ['Passport', 'Voter ID', 'Aadhar', 'PAN', 'Driving license'];

    try {
      const base64 = await this.fetchFileAsBase64(s3Url);
      const content = await this.processDocumentWithDocumentIntelligence(base64);

      const documentType = await this._getIdentityDocumentType(content);

      if (!allowedTypes.includes(documentType)) {
        return {
          statusCode: 400,
          message: 'The uploaded document is not a supported identity document.',
          data: null,
        };
      }

      const extractedData = await this._extractIdentityDocumentFields(content, documentType);

      return {
        statusCode: 200,
        message: 'Identity document data extracted successfully',
        data: extractedData,
      };
    } catch (err: any) {
      console.error('[ERROR] Identity document analysis failed:', err.message);
      throw new InternalServerErrorException({
        error: 'Failed to analyze identity document',
        details: err.message,
      });
    }
  }

  private async _getIdentityDocumentType(content: string): Promise<string> {
    const systemPrompt = extractInvoiceDataSystemPrompt;
    const userPrompt = extractInvoiceDataUserPrompt(content);

    const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0, 100);
    let docTypeResult: any;
    try {
      docTypeResult = this.parseOpenAIJsonResponse(raw);
    } catch {
      throw new InternalServerErrorException('Failed to parse document type from OpenAI response');
    }
    return docTypeResult.documentType?.trim() || '';
  }

  private async _extractIdentityDocumentFields(
    content: string,
    documentType: string,
  ): Promise<any> {
    const systemPrompt = extractIdentityDocumentSystemPrompt;

    const userPrompt = extractIdentityDocumentUserPrompt(content, documentType);

    const raw = await this.callOpenAIChatCompletion(systemPrompt, userPrompt, 0, 300);
    let extractedData: any;
    try {
      extractedData = this.parseOpenAIJsonResponse(raw);
    } catch {
      throw new InternalServerErrorException(
        'Failed to parse extracted fields from OpenAI response',
      );
    }
    return extractedData;
  }
}
