# API Development & Best Practices Guide

This document provides a comprehensive guide for developing APIs and NestJS entities, ensuring best practices for maintainability, security, and scalability.

## API Development Guidelines

### 1. **Data Transfer Objects for API Request and Response**

- Clearly define data models for both request and response.
- Ensure type safety by using TypeScript interfaces or classes.
- Include nested objects and arrays as necessary, adhering to the JSON format.

### 2. **API Request Validation**

- Use libraries like `class-validator` and `class-transformer` for request payload validation.
- Define DTOs (Data Transfer Objects) for API request.
- Enforce strict validation rules (e.g., required fields, data types, regex patterns).
- For the UPDATE/PUT/PATCH APIs add the optional validators for the required fields.

### 3. **Error Handling**

- Include proper HTTP status codes (e.g., 400 for Bad Request, 404 for Not Found, 500 for Internal Server Error, etc).
- Implement global error handling middleware for consistent error response format.
- Provide meaningful error codes and messages in the response body.
- Create a JSON with continuous numbers as indexes and Standardize error messages as values For e.g { 101: "Product not found.", 102: "Invalid product ID format.", 103: "Product creation failed due to validation errors.", 104: "Product deletion failed. Product is associated with existing orders.", 105: "Unauthorized access to product resource.", 201: "Order not found.", 202: "Order creation failed due to validation errors.", 203: "Order update failed due to validation errors.", 204: "Order retrieval failed due to server error." }

### 4. **Swagger Documentation**

Use `@nestjs/swagger` for generating Swagger documentation. Ensure the following:

- **Description**: Provide a clear description of each API endpoint.
- **Request Attributes**: Document all request parameters, headers, and bodies.
- **Response Attributes**: Detail the structure of the response body.
- **Response Messages**: Include sample responses for various scenarios (e.g., success, validation errors).
- **Status Codes**: Enumerate all possible status codes and their meanings.

## Example Swagger Documentation Configuration

```typescript
@ApiOperation({ summary: 'Get a list of users' })
@ApiResponse({ status: 200, description: 'List of users retrieved successfully.' })
@ApiResponse({ status: 400, description: 'Invalid request parameters.' })
@ApiResponse({ status: 500, description: 'Internal server error.' })

```

### 5. **Code Documentation**

- Document the purpose, inputs, outputs, and logic of each method using JSDoc format.
- Document the each method/function.
- Follow @use JSDoc
- For example: ```typescript
    /**
     * Fetches a list of all users.
     * @returns {Promise<User[]>} A promise resolving to an array of user objects.
     */
    async getUsers(): Promise<User[]> {
        // Logic here
    }
    
    ```
- Document the complex functionality with in the API where ever required.

### 6. **Logging**

- Implement logging with different levels (debug, info, warning, error, critical).
- Use the built-in `Logger` service provided by NestJS.
- Include contextual information (e.g., timestamp, request ID, user ID) in logs.

Example:

```typescript
import { Logger } from '@nestjs/common';

@Injectable()
export class UserService {
  private readonly logger = new Logger(UserService.name);

  async getUsers(): Promise<User[]> {
    this.logger.debug('Fetching all users');
    try {
      const users = await this.userRepository.find();
      this.logger.log('Users retrieved successfully');
      return users;
    } catch (error) {
      this.logger.error('Failed to fetch users', error.stack);
      throw new InternalServerErrorException();
    }
  }
}

```

### 7. **Linting and Code Quality**

- Follow a standard linting configuration (e.g., ESLint with Airbnb or NestJS presets).
- Ensure code adheres to consistent formatting and structure.

### 8. **Security Best Practices**

- Validate all incoming inputs to prevent SQL injection and XSS attacks.
- Use environment variables(process.env) for sensitive data (e.g., API keys, database credentials).
- Implement HTTPS and enforce secure cookies.
- Use authentication, and authorization mechanisms.
- Avoid exposing sensitive information in error messages.

### 9. **Testing**

- Write unit and integration tests for APIs using libraries like `Jest` or `Supertest`.
- Ensure test coverage for edge cases and error handling scenarios.

---

## NestJS Entity Development Guidelines

### 1. **Entity Structure**

- Follow camel casing for attribute names.
- Use the same column name as the attribute.

### 2. **Attribute Properties**

- Define attributes with properties like: 
    - `name`: Specify the database column name.
    - `default`: Set default values where applicable.
    - `nullable`: Indicate whether the attribute is optional.
    - `type`: Define the attribute's data type (e.g., `string`, `number`, `boolean`, `Date`).
    - Example:
    
    ```typescript
    @Column({ name: 'first_name', type: 'varchar', nullable: false })
    firstName: string;
    
    ```

### 3. **Constructor in Entity Classes**

- Include a constructor for initializing attributes.
- Example: ```typescript
    constructor(partial: Partial<User>) {
        Object.assign(this, partial);
    }
    
    ```

### 4. **Relationships**

- Define relationships (e.g., OneToOne, OneToMany, ManyToOne, ManyToMany) explicitly.
- Include cascading and eager loading as needed.
- Example: ```typescript
    @OneToMany(() => Post, (post) => post.user, { cascade: true })
    posts: Post[];
    
    ```

### 5. **Appropriate Decorators**

- Use appropriate decorators like `@PrimaryColumn`, `@PrimaryGeneratedColumn`, `@CreateDateColumn`, etc., to define column properties.
- Example: ```typescript
        @PrimaryGeneratedColumn()
        id: number;
        @CreateDateColumn()
        createdAt: Date;
    
    ```

## Additional Best Practices

### 1. **Versioning APIs**

- Use versioning in API routes (e.g., `/api/v1/resource`).

### 2. **Performance Optimization**

- Implement pagination for large data sets.
- Use caching strategies for frequently accessed data.
- Implement Search &amp; Filtering, Sorting in GET API.