import { Injectable } from '@nestjs/common';
import { DataSource, EntityManager } from 'typeorm';
import { CreateUserProgramExperienceDto } from './dto/create-user-program-experience.dto';
import { UpdateUserProgramExperienceDto } from './dto/update-user-program-experience.dto';
import { UserProgramExperienceRepository } from './user-program-experience.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { userProgramExperienceConstMessages } from 'src/common/constants/strings-constants';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { LookupData, ProgramRegistration, User, UserProgramExperience } from 'src/common/entities';

@Injectable()
export class UserProgramExperienceService {
  constructor(
    private readonly repository: UserProgramExperienceRepository,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
  ) {}

  async create(dto: CreateUserProgramExperienceDto): Promise<UserProgramExperience[]> {
    this.logger.log(userProgramExperienceConstMessages.CREATE_REQUEST_RECEIVED(dto.userId), dto);
    try {
      return await this.dataSource.transaction(async (manager) => {
        const { user, createdBy, updatedBy, registration, lookupMap } = await this.prepareEntities(
          dto.userId,
          dto.createdBy,
          dto.updatedBy,
          dto.registrationId,
          dto.lookupDataIds,
          manager,
        );

        const registrationId = dto.registrationId ?? null;
        const existing = await this.repository.findActiveExperiences(
          dto.userId,
          registrationId,
          manager,
        );
        const existingLookupIds = new Set(existing.map((experience) => experience.lookupData.id));
        const idsToCreate = dto.lookupDataIds.filter((id) => !existingLookupIds.has(id));

        if (idsToCreate.length === 0) {
          return existing;
        }

        const experiences = idsToCreate.map((id) => {
          const experience = this.repository.createExperience(
            {
              user,
              registration,
              lookupData: lookupMap.get(id) as LookupData,
              createdBy,
              updatedBy,
            },
            manager,
          );
          this.applyOptionalTimestamps(experience, dto);
          return experience;
        });

        await this.repository.saveExperiences(experiences, manager);
        return await this.repository.findActiveExperiences(dto.userId, registrationId, manager);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PROGRAM_EXPERIENCE_SAVE_FAILED, error);
    }
  }

  async update(
    userId: number,
    dto: UpdateUserProgramExperienceDto,
  ): Promise<UserProgramExperience[]> {
    this.logger.log(userProgramExperienceConstMessages.UPDATE_REQUEST_RECEIVED(userId), dto);
    try {
      return await this.dataSource.transaction(async (manager) => {
        // Deduplicate incoming lookup IDs
        const uniqueLookupIds = [...new Set(dto.lookupDataIds)];
        
        const { user, updatedBy, registration, lookupMap } = await this.prepareEntities(
          userId,
          dto.updatedBy,
          dto.updatedBy,
          dto.registrationId,
          uniqueLookupIds, // Use deduplicated IDs
          manager,
        );
  
        const registrationId = dto.registrationId ?? null;
        await this.repository.softDeleteByUserAndRegistration(userId, registrationId, manager);
  
        const experiences = uniqueLookupIds.map((id) => { // Use deduplicated IDs
          const experience = this.repository.createExperience(
            {
              user,
              registration,
              lookupData: lookupMap.get(id) as LookupData,
              createdBy: updatedBy,
              updatedBy,
            },
            manager,
          );
          this.applyOptionalTimestamps(experience, dto);
          return experience;
        });
  
        await this.repository.saveExperiences(experiences, manager);
        return await this.repository.findActiveExperiences(userId, registrationId, manager);
      });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.USER_PROGRAM_EXPERIENCE_UPDATE_FAILED, error);
    }
  }

  private async prepareEntities(
    userId: number,
    createdById: number,
    updatedById: number,
    registrationId: number | null | undefined,
    lookupIds: number[],
    manager?: EntityManager,
  ): Promise<{
    user: User;
    createdBy: User;
    updatedBy: User;
    registration: ProgramRegistration | null;
    lookupMap: Map<number, LookupData>;
  }> {
    const user = await this.ensureUser(userId, manager);
    const createdBy = await this.ensureUser(createdById, manager);
    const updatedBy = await this.ensureUser(updatedById, manager);
    const registration = await this.ensureRegistration(userId, registrationId, manager);
    const lookupRecords = await this.ensureLookupData(lookupIds, manager);
    const lookupMap = new Map(lookupRecords.map((record) => [record.id, record]));

    return { user, createdBy, updatedBy, registration, lookupMap };
  }

  private async ensureUser(userId: number, manager?: EntityManager): Promise<User> {
    const user = await this.repository.findUserById(userId, manager);
    if (!user) {
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniNotFoundException(ERROR_CODES.USER_NOTFOUND, null, null, userId.toString());
    }
    return user;
  }

  private async ensureRegistration(
    userId: number,
    registrationId: number | null | undefined,
    manager?: EntityManager,
  ): Promise<ProgramRegistration | null> {
    if (registrationId === null || registrationId === undefined) {
      return null;
    }

    const registration = await this.repository.findRegistrationById(registrationId, manager);
    if (!registration) {
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }

    if (registration.userId && registration.userId !== userId) {
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniBadRequestException(
        ERROR_CODES.USER_PROGRAM_EXPERIENCE_REGISTRATION_MISMATCH,
        null,
        null,
        registrationId.toString(),
        userId.toString(),
      );
    }

    return registration;
  }

  private async ensureLookupData(
    lookupIds: number[],
    manager?: EntityManager,
  ): Promise<LookupData[]> {
    const records = await this.repository.findLookupDataByIds(lookupIds, manager);
    const foundIds = new Set(records.map((record) => record.id));
    const missing = lookupIds.filter((id) => !foundIds.has(id));

    if (missing.length > 0) {
      // eslint-disable-next-line @typescript-eslint/only-throw-error
      throw new InifniBadRequestException(
        ERROR_CODES.USER_PROGRAM_EXPERIENCE_INVALID_LOOKUP,
        null,
        null,
        missing.join(', '),
      );
    }

    return records;
  }

  private applyOptionalTimestamps(
    experience: UserProgramExperience,
    dto: { createdAt?: string; updatedAt?: string; deletedAt?: string },
  ): void {
    if (dto.createdAt) {
      experience.createdAt = new Date(dto.createdAt);
    }

    if (dto.updatedAt) {
      experience.updatedAt = new Date(dto.updatedAt);
    }

    if (dto.deletedAt) {
      experience.deletedAt = new Date(dto.deletedAt);
    }
  }
}
