import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, EntityManager } from 'typeorm';
import { CreateUserProfileExtensionDto } from './dto/create-user-profile-extension.dto';
import { UpdateUserProfileExtensionDto } from './dto/update-user-profile-extension.dto';
import { UserProfileExtension } from 'src/common/entities/user-profile-extension.entity';
import { CommonDataService } from 'src/common/services/commonData.service';

@Injectable()
export class UserProfileExtensionRepository {
  constructor(
    @InjectRepository(UserProfileExtension)
    private readonly userProfileExtensionRepo: Repository<UserProfileExtension>,
    private readonly commonDataService: CommonDataService,
  ) {}

  async create(
    createUserProfileExtensionDto: CreateUserProfileExtensionDto,
    manager?: EntityManager,
  ): Promise<UserProfileExtension> {
    const repo = manager ? manager.getRepository(UserProfileExtension) : this.userProfileExtensionRepo;
    
    // Transform DTO to entity format, handling relations
    const entityData: any = { ...createUserProfileExtensionDto };
    
    // Handle user relation
    if (createUserProfileExtensionDto.userId) {
      entityData.user = { id: parseInt(createUserProfileExtensionDto.userId) };
      delete entityData.userId;
    }
    
    // Handle infinitheismContact relation
    if (createUserProfileExtensionDto.infinitheismContactId) {
      entityData.infinitheismContact = { id: createUserProfileExtensionDto.infinitheismContactId };
    }
    
    const userProfileExtension = repo.create(entityData as Partial<UserProfileExtension>);
    return repo.save(userProfileExtension as UserProfileExtension);
  }

  async createEmpty(userId: number, manager?: EntityManager): Promise<UserProfileExtension> {
    const emptyProfileExtension = { user: { id: userId } };
    return this.create(emptyProfileExtension as CreateUserProfileExtensionDto, manager);
  }

  async findByUserId(userId: number): Promise<UserProfileExtension | null> {
    return this.userProfileExtensionRepo.findOne({
      where: { user: { id: userId } },
    });
  }

  async updateByUserId(
    userId: number,
    updateUserProfileExtensionDto: UpdateUserProfileExtensionDto,
    manager?: EntityManager,
  ): Promise<UserProfileExtension> {
    const repo = manager ? manager.getRepository(UserProfileExtension) : this.userProfileExtensionRepo;
    
    // Get existing record to preserve data
    const existingRecord = await repo.findOne({ where: { user: { id: userId } } });
    if (!existingRecord) {
      throw new Error(`Profile extension not found for user ${userId}`);
    }
    
    // Transform DTO to entity format, handling relations
    const updateData: any = { ...updateUserProfileExtensionDto };
    
    // Handle infinitheismContact relation
    if (updateUserProfileExtensionDto.infinitheismContactId) {
      updateData.infinitheismContact = { id: updateUserProfileExtensionDto.infinitheismContactId };
    }
    
    // Preserve existing data for critical fields if not provided in update
    const preservedData = {
      ...updateData,
      // For TAN, GST number, and GST registration - only update if existing value is null
      gstin: (existingRecord.gstin !== null && existingRecord.gstin !== undefined) 
        ? existingRecord.gstin 
        : (updateData.gstin !== undefined ? updateData.gstin : existingRecord.gstin),
      tan: (existingRecord.tan !== null && existingRecord.tan !== undefined) 
        ? existingRecord.tan 
        : (updateData.tan !== undefined ? updateData.tan : existingRecord.tan),
      areYouRegisteredUnderGst: (existingRecord.areYouRegisteredUnderGst !== null && existingRecord.areYouRegisteredUnderGst !== undefined) 
        ? existingRecord.areYouRegisteredUnderGst 
        : (updateData.areYouRegisteredUnderGst !== undefined ? updateData.areYouRegisteredUnderGst : existingRecord.areYouRegisteredUnderGst),
      pincode: (existingRecord.pincode !== null && existingRecord.pincode !== undefined) 
        ? existingRecord.pincode 
        : (updateData.pincode !== undefined ? updateData.pincode : existingRecord.pincode)
    };
    
    await repo.update({ user: { id: userId } }, preservedData);
    const updated = await repo.findOne({ where: { user: { id: userId } } });
    
    if (!updated) {
      throw new Error(`Profile extension not found for user ${userId}`);
    }
    return updated;
  }

  async remove(id: number, manager?: EntityManager): Promise<void> {
    const repo = manager ? manager.getRepository(UserProfileExtension) : this.userProfileExtensionRepo;
    await repo.delete(id);
  }
}