import {
  Controller,
  Get,
  Param,
  Query,
  ParseIntPipe,
  UseGuards,
  HttpStatus,
  Res,
} from '@nestjs/common';
import {
  ApiTags,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiBearerAuth,
} from '@nestjs/swagger';
import { Response } from 'express';
import { CombinedAuthGuard } from '../auth/combined-auth.guard';
import { UserParticipationService } from './user-participation.service';
import {
  UserParticipationSummaryDto,
  ProgramParticipationDto,
  GetUserParticipationQueryDto,
} from './dto';
import { ResponseService } from '../common/response-handling/response-handler';
import { AppLoggerService } from '../common/services/logger.service';
import { handleControllerError } from '../common/utils/controller-response-handling';
import { userParticipationConstMessages } from '../common/constants/strings-constants';

@ApiTags('User Participation')
// @ApiBearerAuth()
// @UseGuards(CombinedAuthGuard)
@Controller('user-participation')
export class UserParticipationController {
  constructor(
    private readonly userParticipationService: UserParticipationService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService,
  ) {}

    @Get('summary/:userId')
  @ApiOperation({
    summary: userParticipationConstMessages.API_OPERATION_SUMMARY,
  })
  @ApiParam({
    name: 'userId',
    description: userParticipationConstMessages.API_PARAM_USER_ID,
    type: Number,
  })
  @ApiQuery({
    name: 'year',
    description: userParticipationConstMessages.API_QUERY_YEAR,
    required: false,
    type: Number,
  })
  @ApiResponse({
    status: 200,
    description: userParticipationConstMessages.API_RESPONSE_SUCCESS,
    type: UserParticipationSummaryDto,
  })
  @ApiResponse({
    status: 404,
    description: userParticipationConstMessages.ERROR_USER_NOT_FOUND,
  })
  async getUserParticipationSummary(
    @Param('userId', ParseIntPipe) userId: number,
    @Query() query: GetUserParticipationQueryDto,
    @Res() res: Response,
  ): Promise<void> {
    try {
      this.logger.debug(
        `${userParticipationConstMessages.METHOD_GET_USER_PARTICIPATION_SUMMARY} - userId: ${userId}`,
      );

      const summary = await this.userParticipationService.getUserParticipation(
        userId,
        query
      );

      await this.responseService.success(
        res,
        userParticipationConstMessages.API_RESPONSE_SUCCESS,
        summary,
        HttpStatus.OK,
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get(':userId/year/:year')
  @ApiOperation({ 
    summary: 'Get user participation for a specific year',
    description: 'Fetch all programs participated by a user in a specific year'
  })
  @ApiParam({ name: 'userId', type: Number, description: 'User ID' })
  @ApiParam({ name: 'year', type: Number, description: 'Year (e.g., 2024)' })
  @ApiResponse({
    status: HttpStatus.OK,
    description: 'User participation data for the year retrieved successfully',
    type: [ProgramParticipationDto],
  })
  @ApiResponse({
    status: HttpStatus.NOT_FOUND,
    description: 'No participation records found for the user in the specified year',
  })
  async getUserParticipationByYear(
    @Param('userId', ParseIntPipe) userId: number,
    @Param('year', ParseIntPipe) year: number,
  ): Promise<ProgramParticipationDto[]> {
    return this.userParticipationService.getUserParticipationByYear(userId, year);
  }

    @Get('programs/:userId')
  @ApiOperation({
    summary: 'Get detailed program participation history for a user',
  })
  @ApiParam({
    name: 'userId',
    description: userParticipationConstMessages.API_PARAM_USER_ID,
    type: Number,
  })
  @ApiQuery({
    name: 'year',
    description: userParticipationConstMessages.API_QUERY_YEAR,
    required: false,
    type: Number,
  })
  @ApiQuery({
    name: 'programName',
    description: 'Filter by program name (optional)',
    required: false,
    type: String,
  })
  @ApiResponse({
    status: 200,
    description: userParticipationConstMessages.DETAILED_PROGRAMS_RETRIEVED,
    type: [ProgramParticipationDto],
  })
  @ApiResponse({
    status: 404,
    description: userParticipationConstMessages.ERROR_USER_NOT_FOUND,
  })
  async getProgramParticipationDetails(
    @Param('userId', ParseIntPipe) userId: number,
    @Query() query: GetUserParticipationQueryDto,
    @Res() res: Response,
  ): Promise<void> {
    try {
      this.logger.debug(
        `${userParticipationConstMessages.METHOD_GET_PROGRAM_PARTICIPATION_DETAILS} - userId: ${userId}`,
      );

      const programs = await this.userParticipationService.getUserParticipation(
        userId,
        query
      );

      await this.responseService.success(
        res,
        userParticipationConstMessages.DETAILED_PROGRAMS_RETRIEVED,
        programs.allPrograms,
        HttpStatus.OK,
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  @Get('stats/:userId')
  @ApiOperation({
    summary: 'Get participation statistics for a user',
  })
  @ApiParam({
    name: 'userId',
    description: userParticipationConstMessages.API_PARAM_USER_ID,
    type: Number,
  })
  @ApiResponse({
    status: 200,
    description: userParticipationConstMessages.USER_PARTICIPATION_STATS_RETRIEVED,
  })
  @ApiResponse({
    status: 404,
    description: userParticipationConstMessages.ERROR_USER_NOT_FOUND,
  })
  async getParticipationStatistics(
    @Param('userId', ParseIntPipe) userId: number,
    @Res() res: Response,
  ): Promise<void> {
    try {
      this.logger.debug(
        `${userParticipationConstMessages.METHOD_GET_PARTICIPATION_STATISTICS} - userId: ${userId}`,
      );

      const stats = await this.userParticipationService.getUserParticipation(userId, {});

      await this.responseService.success(
        res,
        userParticipationConstMessages.USER_PARTICIPATION_STATS_RETRIEVED,
        {
          totalProgramsAttended: stats.totalProgramsAttended,
          yearWiseBreakdown: stats.yearWiseParticipation,
        },
        HttpStatus.OK,
      );
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}
