import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { EntityManager, IsNull, Repository } from 'typeorm';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { SeekerProgramExperience, ProgramRegistration, User } from 'src/common/entities';
import { seekerExperienceRecordMessages } from 'src/common/constants/strings-constants';
import { FilterSeekerProgramExperienceDto } from './dto/filter-seeker-program-experience.dto';

@Injectable()
export class SeekerProgramExperienceRepository {
  constructor(
    @InjectRepository(SeekerProgramExperience)
    private readonly recordRepository: Repository<SeekerProgramExperience>,
    @InjectRepository(User)
    private readonly userRepository: Repository<User>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepository: Repository<ProgramRegistration>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  private getRecordRepository(manager?: EntityManager) {
    return manager ? manager.getRepository(SeekerProgramExperience) : this.recordRepository;
  }

  private getUserRepository(manager?: EntityManager) {
    return manager ? manager.getRepository(User) : this.userRepository;
  }

  private getRegistrationRepository(manager?: EntityManager) {
    return manager ? manager.getRepository(ProgramRegistration) : this.registrationRepository;
  }

  async findUser(userId: number, manager?: EntityManager): Promise<User | null> {
    return this.commonDataService.findOneById(this.getUserRepository(manager), userId, false);
  }

  async findRegistration(registrationId: number, manager?: EntityManager): Promise<ProgramRegistration | null> {
    return this.commonDataService.findOneById(this.getRegistrationRepository(manager), registrationId, false);
  }

  createRecord(partial: Partial<SeekerProgramExperience>, manager?: EntityManager) {
    return this.getRecordRepository(manager).create(partial);
  }

  async saveRecord(record: SeekerProgramExperience, manager?: EntityManager): Promise<SeekerProgramExperience> {
    try {
      return await this.getRecordRepository(manager).save(record);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_SAVE_FAILED, error);
    }
  }

  async findById(id: number, includeDeleted = false, manager?: EntityManager): Promise<SeekerProgramExperience | null> {
    try {
      const repo = this.getRecordRepository(manager);
      const qb = repo.createQueryBuilder('record')
        .select([
          'record.id',
          'record.type',
          'record.message',
          'record.mediaUrl',
          'record.isPrivate',
          'record.isViewed',
          'record.isApproved',
          'record.createdAt',
          'record.updatedAt',
          'record.deletedAt',
          'record.registrationId',
          'record.userId',
          'user.id',
          'user.legalFullName',
          'user.email',
          'registration.id',
          'registration.programId',
          'registration.fullName',
          'registration.mobileNumber',
          'registration.allocatedProgramId',
          'registration.registrationStatus',
          'registration.profileUrl',
          'registration.gender',
          'registration.emailAddress',
          'allocatedProgram.name',
        ])
        .leftJoin('record.user', 'user')
        .leftJoin('record.registration', 'registration')
        .leftJoin('registration.allocatedProgram', 'allocatedProgram')
        .where('record.id = :id', { id });
      if (!includeDeleted) {
        qb.andWhere('record.deletedAt IS NULL');
      } else {
        qb.withDeleted();
      }
      return await qb.getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_FETCH_FAILED, error);
    }
  }

  async findByRegId(
    id: number,
    includeDeleted = false,
    manager?: EntityManager,
  ): Promise<SeekerProgramExperience | null> {
    try {
      const repo = this.getRecordRepository(manager);
      const qb = repo.createQueryBuilder('record')
        .select([
          'record.id',
          'record.type',
          'record.message',
          'record.mediaUrl',
          'record.isPrivate',
          'record.isViewed',
          'record.isApproved',
          'record.createdAt',
          'record.updatedAt',
          'record.deletedAt',
          'record.registrationId',
          'record.userId',
          'user.id',
          'user.legalFullName',
          'user.email',
          'registration.id',
          'registration.fullName',
          'registration.mobileNumber',
          'registration.programId',
          'registration.allocatedProgramId',
          'registration.registrationStatus',
          'registration.profileUrl',
          'registration.gender',
          'registration.emailAddress',
          'allocatedProgram.name',
        ])
        .leftJoin('record.user', 'user')
        .leftJoin('record.registration', 'registration')
        .leftJoin('registration.allocatedProgram', 'allocatedProgram')
        .where('record.registrationId = :registrationId', { registrationId: id });
      if (!includeDeleted) {
        qb.andWhere('record.deletedAt IS NULL');
      } else {
        qb.withDeleted();
      }
      return await qb.getOne();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  async fetchAll(
    filter: FilterSeekerProgramExperienceDto,
    manager?: EntityManager,
  ): Promise<{ data: SeekerProgramExperience[]; pagination: any; total: number }> {
    try {
      const repo = this.getRecordRepository(manager);
      const qb = repo
        .createQueryBuilder('record')
        .select([
          'record.id',
          'record.type',
          'record.message',
          'record.mediaUrl',
          'record.isPrivate',
          'record.isViewed',
          'record.isApproved',
          'record.createdAt',
          'record.updatedAt',
          'record.deletedAt',
          'record.registrationId',
          'record.userId',
          'user.id',
          'user.legalFullName',
          'user.email',
          'registration.id',
          'registration.programId',
          'registration.allocatedProgramId',
          'registration.registrationStatus',
          'registration.fullName',
          'registration.mobileNumber',
          'registration.profileUrl',
          'registration.gender',
          'registration.emailAddress',
          'allocatedProgram.name',
          'approvals.id',
          'approvals.approvalStatus',
        ])
        .leftJoin('record.user', 'user')
        .leftJoin('record.registration', 'registration')
        .leftJoin('registration.allocatedProgram', 'allocatedProgram')
        .leftJoin('registration.approvals', 'approvals');

      if (!filter.includeDeleted) {
        qb.andWhere('record.deletedAt IS NULL');
      } else {
        qb.withDeleted();
      }

      if (filter.userId) {
        qb.andWhere('user.id = :userId', { userId: filter.userId });
      }

      if (filter.registrationId) {
        qb.andWhere('registration.id = :registrationId', { registrationId: filter.registrationId });
      }

      if (filter.programId) {
        qb.andWhere('registration.programId = :programId', { programId: filter.programId });
      }

      if (filter.allocatedProgramId !== undefined) {
        qb.andWhere('registration.allocatedProgramId = :allocatedProgramId', { allocatedProgramId: filter.allocatedProgramId });
      }

      if (filter.type) {
        qb.andWhere('LOWER(record.type) = LOWER(:type)', { type: filter.type });
      }

      if (filter.isPrivate !== undefined) {
        qb.andWhere('record.isPrivate = :isPrivate', { isPrivate: filter.isPrivate });
      }

      if (filter.isApproved !== undefined) {
        qb.andWhere('record.isApproved = :isApproved', { isApproved: filter.isApproved });
      }

      if (filter.isViewed !== undefined) {
        qb.andWhere('record.isViewed = :isViewed', { isViewed: filter.isViewed });
      }

      if (filter.approvalStatus) {
        // Convert single string to array for consistent handling
        const statusArray = Array.isArray(filter.approvalStatus) 
          ? filter.approvalStatus 
          : [filter.approvalStatus];
        if (statusArray.length > 0) {
          qb.andWhere('approvals.approvalStatus IN (:...approvalStatuses)', { approvalStatuses: statusArray });
        }
      }

      if (filter.search) {
        // search reg full name
        qb.andWhere(
          'LOWER(registration.fullName) LIKE LOWER(:search)',
          { search: `%${filter.search}%` },
        );
      }

      if (filter.sortKey) {
        // Only allow sorting by whitelisted fields for security
        const allowedSortKeys = [
          'createdAt', 'updatedAt', 'type', 'isApproved', 'isViewed', 'isPrivate', 'id',
          'registrationId', 'userId', 'message', 'mediaUrl'
        ];
        const key = allowedSortKeys.includes(filter.sortKey) ? filter.sortKey : 'createdAt';
        let direction = 'DESC';
        if (filter.sortOrder) {
          if (typeof filter.sortOrder === 'string') {
            const dir = filter.sortOrder.toUpperCase();
            if (dir === 'ASC' || dir === 'DESC') {
              direction = dir;
            }
          }
        }
        qb.orderBy(`record.${key}`, direction as 'ASC' | 'DESC');
      } else {
        qb.orderBy('record.createdAt', 'ASC');
      }
      qb.skip(filter.offset ?? 0);
      qb.take(filter.limit ?? 10);

      const [data, total] = await qb.getManyAndCount();
      this.logger.log(
        seekerExperienceRecordMessages.LISTING(
          filter.limit ?? 10,
          filter.offset ?? 0,
          filter.search,
        ),
      );
      const limit = filter.limit ?? 10;
      const offset = filter.offset ?? 0;
      const totalPages = Math.ceil(total / limit);
      const pageNumber = Math.floor(offset / limit) + 1;
      return {
        data,
        pagination: {
          totalPages,
          pageNumber,
          pageSize: limit,
          totalRecords: total,
          numberOfRecords: Math.min(limit, total - offset),
        },
        total,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.SEEKER_EXPERIENCE_FETCH_FAILED, error);
    }
  }
}