import { Injectable } from '@nestjs/common';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { UserRepository } from 'src/user/user.repository';
import { ROLE_KEYS } from 'src/common/constants/strings-constants';
import { ProgramRegistrationService } from 'src/program-registration/program-registration.service';
import { ProgramRegistrationRepository } from 'src/program-registration/program-registration.repository';
import { ProgramRepository } from 'src/program/program.repository';
import { Between, In, IsNull, LessThan, Not, Or, Repository } from 'typeorm';
import { ProgramRegistration, User } from 'src/common/entities';
import { InjectRepository } from '@nestjs/typeorm';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { ProgramStatusEnum } from 'src/common/enum/program-status.enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { CommunicationService } from 'src/communication/communication.service';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { ConfigService } from '@nestjs/config';
import { RegistrationService } from 'src/registration/registration.service';
import { SchedulerLogRepository } from './scheduler-log.repository';
import { FEATURE_FLAG_KEYS } from 'src/common/constants/constants';
import { FeatureFlagService } from 'src/feature-flag/feature-flag.service';

@Injectable()
export class SchedulerService {
  constructor(
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
    private readonly userRepository: UserRepository,
    private readonly programRepository: ProgramRepository,
    @InjectRepository(ProgramRegistration)
    private readonly programRegistrationRepo: Repository<ProgramRegistration>,
    private readonly communicationService: CommunicationService,
    private readonly configService: ConfigService,
    private readonly registrationService: RegistrationService,
    private readonly schedulerLogRepository: SchedulerLogRepository,
    private readonly featureFlagService: FeatureFlagService,
  ) {}

  async sendPaymentReminders(): Promise<void> {
    const featureFlag = await this.featureFlagService.isActiveFeatureEnabled(
      FEATURE_FLAG_KEYS.SEND_PAYMENT_REMINDERS_TO_RM,
    );
    if (!featureFlag) {
      this.logger.log(
        'Feature flag for sending payment reminders to RM is disabled. Exiting task.',
      );
      return;
    }
    this.logger.log('Sending payment reminders...');
    try {
      const rmUsers = await this.userRepository.getUsersByRoleKey(ROLE_KEYS.RELATIONAL_MANAGER);
      const futureDates: Date[] = [];
      for (let i = 0; i < 6; i++) {
        const days = 10 + i * 2;
        futureDates.push(new Date(Date.now() + days * 24 * 60 * 60 * 1000));
      }
      const dateConditions = futureDates.map((date) => {
        const startOfDay = new Date(
          Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate(), 0, 0, 0, 0),
        );
        const endOfDay = new Date(
          Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate(), 23, 59, 59, 999),
        );
        return Between(startOfDay, endOfDay);
      });
      const programs = await this.programRepository.findAllPrimaryPrograms({
          status: ProgramStatusEnum.PUBLISHED,
          startsAt: Or(...dateConditions),
        },
        'all',
        'all',
      );
      if (!programs || !programs.data || programs.data.length === 0) {
        this.logger.log('No published programs found for upcoming dates');
        return;
      }

      for (const program of programs.data) {
        for (const rmUser of rmUsers) {
          const whereClause = {
            rmContact: rmUser.id,
            // Only get registrations where payment is required (not free seats)
            isFreeSeat: false,
            // Filter for registrations with payment details that are pending or incomplete
            // Consider registrations with no payment details or with pending/incomplete payment status
            paymentDetails: [
              {
                paymentStatus: In([
                  PaymentStatusEnum.ONLINE_PENDING,
                  PaymentStatusEnum.OFFLINE_PENDING,
                  PaymentStatusEnum.DRAFT,
                  PaymentStatusEnum.FAILED,
                ]),
                deletedAt: IsNull(), // Ensure payment details are not soft deleted
              },
              {},
            ],
            // Exclude cancelled and draft registrations
            registrationStatus: Not(
              In([RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED]),
            ),
            program: {
              id: program.id,
              deletedAt: IsNull(), // Ensure program is not soft deleted
              status: In([ProgramStatusEnum.PUBLISHED]),
              type: {
                key: 'PT_HDBMSD', // Only get registrations for PT_HDBMSD program type
              },
              // Filter for programs starting between 10 and 20 days from now
              startsAt: Or(...dateConditions),
            },
            deletedAt: IsNull(), // Ensure registration is not soft deleted
          };
          const relations = [
            'program',
            'program.type',
            'programSession',
            'invoiceDetails',
            'paymentDetails',
            'travelInfo',
            'travelPlans',
            'user',
            'approvals',
            'allocatedProgram',
            'allocatedSession',
            'swapsRequests',
            'swapsRequests.requestedPrograms',
            'ratings',
            'rmContactUser',
            'preferences',
            'preferences.preferredProgram',
            'preferences.preferredSession',
            'recommendation',
            'goodies',
          ];
          const registrations = await this.commonDataService.get(
            this.programRegistrationRepo,
            undefined,
            whereClause,
            'all',
            'all',
            undefined,
            undefined,
            relations,
          );
          if (registrations.length === 0) {
            continue; // No registrations found for this RM, skip to next
          }
          this.logger.log(`Sending payment reminder for RM: ${rmUser.email}`);
          const templateData: SendTemplateMessageDto = {
            whatsappNumber: (rmUser.countryCode + rmUser.phoneNumber).replace('+', ''),
            templateName: process.env.WATI_RM_PAYMENT_PENDING_REMINDER_LIST_TEMPLATE_ID || '',
            broadcastName: process.env.WATI_RM_PAYMENT_PENDING_REMINDER_LIST_TEMPLATE_ID || '',
            parameters: [
              {
                name: 'rm_name',
                value:
                  rmUser.firstName && rmUser.lastName
                    ? rmUser.firstName + ' ' + rmUser.lastName
                    : rmUser.fullName,
              },
              {
                name: 'reg_count',
                value: registrations.length.toString(),
              },
            ],
          };
          try {
            await this.communicationService.sendTemplateMessage(templateData);
            this.logger.log(`Payment reminder sent to RM: ${rmUser.email}`);
          } catch (error) {
            this.logger.log(`Error sending payment reminder for RM:`, { error });
          }
        }
      }
    } catch (error) {
      this.logger.log(`Error sending payment reminder for RM:`, { error });
    }
  }

  /**
   * Update signed URLs for registrations
   * This is a scheduled job that runs periodically to refresh S3 signed URLs
   */
  async updateSignedUrlsScheduled(): Promise<{
    totalPrograms: number;
    successCount: number;
    failedCount: number;
    totalUpdates: number;
    executionTime: number;
  }> {
    const startTime = Date.now();
    this.logger.log('Starting scheduled signed URLs update job...');

    // Create initial log entry
    const schedulerLog = await this.schedulerLogRepository.createSchedulerLog({
      jobName: 'signed-urls-update',
      jobType: 'signed-urls-update',
      executionStatus: 'in_progress',
      triggeredByUserId: -2, // System user
      triggeredType: 'scheduled',
      attemptNumber: 1,
      totalAttempts: 1,
    });

    const scheduledJobEnabled = this.configService.get<string>('SIGNED_URLS_SCHEDULED_JOB_ENABLED', 'false') === 'true';
    
    if (!scheduledJobEnabled) {
      this.logger.log('Scheduled signed URLs update job is disabled');
      
      // Update log as skipped
      await this.schedulerLogRepository.updateSchedulerLog(schedulerLog.id, {
        executionStatus: 'success',
        executionTimeMs: Date.now() - startTime,
        completedAt: new Date(),
        responseData: { message: 'Job is disabled', totalPrograms: 0 },
      });
      
      return {
        totalPrograms: 0,
        successCount: 0,
        failedCount: 0,
        totalUpdates: 0,
        executionTime: 0,
      };
    }

    try {
      // Get program IDs from environment variable or fetch active programs
      const programIdsConfig = this.configService.get<string>('SIGNED_URLS_SCHEDULED_PROGRAM_IDS', '');
      let programIds: number[] = [];

      if (programIdsConfig) {
        programIds = programIdsConfig.split(',').map(id => parseInt(id.trim(), 10)).filter(id => !isNaN(id));
      }

      if (programIds.length === 0) {
        this.logger.log('No program IDs configured for scheduled signed URLs update');
        
        // Update log as success with no programs
        await this.schedulerLogRepository.updateSchedulerLog(schedulerLog.id, {
          executionStatus: 'success',
          executionTimeMs: Date.now() - startTime,
          completedAt: new Date(),
          responseData: { message: 'No programs configured', totalPrograms: 0 },
        });
        
        return {
          totalPrograms: 0,
          successCount: 0,
          failedCount: 0,
          totalUpdates: 0,
          executionTime: Date.now() - startTime,
        };
      }

      this.logger.log(`Found ${programIds.length} programs for signed URLs update`);

      let successCount = 0;
      let failedCount = 0;
      let totalUpdates = 0;
      const programResults: any[] = [];

      // System user for scheduled operations
      const systemUser = { id: -2, fullName: 'System User' } as User;

      for (const programId of programIds) {
        try {
          this.logger.log(`Processing program ${programId} for signed URLs update`);
          
          // Call registration service to update signed URLs
          const result = await this.registrationService.updateSignedUrlsForRegistrations(programId, undefined, systemUser);
          totalUpdates += result.updatedCount.registration + result.updatedCount.travelInfo + result.updatedCount.travelPlan;
          
          successCount++;
          programResults.push({
            programId,
            status: 'success',
            updatedCount: result.updatedCount,
          });
          
          this.logger.log(`Successfully updated signed URLs for program ${programId}. Updated: ${result.updatedCount.registration} registrations, ${result.updatedCount.travelInfo} travel info, ${result.updatedCount.travelPlan} travel plans`);
        } catch (error) {
          failedCount++;
          programResults.push({
            programId,
            status: 'failed',
            error: error.message,
          });
          this.logger.error(`Failed to update signed URLs for program ${programId}:`, error);
        }
      }

      const executionTime = Date.now() - startTime;
      this.logger.log(
        `Scheduled signed URLs update job completed. ` +
        `Programs: ${programIds.length}, ` +
        `Success: ${successCount}, ` +
        `Failed: ${failedCount}, ` +
        `Total updates: ${totalUpdates}, ` +
        `Time: ${executionTime}ms`
      );

      // Update log as success
      await this.schedulerLogRepository.updateSchedulerLog(schedulerLog.id, {
        executionStatus: failedCount > 0 && successCount === 0 ? 'failed' : 'success',
        executionTimeMs: executionTime,
        completedAt: new Date(),
        responseData: {
          totalPrograms: programIds.length,
          successCount,
          failedCount,
          totalUpdates,
          programResults,
        },
      });

      return {
        totalPrograms: programIds.length,
        successCount,
        failedCount,
        totalUpdates,
        executionTime,
      };
    } catch (error) {
      this.logger.error('Scheduled signed URLs update job failed:', error);
      
      // Update log as failed
      await this.schedulerLogRepository.updateSchedulerLog(schedulerLog.id, {
        executionStatus: 'failed',
        executionTimeMs: Date.now() - startTime,
        failedAt: new Date(),
        errorMessage: error.message,
        errorStack: error.stack,
      });
      
      throw error;
    }
  }
}
