import { Injectable } from '@nestjs/common';
import { EntityManager } from 'typeorm';
import { RoomInventoryRepository } from './room-inventory.repository';
import { ProgramRoomInventoryMapRepository } from './program-room-inventory-map.repository';
import { ProgramRoomInventoryMap } from '../common/entities/program-room-inventory-map.entity';
import { RoomStatus } from '../common/enums/room-status.enum';
import { InifniNotFoundException } from '../common/exceptions/infini-notfound-exception';
import { ERROR_CODES } from '../common/constants/error-string-constants';
import { AppLoggerService } from '../common/services/logger.service';
import { handleKnownErrors } from '../common/utils/handle-error.util';
import InifniBadRequestException from '../common/exceptions/infini-badrequest-exception';

/**
 * Service for managing program room inventory map operations
 * Provides business logic for room inventory validation and management
 */
@Injectable()
export class ProgramRoomInventoryMapService {
  constructor(
    private readonly roomInventoryRepository: RoomInventoryRepository,
    private readonly programRoomInventoryMapRepository: ProgramRoomInventoryMapRepository,
    private readonly logger: AppLoggerService,
  ) {}

  /**
   * Calculate current occupancy based on room allocations
   * @param roomInventory - Room inventory entity with loaded roomAllocations
   * @returns Current number of allocated beds
   */
  private calculateCurrentOccupancy(roomInventory: ProgramRoomInventoryMap): number {
    return roomInventory.roomAllocations?.length || 0;
  }

  /**
   * Calculate available beds in the room
   * @param roomInventory - Room inventory entity with loaded room and roomAllocations
   * @returns Number of available beds
   */
  private calculateAvailableBeds(roomInventory: ProgramRoomInventoryMap): number {
    if (!roomInventory.room?.occupancy) return 0;
    return roomInventory.room.occupancy - this.calculateCurrentOccupancy(roomInventory);
  }

  /**
   * Check if room is fully occupied
   * @param roomInventory - Room inventory entity with loaded room and roomAllocations
   * @returns True if room is at full capacity
   */
  private isRoomFullyOccupied(roomInventory: ProgramRoomInventoryMap): boolean {
    return this.calculateCurrentOccupancy(roomInventory) >= (roomInventory.room?.occupancy || 0);
  }

  /**
   * Determine room status based on current occupancy and reservation status
   * @param roomInventory - Room inventory entity with loaded room and roomAllocations
   * @returns Appropriate room status
   */
  private determineRoomStatus(roomInventory: ProgramRoomInventoryMap): RoomStatus {
    const currentOccupancy = this.calculateCurrentOccupancy(roomInventory);
    const totalCapacity = roomInventory.room?.occupancy || 0;

    if (roomInventory.isReserved) {
      return RoomStatus.RESERVED;
    }

    if (currentOccupancy === 0) {
      return RoomStatus.AVAILABLE;
    } else if (currentOccupancy >= totalCapacity) {
      return RoomStatus.ALLOTTED;
    } else {
      return RoomStatus.PARTIALLY_ALLOTTED;
    }
  }

  /**
   * Validate room inventory mapping for a program
   * @param roomInventoryId - Room inventory map ID
   * @param programId - Program ID to validate against
   * @param subProgramId - Optional sub-program ID to validate against
   * @returns Promise<ProgramRoomInventoryMap>
   * @throws InifniNotFoundException if mapping not found or invalid
   */
  async validateRoomInventoryMapping(
    roomInventoryId: number,
    programId: number,
    subProgramId?: number,
  ): Promise<ProgramRoomInventoryMap> {
    this.logger.log('Room inventory mapping validation request received', { 
      roomInventoryId, 
      programId, 
      subProgramId 
    });

    try {
      let roomInventory: ProgramRoomInventoryMap | null;

      if (subProgramId) {
        // If sub-program ID is provided, validate against sub-program
        roomInventory = await this.programRoomInventoryMapRepository.findByIdAndSubProgramId(
          roomInventoryId,
          subProgramId,
        );
        
        if (!roomInventory) {
          this.logger.error('Room inventory not found for sub-program', JSON.stringify({ 
            roomInventoryId, 
            subProgramId 
          }));
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_SUBPROGRAM_MISMATCH,
            null,
            null,
            subProgramId.toString(),
          );
        }
      } else {
        // If no sub-program ID, validate against main program
        roomInventory = await this.programRoomInventoryMapRepository.findByIdAndProgramId(
          roomInventoryId,
          programId,
        );
        
        if (!roomInventory) {
          this.logger.error('Room inventory not found for program: ' + JSON.stringify({ 
            roomInventoryId, 
            programId 
          }));
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_PROGRAM_MISMATCH,
            null,
            null,
            programId.toString(),
          );
        }
      }

      this.logger.log('Room inventory mapping validated successfully', {
        roomInventoryId,
        programId,
        subProgramId
      });

      return roomInventory;
    } catch (error) {
      this.logger.error('Error in room inventory mapping validation', error.stack, {
        roomInventoryId,
        programId,
        subProgramId
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_VALIDATION_FAILED, error);
    }
  }

  /**
   * Check if room has sufficient capacity for allocations
   * @param roomInventoryId - Room inventory map ID
   * @param requiredCapacity - Number of allocations needed
   * @returns Promise<boolean>
   */
  async checkRoomCapacity(roomInventoryId: number, requiredCapacity: number): Promise<boolean> {
    this.logger.log('Room capacity check request received', {
      roomInventoryId,
      requiredCapacity
    });

    try {
      const roomInventory = await this.programRoomInventoryMapRepository.findById(roomInventoryId);
      
      if (!roomInventory) {
        this.logger.error('Room inventory not found for capacity check: ' + JSON.stringify({ roomInventoryId }));
        return false;
      }

      const hasCapacity = roomInventory.remainingOccupancy >= requiredCapacity;

      this.logger.log('Room capacity check completed', {
        roomInventoryId,
        requiredCapacity,
        remainingOccupancy: roomInventory.remainingOccupancy,
        hasCapacity
      });

      return hasCapacity;
    } catch (error) {
      this.logger.error('Error checking room capacity', error.stack, {
        roomInventoryId,
        requiredCapacity
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_CAPACITY_CHECK_FAILED, error);
    }
  }

  /**
   * Update room occupancy and status after allocation
   * @param roomInventoryId - Room inventory map ID
   * @param allocatedCount - Number of new allocations
   * @returns Promise<ProgramRoomInventoryMap>
   */
  async updateRoomOccupancy(roomInventoryId: number, allocatedCount: number): Promise<ProgramRoomInventoryMap> {
    this.logger.log('Room occupancy update request received', { 
      roomInventoryId, 
      allocatedCount 
    });

    try {
      const roomInventory = await this.programRoomInventoryMapRepository.findWithAllocations(roomInventoryId);
      
      if (!roomInventory) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_NOT_FOUND,
          null,
          null,
          roomInventoryId.toString(),
        );
      }

      // Validate sufficient capacity
      if (roomInventory.remainingOccupancy < allocatedCount) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_INSUFFICIENT_CAPACITY,
          null,
          null,
          allocatedCount.toString(),
          roomInventory.remainingOccupancy.toString()
        );
      }

      // Update remaining occupancy
      const newRemainingOccupancy = roomInventory.remainingOccupancy - allocatedCount;
      roomInventory.remainingOccupancy = newRemainingOccupancy;

      // Update room status based on remaining occupancy
      if (roomInventory.isReserved) {
        roomInventory.roomStatus = RoomStatus.RESERVED;
      } else if (newRemainingOccupancy === (roomInventory.room?.occupancy || 0)) {
        // All seats are available
        roomInventory.roomStatus = RoomStatus.AVAILABLE;
      } else if (newRemainingOccupancy <= 0) {
        // No seats remaining - fully allotted
        roomInventory.roomStatus = RoomStatus.ALLOTTED;
      } else {
        // Some seats remaining - partially allotted
        roomInventory.roomStatus = RoomStatus.PARTIALLY_ALLOTTED;
      }

      // Save the updated inventory using repository
      const updatedInventory = await this.programRoomInventoryMapRepository.save(roomInventory);
      
      this.logger.log('Room occupancy updated successfully', { 
        roomInventoryId, 
        allocatedCount, 
        newRemainingOccupancy,
        newStatus: roomInventory.roomStatus
      });

      return updatedInventory;
    } catch (error) {
      this.logger.error('Error updating room occupancy', error.stack, {
        roomInventoryId,
        allocatedCount
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_OCCUPANCY_UPDATE_FAILED, error);
    }
  }

  /**
   * Update room occupancy and status after deallocation (when removing allocations)
   * @param roomInventoryId - Room inventory map ID
   * @param deallocatedCount - Number of removed allocations
   * @param transactionManager - Optional EntityManager for transactional operations
   * @returns Promise<ProgramRoomInventoryMap>
   */
  async updateRoomOccupancyAfterDeallocation(
    roomInventoryId: number, 
    deallocatedCount: number,
    transactionManager?: EntityManager,
  ): Promise<ProgramRoomInventoryMap> {
    this.logger.log('Room occupancy deallocation update request received', { 
      roomInventoryId, 
      deallocatedCount 
    });

    try {
      // Use transaction manager if provided, otherwise use repository directly
      const roomInventory = transactionManager
        ? await transactionManager.findOne(ProgramRoomInventoryMap, {
            where: { id: roomInventoryId },
            relations: ['room', 'program', 'roomAllocations'],
          })
        : await this.programRoomInventoryMapRepository.findWithAllocations(roomInventoryId);
      
      if (!roomInventory) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_NOT_FOUND,
          null,
          null,
          roomInventoryId.toString(),
        );
      }

      // Update remaining occupancy
      const newRemainingOccupancy = roomInventory.remainingOccupancy + deallocatedCount;
      roomInventory.remainingOccupancy = newRemainingOccupancy;

      // Calculate current occupancy based on remaining capacity (more reliable after deallocation)
      const totalCapacity = roomInventory.room?.occupancy || 0;
      const currentOccupancy = totalCapacity - newRemainingOccupancy;   
      // Update room status based on new occupancy
      if (roomInventory.isReserved) {
        roomInventory.roomStatus = RoomStatus.RESERVED;
      } else if (currentOccupancy === 0) {
        roomInventory.roomStatus = RoomStatus.AVAILABLE;
      } else if (currentOccupancy >= totalCapacity) {
        roomInventory.roomStatus = RoomStatus.ALLOTTED;
      } else {
        roomInventory.roomStatus = RoomStatus.PARTIALLY_ALLOTTED;
      }

      // Save the updated inventory using transaction manager or repository
      const updatedInventory = transactionManager
        ? await transactionManager.save(ProgramRoomInventoryMap, roomInventory)
        : await this.programRoomInventoryMapRepository.save(roomInventory);
      
      this.logger.log('Room occupancy updated after deallocation successfully', { 
        roomInventoryId, 
        deallocatedCount, 
        newRemainingOccupancy,
        newStatus: roomInventory.roomStatus,
        currentOccupancy,
        totalCapacity
      });

      return updatedInventory;
    } catch (error) {
      this.logger.error('Error updating room occupancy after deallocation', error.stack, {
        roomInventoryId,
        deallocatedCount
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_DEALLOCATION_FAILED, error);
    }
  }

  /**
   * Get room occupancy details with status information
   * @param roomInventoryId - Room inventory map ID
   * @returns Promise<object> Room occupancy details
   */
  async getRoomOccupancyDetails(roomInventoryId: number): Promise<{
    roomInventoryId: number;
    totalCapacity: number;
    currentOccupancy: number;
    remainingOccupancy: number;
    roomStatus: RoomStatus;
    isReserved: boolean;
    reservedFor?: string;
    availableBeds: number;
    isFullyOccupied: boolean;
  }> {
    this.logger.log('Room occupancy details request received', { roomInventoryId });

    try {
      const roomInventory = await this.programRoomInventoryMapRepository.findWithAllocations(roomInventoryId);
      
      if (!roomInventory) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_NOT_FOUND,
          null,
          null,
          roomInventoryId.toString(),
        );
      }

      const totalCapacity = roomInventory.room?.occupancy || 0;
      const currentOccupancy = this.calculateCurrentOccupancy(roomInventory);
      const availableBeds = this.calculateAvailableBeds(roomInventory);
      const isFullyOccupied = this.isRoomFullyOccupied(roomInventory);

      const details = {
        roomInventoryId,
        totalCapacity,
        currentOccupancy,
        remainingOccupancy: roomInventory.remainingOccupancy,
        roomStatus: roomInventory.roomStatus,
        isReserved: roomInventory.isReserved,
        reservedFor: roomInventory.reservedFor,
        availableBeds,
        isFullyOccupied
      };

      this.logger.log('Room occupancy details retrieved successfully', {
        roomInventoryId,
        totalCapacity,
        currentOccupancy,
        remainingOccupancy: roomInventory.remainingOccupancy
      });

      return details;
    } catch (error) {
      this.logger.error('Error retrieving room occupancy details', error.stack, {
        roomInventoryId
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_OCCUPANCY_DETAILS_FAILED, error);
    }
  }

  /**
   * Recalculate and update room status for a specific room
   * Useful for data consistency checks and corrections
   * @param roomInventoryId - Room inventory map ID
   * @returns Promise<ProgramRoomInventoryMap>
   */
  async recalculateRoomStatus(roomInventoryId: number): Promise<ProgramRoomInventoryMap> {
    this.logger.log('Room status recalculation request received', { roomInventoryId });

    try {
      const roomInventory = await this.programRoomInventoryMapRepository.findWithAllocations(roomInventoryId);
      
      if (!roomInventory) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_NOT_FOUND,
          null,
          null,
          roomInventoryId.toString(),
        );
      }

      // Recalculate based on actual allocations
      const actualOccupancy = this.calculateCurrentOccupancy(roomInventory);
      const totalCapacity = roomInventory.room?.occupancy || 0;

      // Update remaining occupancy based on actual data
      roomInventory.remainingOccupancy = totalCapacity - actualOccupancy;

      // Update status based on actual occupancy
      roomInventory.roomStatus = this.determineRoomStatus(roomInventory);

      // Save the recalculated inventory using repository
      const updatedInventory = await this.programRoomInventoryMapRepository.save(roomInventory);
      
      this.logger.log('Room status recalculated successfully', { 
        roomInventoryId, 
        actualOccupancy,
        totalCapacity,
        updatedStatus: roomInventory.roomStatus,
        remainingOccupancy: roomInventory.remainingOccupancy
      });

      return updatedInventory;
    } catch (error) {
      this.logger.error('Error recalculating room status', error.stack, {
        roomInventoryId
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_ROOM_INVENTORY_MAP_STATUS_RECALCULATION_FAILED, error);
    }
  }
}