import { Injectable, Logger } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { ProgramRoomInventoryMap } from '../common/entities/program-room-inventory-map.entity';

/**
 * Repository for managing program room inventory map operations
 * Provides data access methods for room inventory mapping
 */
@Injectable()
export class ProgramRoomInventoryMapRepository {
  private readonly logger = new Logger(ProgramRoomInventoryMapRepository.name);

  constructor(
    @InjectRepository(ProgramRoomInventoryMap)
    private readonly repository: Repository<ProgramRoomInventoryMap>,
  ) {}

  /**
   * Find a room inventory mapping by ID and program ID
   * @param id - Room inventory map ID
   * @param programId - Program ID to validate mapping
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findByIdAndProgramId(id: number, programId: number): Promise<ProgramRoomInventoryMap | null> {
    this.logger.debug('Finding room inventory by ID and program ID', { id, programId });
    
    return await this.repository.findOne({
      where: {
        id,
        programId,
      },
      relations: ['room', 'program'],
    });
  }

  /**
   * Find a room inventory mapping by ID and sub-program ID
   * @param id - Room inventory map ID
   * @param subProgramId - Sub-program ID to validate mapping
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findByIdAndSubProgramId(id: number, subProgramId: number): Promise<ProgramRoomInventoryMap | null> {
    this.logger.debug('Finding room inventory by ID and sub-program ID', { id, subProgramId });
    
    return await this.repository.findOne({
      where: {
        id,
        subProgramId,
      },
      relations: ['room', 'program', 'subProgram'],
    });
  }

  /**
   * Find a room inventory mapping by ID with all relations
   * @param id - Room inventory map ID
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findByIdWithRelations(id: number): Promise<ProgramRoomInventoryMap | null> {
    this.logger.debug('Finding room inventory by ID with relations', { id });
    
    return await this.repository.findOne({
      where: { id },
      relations: ['room', 'program', 'subProgram', 'roomAllocations'],
    });
  }

  /**
   * Find a room inventory mapping by ID
   * @param id - Room inventory map ID
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findById(id: number): Promise<ProgramRoomInventoryMap | null> {
    this.logger.debug('Finding room inventory by ID', { id });
    
    return await this.repository.findOne({
      where: { id },
      relations: ['room', 'program', 'subProgram'],
    });
  }

  /**
   * Update remaining occupancy for a room inventory mapping
   * @param id - Room inventory map ID
   * @param remainingOccupancy - New remaining occupancy value
   * @returns Promise<void>
   */
  async updateRemainingOccupancy(id: number, remainingOccupancy: number): Promise<void> {
    this.logger.debug('Updating remaining occupancy', { id, remainingOccupancy });
    
    await this.repository.update(id, { remainingOccupancy });
  }

  /**
   * Save a room inventory mapping entity
   * @param roomInventory - Room inventory entity to save
   * @returns Promise<ProgramRoomInventoryMap>
   */
  async save(roomInventory: ProgramRoomInventoryMap): Promise<ProgramRoomInventoryMap> {
    this.logger.debug('Saving room inventory mapping', { id: roomInventory.id });
    
    return await this.repository.save(roomInventory);
  }

  /**
   * Find room inventory with room allocations for occupancy calculations
   * @param id - Room inventory map ID
   * @returns Promise<ProgramRoomInventoryMap | null>
   */
  async findWithAllocations(id: number): Promise<ProgramRoomInventoryMap | null> {
    this.logger.debug('Finding room inventory with allocations', { id });
    
    return await this.repository.findOne({
      where: { id },
      relations: ['room', 'roomAllocations', 'program', 'subProgram'],
    });
  }
}