import { Injectable } from '@nestjs/common';
import { DataSource, In, Not, IsNull } from 'typeorm';
import { CreateRegistrationDto } from './dto/create-registration.dto';
import { UpdateParentalConsentDto, UpdateRegistrationDto } from './dto/update-registration.dto';
import { allowedRolesForExperienceTags, KPI_CATEGORIES, KPI_FILTERS, REGISTRATION_BINDING_KEYS, SORT_ORDER } from 'src/common/constants/constants';
import { RegistrationRepository } from './registration.repository';
import { 
  REGISTRATION_LIST_REQUIRED_RELATIONS,
  REGISTRATION_LIST_SELECT_FIELDS,
  REGISTRATION_LIST_RELATION_SELECT
} from 'src/common/config/registration-list-view.config';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { formatDate, formatDateTimeIST, calculateAge, formatBooleanValue, replaceUnderscoreWithSpace, fetchPdfAsBase64, generatePDF, formatDateIST, getAirlineFullName, toTitleCase, formatTravelType, formatLocation, formatDateTimeISTWithoutSec, formatSizeSelection, formatRatriaPillar, getEarliestAddedDate, getLatestModifiedDate, formatGoodiesStatus, replaceStringPlaceholders, shouldExcludeFromSignedUrl, formatSwapStatus, getTravelOverAllStatus, getRegistrationStatusDateTime, getStatusDateTimeLabel } from 'src/common/utils/common.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { registrationMessages, stateGstCodes, zeptoEmailCreadentials, ROLE_KEYS, ENVIRONMENTS, ROLE_VALUES, CLEARANCE_REASONS } from 'src/common/constants/strings-constants';
import { allowedRolesForRMFilter, citiesList, phoneNumberType, REPORT_ACCESS_MATRIX, USER_PROFILE_EXTENSION_BINDING_KEYS } from 'src/common/constants/constants';
import {
  FormSection,
  Program,
  ProgramQuestion,
  ProgramSession,
  Question,
  RegistrationCustomResponse,
  RegistrationGroupMap,
  RegistrationPairMap,
  RoomAllocation,
  User,
  ProgramRegistration,
} from 'src/common/entities';
import { UserRepository } from 'src/user/user.repository';
import { UserService } from 'src/user/user.service';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { RegistrationBasicStatusEnum } from 'src/common/enum/registration-basic-status.enum';
import { CancelRegistrationDto } from './dto/cancel-registration.dto';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { CommunicationService } from 'src/communication/communication.service';
import { SendBulkEmailDto, SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { TravelStatusEnum } from 'src/common/enum/travel-status.enum';
import { GenderEnum } from 'src/common/enum/gender.enum';
import { normalizeFilterValues } from './utils/filter-normalization.util';
import { TravelTypeEnum } from 'src/common/enum/travel-type.enum';
import { RegistrationAnswerDto } from './dto/create-registration.dto';
import { MajorCities, SEEKER_FE_EDIT_REGISTRATION_PATH, SEEKER_FE_REG_PATH, USER_PROFILE_BINDING_KEYS } from 'src/common/constants/constants';
import { ExcelService } from 'src/common/services/excel.service';
import { areGSTsFromSameState } from 'src/common/utils/common.util';
import { ProgramRegistrationSwapRepository } from 'src/program-registration/repositories/program-registration-swap.repository';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { ProgramRegistrationService } from 'src/program-registration/program-registration.service';
import { FILTER_HIERARCHY_CONFIG, REPORT_DEFINITIONS } from 'src/common/config/filter-hierarchy.config';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';
import { RecommendationLevel } from 'src/common/enum/recommendation-level.enum';
import { GoodiesStatusEnum } from 'src/common/enum/goodies-status.enum';
import { CommunicationTypeEnum } from 'src/common/enum/communication-type.enum';
import { CommunicationTemplatesKeysEnum } from 'src/common/enum/communication-template-keys.enum';
import { CommunicationCategoryEnum } from 'src/common/enum/communication-category.enum';
import { RegistrationSortKey } from 'src/common/enum/registration-sort-key.enum';
import { UpdateUserDto } from 'src/user/dto/update-user.dto';
import { CLEAR_REGISTRATION_MESSAGES } from 'src/common/constants/strings-constants';
import { UpdateUserProfileExtensionDto } from 'src/user-profile-extension/dto/update-user-profile-extension.dto';
import { UserProfileExtensionService } from 'src/user-profile-extension/user-profile-extension.service';
import { PersonTypeEnum } from 'src/common/enum/person-type.enum';
import { PARENTAL_DECLARATION, TERMS_AND_CONDITIONS } from 'src/common/templates/templates';
import { ParentalDeclaration, TermsAndCconditionsTemplate } from 'src/audit-history/interfaces/templates.interface';
import { AwsS3Service } from 'src/common/services/awsS3.service';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import { Msg91Service } from 'src/common/services/msg91.service';
import { ConfigService } from '@nestjs/config';
import { ViewTypeEnum } from 'src/common/enum/view-type.enum';
import { SwapRequirementEnum } from 'src/common/enum/swap-requirement.enum';
import { SEAT_ALLOCATION_RELATION_SELECT, SEAT_ALLOCATION_REQUIRED_RELATIONS, SEAT_ALLOCATION_SELECT_FIELDS } from 'src/common/config/seat-allocation.config';
import { AllocationClearingService } from 'src/common/services/allocation-clearing.service';
// Role-based report access matrix for Phase 1 implementation


// Define comprehensive return type interface for registration operations
interface RegistrationResult {
  registrationId: number;
  registrationLevel?: string;
  registrationStatus?: RegistrationStatusEnum;
  basicDetailsStatus?: RegistrationBasicStatusEnum;
  
  // Waitlist related properties
  waitlisted?: boolean;
  waitlistPosition?: number;
  participantRemainingSeats?: number;
  
  // Approval related properties
  pendingApproval?: boolean;
  approvalType?: ApprovalStatusEnum;
  
  // Direct registration properties
  registrationSeqNumber?: string;
  remainingSeats?: number;
  
  // Update related properties
  formSection?: string;
  formSectionId?: number;
  updated?: boolean;
  
  // Cancellation related properties
  cancelled?: boolean;
}

// KPI interfaces for Default List View
interface DefaultKPIResult {
  registrations: {
    categoryLabel: string;
    items: KPIItem[];
  };
  programs?: {
    categoryLabel: string;
    items: KPIItem[];
  };
  payments?: {
    categoryLabel: string;
    items: KPIItem[];
  };
  invoices?: {
    categoryLabel: string;
    items: KPIItem[];
  };
  travelAndLogistics?: {
    categoryLabel: string;
    items: KPIItem[];
  };
}

interface KPIItem {
  label: string;
  value: number;
  kpiCategory: string;
  kpiFilter: string;
  order : number
  communicationCategory?: CommunicationCategoryEnum;
  orgCount?: number;
}

// KPI interfaces for Mahatria
interface AllocatedProgramMetrics {
  programs: {
    programId: number;
    programName: string;
    allocatedCount: number;
    maleCount: number;
    femaleCount: number;
    organisationUserCount: number;
    totalSeatsCount: number;
    maleOrganisationUserCount: number;
    femaleOrganisationUserCount: number;
    roomMatePreferenceCount: number;
    swapRequestsCount: number;
    startsAt: Date;
    totalBedCount: number;
  }[];
  holdCount: number;
  rejectCount: number;
  holdMaleCount: number;
  holdFemaleCount: number;
  holdOrganisationUserCount: number;
  rejectMaleCount: number;
  rejectFemaleCount: number;
  rejectOrganisationUserCount: number;
  holdMaleOrganisationUserCount: number;
  holdFemaleOrganisationUserCount: number;
  rejectMaleOrganisationUserCount: number;
  rejectFemaleOrganisationUserCount: number;
  holdProgramPreferenceCount: { programId: number; programName: string; count: number; }[];
  rejectProgramPreferenceCount: { programId: number; programName: string; count: number; }[];
  blessedCount: number;
  blessedMaleCount: number;
  blessedFemaleCount: number;
  blessedOrganisationUserCount: number;
  blessedMaleOrganisationUserCount: number;
  blessedFemaleOrganisationUserCount: number;
  swapRequestsCount: number;
  roomMatePreferenceCount: number;
}

interface UnallocatedProgramMetrics {
  programs: {
    programId: number;
    programName: string;
    count: number;
    maleCount: number;
    femaleCount: number;
    organisationUserCount: number;
    maleOrganisationUserCount: number;
    femaleOrganisationUserCount: number;
    totalSeatsCount: number;
  }[];
  totalMaleCount: number;
  totalFemaleCount: number;
  totalUnallocatedCount: number;
  totalMahatriaChoiceCount: number;
  totalOrganisationUserCount: number;
  totalMaleOrganisationUserCount: number;
  totalFemaleOrganisationUserCount: number;
  totalMahatriaChoiceOrganisationUserCount: number;
}

interface AllRegistrationMetrics{
  totalRegistrations: number;
  mahatriaChoice: number;
  preferredPrograms: {
    id:string;
    name:string;
    count: number;
  }[];
  maleCount: number;
  femaleCount: number;
}

interface MahatriaKPIResult {
  isGroupedProgram: boolean;
  primaryProgramId?: number;
  primaryProgramName?: string;
  allocated?: AllocatedProgramMetrics;
  unallocated?: UnallocatedProgramMetrics;
  totalRegistrations?: number;
}

@Injectable()
export class RegistrationService {
  constructor(
    private readonly dataSource: DataSource,
    private readonly repository: RegistrationRepository,
    private readonly logger: AppLoggerService,
    private readonly communicationService: CommunicationService,
    private readonly excelService: ExcelService,
    private readonly programRegistrationSwapRepository: ProgramRegistrationSwapRepository,
    private readonly programRegistrationService: ProgramRegistrationService,
    private readonly userRepository: UserRepository,
    private readonly userService: UserService,
    private readonly userProfileExtensionService: UserProfileExtensionService,
    private readonly awsS3Service: AwsS3Service,
    private readonly msg91Service: Msg91Service,
    private readonly configService: ConfigService,
    private readonly allocationClearingService: AllocationClearingService,
  ) {
    this.msg91RmRegNotifyMessageTemplateId = this.configService.get<string>('MSG91_RM_REG_NOTIFY_TEMPLATE_ID') ?? '';
  }
  private msg91RmRegNotifyMessageTemplateId : string;

  /**
   * Send MSG91 SMS notification to RM about a new or updated registration
   * @param rmContactUser - The RM contact user object
   * @param registrationName - Name of the person who registered
   * @param registrationId - ID of the registration for logging
   */
  private async sendRmMsg91Notification(
    rmContactUser: { countryCode: string; phoneNumber: string },
    registrationName: string,
    registrationId: number,
  ): Promise<void> {
    if (!rmContactUser?.countryCode || !rmContactUser?.phoneNumber) {
      this.logger.warn(`Missing RM contact details for registration ${registrationId}`);
      return;
    }

    try {
      const phoneNumber = `${rmContactUser.countryCode}${rmContactUser.phoneNumber}`.replace('+', '');
      await this.msg91Service.sendSms(
        phoneNumber,
        { var1: registrationName },
        this.msg91RmRegNotifyMessageTemplateId,
      );
    } catch (error) {
      this.logger.log(
        `Error sending MSG91 notification to RM for registration ${registrationId}: ${error}`,
      );
    } finally {
      return;
    }
  }

  async register(
    dto: CreateRegistrationDto,
    user: User,
    isSelfRegister: boolean,
  ): Promise<RegistrationResult> {
    this.logger.log(registrationMessages.CREATE_REQUEST_RECEIVED, dto);
    try {
      const userId = user.id;
      await this.validateUserRegistrationDetails(user, dto.answers);
      // STEP 1: EARLY VALIDATION - Validate and get program details first
      const program = await this.validateAndGetProgram(dto.programId);
      const registrationLevel = this.getRegistrationLevel(program);

      // STEP 2: EARLY VALIDATION - Validate session if required
      const session = await this.validateSessionIfRequired(dto, program, registrationLevel);

      // STEP 3: EARLY VALIDATION - Validate question answers FIRST
      // This will return immediately if there are any question validation errors
      const questionMappings = await this.validateQuestionAnswers(
        dto,
        program.id,
        dto.programSessionId,
      );

      // STEP 3.5: VALIDATE PREFERENCES - Check if preference program start dates are valid
      const preferenceProgramIds = this.extractPreferenceProgramIds(dto.answers);
      await this.validatePreferenceStartDates(preferenceProgramIds);

      // STEP 4: CHECK REGISTRATION DATES - Check if registrations are still open
      await this.validateRegistrationDates(program, session, registrationLevel);

      // STEP 5: Business Logic - Check if user is already registered (self-registration only)
      if (isSelfRegister) {
        await this.validateUserNotAlreadyRegistered(userId, dto.programId, dto.programSessionId);
      }

      // STEP 6: Business Logic - Check seat availability and determine registration path
      const seatInfo = await this.checkSeatAvailability(program, session, registrationLevel);

      // STEP 7: Business Logic - Process answers and group by storage location
      const processedAnswers = this.processQuestionAnswers(dto.answers, questionMappings, userId);
      // STEP 8: Business Logic - Determine registration outcome based on seats and requirements
      const registrationOutcome = this.determineRegistrationOutcome(seatInfo, program);

      // STEP 9: Delegate to repository for data persistence
      const registration = await this.dataSource.transaction(async (manager) => {
        return await this.repository.createRegistration({
          dto,
          userId,
          isSelfRegister,
          program,
          session,
          registrationLevel,
          seatInfo,
          processedAnswers,
          registrationOutcome,
          registrationStatus: dto.registrationStatus,
        }, manager);
      });


      const programRegistration = await this.repository.findRegistrationById(
        registration.registrationId,
      );


      // Prepare user profile data from answers for potential update
      const userProfileUpdates = programRegistration.userId ? this.extractUserProfileUpdates(dto.answers, programRegistration.user, dto.countryCode) : {};

      const userProfileExtensionUpdated = programRegistration.userId ? this.extractUserProfileExtensionUpdates(dto.answers) : {};


      // Update user profile if required
      if (
        programRegistration.userId &&
        dto.updateUserProfileData &&
        dto.registrationStatus !== RegistrationStatusEnum.SAVE_AS_DRAFT &&
        program.type?.key === 'PT_HDBMSD' &&
        Object.keys(userProfileUpdates).length > 0
      ) {
        await this.userRepository.updateUser(programRegistration.userId, userProfileUpdates);
      }

      // Update user profile extension if required
      if (
        programRegistration.userId &&
        dto.updateUserProfileData &&
        dto.registrationStatus !== RegistrationStatusEnum.SAVE_AS_DRAFT &&
        Object.keys(userProfileExtensionUpdated).length > 0
      ) {
        try {
          await this.userProfileExtensionService.updateByUserId(programRegistration.userId, userProfileExtensionUpdated);
        } catch (error) {
          this.logger.error(`Failed to update profile extension for user ${programRegistration.userId}: ${error.message}`);
        }
      }

      if (registration.basicDetailsStatus === RegistrationBasicStatusEnum.COMPLETED) {
        const mergeInfo = {
          reg_name: programRegistration.fullName,
          reg_edit_link: this.registrationEditLink(programRegistration.program.id, userId),
        };
        const attachments: { name: string; content: string; mime_type: string }[] = [];
        // if (programRegistration.personType === PersonTypeEnum.CHILD) {
        //   const parentalFormData: ParentalDeclaration = {
        //     name: programRegistration.fullName,
        //   };
        //   const parentConsentPdfBuffer = await generatePDF(
        //     parentalFormData,
        //     PARENTAL_DECLARATION,
        //     'parental_declaration',
        //   );

        //   const parentConsentPdfURL = await this.uploadBufferToAWS(
        //     parentConsentPdfBuffer.buffer,
        //     'parental_declaration.pdf',
        //   );

        //   const parentConsentBase64Content = await fetchPdfAsBase64(parentConsentPdfURL.url);
        //   const parentConsentAttachments = {
        //     name: 'parental_declaration.pdf',
        //     content: parentConsentBase64Content,
        //     mime_type: 'application/pdf',
        //   };
        //   attachments.push(parentConsentAttachments);


        //   const termsAndConditionsData: TermsAndCconditionsTemplate = {
        //     name: programRegistration.fullName,
        //     email: programRegistration.emailAddress,
        //   };
        //   const termsAndCconditionsPdf = await generatePDF(
        //     termsAndConditionsData,
        //     TERMS_AND_CONDITIONS,
        //     'terms_and_conditions',
        //   );
        //   const termsAndConditionsPdfUrl = await this.uploadBufferToAWS(
        //     termsAndCconditionsPdf.buffer,
        //     'terms_and_conditions.pdf',
        //   );
        //   const tAndCBase64Content = await fetchPdfAsBase64(termsAndConditionsPdfUrl.url);
        //   const tandcAttachments = {
        //     name: 'terms_and_conditions.pdf',
        //     content: tAndCBase64Content,
        //     mime_type: 'application/pdf',
        //   };
        //   attachments.push(tandcAttachments);
        // }
        const emailData: SendSingleEmailDto = {
          templateKey: process.env.ZEPTO_REGISTRATION_COMPLETED_EMAIL_TEMPLATE_ID,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_EMAIL,
            name: program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: programRegistration.emailAddress,
            name: programRegistration.fullName,
          },
          mergeInfo: mergeInfo,
          attachments: attachments ? attachments : [],
          subject: '',
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        try {
          await this.communicationService.sendSingleEmail(emailData);
        } catch (error) {
          this.logger.log(
            `Error sending email for registration ${programRegistration.id}: ${error}`,
          );
        }
        const messageData: SendTemplateMessageDto = {
          whatsappNumber: programRegistration.mobileNumber.slice(0), // Remove + prefix
          templateName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          parameters: [{ name: 'reg_name', value: programRegistration.fullName }],
          trackinfo: {
            registrationId: programRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        try {
          await this.communicationService.sendTemplateMessage(messageData);
        } catch (error) {
          this.logger.log(
            `Error sending WhatsApp notification for registration ${programRegistration.id}: ${error}`,
          );
        }

        const rmMessageData: SendTemplateMessageDto = {
          whatsappNumber: `${programRegistration.rmContactUser.countryCode}${programRegistration.rmContactUser.phoneNumber}`.slice(0), // Remove + prefix
          templateName: process.env.WATI_RM_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_RM_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          parameters: [{ name: 'reg_name', value: programRegistration.fullName }],
          trackinfo: {
            registrationId: programRegistration.id,
          },
        };
        try {
          await this.communicationService.sendTemplateMessage(rmMessageData);
        } catch (error) {
          this.logger.log(
            `Error sending RM notification for registration ${programRegistration.id}: ${error}`,
          );
        }
        // Send MSG91 message to RM
        await this.sendRmMsg91Notification(
          {
            countryCode: programRegistration.rmContactUser.countryCode,
            phoneNumber: programRegistration.rmContactUser.phoneNumber,
          },
          programRegistration.fullName,
          programRegistration.id,
        );
      }

      return registration;
    } catch (error) {
      this.logger.error('Error in registration create', error?.stack, { error, dto });
      handleKnownErrors(ERROR_CODES.REGISTRATION_CREATION_FAILED, error);
    }
  }

  private decodePreference(value: any): any[] {
    if (!value) {
      return [];
    }
    try {
      const decoded =
        typeof value === 'string' ? decodeURIComponent(value) : value;
      return typeof decoded === 'string' ? JSON.parse(decoded) : decoded;
    } catch (e) {
      this.logger.error('Failed to decode preference value', e);
      return [];
    }
  }

  private extractPreferenceProgramIds(answers: RegistrationAnswerDto[]): number[] {
    if (!Array.isArray(answers) || answers.length === 0) {
      return [];
    }

    const preferenceAnswer = answers.find((answer) => answer.bindingKey === 'preference');

    if (!preferenceAnswer?.answer) {
      return [];
    }

    const preferences = this.decodePreference(preferenceAnswer.answer);

    if (!Array.isArray(preferences) || preferences.length === 0) {
      return [];
    }

    const programIds = preferences
      .map((preference: any) => {
        const id = preference?.sessionId ?? preference?.programId ?? preference?.id;
        if (id === undefined || id === null) {
          return null;
        }
        const numericId = typeof id === 'number' ? id : Number(id);
        return Number.isFinite(numericId) ? numericId : null;
      })
      .filter((id: number | null): id is number => typeof id === 'number' && Number.isFinite(id));

    return Array.from(new Set(programIds));
  }

  private arePreferencesEqual(prev: any[], curr: any[]): boolean {
    const sortByPriority = (arr: any[]) =>
      [...arr].sort(
        (a, b) => (a.priorityOrder ?? 0) - (b.priorityOrder ?? 0),
      );
    const a = sortByPriority(prev);
    const b = sortByPriority(curr);
    if (a.length !== b.length) {
      return false;
    }
    for (let i = 0; i < a.length; i++) {
      if (a[i].sessionId !== b[i].sessionId) {
        return false;
      }
    }
    return true;
  }

  private formatPreferenceForMessage(pref: any[]): string {
    if (!pref || pref.length === 0) {
      return 'Any HDB/MSD';
    }
    const sorted = [...pref].sort(
      (a, b) => (a.priorityOrder ?? 0) - (b.priorityOrder ?? 0),
    );
    return sorted.map((p) => p.sessionName).join(', ');
  }

  async update(dto: UpdateRegistrationDto, user: User,isAdmin: boolean): Promise<RegistrationResult> {
    this.logger.log(registrationMessages.UPDATE_REQUEST_RECEIVED, dto);
    try {
      // Validate user registration details
      await this.validateUserRegistrationDetails(user, dto.answers);

      // Business Logic: Validate form section
      let formSection: FormSection | null = null;
      if (dto.formSectionId) {
        formSection = await this.validateFormSection(dto.formSectionId);
      }
      let updateDto:any = { ...dto };
      if(formSection){
        updateDto = {
          ...dto,
          formSection: Object.assign({}, formSection),
        }
      }
      // Business Logic: Validate registration exists and can be updated
      const registration = await this.validateRegistrationForUpdate(
        dto.programRegistrationId,
        user,
        updateDto,
        isAdmin
      );


      // if role of user is viewer then form section id should be provided
      if (
        user.userRoleMaps.length === 1 &&
        user.userRoleMaps.some((urm) => urm.role.name === 'viewer') &&
        !dto.formSectionId
      ) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_INVALID_STATUS,
          null,
          null,
          'Form section ID is required for viewer role',
        );
      }


      // Business Logic: Check approval status
      if (
        formSection &&
        formSection.key !== 'FS_BASICDETAILS' &&
        formSection.key !== 'FS_MAHATRIAQUESTIONS'
      ) {
        await this.validateApprovalStatus(registration);
      }

      // Business Logic: Validate program and session
      const program = await this.validateProgramForUpdate(dto.programId, registration);
      const session = await this.validateSessionForUpdate(dto, registration, program);

      if (
        formSection &&
        formSection.key === 'FS_TRAVELPLAN')
      {
        // #codebase if registration_status is save_As_draft then in upserting question check for the section of the question if its section_key is FS_travel.... then check if record is there if there mark that status as save_as_draft and if not there then add a new record there with save as draft directly insert first
      const existingTravelPlan = await this.repository.findTravelPlanByRegistrationId(
        registration.id,
      );
      if (existingTravelPlan) {
        // Update existing travel plan with draft status
        await this.repository.updateTravelPlanStatus(
          existingTravelPlan.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id,
        );
      } else {
        // Create new travel plan with draft status
        await this.repository.createTravelPlan(
          registration.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id);
      }

      const existingTravelInfo = await this.repository.findTravelInfoByRegistrationId(
        registration.id,
      );
      if (existingTravelInfo) {
        // Update existing travel info with draft status
        await this.repository.updateTravelInfoStatus(
          existingTravelInfo.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id,
        );
      } else {
        // Create new travel info with draft status
        await this.repository.createTravelInfo(
          registration.id,
          dto.registrationStatus === RegistrationStatusEnum.SAVE_AS_DRAFT ? TravelStatusEnum.SAVE_AS_DRAFT : TravelStatusEnum.PENDING,
          user.id);
      }

      }

      // Business Logic: Validate and process section-specific questions
      const sectionQuestions = await this.validateSectionQuestions(
        dto,
        program.id,
        dto.programSessionId,
        formSection,
      );

      const processedAnswers = registration.user ? this.processQuestionAnswers(dto.answers, sectionQuestions, registration.userId) : {};

      // Prepare user profile updates from answers
      const userProfileUpdates = registration.user ? this.extractUserProfileUpdates(dto.answers, registration.user, dto.countryCode) : {};

      // prepare user extension profile updates
      const userProfileExtensionUpdated = this.extractUserProfileExtensionUpdates(dto.answers);

      let preferenceChange: { oldPref: string; newPref: string } | null = null;
      const preferenceQuestion = dto.answers.find(
        (m) => m.bindingKey === 'preference',
      );
      if (preferenceQuestion) {
          const prevAns = await this.repository.findQuestionAnswer(
            registration.id,
            preferenceQuestion.questionId,
          );
          const prevPref = this.decodePreference(prevAns?.answerValue);
          const newPref = this.decodePreference(preferenceQuestion?.answer);
          if (!this.arePreferencesEqual(prevPref, newPref)) {
            preferenceChange = {
              oldPref: this.formatPreferenceForMessage(prevPref),
              newPref: this.formatPreferenceForMessage(newPref),
            };
          }
          if (!isAdmin) {
            const updatedPreferenceProgramIds = this.extractPreferenceProgramIds(dto.answers);
            await this.validatePreferenceStartDates(updatedPreferenceProgramIds);
          }
      }
      
      // Delegate to repository for data persistence
      const updateRegistrationData = await this.dataSource.transaction(async (manager) => {
        return await this.repository.updateRegistrationData({
          registration,
          processedAnswers,
          userId: user.id,
          registrationStatus: dto.registrationStatus,
        }, manager);
      });

      // Update user profile if required
      if (
        dto.updateUserProfileData &&
        dto.registrationStatus !== RegistrationStatusEnum.SAVE_AS_DRAFT &&
        program.type?.key === 'PT_HDBMSD' &&
        formSection?.key === 'FS_BASICDETAILS' &&
        Object.keys(userProfileUpdates).length > 0 &&
        registration.userId
      ) {
        await this.userRepository.updateUser(registration.userId, userProfileUpdates);
      }

      // Update user profile extension if required
      if (
        dto.updateUserProfileData &&
        dto.registrationStatus !== RegistrationStatusEnum.SAVE_AS_DRAFT &&
        Object.keys(userProfileExtensionUpdated).length > 0 &&
        registration.userId
      ) {
        try {
          await this.userProfileExtensionService.updateByUserId(registration.userId, userProfileExtensionUpdated);
        } catch (error) {
          this.logger.error(`Failed to update profile extension for user ${registration.userId}: ${error.message}`);
        }
      }

      const updatedRegistration = await this.repository.findRegistrationById(
        updateRegistrationData.registrationId,
      );
      if (
        preferenceChange &&
        updatedRegistration.rmContactUser?.phoneNumber
      ) {
        const rmMsg: SendTemplateMessageDto = {
          whatsappNumber: `${updatedRegistration.rmContactUser.countryCode}${updatedRegistration.rmContactUser.phoneNumber}`.slice(0),
          templateName:
            process.env.WATI_RM_PREFERENCE_EDITED_TEMPLATE_ID || '',
          broadcastName:
            process.env.WATI_RM_PREFERENCE_EDITED_TEMPLATE_ID || '',
          parameters: [
            { name: 'rm_name', value: updatedRegistration.rmContactUser.orgUsrName || updatedRegistration.rmContactUser.fullName },
            { name: 'reg_name', value: updatedRegistration.fullName },
            { name: 'reg_id', value: String(updatedRegistration.id) },
            { name: 'old_pref', value: preferenceChange.oldPref },
            { name: 'new_pref', value: preferenceChange.newPref },
          ],
          trackinfo: { registrationId: updatedRegistration.id },
        };
        await this.communicationService.sendTemplateMessage(rmMsg);
      }

      if (registration.basicDetailsStatus === RegistrationBasicStatusEnum.DRAFT && updatedRegistration.basicDetailsStatus === RegistrationBasicStatusEnum.COMPLETED) {
        const mergeInfo = {
          reg_name: updatedRegistration.fullName,
          reg_edit_link: this.registrationEditLink(updatedRegistration.program.id, user.id),
        };
        const attachments: { name: string; content: string; mime_type: string }[] = [];
        // if (updatedRegistration.personType === PersonTypeEnum.CHILD) {
        //   const parentalFormData: ParentalDeclaration = {
        //     name: updatedRegistration.fullName,
        //   };
        //   const parentConsentPdfBuffer = await generatePDF(
        //     parentalFormData,
        //     PARENTAL_DECLARATION,
        //     'parental_declaration',
        //   );
        //   const parentConsentPdfURL = await this.uploadBufferToAWS(
        //     parentConsentPdfBuffer.buffer,
        //     'parental_declaration.pdf',
        //   );

        //   const consentBase64Content = await fetchPdfAsBase64(parentConsentPdfURL.url);
        //   const parentConsentAttachment = {
        //     name: 'parental_declaration.pdf',
        //     content: consentBase64Content,
        //     mime_type: 'application/pdf',
        //   };
        //   attachments.push(parentConsentAttachment);

        //   const termsAndConditionsData: TermsAndCconditionsTemplate = {
        //     name: updatedRegistration.fullName,
        //     email: updatedRegistration.emailAddress,
        //   };
        //   const termsAndCconditionsPdf = await generatePDF(
        //     termsAndConditionsData,
        //     TERMS_AND_CONDITIONS,
        //     'terms_and_conditions',
        //   );
        //   const termsAndConditionsPdfUrl = await this.uploadBufferToAWS(
        //     termsAndCconditionsPdf.buffer,
        //     'terms_and_conditions.pdf',
        //   );
        //   const tAndCBase64Content = await fetchPdfAsBase64(termsAndConditionsPdfUrl.url);
        //   const tandcAttachments = {
        //     name: 'terms_and_conditions.pdf',
        //     content: tAndCBase64Content,
        //     mime_type: 'application/pdf',
        //   };
        //   attachments.push(tandcAttachments);
        // }
        const emailData: SendSingleEmailDto = {
          templateKey: process.env.ZEPTO_REGISTRATION_COMPLETED_EMAIL_TEMPLATE_ID,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_EMAIL,
            name: program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: updatedRegistration.emailAddress,
            name: updatedRegistration.fullName,
          },
          mergeInfo: mergeInfo,
          attachments: attachments ? attachments : [],
          subject: '',
          trackinfo: {
            registrationId: updatedRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        try {
          await this.communicationService.sendSingleEmail(emailData);
        } catch (error) {
          this.logger.log(
            `Error sending email notification for registration ${updatedRegistration.id}: ${error}`,
          );
        }
        // Send WhatsApp message to registered user
        const messageData: SendTemplateMessageDto = {
          whatsappNumber: updatedRegistration.mobileNumber.slice(0), // Remove + prefix
          templateName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_SEEKER_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          parameters: [{ name: 'reg_name', value: updatedRegistration.fullName }],
          trackinfo: {
            registrationId: updatedRegistration.id,
            // createdBy: userId,
            // updatedBy: userId,
          },
        };

        try {
          await this.communicationService.sendTemplateMessage(messageData);
        } catch (error) {
          this.logger.log(
            `Error sending WhatsApp notification for registration ${updatedRegistration.id}: ${error}`,
          );
        }

        //  Send message to RM
        const rmMessageData: SendTemplateMessageDto = {
          whatsappNumber:
            `${updatedRegistration.rmContactUser.countryCode}${updatedRegistration.rmContactUser.phoneNumber}`.replace(
              '+',
              '',
            ), // Remove + prefix
          templateName: process.env.WATI_RM_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_RM_REGISTRATION_COMPLETED_TEMPLATE_ID || '',
          parameters: [{ name: 'reg_name', value: updatedRegistration.fullName }],
          trackinfo: {
            registrationId: updatedRegistration.id,
          },
        };

        try {
          await this.communicationService.sendTemplateMessage(rmMessageData);
        } catch (error) {
          this.logger.log(
            `Error sending RM notification for registration ${updatedRegistration.id}: ${error}`,
          );
        }

        // Send MSG91 message to RM
        await this.sendRmMsg91Notification(
          {
            countryCode: updatedRegistration.rmContactUser.countryCode,
            phoneNumber: updatedRegistration.rmContactUser.phoneNumber,
          },
          updatedRegistration.fullName,
          updatedRegistration.id,
        );
      }
      return updateRegistrationData;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_CREATION_FAILED, error);
    }
  }

  /**
   * Get registrations list view with KPIs based on user roles
   */
  async getRegistrationsListView(
    limit: number,
    offset: number,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles: string[],
    filterRegistrationsByUserId?: number | null,
    downloadType?: string,
  ) {
    this.logger.log('Getting registrations list view', {
      limit,
      offset,
      programId,
      programSessionId,
      searchText,
      parsedFilters,
      userRoles,
      downloadType,
    });
  
    try {
      const isRMUser = userRoles.includes('relational_manager');
  
      // Handle download case - return all data without pagination
      if (downloadType) {
        const registrationData = await this.repository.findRegistrations(
          'all',
          'all',
          programId,
          programSessionId,
          downloadType === 'filtered' ? searchText : '',
          downloadType === 'filtered' ? this.extractDataFilters(parsedFilters) : {},
          userRoles,
          isRMUser ? filterRegistrationsByUserId : undefined,
        );
        
        return {
          data: registrationData.data,
          pagination: {
            totalPages: 1,
            pageNumber: 1,
            pageSize: registrationData.data.length,
            totalRecords: registrationData.data.length,
            numberOfRecords: registrationData.data.length,
          },
        };
      }
  
      // ✅ FIXED: Separate filters for KPIs and data
      const kpiFilters = this.extractKPIFilters(parsedFilters);
      const dataFilters = this.extractDataFilters(parsedFilters);

      // Get registration data with BOTH search + KPI filters applied
      const result = await this.repository.findRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText, // ✅ Search applies to data table
        dataFilters,
        userRoles,
        isRMUser ? filterRegistrationsByUserId : undefined,
      );
  
      // ✅ CRITICAL FIX: Generate KPIs WITHOUT search text - always show total state
      const kpis = await this.generateDefaultKPIs({
        programId,
        programSessionId,
        searchText: '', // ✅ NO SEARCH TEXT for KPIs
        parsedFilters: this.removeKPIFilters(parsedFilters), // ✅ Remove KPI filters from KPI generation
        userRoles,
        filterRegistrationsByUserId: isRMUser ? filterRegistrationsByUserId : undefined,
        activeKpiCategory: parsedFilters?.kpiCategory || null,
      });
  
      return {
        data: result.data,
        pagination: result.pagination,
        kpis: {
          default: kpis,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Transform registration list data for the controller response
   */
  async transformRegistrationListData(result: any, userRoles: string[], parentFilter?: string, kpiFilter?: string, programId?: number | null, view?: string, kpiCategory?: string) {
    this.logger.debug(`Transforming registration list data with ${result?.data?.length || 0} records`);
    
    try {
      if (!result || !result.data) {
        this.logger.warn('Invalid result data provided for transformation');
        return {
          data: [],
          tableHeaders: this.getTableHeaders(userRoles || [], view || ViewTypeEnum.REGISTRATIONS),
          quickViewHeaders: this.getQuickViewHeaders(userRoles || []),
          pagination: { totalRecords: 0, currentPage: 1, totalPages: 0, pageSize: 10 },
          kpis: {},
          viewList: this.getViewList()
        };
      }

      const transformedData = {
        data: result.data.map((registration, index) => {
          try {
            // Pre-compute reusable values to avoid multiple accesses and sorting
            const paymentStatus = registration?.paymentDetails?.[0]?.paymentStatus;
            const approvalRecord = registration?.approvals && registration.approvals.length > 0 ? registration.approvals[0] : null;
            const sortedSwapRequests = Array.isArray(registration?.swapsRequests) && registration.swapsRequests.length > 0
              ? [...registration.swapsRequests].sort((a, b) => b.id - a.id)
              : [];
            const latestSwapRequest = sortedSwapRequests.length > 0 ? sortedSwapRequests[0] : null;
            
            // Pre-compute status checks to avoid repetitive evaluation
            const isInactiveStatus = [RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED, RegistrationStatusEnum.SAVE_AS_DRAFT].includes((registration?.registrationStatus ?? '')) ||
              [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD].includes((approvalRecord?.approvalStatus ?? ''));
            
            const activeSwapReq = isInactiveStatus
              ? false
              : (sortedSwapRequests.some(swap => swap.status === SwapRequestStatus.ACTIVE) || false);
            
            // order the swaps by id desc and get the latest one and check its swap requiremnet is swap_demand and status is on_hold
            const latestSwapDemand = latestSwapRequest && latestSwapRequest.type === SwapType.WantsSwap && latestSwapRequest.status === SwapRequestStatus.ON_HOLD && latestSwapRequest.swapRequirement === SwapRequirementEnum.SWAP_DEMAND ? latestSwapRequest: null;       
            return {
              id: registration.id,
              userId: registration?.user?.id,
              seekerName: this.capitalizeWords(registration.fullName) ?? '',
              age: calculateAge(registration.dob),
              gender: registration?.gender ?? '',
              location: (registration?.city && registration?.city.trim().toLowerCase() === 'other')
                ? (registration?.otherCityName ?? '')
                : (registration?.city ?? ''),
              blessedWith: registration?.allocatedProgram?.name ?? '',
              approvalStatus: approvalRecord?.approvalStatus ?? '',
              numberOfHDBs: registration?.noOfHDBs ?? 0,
              rmComments: registration?.rmReview || '',
              paymentStatus: this.formatPaymentStatus(paymentStatus ?? '', registration?.allocatedProgram?.id, registration?.isFreeSeat),
              invoiceStatus: this.formatInvoiceStatus(registration?.invoiceDetails?.[0]?.invoiceStatus ?? '', registration?.allocatedProgram?.id, registration?.isFreeSeat, paymentStatus),
              invoiceNumber: registration?.invoiceDetails[0]?.invoiceSequenceNumber ?? '',
              invoiceUrl: registration?.invoiceDetails?.[0]?.invoicePdfUrl ?? '',
              travelPlanStatus: getTravelOverAllStatus(registration),
              travelUpdatedAt: registration?.travelPlans?.[0]?.updatedAt ? registration.travelPlans[0].updatedAt : null,
              allocatedProgramStartsAt: registration?.allocatedProgram?.startsAt ?? null,
              mobileNumber: registration?.mobileNumber ?? '',
              registrationStatus: registration?.registrationStatus,
              averageRating: registration?.averageRating || 0,
              ratings: registration?.ratings || [],
              profileUrl: registration?.profileUrl || '',
              profilePictureUrl: registration?.profileUrl || '',
              emailAddress: registration?.emailAddress,
              programId: registration?.program?.id,
              programSessionId: registration?.programSession?.id,
              preference: registration?.preferences ?? [],
              appliedOn: registration?.registrationDate ? registration.registrationDate : null,
              registrationDate: registration?.registrationDate ? registration.registrationDate : null,
              dob: registration?.dob || null,
              countryName: registration?.countryName || '',
              hdbAssociationSince: registration?.hdbAssociationSince || '',
              registrationLastModified: registration?.updatedAt || null,
              blessedDate: approvalRecord?.approvalStatus === ApprovalStatusEnum.APPROVED ? approvalRecord?.approvalDate || null : null,
              holdDate: approvalRecord?.approvalStatus === ApprovalStatusEnum.REJECTED ? approvalRecord?.updatedAt || null : null,
              swapDemandDate: approvalRecord?.approvalStatus === ApprovalStatusEnum.ON_HOLD ? approvalRecord?.updatedAt || null : null,
              cancelledDate: registration?.registrationStatus === RegistrationStatusEnum.CANCELLED ? registration?.cancellationDate || null : null,
              swapRequestDate: latestSwapRequest ? latestSwapRequest?.createdAt : null, 
              pendingDate: (approvalRecord?.approvalStatus === ApprovalStatusEnum.ON_HOLD || approvalRecord?.approvalStatus === ApprovalStatusEnum.REJECTED) ? approvalRecord?.updatedAt || null : null,
              statusDateTime: getRegistrationStatusDateTime(registration, approvalRecord, latestSwapRequest, kpiFilter, kpiCategory),
              allStatusDate: getRegistrationStatusDateTime(registration, approvalRecord, latestSwapRequest, kpiFilter, kpiCategory), 
              statusDateTimeLabel: getStatusDateTimeLabel(registration, approvalRecord, latestSwapRequest, kpiFilter, kpiCategory),
              isSwapRequestActive: activeSwapReq,
              canShiftReqActive: isInactiveStatus
              ? false
              : (sortedSwapRequests.some(swap => swap.type === SwapType.CanShift && swap.status === SwapRequestStatus.ACTIVE) || false),
              wantsSwapReqActive: isInactiveStatus
              ? false
              : (sortedSwapRequests.some(swap => swap.type === SwapType.WantsSwap && swap.status === SwapRequestStatus.ACTIVE) || false),
              latestSwapRequestedPrograms: (activeSwapReq || approvalRecord?.approvalStatus === ApprovalStatusEnum.ON_HOLD) 
                ? (latestSwapRequest?.swapRequestedProgram?.map(sp => ({
                    id: sp.program?.id,
                    name: sp.program?.name
                  })).filter(item => item.id && item.name) || [])
                : [],
              swapPreference: (activeSwapReq || approvalRecord?.approvalStatus === ApprovalStatusEnum.ON_HOLD) 
                ? (latestSwapRequest?.swapRequestedProgram?.map(sp => sp.program?.name).filter(Boolean).join(', ') || '')
                : '',
              swapDemandFrom: latestSwapDemand ? latestSwapDemand?.currentProgram?.name  || null: null,
              swapDemand: latestSwapDemand && latestSwapDemand.swapRequestedProgram?.length > 0 ? latestSwapDemand.swapRequestedProgram.sort((a, b) => a.id - b.id).map(sp => sp.program?.name).filter(Boolean).join(', ') : null,
              swapDemandComments: latestSwapDemand ? latestSwapDemand.comment || null : null,
              isOrgUser: registration?.user?.userType === UserTypeEnum.ORG ? true : false,
              cancellationReason: registration?.cancellationReason 
                ? this.capitalizeFirstLetter(registration.cancellationReason.toLowerCase().replace(/_/g, ' '))
                : '',
              cancellationComments: registration?.cancellationComments || '',
              proFormaInvoicePdfUrl: registration?.proFormaInvoicePdfUrl || '',
              preferredRoomMate: registration?.preferredRoomMate || '',
              rmContact: (registration.rmContactUser && 
                typeof registration.rmContactUser.orgUsrName === 'string' &&
                registration.rmContactUser.orgUsrName.trim().toLowerCase() === 'other'
                )
                ? registration.otherInfinitheismContact || ''
                : (registration.rmContactUser ? registration.rmContactUser.orgUsrName : ''),
              lastHdbMsd: registration?.lastHdbAttended || '',
              recommendationComments: registration?.recommendation?.[0]?.isRecommended ? registration?.recommendation?.[0]?.recommendationText : '-',
              recommendation: registration?.recommendation?.[0]?.isRecommended ? registration?.recommendation?.[0]?.recommendationKey : '-',
              followUpCount: registration?.recommendation?.[0]?.followUpCount || null,
              activeSwapRequests: registration?.swapsRequests?.filter(swap => swap.status === SwapRequestStatus.ACTIVE)?.map(swap => ({
                id: swap.id,
                type: swap.type,
                comment: swap.comment,
                SwapRequirement: swap.swapRequirement,
                requestedPrograms: swap.requestedPrograms?.map(program => ({
                  id: program.id,
                  name: program.name
                })) || []
              })) || [],
              personType: registration?.personType,
              parentalFormStatus: registration?.parentalFormStatus,
              userProgramExperiences: registration?.user?.programExperiences?.filter(experience => experience.deletedAt === null) || [],
              goodiesStatus: formatGoodiesStatus(registration?.goodies[0], registration?.allocatedProgram?.id, approvalRecord?.approvalStatus),
              tshirt: registration?.goodies?.[0]?.tshirtSize === undefined 
                ? null
                : formatSizeSelection(registration?.goodies?.[0]?.tshirt ?? false, registration?.goodies?.[0]?.tshirtSize),
              jacket: registration?.goodies?.[0]?.jacketSize === undefined 
                ? null 
                : formatSizeSelection(registration?.goodies?.[0]?.jacket ?? false, registration?.goodies?.[0]?.jacketSize),
              ratriaPillar: registration?.goodies?.[0]?.ratriaPillarLeonia === undefined 
                ? null
                : formatRatriaPillar(
                    registration?.goodies?.[0]?.ratriaPillarLeonia, 
                    registration?.goodies?.[0]?.ratriaPillarLocation, 
                    registration?.goodies?.[0]?.ratriaPillarOtherLocation
                  ),
              flask: registration?.goodies?.[0] ? formatBooleanValue(registration?.goodies?.[0]?.flask) : null,
              notebook: registration?.goodies?.[0] ? formatBooleanValue(registration?.goodies?.[0]?.notebook) : null
            };
          } catch (itemError) {
            this.logger.error(`Error transforming registration item at index ${index}`, itemError.stack);
            // Return a minimal valid object to prevent complete failure
            return {
              id: registration?.id || `error-${index}`,
              seekerName: 'Error loading data',
              age: 0,
              gender: '',
              location: '',
              blessedWith: '',
              approvalStatus: '',
              numberOfHDBs: 0,
              rmComments: '',
              paymentStatus: '',
              invoiceStatus: '',
              invoiceNumber: '',
              invoiceUrl: '',
              travelPlanStatus: '',
              allocatedProgramStartsAt: null,
              mobileNumber: '',
              registrationStatus: '',
              averageRating: 0,
              ratings: [],
              profileUrl: '',
              emailAddress: '',
              programId: null,
              programSessionId: null,
              preference: [],
              appliedOn: null,
              isSwapRequestActive: false,
              isOrgUser: false,
              proFormaInvoicePdfUrl: '',
              preferredRoomMate: '',
            };
          }
        }),
        tableHeaders: this.getTableHeaders(userRoles || [], view || ViewTypeEnum.REGISTRATIONS),
        quickViewHeaders: this.getQuickViewHeaders(userRoles || []),
        pagination: result.pagination || { totalRecords: 0, currentPage: 1, totalPages: 0, pageSize: 10 },
        kpis: this.flattenKPIs(result.kpis?.dynamic ?? result.kpis?.default ?? {})
      };
      const validParentFilter = ['hdb1Blessed','hdb2Blessed','hdb3Blessed','msd1Blessed','msd2Blessed'].includes(parentFilter || '');
      const validKpiFilter = ['travelComplete'].includes(kpiFilter || '');
      if( validParentFilter && validKpiFilter) {
        transformedData.tableHeaders.push(
          {
            key: 'travelUpdatedAt',
            label: 'Last updated travel plan',
            sortable: true,
            filterable: true,
            type: 'string',
            order: '7'
          },
        )
        transformedData.tableHeaders = transformedData.tableHeaders.filter((header) => header.key !== 'travelPlanStatus');
      }

      // Add cancellation headers for specific user roles
      const cancellationRoles = ['shoba', 'mahatria', 'relational_manager'];
      const validCancelKpiFilter = ['cancelled'].includes(kpiFilter || '');
      if(cancellationRoles.some(role => userRoles.includes(role)) && validCancelKpiFilter) {
        transformedData.tableHeaders.push(
          {
            key: 'cancellationReason',
            label: 'Cancellation Reason',
            sortable: false,
            filterable: false,
            type: 'string',
            order: '11A'
          },
          {
            key: 'cancellationComments',
            label: 'Cancellation Comments',
            sortable: false,
            filterable: false,
            type: 'string',
            order: '11B'
          }
        );
        // filter recommendation
        transformedData.tableHeaders = transformedData.tableHeaders.filter((header) => header.key !== 'recommendation' && header.key !== 'recommendationComments' && header.key !== 'followUpCount');
      }
      const flattenedKpis = this.flattenKPIs(result.kpis?.dynamic ?? result.kpis?.default ?? {});
      transformedData.kpis = flattenedKpis;
      
      // Apply registration status transformation
      const isUserRoleRMorOM = userRoles.includes('finance_manager') || userRoles.includes('operations_manager');
      if (transformedData.data) {
        transformedData.data.forEach((registration, index) => {
          try {
            // if (isUserRoleRMorOM){
              // delete registration.rmComments;
              // delete registration.ratings;
            // }
            const originalReg = result.data[index];
            if (originalReg && originalReg.registrationStatus === RegistrationStatusEnum.PENDING && 
                originalReg.approvals && 
                originalReg.approvals[0] && 
                originalReg.approvals[0].approvalStatus === ApprovalStatusEnum.PENDING) {
              registration.registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
            }
          } catch (statusError) {
            this.logger.error(`Error applying status transformation for registration at index ${index}`, statusError.stack);
          }
        });
      }

      // Add statusCounts for subProgram-wise registration counts with approval status breakdown
      // Only for mahatria and shoba roles
      let statusCountsData: any = [];
      const isMahatriaOrShoba = userRoles.includes('mahatria') || userRoles.includes('shoba');
      
      if (isMahatriaOrShoba && programId) {
        try {
          statusCountsData = await this.repository.getSubProgramStatusCounts(programId);
        } catch (statusCountsError) {
          this.logger.error('Error fetching statusCounts:', statusCountsError);
          // Continue without statusCounts if there's an error
        }
      }

      // Add status-specific date columns based on KPI context
      this.addStatusDateHeaders(transformedData, kpiCategory, kpiFilter, parentFilter);

      const finalTransformedData = {
        ...transformedData,
        statusCountsData: statusCountsData,
        viewList: this.getViewList()
      };

      this.logger.debug(`Successfully transformed ${transformedData.data.length} registration records with ${statusCountsData.length} statusCounts`);
      return finalTransformedData;
    } catch (error) {
      this.logger.error('Error in transformRegistrationListData', error.stack);
      // Return safe fallback structure
      return {
        data: [],
        tableHeaders: this.getTableHeaders(userRoles || [], view || ViewTypeEnum.REGISTRATIONS),
        quickViewHeaders: this.getQuickViewHeaders(userRoles || []),
        pagination: { totalRecords: 0, currentPage: 1, totalPages: 0, pageSize: 10 },
        kpis: {},
        viewList: this.getViewList()
      };
    }
  }

  // transformDraftRegistrationListData
  async transformDraftRegistrationListData(result: any, userRoles: string[]) {
    const data = {
      data: result.data.map(registration => ({
        id: registration.id,
        seekerName: this.capitalizeWords(registration.fullName) ?? '',
        age: calculateAge(registration.dob),
        gender: registration.gender ?? '',
        location: registration.otherCityName ?? registration.city ?? '',
        numberOfHDBs: registration.noOfHDBs ?? 0,
        mobileNumber: registration.mobileNumber ?? '',
        emailAddress: registration.emailAddress ?? '',
        profileUrl: registration.profileUrl || '',
        rmContact:
          registration.otherInfinitheismContact ??
          (registration.rmContactUser ? registration.rmContactUser.orgUsrName : ''),
        programId: registration.program?.id,
        programSessionId: registration.programSession?.id,
        preferences: registration.preferences ?? [],
      }))
    };
    const transformedData = {
      data: data.data,
      tableHeaders: this.getDraftRegTableHeaders(userRoles),
      pagination: result.pagination,
    };
    return transformedData;
  }

  // transformMahatriaKPIData
  async transformMahatriaKPIData(result: any, userRoles: string[], parsedFilters: any) {

    const transformedData = result.data.map((item: any) => {
      // Extract commonly used data with constants
      const latestApproval = item.approvals && item.approvals.length > 0 ? item.approvals[0] : null;
      const latestRecommendation = item.recommendation?.[0] ?? null;
      const latestSwapRequest = item.swapsRequests && item.swapsRequests.length > 0 ? 
        [...item.swapsRequests].sort((a: any, b: any) => b.id - a.id)[0] : null;
        

      // Handle registration status transformation (avoid double looping)
      let registrationStatus = item.registrationStatus;
      if (registrationStatus === RegistrationStatusEnum.PENDING && 
          latestApproval && 
          latestApproval.approvalStatus === ApprovalStatusEnum.PENDING) {
        registrationStatus = RegistrationStatusEnum.PENDING_APPROVAL;
      }

      return {
        ...item,
        registrationStatus, // Use the transformed status
        isOrganizationUser: item.user?.userType === UserTypeEnum.ORG ? true : false,
        isRecommended: latestRecommendation?.isRecommended ?? null,
        recommendation: latestRecommendation?.recommendationKey ?? null,
        recommendations: item.recommendation ?? null,
        latestSwapRequest: latestSwapRequest ? {
          id: latestSwapRequest.id,
          swapType: latestSwapRequest.swapType,
          createdAt: latestSwapRequest.createdAt,
          status: latestSwapRequest.status,
          comments: latestSwapRequest.comments
        } : null,
        statusDateTime: getRegistrationStatusDateTime(item, latestApproval, latestSwapRequest, parsedFilters?.kpiFilter, parsedFilters?.kpiCategory, true),
        statusDateTimeLabel: getStatusDateTimeLabel(item, latestApproval, latestSwapRequest, parsedFilters?.kpiFilter, parsedFilters?.kpiCategory, true),
      };
    });
    return transformedData;
  }

/**
 * Generate Excel file for download and upload to S3 with new format
 * @param registrations - Raw registration data from repository
 * @param userRoles - User roles for determining visible columns
 * @param downloadType - Type of download (all/filtered)
 * @returns S3 URL of the generated Excel file
 */
async generateExcelDownload(
  registrations: any[], 
  userRoles: string[], 
  downloadType: string
): Promise<string> {
  try {    
    // Transform data for Excel according to the new specified format
    const excelData = registrations.map((registration, index) => {
      const igstCheck = (registration?.invoiceDetails?.[0]?.gstNumber && registration?.program?.gstNumber) ? !areGSTsFromSameState(registration.program?.gstNumber, registration?.invoiceDetails?.[0]?.gstNumbe) : false;
      let transformedReg: any = {
        'S.No.': index + 1,
        'Seeker Email': registration?.emailAddress ?? '',
        'Seeker Name': registration?.fullName ?? '',
        'Blessed': registration?.allocatedProgram?.name ?? '',
        'No of HDB / MSD done': registration?.noOfHDBs ?? 0,
        'Mobile': registration?.mobileNumber ?? '',
        'RM': registration?.rmContactUser?.orgUsrName ?? '',
        // 'Country': registration?.country ?? 'India', 
        'City': registration?.city ?? '',
        'Date of blessing': registration?.allocatedProgram ? formatDateTimeIST(registration?.approvals[0]?.approvalDate) : '',
        // 'Payment link completion date': registration?.paymentDetails?.[0]?.paymentStatus === 'online_completed' ? (registration?.paymentDetails?.[0]?.updatedAt ? new Date(registration.paymentDetails[0].updatedAt).toLocaleDateString('en-GB') : '') : '',
        'Programme Name with period': this.formatProgrammeNameWithPeriod(registration?.program, registration?.allocatedProgram) ?? ''
      };

      if(userRoles.includes('finance_manager')) {
        transformedReg = {
          ...transformedReg,
          'Mode of payment': this.formatPaymentMode(registration?.paymentDetails?.[0]?.paymentMode,registration?.paymentDetails[0]?.offlineMeta) ?? '',
          'Cheque/DD #': registration?.paymentDetails?.[0]?.offlineMeta?.chequeNo ?? '',
          'Cheque/DD Date': formatDate(registration?.paymentDetails?.[0]?.offlineMeta?.chequeDate),
          'UTR/Reference #': registration?.paymentDetails?.[0]?.offlineMeta?.utrNumber ?? registration?.paymentDetails?.[0]?.offlineMeta?.referenceNumber ?? '',
          'NEFT / Bank Transfer Date': formatDate(registration?.paymentDetails?.[0]?.offlineMeta?.bankTransferDate),
          [`CGST ${registration?.program?.cgst}%`]: igstCheck ? '' : this.calculateTaxAmount((registration?.paymentDetails?.[0]?.originalAmount ? Number(registration?.paymentDetails?.[0]?.originalAmount) : 0), (registration?.program?.cgst ?? 9)),
          [`SGST ${registration?.program?.sgst}%`]: igstCheck ? '' : this.calculateTaxAmount((registration?.paymentDetails?.[0]?.originalAmount ? Number(registration?.paymentDetails?.[0]?.originalAmount) : 0), (registration?.program?.sgst ?? 9)),
          [`IGST ${registration?.program?.igst}%`]: igstCheck ? this.calculateTaxAmount(registration?.paymentDetails?.[0]?.originalAmount ? Number(registration?.paymentDetails?.[0]?.originalAmount) : 0, (registration?.program?.igst ?? 18)) : '',
          'Total amount': registration?.paymentDetails?.[0]?.subTotal ?? registration?.paymentDetails?.[0]?.subTotal ?? '',
          'TDS': registration?.paymentDetails?.[0]?.tds ?? '',
          'Amount Received': registration?.paymentDetails?.[0]?.subTotal ? registration?.paymentDetails?.[0]?.subTotal + registration?.paymentDetails?.[0]?.tds : '',
          'Fee Amount': registration?.invoiceDetails?.[0]?.baseAmount ?? registration?.paymentDetails?.[0]?.originalAmount ?? '',
          'GSTIN': registration?.invoiceDetails?.[0]?.gstNumber ?? '',
          'TAN': registration?.invoiceDetails?.[0]?.tanNumber ?? '',
          'Payment ID (Razor pay)': registration?.paymentDetails?.[0]?.offlineMeta?.razropayPaymentId ?? '',
          'Order ID (Razor pay)': registration?.paymentDetails?.[0]?.razorpayId ?? '',
          'Invoice Type': registration?.invoiceDetails?.[0] ? (igstCheck ? 'IGST' : 'CGST AND SGST') : '',
          'Invoice Name': registration?.invoiceDetails?.[0]?.invoiceName ?? '',
          'Invoice Email': registration?.invoiceDetails?.[0]?.invoiceEmail ?? '',
          'Invoice Address': registration?.invoiceDetails?.[0]?.invoiceAddress ?? '',
          'Pincode': registration?.invoiceDetails?.[0]?.zip ?? '',
          'Place of Supply':  registration?.invoiceDetails?.gstNumber ? stateGstCodes.find(code => code.code === registration.invoiceDetails?.gstNumber.substring(0, 2))?.stateName || null : null,
          'Invoice #': registration?.invoiceDetails?.[0]?.invoiceSequenceNumber ?? '',
          // 'Invoice Date': registration?.invoiceDetails?.[0]?.invoiceIssuedDate ? new Date(registration.invoiceDetails[0].invoiceIssuedDate).toLocaleDateString('en-GB') : '',
          'e-Invoice Ack #': registration?.invoiceDetails?.[0]?.einvoiceAckNumber ?? '',
          'e-Invoice Date': registration?.invoiceDetails?.[0]?.einvoiceAckDate ? formatDateTimeIST(registration.invoiceDetails[0].einvoiceAckDate) : '',
          'Invoice Order Type': registration?.paymentDetails?.[0]?.paymentMode === 'online' ? 'Online' : 'Offline',
        }
      } else {
        transformedReg = {
          ...transformedReg,
          'Payment Status': this.formatPaymentStatus(registration?.paymentDetails?.[0]?.paymentStatus ?? '', registration?.allocatedProgram?.id, registration?.isFreeSeat),
          'Invoice Status': this.formatInvoiceStatus(registration?.invoiceDetails?.[0]?.invoiceStatus ?? '', registration?.allocatedProgram?.id, registration?.isFreeSeat, registration?.paymentDetails?.[0]?.paymentStatus),
          'Invoice Created Date Time': registration?.invoiceDetails?.[0]?.createdAt ? formatDateTimeIST(registration.invoiceDetails[0].createdAt) : '',
          'Travel Status': getTravelOverAllStatus(registration) || '',
          'Type of ID': registration.travelInfo?.[0]?.idType ?? '',
          'ID Number': registration.travelInfo?.[0]?.idNumber ?? '',
          'T-Shirt Size': registration.travelInfo?.[0]?.tshirtSize ?? '',
          'Arrival Airline Name': registration.travelPlans?.[0]?.airlineName ?? '',
          'Flight Number': registration.travelPlans?.[0]?.flightNumber ?? '',
          'Arrival Date Time': registration.travelPlans?.[0]?.arrivalDatetime ? formatDateTimeIST(registration.travelPlans[0].arrivalDatetime) : '',
          'Arrival Location': registration.travel,
          'Preferred Roommate Name': registration.preferredRoomMate ?? '',
        }
      }

      return transformedReg;
    });
    // Generate filename
    const timestamp = new Date().toISOString().split('T')[0];
    const filename = `Registrations_${timestamp.replace(/-/g, '')}.xlsx`;

      // Excel options
      const excelOptions = {
        sheetName: 'Registrations Export',
        makeHeaderBold: true,
        autoWidth: true
      };

      // Generate Excel and upload to S3
      const s3Url = await this.excelService.jsonToExcelAndUpload(
        excelData,
        filename,
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        excelOptions
      );

      return s3Url;
    } catch (error) {
      this.logger.error('Error generating Excel download', error);
      throw new Error('Failed to generate Excel file');
    }
  }

/**
 * Helper method to format payment mode for display
 */
private formatPaymentMode(paymentMode: string,offlineMeta:any): string {
  if (!paymentMode) return '';
  
  switch (paymentMode.toLowerCase()) {
    case 'online':
      return 'Online ( Razorpay )';
    case 'offline':
      return offlineMeta?.paymentMode ?? 'Offline';
    default:
      return paymentMode;
  }
}

/**
 * Helper method to calculate tax amount based on base amount and tax percentage
 */
private calculateTaxAmount(baseAmount: number, taxPercentage: number): number {
  if (!baseAmount || !taxPercentage) return 0;
  return (baseAmount * taxPercentage) / 100;
}

/**
 * Helper method to format programme name with period
 */
private formatProgrammeNameWithPeriod(program: any, allocatedProgram: any): string {
  const programName = allocatedProgram?.name || program?.name;
  
  if (!programName) return '';
  
  // You can customize this format based on how you want to display the period
    const startDate = allocatedProgram?.startsAt
      ? formatDate(allocatedProgram.startsAt.toISOString())
      : program?.startsAt
        ? formatDate(program.startsAt.toISOString())
        : '';
    const endDate = allocatedProgram?.endsAt
      ? formatDate(allocatedProgram.endsAt.toISOString())
      : program?.endsAt
        ? formatDate(program.endsAt.toISOString())
        : '';

  if (startDate && endDate) {
    return `${programName} (${startDate} to ${endDate})`;
  } else if (startDate) {
    return `${programName} (From ${startDate})`;
  } else {
    return programName;
  }
}

  /**
   * Get table headers configuration based on user roles and view type
   */
  private getTableHeaders(userRoles: string[] = [], view: string = ViewTypeEnum.REGISTRATIONS) {
    // Delegate to specific view header methods based on view type
    if (view === 'goodies') {
      return this.getGoodiesViewHeaders(userRoles);
    }

    // Default: Registrations view
    let headers = [
      {
        key: 'seekerName',
        label: 'Seeker Name',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '1B'
      },
      {
        key: 'profileUrl',
        label: 'Profile URL',
        sortable: false,
        filterable: false,
        type: 'string',
        order: '1A'
      },
      {
        key: 'age',
        label: 'Age',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '2A'
      },
      {
        key: 'gender',
        label: 'Gender',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '2B'
      },
      {
        key: 'location',
        label: 'Location',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '2C'
      },
      {
        key: 'blessedWith',
        label: 'Status',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '3'
      },
      {
        key: 'numberOfHDBs',
        label: 'HDBs',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '6'
      },
      {
        key: 'rmComments',
        label: 'RM Comments',
        sortable: false,
        filterable: true,
        type: 'string',
        order: '5B'
      }
    ];

    if (userRoles.includes('finance_manager')) {
      headers.push({
        key: 'invoiceStatus',
        label: 'Invoice Status',
        sortable: false,
        filterable: true,
        type: 'string',
        order: '8A',
      });
    }

    if(userRoles.includes('mahatria') || userRoles.includes('shoba')) {
      headers.push(
        {
          key: 'lastHdbMsd',
          label: 'Last HDB/MSD',
          sortable: true,
          filterable: false,
          type: 'string',
          order: '7'
        },
        {
          key: 'rmContact',
          label: 'RM Contact',
          sortable: true,
          filterable: false,
          type: 'string',
          order: '4'
        },
      );
    }
    if (
      userRoles.includes(ROLE_VALUES.OPERATIONAL_MANAGER) ||
      userRoles.includes(ROLE_VALUES.RM_SUPPORT)
    ) {
      headers.push(
        {
          key: 'rmContact',
          label: 'RM Contact',
          sortable: true,
          filterable: false,
          type: 'string',
          order: '4'
        },
      );
    }
    if (!userRoles.includes('mahatria')) {
      headers.push(
        {
          key: 'paymentStatus',
          label: 'Payment Status',
          sortable: false,
          filterable: true,
          type: 'string',
          order: '8'
        },
        {
          key: 'travelPlanStatus',
          label: 'Travel Status',
          sortable: true,
          filterable: true,
          type: 'string',
          order: '9A'
        },
        {
          key: 'goodiesStatus',
          label: 'Goodies Status',
          sortable: false,
          filterable: true,
          type: 'string',
          order: '9B'
        },
      );
    }
    if (userRoles.includes('shoba')) {
      //RM Recommendation 
        headers.push({
          key: 'recommendation',
          label: 'Recommendation',
          sortable: false,
          filterable: false,
          type: 'string',
          order: '8A'
        },
        {
          key: 'recommendationComments',
          label: 'Recommendation Comments',
          sortable: false,
          filterable: false,
          type: 'string',
          order: '8B'
        });
    }
    if(userRoles.includes('mahatria') || userRoles.includes('shoba') || userRoles.includes('relational_manager')) {
      headers.push(
      {
        key: 'averageRating',
        label: 'RM Rating',
        sortable: true,
        filterable: false,
        type: 'number',
        order : '5A'
      });
    }

    // Remove headers for RM and operational manager roles
    const isUserRoleRMorOM =
      userRoles.includes('finance_manager') ||
      userRoles.includes('operational_manger') ||
      userRoles.includes('rm_support');
    if (isUserRoleRMorOM) {
      headers = headers.filter(header => 
        !['rmComments', 'averageRating','recommendation'].includes(header.key)
      );
    }

    // Sort headers by order
    return headers.sort((a, b) => a.order.localeCompare(b.order));
  }

  /**
   * Get Goodies view table headers configuration
   */
  private getGoodiesViewHeaders(userRoles: string[] = []) {
    let headers = [
      {
        key: 'seekerName',
        label: 'Seeker Name',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '1B'
      },
      {
        key: 'profileUrl',
        label: 'Profile URL',
        sortable: false,
        filterable: false,
        type: 'string',
        order: '1A'
      },
      {
        key: 'age',
        label: 'Age',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '2A'
      },
      {
        key: 'gender',
        label: 'Gender',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '2B'
      },
      {
        key: 'location',
        label: 'Location',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '2C'
      },
      {
        key: 'goodiesStatus',
        label: 'Goodies Status',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '5'
      },
      {
        key: 'tshirt',
        label: 'T-Shirt',
        sortable: true,
        filterable: true,
        type: 'goodies',
        order: '6'
      },
      {
        key: 'jacket',
        label: 'Jacket',
        sortable: true,
        filterable: true,
        type: 'goodies',
        order: '7'
      },
      {
        key: 'ratriaPillar',
        label: 'Ratria Pillars',
        sortable: true,
        filterable: true,
        type: 'goodies',
        order: '8'
      },
      {
        key: 'flask',
        label: 'Flask',
        sortable: true,
        filterable: true,
        type: 'goodies',
        order: '9'
      },
      {
        key: 'notebook',
        label: 'Notebook',
        sortable: true,
        filterable: true,
        type: 'goodies',
        order: '10'
      }
    ];

    // Add Coordinator (shoba) specific columns
    if (userRoles.includes(ROLE_VALUES.COORDINATOR)) {
      headers.push(
        {
          key: 'rmContact',
          label: 'RM Contact',
          sortable: true,
          filterable: true,
          type: 'string',
          order: '3'
        },
        {
          key: 'lastHdbMsd',
          label: 'Last HDB/MSD',
          sortable: true,
          filterable: true,
          type: 'string',
          order: '4'
        }
      );
    }

    // Sort headers by order
    return headers.sort((a, b) => a.order.localeCompare(b.order));
  }

  /**
   * Get static view list configuration
   */
  private getViewList() {
    return [
      {
        key: ViewTypeEnum.REGISTRATIONS,
        label: 'Registrations',
        value: ViewTypeEnum.REGISTRATIONS
      },
      {
        key: ViewTypeEnum.GOODIES,
        label: 'Goodies',
        value: ViewTypeEnum.GOODIES
      }
      // ,
      // {
      //   key: ViewTypeEnum.TRAVEL,
      //   label: 'Travel',
      //   value: ViewTypeEnum.TRAVEL
      // }
    ];
  }

  /**
   * Get quick view headers configuration based on user roles
   */
  private getQuickViewHeaders(userRoles: string[] = []) {
    const headers = [
      {
        key: 'seekerName',
        label: 'Seeker name',
        type: 'string',
        order: '1'
      },
      {
        key: 'gender',
        label: 'Gender',
        type: 'string',
        order: '2'
      },
      {
        key: 'age',
        label: 'Age',
        type: 'number',
        order: '3'
      },
      {
        key: 'location',
        label: 'City',
        type: 'string',
        order: '4'
      },
      {
        key: 'profilePictureUrl',
        label: 'Profile picture',
        type: 'string',
        order: '5'
      },
      {
        key: 'numberOfHDBs',
        label: 'Number of HDBs done',
        type: 'number',
        order: '6'
      },
      {
        key: 'preferredRoomMate',
        label: 'Preferred roommate name',
        type: 'string',
        order: '7'
      },
      {
        key: 'appliedOn',
        label: 'Registration date',
        type: 'date',
        order: '8'
      },
      {
        key: 'dob',
        label: 'DOB',
        type: 'date',
        order: '9'
      },
      {
        key: 'mobileNumber',
        label: 'Mobile number',
        type: 'string',
        order: '10'
      },
      {
        key: 'emailAddress',
        label: 'Email address',
        type: 'string',
        order: '11'
      },
      {
        key: 'rmContact',
        label: 'Contact person',
        type: 'string',
        order: '12'
      },
      {
        key: 'lastHdbMsd',
        label: 'When was your last HDB/MSD?',
        type: 'string',
        order: '13'
      },
      {
        key: 'countryName',
        label: 'Country',
        type: 'string',
        order: '14'
      },
      {
        key: 'approvalStatus',
        label: 'Registration status',
        type: 'string',
        order: '15'
      },
      {
        key: 'blessedDate',
        label: 'Blessed date',
        type: 'date',
        order: '16'
      },
      {
        key: 'preference',
        label: 'HDB/MSD preference',
        type: 'array',
        order: '17'
      },
      {
        key: 'swapPreference',
        label: 'Swap preference',
        type: 'string',
        order: '18'
      },
      {
        key: 'hdbAssociationSince',
        label: 'Since when have you been associated with infinitheism',
        type: 'string',
        order: '19'
      },
      {
        key: 'registrationLastModified',
        label: 'Registration last modified',
        type: 'date',
        order: '20'
      },
      {
        key: 'goodiesStatus',
        label: 'Goodies status',
        type: 'string',
        order: '21'
      }
    ];

    if(userRoles.includes('mahatria') || userRoles.includes('shoba') || userRoles.includes('relational_manager')) {
      headers.push(
        {
          key: 'swapDemand',
          label: 'Swap demand',
          type: 'string',
          order: '22A'
        },
        {
          key: 'swapDemandComments',
          label: 'Swap demand comments',
          type: 'string',
          order: '22B'
        },
        {
        key: 'rmComments',
        label: 'RM review',
        type: 'string',
        order: '23A'
      },
      {
        key: 'averageRating',
        label: 'RM rating',
        type: 'string',
        order: '23B'
      },
      {
        key: 'recommendation',
        label: 'RM recommendation',
        type: 'string',
        order: '24A'
      },
      {
        key: 'recommendationComments',
        label: 'RM recommendation comments',
        type: 'string',
        order: '24B'
      });
    }

    return headers;
  }

  /**
   * Flatten KPIs for easier frontend consumption
   */
  private flattenKPIs(kpis: any) {
      const flatKPIs: Array<{ 
          category: string; 
          categoryLabel: string; 
          [key: string]: any; 
      }> = [];

    // Handle both old static structure and new dynamic structure
    Object.keys(kpis).forEach(categoryKey => {
      const category = kpis[categoryKey];
      if (category && category.items && Array.isArray(category.items)) {
        category.items.forEach(item => {
          flatKPIs.push({
            ...item,
            category: categoryKey,
            categoryLabel: category.categoryLabel
          });
        });
      }
    });

    const flatKPIsLength = flatKPIs.length + 1
    const orderedKPIs = flatKPIs.map((kpi) => {kpi.order = kpi.order <0 ? flatKPIsLength + kpi.order : kpi.order; return kpi;}).sort((a, b) => a.order - b.order);
    return orderedKPIs;
  }

   private getDraftRegTableHeaders(userRoles: string[] = []) {
    let headers = [
      {
        key: 'profileUrl',
        label: 'Profile URL',
        sortKey: 'profileUrl',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '1'
      },
      {
        key: 'seekerName',
        label: 'Seeker Name',
        sortKey: 'fullName',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '2'
      },
      {
        key: 'age',
        label: 'Age',
        sortKey: 'dob',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '3'
      },
      {
        key: 'gender',
        label: 'Gender',
        sortKey: 'gender',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '4'
      },
      {
        // email Address
        key: 'emailAddress',
        label: 'Email Address',
        sortKey: 'emailAddress',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '5'
      },
      {
        key: 'location',
        label: 'Location',
        sortKey: 'city',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '6'
      },
      {
        key: 'numberOfHDBs',
        label: 'No. of HDBs',
        sortKey: 'noOfHDBs',
        sortable: true,
        filterable: false,
        type: 'number',
        order: '7'
      },
      {
        key: 'mobileNumber',
        label: 'Mobile Number',
        sortKey: 'mobileNumber',
        sortable: true,
        filterable: true,
        type: 'string',
        order: '8'
      },
      {
        // rm Contact 
        key: 'rmContact',
        label: 'RM Contact',
        sortKey: '',
        sortable: false,
        filterable: false,
        type: 'string',
        order: '9'
      }
    ];

    // if user role is relational_manager then remove rmContact
    if (userRoles.includes('relational_manager')) {
      headers = headers.filter(header => header.key !== 'rmContact');
    }
    // Sort headers by order
    return headers.sort((a, b) => a.order.localeCompare(b.order));
  }

  private getFilterConfiguration(userRoles: string[]): any[] {
    const filters = [
      {
        key: 'approvalStatus',
        label: 'Seeker status', 
        type: 'checkbox',
        options: [
          { label: 'Unassigned', value: 'pending' },
          { label: 'Blessed ', value: 'approved' },
          { label: 'Swap demand', value: 'on_hold' },
          { label: 'On Hold', value: 'rejected' },
          { label: 'Cancelled', value: 'cancelled' }
        ]
      },
      {
        key: 'gender',
        label: 'Gender',
        type: 'checkbox',
        options: [
          { label: 'Male', value: 'Male' },
          { label: 'Female', value: 'Female' }
        ]
      },
      {
        key: 'age',
        label: 'Age',
        type: 'checkbox',
        options: [
          { label: '<=20 years', value: '0-20' },
          { label: '21 - 30 years', value: '21-30' },
          { label: '31 - 50 years', value: '31-50' },
          { label: '51 - 65 years', value: '51-65' },
          { label: '> 65 years', value: '>65' }
        ]
      },
      {
        key: 'numberOfHdbs',
        label: 'Number of HDBs',
        type: 'checkbox',
        options: [
          {
            label: '1st Timer',
            value: '0',
          },
          {
            label: '1 - 10 HDBs/MSDs',
            value: '1-10',
          },
          {
            label: '>10 HDBs/MSDs',
            value: '>10',
          },
        ]
      },
      {
        key: 'location',
        label: 'Location',
        type: 'dropdown',
        options: citiesList
      },
      ...(userRoles.includes('mahatria') || userRoles.includes('shoba') || userRoles.includes('admin') || userRoles.includes('finance_manager') || userRoles.includes('relational_manager')
    ? [{
        key: 'freeSeat',
        label: 'Seat Type',
        type: 'checkbox',
        options: [
          { label: 'Free Seat', value: 'yes' },
          { label: 'Paid Seat', value: 'no' }
        ]
      }]
    : []),
      /** commented out registrationStatus filter because it is not used currently but might require in future
     since we already have individual statuses for travel, payment, and approval, the registration status is no longer required **/
      // {
      //   key: 'registrationStatus',
      //   label: 'Registration status',
      //   type: 'checkbox',
      //   options: [
      //     { label: 'Pending', value: 'pending' },
      //     { label: 'Completed', value: 'completed' },
      //     // { label: 'Cancelled', value: 'cancelled' },
      //     // { label: 'Waitlisted', value: 'waitlisted' }
      //   ]
      // },
    ];

    filters.push({
        key: 'paymentMode',
        label: 'Payment mode',
        type: 'checkbox',
        options: [
          { label: 'Online', value: 'online' },
          { label: 'Offline', value: 'offline' },
        ]
      });
    // Add role-based filters
      filters.push({
        key: 'paymentStatus',
        label: 'Payment status',
        type: 'checkbox',
        options: [
          { label: 'Completed', value: 'payment_completed' },
          { label: 'Pending', value: 'payment_pending' },
          { label: 'No payment', value: 'no_payment' },
        ]
      });
     

      filters.push({
        key: 'invoiceStatus',
        label: 'Invoice status',
        type: 'checkbox',
        options: [
          { label: 'Completed', value: 'invoice_completed' },
          { label: 'Pending', value: 'invoice_pending' },
        ]
      });

      filters.push({
        key: 'travelStatus',
        label: 'Travel status',
        type: 'checkbox',
        options: [
          { label: 'Completed', value: 'travel_completed' },
          { label: 'Pending', value: 'travel_pending' }
        ]
      });

      if (userRoles.includes('admin') || userRoles.includes('mahatria') || userRoles.includes('shoba')) {
          filters.push({
            key: 'dateRange',
            label: 'Seeker registration start & end date',
            type: 'date',
            options: [
              { label: 'Yes', value: 'yes' },
              { label: 'No', value: 'no' }
            ]
          });
      }

      // add preferredRoomMate
      filters.push({
        key: 'preferredRoomMate',
        label: 'Preferred Roommate',
        type: 'checkbox',
        options: [
          { label: 'Yes', value: 'yes' },
          { label: 'No', value: 'no' }
        ]
      });


    if (userRoles.includes('admin') || userRoles.includes('mahatria') || userRoles.includes('shoba')) {
      filters.push({
        key: 'swapRequests',
        label: 'Swap request',
        type: 'checkbox',
        options: [
          { label: 'Can shift', value: 'can_shift' },
          { label: 'Wants swap', value: 'wants_swap' }
        ]
      });
    }

      filters.push(
        {
          key: 'organisation',
          label: 'From Organization',
          type: 'checkbox',
          options: [
            { label : 'Yes', value: 'yes' },
            { label : 'No', value: 'no' }
          ]
        }
      );


      if (
        !userRoles.includes('finance_manager') &&
        !userRoles.includes('operational_manger') &&
        !userRoles.includes('rm_support')
      ) {
        filters.push({
          key: 'rmRating',
          label: 'RM rating',
          type: 'checkbox',
          options: [
            { label: '1 & up', value: '1up' },
            { label: '2 & up', value: '2up' },
            { label: '3 & up', value: '3up' },
            { label: '4 & up', value: '4up' },
            { label: 'Only 5', value: 'only5' },
            { label: 'Yet to review', value: 'unrated' },
          ],
        });
        filters.push({
        key: 'recommendation',
        label: 'RM Recommendation',
        type: 'checkbox',
        options: [
          { label: 'Wholeheartedly', value: RecommendationLevel.WHOLEHEARTEDLY },
          { label: 'Affirmatively', value: RecommendationLevel.AFFIRMATIVELY },
          { label: 'Supportively', value: RecommendationLevel.SUPPORTIVELY },
          { label: 'None', value: 'none' },
        ],
      });
      }

    return filters;
  }

  async getSeatAllocationFiltersConfiguration(programId: number, userRoles: string[]): Promise<any[]> {
    const filters = [
      {
        key: 'age',
        label: 'Age',
        type: 'checkbox',
        options: [
          { label: '<=20 years', value: '0-20' },
          { label: '21 - 30 years', value: '21-30' },
          { label: '31 - 50 years', value: '31-50' },
          { label: '51 - 65 years', value: '51-65' },
          { label: '> 65 years', value: '>65' }
        ]
      },
      {
        key: 'numberOfHdbs',
        label: 'Number of HDBs',
        type: 'multi',
        options: [
          {
            label: 'Equals',
            value: '=',
          },
          {
            label: 'Greater than',
            value: '>',
          },
          {
            label: 'Less than',
            value: '<',
          },
          {
            label: 'Within a range',
            value: '-',
          },
        ]
      },
      {
        key: 'experienceTags',
        label: 'Experiences',
        type: 'dropdown',
        options: await this.getExperienceTagFilterOptions()
      },
      {
        key: 'freeSeat',
        label: 'Seat Type',
        type: 'checkbox',
        options: [
          { label: 'Free Seat', value: 'yes' },
          { label: 'Paid Seat', value: 'no' }
        ]
      },
      {
        key: 'organisation',
        label: 'Organisation',
        type: 'checkbox',
        options: [
          { label : 'Yes', value: 'yes' },
          { label : 'No', value: 'no' }
        ]
      }
    ];

    // Add RM Contact filter 
    const hasAllowedRole = userRoles && userRoles.some(role => allowedRolesForRMFilter.includes(role));
    if (hasAllowedRole) {
      try {
        // Get all relational managers without limit/offset (get all)
        const rmFilters = { roleKey: ROLE_KEYS.RELATIONAL_MANAGER };
        const rmUsers = await this.userService.getAllUsersWithoutPagination('', rmFilters); // Using high limit to get all
        const rmOptions = rmUsers.map(user => ({
          label: user.orgUsrName,
          value: user.id.toString()
        }));

        filters.push({
          key: 'rmContact',
          label: 'RM Contact',
          type: 'dropdown',
          options: rmOptions
        });
      } catch (error) {
        this.logger.error('Error fetching RM users for filter', error);
        // Add empty filter if there's an error
        // filters.push({
        //   key: 'rmContact',
        //   label: 'RM Contact',
        //   type: 'dropdown',
        //   options: []
        // });
      }
    }

    filters.push({
      key: 'rmRating',
      label: 'RM rating',
      type: 'checkbox',
      options: [
        { label: '1 & up', value: '1up' },
        { label: '2 & up', value: '2up' },
        { label: '3 & up', value: '3up' },
        { label: '4 & up', value: '4up' },
        { label: 'Only 5', value: 'only5' },
        { label: 'Yet to review', value: 'unrated' },
      ],
    });

    filters.push({
      key: 'recommendation',
      label: 'RM Recommendation',
      type: 'checkbox',
      options: [  
        { label: 'Wholeheartedly', value: RecommendationLevel.WHOLEHEARTEDLY },
        { label: 'Affirmatively', value: RecommendationLevel.AFFIRMATIVELY },
        { label: 'Supportively', value: RecommendationLevel.SUPPORTIVELY },
        { label: 'None', value: 'none' },
      ],
    });

    filters.push({
      key: 'birthdayListRange',
      label: 'Birthday List Range',
      type: 'checkbox',
      options: await this.getBirthdayFilterOptions(programId)
    });

    filters.push({
        key: 'location',
        label: 'Location',
        type: 'dropdown',
        options: citiesList
      });

    return filters;
  }

  /**
   * Common function to get all sub-programs for a given program ID
   * @param programId - The parent program ID
   * @returns Array of sub-programs with id, name, code, and totalSeats
   */
  async getSubPrograms(programId: number): Promise<{ id: number; name: string; code: string; totalSeats: number }[]> {
    try {
      if (!programId || programId <= 0) {
        this.logger.warn(`Invalid programId provided: ${programId}`);
        return [];
      }

      const subPrograms = await this.repository.getSubPrograms(programId);
      this.logger.log(`Retrieved ${subPrograms.length} sub-programs for programId: ${programId}`);
      
      return subPrograms;
    } catch (error) {
      this.logger.error(`Error fetching sub-programs for programId ${programId}: ${error.message}`, error.stack);
      return [];
    }
  }

  /**
   * Get birthday filter options with sub-program data
   * @param programId - The parent program ID
   * @returns Array of options for birthday filter
   */
  private async getBirthdayFilterOptions(programId: number): Promise<{ label: string; value: string }[]> {
    try {
      const subPrograms = await this.getSubPrograms(programId);
      
      // Map sub-programs to filter options
      const options = subPrograms.map(subProgram => ({
        label: subProgram.name,
        value: subProgram.id.toString()
      }));

      this.logger.log(`Generated ${options.length} birthday filter options for programId: ${programId}`);
      return options;
    } catch (error) {
      this.logger.error(`Error generating birthday filter options for programId ${programId}: ${error.message}`, error.stack);
      // Return empty array if there's an error
      return [];
    }
  }

  private async getExperienceTagFilterOptions(): Promise<{ label: string; value: string }[]> 
  {
    try {
      const experienceTags = await this.repository.getExperienceTags();

      if (!experienceTags || experienceTags.length === 0) {
        this.logger.warn(`No experience tags found`);
        return [];
      }

      return experienceTags.map(tag => ({
        label: tag.lookupLabel || 'Unknown',
        value: tag.lookupKey || 'unknown'
      }));
    } catch (error) {
      this.logger.error(`Error generating experience tag filter options: ${error.message}`, error.stack);
      return [];
    }
  }

  private extractKPIFilters(parsedFilters: Record<string, any>): Record<string, any> {
    const kpiFilters: Record<string, any> = {};

    if (parsedFilters?.kpiCategory && parsedFilters?.kpiFilter) {
      const kpiFilterCondition = this.getKPIFilterCondition(
        parsedFilters.kpiCategory,
        parsedFilters.kpiFilter,
      );

      if (kpiFilterCondition) {
        Object.assign(kpiFilters, kpiFilterCondition);
      }
    }

    return kpiFilters;
  }

  /**
   * Extract data filters (non-KPI filters)
   */
  private extractDataFilters(parsedFilters: Record<string, any>): Record<string, any> {
    const dataFilters = { ...parsedFilters };

    // Remove parent filter - it's only for KPI generation, not data filtering
    delete dataFilters.parentFilter;

    // Apply KPI filters to data if they exist
    if (parsedFilters?.kpiCategory && parsedFilters?.kpiFilter) {
      const kpiFilterCondition = this.getKPIFilterCondition(
        parsedFilters.kpiCategory,
        parsedFilters.kpiFilter,
      );
      if (kpiFilterCondition) {
        Object.assign(dataFilters, kpiFilterCondition);
      }
    }

    return dataFilters;
  }

  /**
   * Extract filters specifically for report generation
   * Reports need different handling of KPI filters compared to regular data filtering
   */
  private extractReportFilters(parsedFilters: Record<string, any>): Record<string, any> {
    const dataFilters = { ...parsedFilters };

    // Remove parent filter - it's only for KPI generation, not data filtering
    delete dataFilters.parentFilter;

    // Apply KPI filters to reports with report-specific logic
    if (parsedFilters?.kpiCategory && parsedFilters?.kpiFilter) {
      const kpiFilterCondition = this.getKPIFilterCondition(
        parsedFilters.kpiCategory,
        parsedFilters.kpiFilter,
      );

      if (kpiFilterCondition) {
        // For reports, modify certain conditions to be more inclusive
        const reportFilterCondition = this.adaptFilterForReports(
          kpiFilterCondition,
          parsedFilters.kpiCategory,
          parsedFilters.kpiFilter
        );
        Object.assign(dataFilters, reportFilterCondition);
      }
      
      // Remove KPI filter parameters after processing
      delete dataFilters.kpiCategory;
      delete dataFilters.kpiFilter;
    }

    return dataFilters;
  }

  /**
   * Adapt existing KPI filter conditions for report generation
   * This allows us to reuse existing logic while making reports more inclusive
   */
  private adaptFilterForReports(
    filterCondition: Record<string, any>, 
    category: string, 
    filter: string
  ): Record<string, any> {
    // For "all" category with program filters, use allocatedProgramId and remove approval status restrictions
    if (category === 'all' && filter.startsWith('program_')) {
      const programId = parseInt(filter.replace('program_', ''));
      return { allocatedProgramId: programId };
    }

    // For "programs" category, use allocatedProgramId and remove approval status restrictions
    if (category === 'programs' && filter.startsWith('program_')) {
      const programId = parseInt(filter.replace('program_', ''));
      return { allocatedProgramId: programId };
    }

    // For most other cases, remove restrictive approval status filters for reports
    const adaptedFilter = { ...filterCondition };
    
    // Remove approval status restrictions for reports to be more inclusive
    if (adaptedFilter.approvalStatus && Array.isArray(adaptedFilter.approvalStatus)) {
      delete adaptedFilter.approvalStatus;
    }

    return adaptedFilter;
  }

  /**
   * Remove KPI filter parameters from filters object
   */
  private removeKPIFilters(parsedFilters: Record<string, any>): Record<string, any> {
    const cleanFilters = { ...parsedFilters };
    delete cleanFilters.kpiCategory;
    delete cleanFilters.kpiFilter;
    delete cleanFilters.parentFilter; // Also remove parentFilter from data queries
    return cleanFilters;
  }

  /**
   * Get Mahatria-specific KPIs for program allocation management with registration data
   */
  async getMahatriaKPIs(
    limit: number,
    offset: number,
    programId: number,
    searchText: string = '',
    parsedFilters: Record<string, any> = {},
    user: any,
    sortKey: string,
    sortOrder: 'ASC' | 'DESC',
  ): Promise<{
    data: any[];
    pagination: any;
    kpis: {
      mahatria: any;
    };
    filters: any;
  }> {
    this.logger.log('Getting Mahatria KPIs with registration data', {
      limit,
      offset,
      programId,
      searchText,
      parsedFilters,
    });

    // Reverse DOB sorting logic: if sorting by DOB, reverse the order because older DOB means older age
    if (sortKey === RegistrationSortKey.DOB) {
      sortOrder = sortOrder.toLocaleLowerCase() === 'asc' ? 'DESC' : 'ASC'; // Reverse DOB sorting
    }

    // Use common default sort function for consistent sorting behavior
    const defaultSort = this.getDefaultSortParameters(sortKey, sortOrder, parsedFilters);
    const finalSortKey = defaultSort.sortKey;
    const finalSortOrder = defaultSort.sortOrder;
    
    this.logger.log(`Final sort key/order for Mahatria KPI: ${finalSortKey}/${finalSortOrder}`);

    try {
      // Validate that the program requires approval for mahatria users
      const program = await this.repository.findProgramWithType(programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }

      const userRoles = user.userRoleMaps.map(roleMap => roleMap.role.name);
      const seatAllocationFilters = await this.getSeatAllocationFiltersConfiguration(programId, userRoles);
      if (!program.requiresApproval) {
        return {
          data: [],
          pagination: {
            totalPages: 0,
            pageNumber: 0,
            pageSize: limit,
            totalRecords: 0,
            numberOfRecords: 0,
          },
          kpis: {
            mahatria: {
              groupedProgramMetrics: {
                isGroupedProgram: false,
                totalRegistrations: 0,
                message: 'No approval required for this program',
              },
            },
          },
          filters: seatAllocationFilters,
        };
      }

      // Normalize filter values (gender, etc.)
      normalizeFilterValues(parsedFilters);

      // Apply KPI filter condition if provided (for Mahatria-specific filters)
      let allKpiCounts = {};
      let cancelledCounts = {};
      if (parsedFilters?.kpiCategory && parsedFilters?.kpiFilter) {
          allKpiCounts = await this.repository.getAllRegistrationMetrics(programId) ?? {}
          cancelledCounts = await this.repository.getCancelledRegistrationMetrics(programId) ?? {}
        const kpiFilterCondition = this.getMahatriaKPIFilterCondition(
          parsedFilters.kpiCategory,
          parsedFilters.kpiFilter,
        );
        if (kpiFilterCondition) {
          Object.assign(parsedFilters, kpiFilterCondition);
        }
      }

      // Get registration data WITH search text (for table filtering)
      const registrationResult = await this.repository.findRegistrations(
        limit,
        offset,
        programId,
        null, // programSessionId - not used for Mahatria KPIs
        searchText, // ✅ Search applies to data table
        parsedFilters,
        [], // userRoles - empty for Mahatria context
        undefined, // filterRegistrationsByUserId - not used for Mahatria
        finalSortKey,
        finalSortOrder,
        SEAT_ALLOCATION_REQUIRED_RELATIONS,
        undefined,
        SEAT_ALLOCATION_SELECT_FIELDS,
        SEAT_ALLOCATION_RELATION_SELECT
      );

      // ✅ CRITICAL FIX: Generate Mahatria-specific KPIs WITHOUT search text
      const mahatriaKPIResult = await this.generateMahatriaKPIs({
        programId,
        searchText: '', // ✅ Empty search text for KPIs
        parsedFilters,
      });

      // Format the KPIs similar to default KPIs structure
      const formattedMahatriaKPIs = this.formatMahatriaKPIs(mahatriaKPIResult);
      // Add all KPIs if available
      formattedMahatriaKPIs['allKpis'] = allKpiCounts;
      // Add cancelled KPIs if available
      formattedMahatriaKPIs['cancelledKpis'] = cancelledCounts;

      const swapData = await this.programRegistrationService.getSwapRequestKPI({programId});
      formattedMahatriaKPIs['swapRequests'] = swapData?.kpis || [];
    
      const swapDemand = await this.programRegistrationService.getSwapDemandKPI({programId});
      formattedMahatriaKPIs['swapDemands'] = swapDemand?.kpis || [];

        const regPending = this.programRegistrationService.getRegistrationPendingKPI({
         formattedKPIs: formattedMahatriaKPIs || [],
        });
        formattedMahatriaKPIs['registrationPendings'] = regPending?.kpis || [];

      return {
        data: registrationResult.data,
        pagination: registrationResult.pagination,
        kpis: {
          mahatria: {
            groupedProgramMetrics: formattedMahatriaKPIs,
          },
        },
        filters: seatAllocationFilters,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  async findOne(id: number) {
    this.logger.log(registrationMessages.GET_BY_ID);
    try {
      return await this.repository.findRegistrationById(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  async getQuestionResponses(id: number) {
    this.logger.log(registrationMessages.GET_QA_REQUEST_RECEIVED, { id });
    try {
      return await this.repository.findQuestionResponsesByRegistrationId(id);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_QUESTION_RESPONSE_GET_FAILED, error);
    }
  }

  async getRegistrationStatuses(id: number) {
    this.logger.log(registrationMessages.GET_STATUSES, { id });
    try {
      const registration = await this.repository.findRegistrationById(id);
      if (!registration) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          id.toString(),
        );
      }

      const sortLatest = (arr?: any[]) => {
        if (!arr || arr.length === 0) return null;
        return arr.sort((a, b) => {
          const dateA = new Date(a.updatedAt ?? a.createdAt ?? 0).getTime();
          const dateB = new Date(b.updatedAt ?? b.createdAt ?? 0).getTime();
          return dateB - dateA;
        })[0];
      };

      // Get the latest travel plan and travel info
      const latestTravelPlan = sortLatest(registration.travelPlans);
      const latestTravelInfo = sortLatest(registration.travelInfo);

      // Determine travel plan status based on travelType and returnTravelType
      let effectiveTravelPlanStatus: string | null = null;
      if (latestTravelPlan) {
        // If both travelType and returnTravelType are null, travel plan status should be null
        if(latestTravelPlan.travelType && latestTravelPlan.returnTravelType)
        {
           effectiveTravelPlanStatus = "completed";
        }
        else if (latestTravelPlan.travelType === null && latestTravelPlan.returnTravelType === null && latestTravelPlan.onwardAdditionalInfo === null && latestTravelPlan.returnAdditionalInfo === null) {
          effectiveTravelPlanStatus = null;
        }
        else if (latestTravelPlan.travelType || latestTravelPlan.returnTravelType || latestTravelPlan.onwardAdditionalInfo || latestTravelPlan.returnAdditionalInfo ) {
          effectiveTravelPlanStatus = "pending";
        }
         else {
          effectiveTravelPlanStatus = latestTravelPlan.travelPlanStatus;
        }
      }

      // Determine travel info status based on internationalIdPictureUrl
      let effectiveTravelInfoStatus: string | null = null;
      if (latestTravelInfo) {
        // If internationalIdPictureUrl is null, travel info status should be null
        if(latestTravelInfo.internationalIdPictureUrl)
        {
           effectiveTravelInfoStatus = "completed";
        }
        else if (latestTravelInfo.internationalIdPictureUrl === null && registration.alternatePhoneNumber === null) {
          effectiveTravelInfoStatus = null;
        }
        else if (latestTravelInfo.internationalIdPictureUrl || registration.alternatePhoneNumber ) {
          effectiveTravelInfoStatus = "pending";
        } 
         else {
          effectiveTravelInfoStatus = latestTravelInfo.travelInfoStatus;
        }
      }

      // Calculate overall travel status
      let travelStatus: string | null = null;
      if (effectiveTravelPlanStatus === 'completed' && effectiveTravelInfoStatus === 'completed') {
        travelStatus = 'completed';
      } else if (effectiveTravelPlanStatus === 'pending' || effectiveTravelInfoStatus === 'pending') {
        travelStatus = 'pending';
      } else {
        // At least one has a status but not both completed - consider as pending if any has actual status
        if (effectiveTravelPlanStatus !== null || effectiveTravelInfoStatus !== null) {
          travelStatus = 'pending';
        }
      }

      const latestGoodies = sortLatest(registration.goodies);

      let goodiesStatus: GoodiesStatusEnum | null = null;
      if (latestGoodies) {
        const goodiesFields = [
          latestGoodies.notebook,
          latestGoodies.ratriaPillarLeonia,
          latestGoodies.flask,
          latestGoodies.tshirt,
          latestGoodies.jacket,
        ];
        const filledFields = goodiesFields.filter((field) => field !== null && field !== undefined);
        if (filledFields.length === goodiesFields.length) {
          goodiesStatus = GoodiesStatusEnum.COMPLETED;
        } else if (filledFields.length > 0) {
          goodiesStatus = GoodiesStatusEnum.PENDING;
        } else {
          goodiesStatus = GoodiesStatusEnum.DRAFT;
        }
      } else {
        goodiesStatus = null;
      }

      return {
        registrationStatus: registration.registrationStatus ?? null,
        basicDetailsRegistrationStatus: registration.basicDetailsStatus ?? null,
        approvalStatus: sortLatest(registration.approvals)?.approvalStatus ?? null,
        paymentStatus: sortLatest(registration.paymentDetails)?.paymentStatus ?? null,
        invoiceStatus: sortLatest(registration.invoiceDetails)?.invoiceStatus ?? null,
        travelInfoStatus: effectiveTravelInfoStatus,
        travelPlanStatus: effectiveTravelPlanStatus,
        travelStatus,
        goodiesStatus,
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_FIND_BY_ID_FAILED, error);
    }
  }

  // ===== DEFAULT KPI GENERATION METHODS =====

  /**
   * Generate default KPIs for list view based on user roles
   */
  private async generateDefaultKPIs(parameters: {
    programId?: number | null;
    programSessionId?: number | null;
    searchText?: string; // ✅ This should be empty string when called
    parsedFilters?: Record<string, any>;
    userRoles: string[];
    filterRegistrationsByUserId?: number | null;
    activeKpiCategory?: string;
  }): Promise<DefaultKPIResult> {
    try {
      const {
        programId,
        programSessionId,
        searchText, // ✅ This will be empty string
        parsedFilters,
        userRoles,
        filterRegistrationsByUserId,
        activeKpiCategory,
      } = parameters;

      const kpis: DefaultKPIResult = {
        registrations: {
          categoryLabel: 'Registration Metrics',
          items: [],
        },
      };

      // ✅ FIXED: Get seats metrics WITHOUT search text
      const seatsMetrics = await this.repository.getSeatsMetrics(
        programId,
        programSessionId,
        parsedFilters,
        activeKpiCategory === 'registrations' ? searchText : '',
        filterRegistrationsByUserId,
      );


      // Build registration KPIs
      kpis.registrations.items = [
        {
          label: 'All',
          value: parseInt(seatsMetrics?.all ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'all',
          order: 1,
          communicationCategory: CommunicationCategoryEnum.HDB_ALL,
        },
        {
          label: 'Unassigned',
          value: parseInt(seatsMetrics?.pending ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'newSeekersPending',
          order: 2,
          communicationCategory: CommunicationCategoryEnum.HDB_SEEKER_PENDING,
        },
        {
          label: 'Blessed',
          value: parseInt(seatsMetrics?.approved ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'approvedSeekers',
          order: 3,
          communicationCategory: CommunicationCategoryEnum.HDB_BLESSED,
        },
        {
          label: 'Swap demand',
          value: parseInt(seatsMetrics?.onHold ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'onHold',
          order: -3,
          communicationCategory: CommunicationCategoryEnum.HDB_HOLD,
        },
        {
          label: 'Hold',
          value: parseInt(seatsMetrics?.rejected ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'rejected',
          order: -2,
        },
        {
          label: 'Cancelled',
          value: parseInt(seatsMetrics?.cancelled ?? '0'),
          kpiCategory: 'registrations',
          kpiFilter: 'cancelled',
          order: -1,
          communicationCategory: CommunicationCategoryEnum.HDB_REJECT,
        },
      ];

      // Add program-specific KPIs if looking at a specific program
      if (programId) {
        // ✅ FIXED: Call without search text
        const programKPIs = await this.generateProgramKPIs(
          programId,
          parsedFilters,
          filterRegistrationsByUserId,
          activeKpiCategory === 'programs' ? searchText : '',
          4,
        );
        if (programKPIs.length > 0) {
          kpis.programs = {
            categoryLabel: 'Program Allocation',
            items: programKPIs,
          };
        }
      }

      return kpis;
    } catch (error) {
      this.logger.error('Error generating default KPIs', error);
      throw error;
    }
  }

  /**
   * Generate program-specific KPIs (HDB1, HDB2, MSD1, etc.)
   */
  private async generateProgramKPIs(
    programId: number,
    parsedFilters?: Record<string, any>,
    filterRegistrationsByUserId?: number | null,
    searchText: string = '',
    orderStartNumber: number = 0,
  ): Promise<KPIItem[]> {
    try {
      // Check if this is a grouped program
      const mainProgram = await this.repository.findProgramWithType(programId);
      if (!mainProgram?.isGroupedProgram) {
        return [];
      }

      // Get all programs in this group
      const groupedPrograms = await this.repository.findGroupedPrograms(programId);
      const programKPIs: KPIItem[] = [];
      for (const [index, program] of groupedPrograms.entries()) {
        // ✅ FIXED: Get registration count WITHOUT search text
        const count = await this.repository.getTotalAllocatedRegistrations(
          mainProgram.id,
          program.id,
          parsedFilters,
          searchText,
          filterRegistrationsByUserId,
        );


        programKPIs.push({
          label: program.name,
          value: count,
          kpiCategory: 'programs',
          kpiFilter: `program_${program.id}`,
          order: orderStartNumber + index || 0,
          communicationCategory: CommunicationCategoryEnum.HDB_BLESSED,
        });
      }

      return programKPIs;
    } catch (error) {
      this.logger.error('Error generating program KPIs', error);
      return [];
    }
  }

  /**
   * Role-based visibility methods
   */
  private shouldShowPaymentKPIs(userRoles: string[]): boolean {
    const paymentRoles = ['admin', 'finance_manager'];
    return userRoles.some((role) => paymentRoles.includes(role));
  }

  private shouldShowInvoiceKPIs(userRoles: string[]): boolean {
    const invoiceRoles = ['admin', 'finance_manager'];
    return userRoles.some((role) => invoiceRoles.includes(role));
  }

  private shouldShowTravelKPIs(userRoles: string[]): boolean {
    const travelRoles = ['admin', 'operational_manager', 'rm_support'];
    return userRoles.some((role) => travelRoles.includes(role));
  }

  // ===== PRIVATE BUSINESS LOGIC METHODS =====

  private async validateUserRegistrationDetails(
    user: User,
    answers: RegistrationAnswerDto[],
  ): Promise<void> {
    const isViewer =
      user.userRoleMaps?.length === 1 &&
      user.userRoleMaps.some((urm) => urm.role.name === 'viewer');
    if (!isViewer) {
      this.logger.log(
        `Skipping validation for user ${user.id} as they are not a viewer`,
      );
      return;
    }
    const mobileNumberTypeAnswer = answers.find((a) => a.bindingKey === 'phoneNumberType');
    if (mobileNumberTypeAnswer?.answer && mobileNumberTypeAnswer?.answer === phoneNumberType.differentNumber) {
      return;
    }

    const emailAnswer = answers.find((a) => a.bindingKey === 'email');
    if (
      emailAnswer?.answer && user?.email && user?.email?.toLowerCase() !== String(emailAnswer?.answer)?.toLowerCase()
    ) {
      throw new InifniBadRequestException(
        ERROR_CODES.USER_EMAIL_MISMATCH,
        null,
        null,
        user.email,
      );
    }

    const mobileAnswer = answers.find((a) => a.bindingKey === 'mobileNumber');
    if ( mobileAnswer?.answer && user?.countryCode && user?.phoneNumber && user?.countryCode?.concat(user?.phoneNumber) !== mobileAnswer.answer) {
      throw new InifniBadRequestException(
        ERROR_CODES.USER_PHONE_MISMATCH,
        null,
        null,
        `${user.countryCode}${user.phoneNumber}`,
      );
    }

    if (!user?.phoneNumber && !user?.countryCode && user?.email) {
      if (mobileAnswer?.answer) {
          const existingUser = await this.userRepository.fetchUserByCombinedPhoneNumber(
            mobileAnswer?.answer,
          );
          if (existingUser && existingUser.id !== user.id) {
            throw new InifniBadRequestException(
              ERROR_CODES.USER_PHONE_ALREADY_EXISTS,
              null,
              null,
              `${mobileAnswer.answer}`,
            );
          }
        } 
    }

    if (!user?.email && user?.countryCode && user?.phoneNumber) {
      if (emailAnswer?.answer) {
        const email = String(emailAnswer?.answer).toLowerCase();
          const existing = await this.userRepository.fetchUsersByEmail(email);
          if (existing && existing.some((u) => u.id !== user.id)) {
            throw new InifniBadRequestException(
              ERROR_CODES.USER_EMAIL_ALREADY_EXISTS,
              null,
              null,
              email,
            );
        }
      }
    }
  }

  private async validateUserNotAlreadyRegistered(
    userId: number,
    programId: number,
    programSessionId?: number,
  ): Promise<void> {
    const existingRegistration = await this.repository.findExistingUserRegistration(
      userId,
      programId,
      programSessionId,
    );
    if (existingRegistration) {
      throw new InifniBadRequestException(
        ERROR_CODES.USER_ALREADY_REGISTERED,
        null,
        null,
        `User ${userId} is already registered for program ${programId}${programSessionId ? ` session ${programSessionId}` : ''}`,
      );
    }
  }

  private async validateAndGetProgram(programId: number): Promise<Program> {
    const program = await this.repository.findProgramWithType(programId);
    if (!program) {
      throw new InifniNotFoundException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        programId.toString(),
      );
    }
    return program;
  }

  private getRegistrationLevel(program: Program): string {
    const registrationLevel = program.type?.registrationLevel;
    if (!registrationLevel) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
        null,
        null,
        'Program type must have a registration level defined',
      );
    }
    return registrationLevel;
  }

  private async validateSessionIfRequired(
    dto: CreateRegistrationDto,
    program: Program,
    registrationLevel: string,
  ): Promise<ProgramSession | null> {
    if (registrationLevel !== 'session') {
      return null;
    }

    if (!dto.programSessionId) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
        null,
        null,
        'Program session ID is required for session-level registration',
      );
    }

    const session = await this.repository.findSessionForProgram(dto.programSessionId, program.id);
    if (!session) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_MISMATCH,
        null,
        null,
        dto.programSessionId.toString(),
      );
    }

    return session;
  }


  async cancel(dto: CancelRegistrationDto, user: User) {
    const registration = await this.validateRegistrationForUpdate(
      dto.programRegistrationId,
      user,
      {},
    );

    const allowedRoles = [
      ROLE_KEYS.ADMIN,
      ROLE_KEYS.SHOBA,
      ROLE_KEYS.OPERATIONAL_MANAGER,
    ];
    const hasPermission = user.userRoleMaps.some((urm) =>
      allowedRoles.includes(urm.role.roleKey),
    );
    if (!hasPermission) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_CANCEL_UNAUTHORIZED,
      );
    }

    // Use centralized allocation clearing service
    await this.allocationClearingService.clearRegistrationAllocations(
      registration.id,
      user.id,
      CLEARANCE_REASONS.REGISTRATION_CANCELLED(registration.id)
    );

    const programStart =
      registration.allocatedProgram?.startsAt;
    const cancellationDate = new Date();
    if (programStart && cancellationDate >= new Date(programStart)) {
      this.logger.warn(
        `Cancellation date ${cancellationDate.toISOString()} is after program start date ${programStart.toISOString()}.`,
      );
    }

    const result = await this.repository.cancelRegistration(
      registration.id,
      cancellationDate,
      user.id,
      dto.cancellationReason,
      dto.cancellationComments,
    );
    // uncomment the following block to enable email notifications on cancellation
    // try {
    //   await this.sendRefundCommunication(registration.id);
    // } catch (error) {
    //   this.logger.error('Error sending payment settlement communication', error);
    // }

    // uncomment the following block to enable email notifications on seat cancellation
    // try {
    //   await this.sendRegistrationCancelCommunication(registration.id);
    // } catch (error) {
    //   this.logger.error('Error sending registration cancellation communication', error);
    // }
    

    // await this.sendCancellationNotifications(
    //   registration,
    //   {
    //     cancellationReason: dto.cancellationReason,
    //     cancellationComments: dto.cancellationComments,
    //     cancellationDate,
    //   },
    //   user,
    // );
    return result;
  }

  async findAll(
    limit: number,
    offset: number,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles: string[],
    filterRegistrationsByUserId?: number | null,
  ) {
    this.logger.log(registrationMessages.GET_ALL, {
      limit,
      offset,
      programId,
      programSessionId,
      searchText,
      parsedFilters,
      userRoles,
    });
    try {
      // Normalize filter values (gender, etc.)
      normalizeFilterValues(parsedFilters);

      if (userRoles.includes('mahatria')) {
        if (programId) {
          const program = await this.repository.findProgramWithType(programId);
          if (!program) {
            throw new InifniNotFoundException(
              ERROR_CODES.PROGRAM_NOTFOUND,
              null,
              null,
              programId.toString(),
            );
          }

          if (!program.requiresApproval) {
            return {
              message: 'No approval required for this program',
              data: [],
              pagination: {
                totalPages: 0,
                pageNumber: 1,
                pageSize: limit,
                totalRecords: 0,
                numberOfRecords: 0,
              },
            };
          }
        }
      }

      // Check if user is RM
      const isRMUser = userRoles.includes('relational_manager');
      return await this.repository.findRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText,
        parsedFilters,
        userRoles,
        isRMUser ? filterRegistrationsByUserId : undefined,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  private async validateRegistrationDates(
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
  ): Promise<void> {
    const currentTimeUTC = new Date().getTime(); // current UTC timestamp

    let registrationStartDateTime: Date | null = null;
    let registrationEndDateTime: Date | null = null;

    if (registrationLevel === 'session' && session) {
      registrationStartDateTime = session.registrationStartsAt;
      registrationEndDateTime = session.registrationEndsAt;
    } else {
      registrationStartDateTime = program.registrationStartsAt;
      registrationEndDateTime = program.registrationEndsAt;
    }

    if (registrationStartDateTime && currentTimeUTC < registrationStartDateTime.getTime()) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_NOT_STARTED,
        null,
        null,
        `Registration will start on ${registrationStartDateTime.toLocaleString('en-IN', { timeZone: 'Asia/Kolkata' })}`,
      );
    }

    if (registrationEndDateTime && currentTimeUTC > registrationEndDateTime.getTime()) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_CLOSED,
        null,
        null,
        `Registration closed on ${registrationEndDateTime.toLocaleString('en-IN', { timeZone: 'Asia/Kolkata' })}`,
      );
    }
  }

  async checkSeatAvailability(
    program: Program,
    session: ProgramSession | null,
    registrationLevel: string,
  ) {
    const seatsInfo = registrationLevel === 'session' ? session : program;
    const isLimitedSeats = seatsInfo?.limitedSeats === true;
    const isWaitlistEnabled = seatsInfo?.waitlistApplicable === true;

    // if program or session has requiresApproval, we don't need to check seats
    if (program.requiresApproval || (session && session.requiresApproval)) {
      return {
        isWaitlistTriggered: false,
        canRegister: true,
        shouldWaitlist: false,
      };
    }

    if (!isLimitedSeats) {
      return {
        isWaitlistTriggered: false,
        canRegister: true,
        shouldWaitlist: false,
      };
    } else if (!isWaitlistEnabled) {
      return {
        isWaitlistTriggered: false,
        canRegister:
          registrationLevel === 'session'
            ? (session?.reservedSeats ?? 0) < (session?.totalSeats ?? 0)
            : (program?.filledSeats || 0) < (seatsInfo?.totalSeats || 0),
        shouldWaitlist: false,
      };
    }
    const totalSeats = seatsInfo?.totalSeats || 0;
    const waitlistTriggerCount = seatsInfo?.waitlistTriggerCount || 0;
    // Only Program entity has filledSeats, ProgramSession doesn't
    const filledSeats = registrationLevel === 'session' ? 0 : program?.filledSeats || 0;

    const isWaitlistTriggered = filledSeats >= waitlistTriggerCount;
    const canDirectRegister = filledSeats < waitlistTriggerCount && !isWaitlistTriggered;
    const canRegister = filledSeats < totalSeats;
    const shouldWaitlist = !canDirectRegister && canRegister;

    return {
      totalSeats,
      waitlistTriggerCount,
      filledSeats,
      isWaitlistTriggered,
      canRegister,
      shouldWaitlist,
    };
  }

  private async validateQuestionAnswers(
    dto: CreateRegistrationDto,
    programId: number,
    sessionId?: number,
  ) {
    const questionMappings = await this.repository.findProgramQuestions(programId, sessionId);
    const validQuestionIds = questionMappings.map((m) => m.question.id);
    const invalid = dto.answers.filter((a) => !validQuestionIds.includes(a.questionId));

    if (invalid.length) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_INVALID_QUESTION,
        null,
        null,
        `Invalid question IDs: ${invalid.map((i) => i.questionId).join(', ')}`,
      );
    }

    return questionMappings;
  }

  private processQuestionAnswers(answers: any[], questionMappings: any[], userId: number) {
    const questionMap = new Map<number, Question>();
    questionMappings.forEach((m) => questionMap.set(m.question.id, m.question));

    const customResponses: RegistrationCustomResponse[] = [];
    const groupedDirectLocationAnswers: Map<
      string,
      { columnName: string; value: any; question: Question }[]
    > = new Map();
    const qaMappings: { questionId: number; answerValue: any }[] = [];

    let isTravelUpdateNow: boolean | undefined = undefined;
    let travelPlanReturn: any = null;
    let travelPlanOnward: any = null;

    for (const ans of answers) {
      const question = questionMap.get(ans.questionId);
      const location = question?.answerLocation;

      // Check for travelDetails bindingKey
      if (question?.bindingKey === 'travelDetails') {
        if (typeof ans.answer === 'string' && ans.answer.trim().toLowerCase() === 'i want to update now') {
          isTravelUpdateNow = true;
        } else {
          isTravelUpdateNow = false;
        }
      }

      // Capture travelPlanReturn and travelPlanOnward answers
      if (question?.bindingKey === REGISTRATION_BINDING_KEYS.TRAVEL_PLAN_RETURN) {
        travelPlanReturn = ans.answer;
      }
      if (question?.bindingKey === REGISTRATION_BINDING_KEYS.TRAVEL_PLAN_ONWARD) {
        travelPlanOnward = ans.answer;
      }

      if (location) {
        const [tableName, columnName] = location.split('.');
        if (!groupedDirectLocationAnswers.has(tableName)) {
          groupedDirectLocationAnswers.set(tableName, []);
        }
        groupedDirectLocationAnswers
          .get(tableName)!
          .push({ columnName, value: ans.answer, question });
      } else {
        customResponses.push({
          questionId: ans.questionId,
          responseValue: ans.answer,
          createdBy: { id: userId } as any,
          updatedBy: { id: userId } as any,
        } as RegistrationCustomResponse);
      }
      qaMappings.push({ questionId: ans.questionId, answerValue: ans.answer });
    }

    // If isTravelUpdateNow is true, check travelPlanReturn and travelPlanOnward
    let travelUpdateStatus: TravelStatusEnum.COMPLETED | TravelStatusEnum.PENDING | undefined = undefined;
      if (travelPlanReturn && travelPlanOnward) {
        travelUpdateStatus = TravelStatusEnum.COMPLETED;
      } else {
        travelUpdateStatus = TravelStatusEnum.PENDING;
      }
      
    return {
      customResponses,
      groupedDirectLocationAnswers,
      qaMappings,
      ...(isTravelUpdateNow !== undefined ? { isTravelUpdateNow } : {}),
      ...(travelUpdateStatus ? { travelUpdateStatus } : {}),
    };
  }

  private determineRegistrationOutcome(seatInfo: any, program: Program) {
    const requiresPayment = program.requiresPayment === true;
    const approvalRequired = program.requiresApproval === true;

    // Determine registration path
    if (seatInfo.isWaitlistTriggered && seatInfo.shouldWaitlist) {
      return { type: 'waitlist', seatInfo };
    }

    if (!seatInfo.canRegister) {
      return { type: 'no_space', seatInfo };
    }

    if (approvalRequired) {
      return { type: 'pending_approval', requiresPayment };
    }

    return {
      type: 'direct_registration',
      requiresPayment,
      finalStatus: requiresPayment
        ? RegistrationStatusEnum.PENDING
        : RegistrationStatusEnum.COMPLETED,
    };
  }

  private async validateRegistrationForUpdate(registrationId: number, user: User,updateDto, isAdmin: boolean = false): Promise<ProgramRegistration> {
    this.logger.log(`Validating registration for update: ${registrationId}`);
    const registration = await this.repository.findRegistrationWithRelations(registrationId);
    if (!registration) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
        null,
        null,
        registrationId.toString(),
      );
    }

    const userRoles = user.userRoleMaps.map((urm) => urm.role.roleKey);
    const approval = await this.repository.findApprovalStatus(registration.id);
    const isPrivilegedRole = userRoles.some(role =>
      ['ROLE_ADMIN', 'ROLE_SHOBA', 'ROLE_RELATIONAL_MANAGER'].includes(role)
    );
    this.logger.log('isAdmin:', {isAdmin, isPrivilegedRole, userRoles, approvalStatus: approval?.approvalStatus, registrationStatus: registration.registrationStatus});
    const isRejectedOrCancelledOrOnHold =
      (approval?.approvalStatus !== undefined && [ApprovalStatusEnum.REJECTED].includes(approval.approvalStatus)) ||
      [RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.ON_HOLD].includes(registration.registrationStatus);
    const isCancelled = registration.registrationStatus === RegistrationStatusEnum.CANCELLED;


    // if (!isAdmin && !isPrivilegedRole && isRejectedOrCancelledOrOnHold) {
    //   throw new InifniBadRequestException(
    //   ERROR_CODES.REGISTRATION_CANT_BE_UPDATED,
    //   null,
    //   null,
    //   registrationId.toString()
    //   );
    // }
    if (isCancelled) {
      throw new InifniBadRequestException(
      ERROR_CODES.REGISTRATION_CANT_BE_UPDATED,
      null,
      null,
      registrationId.toString()
      );
    }

    // Check if user is a viewer
    const isViewer = user.userRoleMaps.length === 1 && user.userRoleMaps.some((urm) => urm.role.name === 'viewer')
    if (isViewer) {
      const formSection = updateDto?.formSection;
      await this.validateViewerRegistrationUpdate(registration,formSection);
    }

    return registration;
  }

  /**
   * Validates if a viewer can update the registration
   * @param registration - The registration to validate
   * @throws InifniBadRequestException if validation fails
   */
  private async validateViewerRegistrationUpdate(
    registration: ProgramRegistration,
    formSection: FormSection | null,
  ): Promise<void> {
    // Check for cancelled status
    if (registration.registrationStatus === RegistrationStatusEnum.CANCELLED) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_INVALID_STATUS,
        null,
        null,
        'Cannot update cancelled registration'
      );
    }

    // Check for completed status
    if (registration.registrationStatus === RegistrationStatusEnum.COMPLETED) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_INVALID_STATUS,
        null,
        null,
        'Cannot update completed registration'
      );
    }

    // Check for approved status
    const approval = await this.repository.findApprovalStatus(registration.id);
    const formSectionKey = formSection?.key || '';
    if (
      (formSectionKey === 'FS_BASICDETAILS' || formSectionKey === 'FS_MAHATRIAQUESTIONS') &&
      approval?.approvalStatus === ApprovalStatusEnum.APPROVED
    ) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_ALREADY_APPROVED,
        null,
        null
      );
    }
  }

  private async sendCancellationNotifications(
    registration: ProgramRegistration,
    dto: {
      cancellationReason: string;
      cancellationComments?: string;
      cancellationDate: Date;
    },
    cancelledBy: User,
  ): Promise<void> {
    try {
      const programStart =
        registration.allocatedProgram?.startsAt ||
        registration.program?.startsAt ||
        registration.programSession?.startsAt;
      const programName = registration.program?.name || '';
      const mergeInfo = {
        program_name: programName,
        program_start_date: programStart
          ? new Date(programStart).toLocaleDateString('en-GB')
          : '',
        cancellation_reason: dto.cancellationReason,
        cancellation_comments: dto.cancellationComments || '',
        cancelled_by: cancelledBy.fullName || '',
        cancellation_date: new Date(dto.cancellationDate).toLocaleDateString('en-GB'),
      };

      const from = {
        address: zeptoEmailCreadentials.ZEPTO_EMAIL,
        name:
          registration.program?.emailSenderName ||
          zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
      };

      // Notify seeker
      const seekerEmail: SendSingleEmailDto = {
        templateKey:
          process.env.ZEPTO_REGISTRATION_CANCELLED_EMAIL_TEMPLATE_ID || '',
        from,
        to: {
          emailAddress: registration.emailAddress,
          name: registration.fullName,
        },
        mergeInfo,
        attachments: [],
        subject: '',
        trackinfo: { registrationId: registration.id },
      };
      await this.communicationService.sendSingleEmail(seekerEmail);

      const seekerMsg: SendTemplateMessageDto = {
        whatsappNumber: registration.mobileNumber.slice(0),
        templateName:
          process.env.WATI_SEEKER_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
        broadcastName:
          process.env.WATI_SEEKER_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
        parameters: [
          { name: 'reg_name', value: registration.fullName },
          { name: 'program_name', value: programName },
        ],
        trackinfo: { registrationId: registration.id },
      };
      await this.communicationService.sendTemplateMessage(seekerMsg);

      // Notify RM
      if (registration.rmContactUser) {
        const rm = registration.rmContactUser;
        const rmEmail: SendSingleEmailDto = {
          templateKey:
            process.env.ZEPTO_RM_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
          from,
          to: { emailAddress: rm.email, name: rm.fullName },
          mergeInfo,
          attachments: [],
          subject: '',
          trackinfo: { registrationId: registration.id },
        };
        await this.communicationService.sendSingleEmail(rmEmail);

        const rmMsg: SendTemplateMessageDto = {
          whatsappNumber: `${rm.countryCode}${rm.phoneNumber}`.slice(0),
          templateName:
            process.env.WATI_RM_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
          broadcastName:
            process.env.WATI_RM_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
          parameters: [
            { name: 'reg_name', value: registration.fullName },
            { name: 'program_name', value: programName },
          ],
          trackinfo: { registrationId: registration.id },
        };
        await this.communicationService.sendTemplateMessage(rmMsg);
      }

      // Notify Finance Managers
      try {
        const financeUsers = await this.userRepository.getUsersByRoleKey(
          ROLE_KEYS.FINANCE_MANAGER,
        );
        for (const fm of financeUsers) {
          const fmEmail: SendSingleEmailDto = {
            templateKey:
              process.env.ZEPTO_FINANCE_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
            from,
            to: { emailAddress: fm.email, name: fm.fullName },
            mergeInfo,
            attachments: [],
            subject: '',
            trackinfo: { registrationId: registration.id },
          };
          await this.communicationService.sendSingleEmail(fmEmail);

          const fmMsg: SendTemplateMessageDto = {
            whatsappNumber: `${fm.countryCode}${fm.phoneNumber}`.slice(0),
            templateName:
              process.env.WATI_FINANCE_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
            broadcastName:
              process.env.WATI_FINANCE_REGISTRATION_CANCELLED_TEMPLATE_ID || '',
            parameters: [
              { name: 'reg_name', value: registration.fullName },
              { name: 'program_name', value: programName },
            ],
            trackinfo: { registrationId: registration.id },
          };
          await this.communicationService.sendTemplateMessage(fmMsg);
        }
      } catch (err) {
        this.logger.error(
          'Failed to send finance manager cancellation notifications',
          err,
        );
      }
    } catch (error) {
      this.logger.error('Failed to send cancellation notifications', error);
    }
  }

  private async validateApprovalStatus(registration: any): Promise<void> {
    if (!registration.program?.requiresApproval) {
      return;
    }

    const pendingApproval = await this.repository.findPendingApproval(registration.id);
    if (pendingApproval) {
      throw new InifniBadRequestException(
        ERROR_CODES.REGISTRATION_PENDING_APPROVAL,
        null,
        null,
        registration.id.toString(),
      );
    }
  }

  private async validateFormSection(formSectionId: number) {
    const formSection = await this.repository.findFormSection(formSectionId);
    if (!formSection) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_NOTFOUND,
        null,
        null,
        `Form section with ID ${formSectionId} not found`,
      );
    }
    return formSection;
  }

  private async validateProgramForUpdate(programId: number, registration: any): Promise<Program> {
    if (!registration.program || registration.program.id !== programId) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_NOTFOUND,
        null,
        null,
        programId.toString(),
      );
    }
    return registration.program;
  }

  private async validateSessionForUpdate(
    dto: UpdateRegistrationDto,
    registration: any,
    program: Program,
  ): Promise<ProgramSession | null> {
    const registrationLevel = program.type?.registrationLevel;

    if (registrationLevel !== 'session') {
      return null;
    }

    const sessionId = dto.programSessionId ?? registration.programSession?.id;
    if (!sessionId) {
      throw new InifniBadRequestException(ERROR_CODES.PROGRAM_SESSION_NOTFOUND);
    }

    const session = await this.repository.findSessionForProgram(sessionId, program.id);
    if (!session) {
      throw new InifniBadRequestException(
        ERROR_CODES.PROGRAM_SESSION_MISMATCH,
        null,
        null,
        sessionId.toString(),
      );
    }

    return session;
  }

  private async validateSectionQuestions(
    dto: UpdateRegistrationDto,
    programId: number,
    sessionId: number | undefined,
    formSection: FormSection | null,
  ) {
    const questionMappings = await this.repository.findProgramQuestions(programId, sessionId);
    let sectionQuestions: ProgramQuestion[] = questionMappings;
    if (
      formSection &&
      formSection.key !== 'FS_BASICDETAILS' &&
      formSection.key !== 'FS_MAHATRIAQUESTIONS'
    ) {
      sectionQuestions = questionMappings.filter(
        (m) => m.question.formSection?.id === formSection.id,
      );

      if (sectionQuestions.length === 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_INVALID_QUESTION,
          null,
          null,
          `No questions found for form section ID: ${formSection.id}`,
        );
      }

      const validQuestionIds = sectionQuestions.map((m) => m.question.id);
      const invalid = dto.answers.filter((a) => !validQuestionIds.includes(a.questionId));
      if (invalid.length) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_INVALID_QUESTION,
          null,
          null,
          invalid.map((i) => i.questionId).join(','),
        );
      }
    }

    return sectionQuestions;
  }

  /**
   * Format Mahatria KPIs to match the structure of default KPIs
   */
  private formatMahatriaKPIs(kpis: MahatriaKPIResult): any {
    if (!kpis.isGroupedProgram) {
      return {
        isGroupedProgram: false,
        categoryLabel: 'Program Registrations',
        totalRegistrations: kpis.totalRegistrations || 0,
      };
    }

    const formatted = {
      isGroupedProgram: true,
      primaryProgramId: kpis.primaryProgramId,
      primaryProgramName: kpis.primaryProgramName,
      allocated: {
        categoryLabel: 'Allocated Programs',
        programs:
          kpis.allocated?.programs?.map((program) => ({
            programId: program.programId,
            programName: program.programName,
            allocatedCount: program.allocatedCount,
            maleCount: program.maleCount,
            femaleCount: program.femaleCount,
            organisationUserCount: program.organisationUserCount,
            totalSeatsCount: program.totalSeatsCount,
            maleOrganisationUserCount: program.maleOrganisationUserCount,
            femaleOrganisationUserCount: program.femaleOrganisationUserCount,
            roomMatePreferenceCount: program.roomMatePreferenceCount,
            label: `${program.programName} (${program.allocatedCount})`,
            kpiCategory: 'allocated',
            kpiFilter: `program_${program.programId}`,
            swapRequestsCount: program.swapRequestsCount,
            startsAt: program.startsAt,
            totalBedCount: program.totalBedCount,
          })) || [],
        totals: {
          holdCount: {
            label: 'Swap demand',
            value: Number(kpis.allocated?.holdCount || 0),
            kpiCategory: 'allocated',
            kpiFilter: 'hold',
            femaleCount: Number(kpis.allocated?.holdFemaleCount || 0),
            maleCount: Number(kpis.allocated?.holdMaleCount || 0),
            maleOrganisationUserCount: Number(kpis.allocated?.holdMaleOrganisationUserCount || 0),
            femaleOrganisationUserCount: Number(kpis.allocated?.holdFemaleOrganisationUserCount || 0),
            organisationUserCount: Number(kpis.allocated?.holdOrganisationUserCount || 0),
            preferredPrograms: kpis.allocated?.holdProgramPreferenceCount || [],
          },
          rejectCount: {
            label: 'On Hold',
            value: Number(kpis.allocated?.rejectCount || 0),
            kpiCategory: 'allocated',
            kpiFilter: 'reject',
            femaleCount: Number(kpis.allocated?.rejectFemaleCount || 0),
            maleCount: Number(kpis.allocated?.rejectMaleCount || 0),
            maleOrganisationUserCount: Number(kpis.allocated?.rejectMaleOrganisationUserCount || 0),
            femaleOrganisationUserCount: Number(kpis.allocated?.rejectFemaleOrganisationUserCount || 0),
            organisationUserCount: Number(kpis.allocated?.rejectOrganisationUserCount || 0),
            preferredPrograms: kpis.allocated?.rejectProgramPreferenceCount || [],
          },
          blessedCount: {
            label: 'Blessed',
            allocatedCount: Number(kpis.allocated?.blessedCount || 0),
            kpiCategory: 'allocated',
            kpiFilter: 'blessed',
            femaleCount: Number(kpis.allocated?.blessedFemaleCount || 0),
            maleCount: Number(kpis.allocated?.blessedMaleCount || 0),
            organisationUserCount: Number(kpis.allocated?.blessedOrganisationUserCount || 0),
            organisationMaleCount: Number(kpis.allocated?.blessedMaleOrganisationUserCount || 0),
            organisationFemaleCount: Number(kpis.allocated?.blessedFemaleOrganisationUserCount || 0),
            swapRequestsCount: Number(kpis.allocated?.swapRequestsCount || 0),
            roomMatePreferenceCount: Number(kpis.allocated?.roomMatePreferenceCount || 0),
          }
        },
      },
      unallocated: {
        categoryLabel: 'Unallocated Programs',
        programs:
          kpis.unallocated?.programs?.map((program) => ({
            programId: program.programId,
            programName: program.programName,
            count: program.count,
            maleCount: program.maleCount,
            femaleCount: program.femaleCount,
            organisationUserCount: program.organisationUserCount,
            maleOrganisationUserCount: program.maleOrganisationUserCount,
            femaleOrganisationUserCount: program.femaleOrganisationUserCount,
            totalSeatsCount: program.totalSeatsCount,
            label: `${program.programName} (${program.count})`,
            kpiCategory: 'unallocated',
            kpiFilter: `program_${program.programId}`,
          })) || [],
        totals: {
          totalMaleCount: {
            label: 'Total Male',
            value: kpis.unallocated?.totalMaleCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'male',
            organisationUserCount: kpis.unallocated?.totalMaleOrganisationUserCount || 0,
          },
          totalFemaleCount: {
            label: 'Total Female',
            value: kpis.unallocated?.totalFemaleCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'female',
            organisationUserCount: kpis.unallocated?.totalFemaleOrganisationUserCount || 0,
          },
          totalUnallocatedCount: {
            label: 'Total Unallocated',
            value: kpis.unallocated?.totalUnallocatedCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'total_unallocated',
            organisationUserCount: kpis.unallocated?.totalOrganisationUserCount || 0,
          },
          totalMahatriaChoiceCount: {
            label: 'Mahatria Choice',
            value: kpis.unallocated?.totalMahatriaChoiceCount || 0,
            kpiCategory: 'unallocated',
            kpiFilter: 'mahatria_choice',
            organisationUserCount: kpis.unallocated?.totalMahatriaChoiceOrganisationUserCount || 0,
          },
        },
      },
    };

    return formatted;
  }

  /**
   * Generate Mahatria-specific KPIs for program allocation management
   */
  private async generateMahatriaKPIs(parameters: {
    programId: number;
    searchText?: string; // ✅ This should be empty string when called
    parsedFilters?: Record<string, any>;
  }): Promise<MahatriaKPIResult> {
    try {
      const { programId, parsedFilters, searchText } = parameters;

      // Get the main program to check if it's a grouped program
      const mainProgram = await this.repository.findProgramWithType(programId);

      if (!mainProgram) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }

      // Check if this is a grouped program
      const isGroupedProgram = mainProgram.isGroupedProgram || false;

      if (isGroupedProgram) {
        // Step 1: Get ALL programs first
        const allPrograms = await this.repository.findGroupedPrograms(programId);

        // Initialize metrics objects
        const allocated: AllocatedProgramMetrics = {
          programs: [],
          holdCount: 0,
          rejectCount: 0,
          holdMaleCount: 0,
          holdMaleOrganisationUserCount: 0,
          holdFemaleOrganisationUserCount: 0,
          holdFemaleCount: 0,
          rejectFemaleCount: 0,
          rejectMaleOrganisationUserCount: 0,
          rejectFemaleOrganisationUserCount: 0,
          rejectMaleCount: 0,
          holdOrganisationUserCount: 0,
          rejectOrganisationUserCount: 0,
          holdProgramPreferenceCount: [],
          rejectProgramPreferenceCount: [],
          blessedCount: 0,
          blessedFemaleCount: 0,
          blessedMaleCount: 0,
          blessedOrganisationUserCount: 0,
          blessedFemaleOrganisationUserCount: 0,
          blessedMaleOrganisationUserCount: 0,
          swapRequestsCount: 0,
          roomMatePreferenceCount: 0,
        };

        const unallocated: UnallocatedProgramMetrics = {
          programs: [],
          totalMaleCount: 0,
          totalFemaleCount: 0,
          totalUnallocatedCount: 0,
          totalMahatriaChoiceCount: 0,
          totalOrganisationUserCount: 0,
          totalMaleOrganisationUserCount: 0,
          totalFemaleOrganisationUserCount: 0,
          totalMahatriaChoiceOrganisationUserCount: 0,
        };

        // Add Mahatria Choice metrics
        const mahatriaChoice = {
          count: 0,
          maleCount: 0,
          femaleCount: 0,
          organisationUserCount: 0,
        };

        let totalUnallocatedMaleCount = 0;
        let totalUnallocatedFemaleCount = 0;
        let totalUnalocatedCount = 0;
        let totalUnallocatedOrganisationCount = 0;
        let totalUnallocatedMaleOrganisationCount = 0;
        let totalUnallocatedFemaleOrganisationCount = 0;
        let totalMahatriaChoiceOrganisationUserCount = 0;

        const unallocatedPrograms: {
          programId: number;
          programName: string;
          count: number;
          maleCount: number;
          femaleCount: number;
          organisationUserCount: number;
          maleOrganisationUserCount: number;
          femaleOrganisationUserCount: number;
          totalSeatsCount: number;
        }[] = [];
        const allocatedPrograms: {
          programId: number;
          programName: string;
          allocatedCount: number;
          maleCount: number;
          femaleCount: number;
          organisationUserCount: number;
          totalSeatsCount: number;
          maleOrganisationUserCount: number;
          femaleOrganisationUserCount: number;
          roomMatePreferenceCount: number;
          swapRequestsCount: number;
          startsAt: Date;
          totalBedCount: number;
        }[] = [];

        // ✅ FIXED: Build base query for approvals WITHOUT search text
        const approvalResult = await this.repository.getApprovalMetrics(
          programId,
          parsedFilters,
          '',
        );

        const holdProgramPreferenceCount = await this.repository.getProgramPreferencesMetrics(
          programId,
          ApprovalStatusEnum.ON_HOLD,
        );

        const rejectedProgramPreferenceCount = await this.repository.getProgramPreferencesMetrics(
          programId,
          ApprovalStatusEnum.REJECTED,
        );

        const blessedCounts = await this.repository.getBlessedCount(programId);
        allocated.blessedCount = Number(blessedCounts?.blessedCount || 0);
        allocated.blessedFemaleCount = Number(blessedCounts?.femaleCount || 0);
        allocated.blessedMaleCount = Number(blessedCounts?.maleCount || 0);
        allocated.blessedOrganisationUserCount = Number(blessedCounts?.orgCount || 0);
        allocated.blessedFemaleOrganisationUserCount = Number(blessedCounts?.femaleOrganisationUserCount || 0);
        allocated.blessedMaleOrganisationUserCount = Number(blessedCounts?.maleOrganisationUserCount || 0);
        allocated.swapRequestsCount = Number(blessedCounts?.swapRequestsCount || 0);
        allocated.roomMatePreferenceCount = Number(blessedCounts?.roomMatePreferenceCount || 0);

        allocated.holdMaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.MALE)[0]?.holdCount || 0);
        allocated.holdFemaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.FEMALE)[0]?.holdCount || 0);
        allocated.holdOrganisationUserCount =
          Number(approvalResult.find((record) => record.gender === GenderEnum.MALE)?.holdOrgCount || 0) +
          Number(approvalResult.find((record) => record.gender === GenderEnum.FEMALE)?.holdOrgCount || 0);
        allocated.holdCount = Number(allocated.holdMaleCount || 0) + Number(allocated.holdFemaleCount || 0) || 0;
        allocated.rejectMaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.MALE)[0]?.rejectedCount || 0);
        allocated.rejectFemaleCount = Number(approvalResult.filter((record) => record.gender === GenderEnum.FEMALE)[0]?.rejectedCount || 0);
        allocated.rejectOrganisationUserCount =
          Number(approvalResult.find((record) => record.gender === GenderEnum.MALE)?.rejectedOrgCount || 0) +
          Number(approvalResult.find((record) => record.gender === GenderEnum.FEMALE)?.rejectedOrgCount || 0);
        allocated.rejectCount = Number(allocated.rejectMaleCount || 0) + Number(allocated.rejectFemaleCount || 0) || 0;
        allocated.holdMaleOrganisationUserCount =
          Number(approvalResult.find((record) =>
            record.gender === GenderEnum.MALE && record.approvalStatus === ApprovalStatusEnum.ON_HOLD,
          )?.holdOrgCount || 0);
        allocated.holdFemaleOrganisationUserCount =
          Number(approvalResult.find((record) =>
            record.gender === GenderEnum.FEMALE && record.approvalStatus === ApprovalStatusEnum.ON_HOLD,
          )?.holdOrgCount || 0);
        allocated.rejectMaleOrganisationUserCount =
          Number(approvalResult.find((record) => 
            record.gender === GenderEnum.MALE && record.approvalStatus === ApprovalStatusEnum.REJECTED,
          )?.rejectedOrgCount || 0);
        allocated.rejectFemaleOrganisationUserCount =
          Number(approvalResult.find((record) => 
            record.gender === GenderEnum.FEMALE && record.approvalStatus === ApprovalStatusEnum.REJECTED,
          )?.rejectedOrgCount || 0);
        allocated.holdProgramPreferenceCount = holdProgramPreferenceCount;
        allocated.rejectProgramPreferenceCount = rejectedProgramPreferenceCount;
        // ✅ FIXED: Query for Mahatria Choice WITHOUT search text
        const mahatriaChoiceUnallocatedResult = await this.repository.getMahatriaChoiceMetrics(
          programId,
          parsedFilters,
          '',
        );
        mahatriaChoice.count = parseInt(mahatriaChoiceUnallocatedResult?.count ?? '0');
        mahatriaChoice.maleCount = parseInt(mahatriaChoiceUnallocatedResult?.maleCount ?? '0');
        mahatriaChoice.femaleCount = parseInt(mahatriaChoiceUnallocatedResult?.femaleCount ?? '0');
        mahatriaChoice.organisationUserCount = parseInt(mahatriaChoiceUnallocatedResult?.orgCount ?? '0');
        const mahatriaChoiceMaleOrgCount = parseInt(
          mahatriaChoiceUnallocatedResult?.maleOrganizationUserCount ?? '0',
        );
        const mahatriaChoiceFemaleOrgCount = parseInt(
          mahatriaChoiceUnallocatedResult?.femaleOrganizationUserCount ?? '0',
        );

        // Add to totals
        totalUnallocatedMaleCount += mahatriaChoice.maleCount;
        totalUnallocatedFemaleCount += mahatriaChoice.femaleCount;
        totalUnalocatedCount += mahatriaChoice.count;
        totalUnallocatedOrganisationCount += mahatriaChoice.organisationUserCount;
        totalUnallocatedMaleOrganisationCount += mahatriaChoiceMaleOrgCount;
        totalUnallocatedFemaleOrganisationCount += mahatriaChoiceFemaleOrgCount;
        totalMahatriaChoiceOrganisationUserCount = mahatriaChoice.organisationUserCount;

        for (const program of allPrograms) {
          // ✅ FIXED: Build allocated query WITHOUT search text
          const allocatedResult = await this.repository.getAllocatedMetrics(
            programId,
            program.id,
            parsedFilters,
            '',
          );
          const allocatedCount = parseInt(allocatedResult?.allocatedCount ?? '0');
          const allocatedMaleCount = parseInt(allocatedResult?.maleCount ?? '0');
          const allocatedFemaleCount = parseInt(allocatedResult?.femaleCount ?? '0');
          const allocatedOrgCount = parseInt(allocatedResult?.orgCount ?? '0');
          const allocatedMaleOrgCount = parseInt(allocatedResult?.maleOrganizationUserCount ?? '0');
          const allocatedFemaleOrgCount = parseInt(allocatedResult?.femaleOrganizationUserCount ?? '0');
          const roomMatePreferenceCount = parseInt(allocatedResult?.roomMatePreferenceCount ?? '0');
          const swapRequestsCount = parseInt(allocatedResult?.swapRequestsCount ?? '0');

          // ✅ FIXED: Build non-allocated preferences query WITHOUT search text
          const nonAllocatedResult = await this.repository.getNonAllocatedMetrics(
            program.id,
            parsedFilters,
            '',
          );
          const nonAllocatedCount = parseInt(nonAllocatedResult?.count ?? '0');
          const nonAllocatedMaleCount = parseInt(nonAllocatedResult?.maleCount ?? '0');
          const nonAllocatedFemaleCount = parseInt(nonAllocatedResult?.femaleCount ?? '0');
          const nonAllocatedOrgCount = parseInt(nonAllocatedResult?.orgCount ?? '0');
          const nonAllocatedMaleOrgCount = parseInt(
            nonAllocatedResult?.maleOrganizationUserCount ?? '0',
          );
          const nonAllocatedFemaleOrgCount = parseInt(
            nonAllocatedResult?.femaleOrganizationUserCount ?? '0',
          );

          totalUnallocatedMaleCount += nonAllocatedMaleCount;
          totalUnallocatedFemaleCount += nonAllocatedFemaleCount;
          totalUnalocatedCount += nonAllocatedCount;
          totalUnallocatedOrganisationCount += nonAllocatedOrgCount;
          totalUnallocatedMaleOrganisationCount += nonAllocatedMaleOrgCount;
          totalUnallocatedFemaleOrganisationCount += nonAllocatedFemaleOrgCount;

          unallocatedPrograms.push({
            programId: program.id,
            programName: program.name,
            count: nonAllocatedCount,
            maleCount: nonAllocatedMaleCount,
            femaleCount: nonAllocatedFemaleCount,
            organisationUserCount: nonAllocatedOrgCount,
            maleOrganisationUserCount: nonAllocatedMaleOrgCount,
            femaleOrganisationUserCount: nonAllocatedFemaleOrgCount,
            totalSeatsCount: program.totalSeats || 0,
          });

          // Right side: Allocated programs
          allocatedPrograms.push({
            programId: program.id,
            programName: program.name,
            allocatedCount,
            maleCount: allocatedMaleCount,
            femaleCount: allocatedFemaleCount,
            organisationUserCount: allocatedOrgCount,
            totalSeatsCount: program.totalSeats || 0,
            maleOrganisationUserCount: allocatedMaleOrgCount || 0,
            femaleOrganisationUserCount: allocatedFemaleOrgCount || 0,
            roomMatePreferenceCount: roomMatePreferenceCount || 0,
            swapRequestsCount: swapRequestsCount || 0,
            startsAt: program.startsAt,
            totalBedCount: program.totalBedCount || 0,
          });
        }

        allocated.programs = allocatedPrograms ?? [];
        unallocated.programs = unallocatedPrograms ?? [];
        unallocated.totalMaleCount = totalUnallocatedMaleCount;
        unallocated.totalFemaleCount = totalUnallocatedFemaleCount;
        unallocated.totalUnallocatedCount = totalUnalocatedCount;
        unallocated.totalMahatriaChoiceCount = mahatriaChoice.count;
        unallocated.totalOrganisationUserCount = totalUnallocatedOrganisationCount;
        unallocated.totalMaleOrganisationUserCount = totalUnallocatedMaleOrganisationCount;
        unallocated.totalFemaleOrganisationUserCount = totalUnallocatedFemaleOrganisationCount;
        unallocated.totalMahatriaChoiceOrganisationUserCount = totalMahatriaChoiceOrganisationUserCount;

        return {
          isGroupedProgram: true,
          primaryProgramId: programId,
          primaryProgramName: mainProgram.name,
          allocated,
          unallocated,
        };
      } else {
        // ✅ FIXED: Build total registrations query WITHOUT search text
        const totalRegistrations = await this.repository.getTotalRegistrations(
          programId,
          parsedFilters,
          '',
        );

        return {
          isGroupedProgram: false,
          totalRegistrations,
        };
      }
    } catch (error) {
      this.logger.error('Error generating Mahatria KPIs', error);
      throw error;
    }
  }

  private getMahatriaKPIFilterCondition(
    category: string,
    filter: string,
  ): Record<string, any> | null {
    switch (category) {
      case 'allocated':
        switch (filter) {
          case 'hold':
            return { approvalStatus: ApprovalStatusEnum.ON_HOLD };
          case 'reject':
            return { approvalStatus: ApprovalStatusEnum.REJECTED };
          case 'blessed':
            return { approvalStatus: ApprovalStatusEnum.APPROVED };
          default:
            // Handle program-specific filters like 'program_123'
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return { allocatedProgramId: programId, approvalStatus: ApprovalStatusEnum.APPROVED };
            }
            return null;
        }

      case 'unallocated':
        switch (filter) {
          case 'male':
            return {
              gender: GenderEnum.MALE,
              approvalStatus: ApprovalStatusEnum.PENDING,
            };
          case 'female':
            return {
              gender: GenderEnum.FEMALE,
              approvalStatus: ApprovalStatusEnum.PENDING,
            };
          case 'total_unallocated':
            return {
              approvalStatus: ApprovalStatusEnum.PENDING,
              allocatedProgram: { id: IsNull() },
            };
          case 'mahatria_choice':
            return {
              approvalStatus: ApprovalStatusEnum.PENDING,
              // Assuming registrations without program preferences are "Mahatria Choice"
              preferences: {
                id: IsNull(),
              },
              allocatedProgram: {
                id: IsNull(),
              },
            };
          default:
            // Handle program-specific filters like 'program_123'
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return {
                preferredProgramId: programId,
                approvalStatus: ApprovalStatusEnum.PENDING,
              };
            }
            return null;
        }

      case 'all':
        switch (filter) {
          case 'all':
            return {
              excludeCancelledRegistrations: false,
              approvalStatus: [
                ApprovalStatusEnum.PENDING,
                ApprovalStatusEnum.ON_HOLD,
                ApprovalStatusEnum.REJECTED,
                ApprovalStatusEnum.APPROVED,
                ApprovalStatusEnum.CANCELLED
              ],
            };
          case 'male':
            return {
              gender: GenderEnum.MALE,
              excludeCancelledRegistrations: false,
              approvalStatus: [
                ApprovalStatusEnum.PENDING,
                ApprovalStatusEnum.ON_HOLD,
                ApprovalStatusEnum.REJECTED,
                ApprovalStatusEnum.APPROVED,
                ApprovalStatusEnum.CANCELLED
              ],
            };
          case 'female':
            return {
              excludeCancelledRegistrations: false,
              gender: GenderEnum.FEMALE,
              approvalStatus: [
                ApprovalStatusEnum.PENDING,
                ApprovalStatusEnum.ON_HOLD,
                ApprovalStatusEnum.REJECTED,
                ApprovalStatusEnum.APPROVED,
                ApprovalStatusEnum.CANCELLED
              ],
            };
          case 'mahatria-choice':
            return {
              mahatriaChoice: true
            }
          case 'swap-requests':
            return {
              swapRequests:[ SwapType.WantsSwap],
              approvalStatus: [
                  ApprovalStatusEnum.PENDING,
                  ApprovalStatusEnum.APPROVED,
                ]
            }
          default:
            // Handle program-specific filters like 'program_123'
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return {
                preferredProgramId: programId,
                approvalStatus: [
                  ApprovalStatusEnum.PENDING,
                  ApprovalStatusEnum.ON_HOLD,
                  ApprovalStatusEnum.REJECTED,
                ],
              };
            }
        }

      case 'cancelled':
        switch (filter) {
          case 'cancelled':
            return {
              excludeCancelledRegistrations: false,
              registrationStatus: RegistrationStatusEnum.CANCELLED
            };
          case 'male':
            return {
              excludeCancelledRegistrations: false,
              gender: GenderEnum.MALE,
              registrationStatus: RegistrationStatusEnum.CANCELLED,
            };
          case 'female':
            return {
              excludeCancelledRegistrations: false,
              gender: GenderEnum.FEMALE,
              registrationStatus: RegistrationStatusEnum.CANCELLED,
            };
          default:
            return null;
        }
      case 'registrations':
        switch (filter) {
          case 'all':
            return { excludeCancelledRegistrations: false };
          case 'male':
            return {
              excludeCancelledRegistrations: false,
              gender: GenderEnum.MALE,
            };
          case 'female':
            return {
              excludeCancelledRegistrations: false,
              gender: GenderEnum.FEMALE,
            };
          case 'regPending':
            return {
              excludeCancelledRegistrations: false,
              approvalStatus: In([ApprovalStatusEnum.ON_HOLD, ApprovalStatusEnum.REJECTED]),
            };
          default:
            return null;
        }

      default:
        return null;
    }
  }

  /**
   * Fix the getKPIFilterCondition method to return proper values instead of TypeORM operators
   */
  private getKPIFilterCondition(category: string, filter: string): Record<string, any> | null {
    this.logger.debug(`Getting KPI filter condition for category: ${category}, filter: ${filter}`);
    
    try {
      if (!category || !filter) {
        this.logger.warn(`Invalid parameters for getKPIFilterCondition: category=${category}, filter=${filter}`);
        return null;
      }

      switch (category) {
        case 'registrations':
          switch (filter) {
            case 'all':
              return { registrationStatus: Not(In([RegistrationStatusEnum.DRAFT, RegistrationStatusEnum.SAVE_AS_DRAFT])) };
            case 'cancelled':
              return { registrationStatus: RegistrationStatusEnum.CANCELLED };
            case 'newSeekersPending':
              return {
                approvalStatus: In([ApprovalStatusEnum.PENDING]),
              };
            case 'regPending':
              return {
                approvalStatus: In([ApprovalStatusEnum.ON_HOLD, ApprovalStatusEnum.REJECTED]),
              };
            case 'rejected':
              return { approvalStatus: ApprovalStatusEnum.REJECTED };
            case 'onHold':
              return { approvalStatus: ApprovalStatusEnum.ON_HOLD };
            case 'approvedSeekers':
              return { approvalStatus: ApprovalStatusEnum.APPROVED };
            case 'rejectedSeekers':
              return { approvalStatus: ApprovalStatusEnum.REJECTED };
            default:
              this.logger.debug(`No filter condition found for registrations category with filter: ${filter}`);
              return null;
          }

        case 'programs':
          if (filter.startsWith('program_')) {
            const programId = parseInt(filter.replace('program_', ''));
            if (isNaN(programId)) {
              this.logger.warn(`Invalid program ID in filter: ${filter}`);
              return null;
            }
            return {
              approvalStatus: ApprovalStatusEnum.APPROVED,
              allocatedProgramId: programId,
            };
          }
          this.logger.debug(`No filter condition found for programs category with filter: ${filter}`);
          return null;

      case 'payments':
        switch (filter) {
          case 'all':
            return {
              isFreeSeat: false,
              paymentDetails: {
                paymentStatus: Not(In([PaymentStatusEnum.DRAFT, PaymentStatusEnum.SAVE_AS_DRAFT])),
              },
            };
          case 'paymentPending':
            return {
              isFreeSeat: false,
              paymentDetails: {
                paymentStatus: In([
                  PaymentStatusEnum.ONLINE_PENDING,
                  PaymentStatusEnum.OFFLINE_PENDING,
                ]),
              },
            };
          case 'paymentCompleted':
            return {
              isFreeSeat: false,
              paymentDetails: {
                paymentStatus: In([
                  PaymentStatusEnum.ONLINE_COMPLETED,
                  PaymentStatusEnum.OFFLINE_COMPLETED,
                ]),
              },
            };
          case 'no_payment':
            return { 
              isFreeSeat: true,
            };
          default:
            return null;
        }

      case 'invoices':
        switch (filter) {
          case 'all':
            return {
              invoiceDetails: {
                invoiceStatus: Not(InvoiceStatusEnum.DRAFT),
              },
            };
          case 'invoicePending':
            return {
              invoiceDetails: {
                invoiceStatus: InvoiceStatusEnum.INVOICE_PENDING,
              },
            };
          case 'invoiceCompleted':
            return {
              invoiceDetails: {
                invoiceStatus: InvoiceStatusEnum.INVOICE_COMPLETED,
              },
            };
          default:
            return null;
        }

      case 'travelAndLogistics':
        switch (filter) {
          case 'all':
            return {
              travelInfo: {
                travelInfoStatus: In([TravelStatusEnum.PENDING, TravelStatusEnum.COMPLETED]),
              },
            };
          case 'travelPending':
            return {
              travelInfo: {
                travelInfoStatus: TravelStatusEnum.PENDING,
              },
            };
          case 'travelCompleted':
            return {
              travelInfo: {
                travelInfoStatus: TravelStatusEnum.COMPLETED,
              },
            };
          case 'logisticsPending':
            return {
              travelPlans: {
                travelPlanStatus: TravelStatusEnum.PENDING,
              },
            };
          case 'logisticsCompleted':
            return {
              travelPlans: {
                travelPlanStatus: TravelStatusEnum.COMPLETED,
              },
            };
          default:
            return null;
        }

      // Handle blessed categories (hdb1Blessed, hdb2Blessed, etc.)
      case 'blessed':
      case 'hdb1Blessed':
      case 'hdb2Blessed':
      case 'hdb3Blessed':
      case 'msd1Blessed':
      case 'msd2Blessed':
        switch (filter) {
          // Program-specific filters
          default:
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            }
            // Payment filters
            else if (filter === 'paymentPending') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                isFreeSeat: false, // Only paid seats for payment filtering
                paymentDetails: {
                  paymentStatus: In([
                    PaymentStatusEnum.ONLINE_PENDING,
                    PaymentStatusEnum.OFFLINE_PENDING,
                  ]),
                },
              };
            }
            else if (filter === 'paymentComplete') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                isFreeSeat: false, // Only paid seats for payment filtering
                paymentDetails: {
                  paymentStatus: In([
                    PaymentStatusEnum.ONLINE_COMPLETED,
                    PaymentStatusEnum.OFFLINE_COMPLETED,
                  ]),
                },
              };
            }
            // Invoice filters
            else if (filter === 'invoicePending') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                invoiceDetails: {
                  invoiceStatus: InvoiceStatusEnum.INVOICE_PENDING,
                },
              };
            }
            else if (filter === 'invoiceComplete') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                invoiceDetails: {
                  invoiceStatus: InvoiceStatusEnum.INVOICE_COMPLETED,
                },
              };
            }
            // Travel filters
            else if (filter === 'travelPending') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                travelInfo: {
                  travelInfoStatus: TravelStatusEnum.PENDING,
                },
              };
            }
            else if (filter === 'travelComplete') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                travelInfo: {
                  travelInfoStatus: TravelStatusEnum.COMPLETED,
                },
              };
            }
            // Swap request filters
            else if (filter === 'swapRequests') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                swapRequests: [SwapType.WantsSwap],
              };
            }
            // Cancelled filters
            else if (filter === 'cancelled') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                registrationStatus: RegistrationStatusEnum.CANCELLED,
              };
            }
            else if (filter === 'approvedSeekers') {
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
              };
            }
            return null;
        }

      case 'swapRequests':
        switch (filter) {
          case 'canShift':
            return {
              swapRequests: [SwapType.CanShift],
            };
          case 'wantsSwap':
            return {
              swapRequests: [SwapType.WantsSwap],
            };
          default:
            return null;
        }

      case 'all':
        switch (filter) {
          case 'all':
            return {
              registrationStatus: Not(In([RegistrationStatusEnum.DRAFT, RegistrationStatusEnum.SAVE_AS_DRAFT])),
            };
          default:
            // Handle program-specific filters like 'program_793'
            if (filter.startsWith('program_')) {
              const programId = parseInt(filter.replace('program_', ''));
              return {
                allocatedProgramId: programId,
              };
            }
            return null;
        }

      default:
        // Handle when kpiCategory starts with 'program_' (new blessed category format)
        if (category.startsWith('program_')) {
          const programId = parseInt(category.replace('program_', ''));
          switch (filter) {
            // Blessed filter for specific program
            case 'blessed':
              return {
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            // Payment filters for specific program
            case 'paymentPending':
              return {
                paymentStatus: ['payment_pending'],
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            case 'paymentComplete':
              return {
                paymentStatus: ['payment_completed'],
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            // Invoice filters for specific program
            case 'invoicePending':
              return {
                invoiceStatus: ['invoice_pending'],
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            case 'invoiceComplete':
              return {
                invoiceStatus: ['invoice_completed'],
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            // Travel filters for specific program
            case 'travelPending':
              return {
                travelStatus: ['travel_pending'],
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            case 'travelComplete':
              return {
                travelStatus: ['travel_completed'],
                approvalStatus: ApprovalStatusEnum.APPROVED,
                allocatedProgramId: programId,
              };
            // Program-specific filters
            default:
              if (filter.startsWith('program_')) {
                const filterProgramId = parseInt(filter.replace('program_', ''));
                return {
                  approvalStatus: ApprovalStatusEnum.APPROVED,
                  allocatedProgramId: filterProgramId,
                };
              }
              return null;
          }
        }
        this.logger.debug(`No filter condition found for unknown category: ${category}`);
        return null;
      }
    } catch (error) {
      this.logger.error(`Error in getKPIFilterCondition for category=${category}, filter=${filter}`, error.stack);
      return null;
    }
  }

  async getProgramDashboardData({
    programId,
    mainProgramDimensions = [],
    subProgramId,
    subProgramDimensions = [],
    rmId,
    userRole,
  }: {
    programId: number;
    mainProgramDimensions: string[];
    subProgramId?: number;
    subProgramDimensions?: string[];
    rmId?: number;
    userRole: string;
  }) {
    // this.logger.log(registrationMessages.GET_PROGRAM_DASHBOARD_DATA, { programId });
    try {
      const program = await this.repository.findProgramWithType(programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString(),
        );
      }
      const subPrograms = await this.repository.getSubPrograms(programId);
      const subProgramIds = subProgramId ? [subProgramId] : subPrograms.map((item) => item.id);
      const dimensions = subProgramId ? subProgramDimensions : mainProgramDimensions;
      let data: Array<{
        displayName: string;
        key: string;
        columnDefinition?: any;
        count?: number;
        values: {
          displayName: string;
          key: string;
          count: any;
          id?: number;
          filters?: {};
          order: number;
        }[];
      }> = [];

      const result = dimensions.map(async (dimension) => {
        const resultData = await this.repository.getRegistrationCountsOfDimension({
          programId: programId,
          dimension,
          subProgramId: subProgramId ? subProgramId : undefined,
          rmId: rmId ? rmId : undefined,
        });
        const ageGroups = [
          { key: 'greaterThanEqual0', label: 'Age', displayName : '<=20', min: 0, max: 20, filters : {age : ['0-20']} },
          { key: 'greaterThanEqual20', label: 'Age', displayName : '21 - 30', min: 21, max: 30, filters : {age : ['21-30']} },
          { key: 'greaterThanEqual30', label: 'Age', displayName : '31 - 50', min: 31, max: 50, filters : {age : ['31-50']} },
          { key: 'greaterThanEqual50', label: 'Age', displayName : '51 - 65', min: 51, max: 65, filters : {age : ['51-65']} },
          { key: 'greaterThanEqual65', label: 'Age', displayName : '> 65', min: 66, max: 200, filters : {age : ['>65']} },
        ];
        switch (dimension) {
          case 'programRegistration':
            let programValues: any = [
              {
                displayName: 'Total registrations',
                key: 'totalRegistrations',
                count: Number(resultData?.totalRegistrations || 0),
                filters: {},
                order: 1,
              },
              {
                displayName: 'Unassigned',
                key: 'newRegistrations',
                count: Number(resultData?.newRegistrations || 0),
                filters: {
                  kpiCategory: 'registrations',
                  kpiFilter: 'newSeekersPending',
                },
                order: 2,
              },
              {
                displayName: 'Blessed seekers',
                key: 'blessedSeekers',
                count: Number(resultData?.blessedSeekers || 0),
                filters: {
                  kpiCategory: 'registrations',
                  kpiFilter: 'approvedSeekers',
                },
                order: 3,
              },
            ];
            if (userRole === 'rm') {
              const rmProgramValues = [
                {
                  displayName: 'Pending details',
                  key: 'pendingDetails',
                  count: Number(resultData?.pendingDetails || 0),
                  filters: {
                    kpiCategory: 'registrations',
                    kpiFilter: 'approvedSeekers',
                    approvalStatus: 'pending',
                    registrationStatus: 'pending',
                    order: 4,
                  },
                },
                {
                  displayName: 'Swap request',
                  key: 'swapRequest',
                  count: Number(resultData?.swapRequest || 0),
                  filters: {
                    swapRequests: ['can_shift', 'wants_swap'],
                    order: 5,
                  },
                },
              ];
              programValues = [...programValues, ...rmProgramValues];
            }
            if (
              [
                'admin',
                'viewer',
                'mahatria',
                'finance_manager',
                'relational_manager',
                'shoba',
                'operational_manger',
                'rm_support',
              ].includes(userRole)
            ) {
              programValues.push({
                displayName: 'Payment pending',
                key: 'paymentPending',
                label: 'Payment status',
                count: Number(resultData?.paymentPending || 0),
                filters: {
                  paymentStatus: 'payment_pending',
                },
                order: 4,
              });
              if (userRole === 'finance_manager') {
                programValues.push({
                  displayName: 'Invoice pending',
                  key: 'invoicePending',
                  label: 'Invoice status',
                  count: Number(resultData?.invoicePending || 0),
                  filters: {
                    invoiceStatus: 'invoice_pending',
                  },
                  order: 5,
                });
              } else if (
                [
                  'admin',
                  'viewer',
                  'mahatria',
                  'relational_manager',
                  'shoba',
                  'operational_manger',
                  'rm_support',
                ].includes(userRole)
              ) {
                programValues.push({
                  displayName: 'Travel pending',
                  key: 'travelPending',
                  label: 'Travel status',
                  count: Number(resultData?.travelPending || 0),
                  filters: {
                    travelStatus: 'travel_pending',
                  },
                  order: 6,
                });
              }
            }

            return {
              displayName: 'Program registrations',
              key: 'programRegistrations',
              values: programValues,
              count: programValues.reduce((sum, item) =>
                sum + Number(item.count || 0), 0),
            };
          
          
          case 'rating':
            const ratingColumnDefinition = [
              {
                displayName: 'User profile',
                key: 'profileUrl',
              },
              {
                displayName: 'Full name',
                key: 'fullName',
              },
              {
                displayName: 'No Of HDBs',
                key: 'noOfHdbs',
              }
            ];
            const ratingValues = resultData?.unratedRegistrations || [];
            return {
              displayName: 'Yet to review',
              key: 'ratings',
              label: 'RM rating',
              columnDefinition: ratingColumnDefinition,
              values: ratingValues,
              filters: {"rmRating":["unrated"]},
              count: Number(resultData?.totalUnRatedRegistrations) || 0,
            };

          case 'programBlessedSeekers':
            const blessedSeekersValues = subProgramIds.map((item) => {
            const subProgram = subPrograms.filter((sp) => sp.id === item);
            const filteredItems = (resultData.filter(r => r.id == item))
              const itemData = filteredItems.length > 0 ? filteredItems[0] : {
                programId: item,
                programName: null,
                count: 0,
              };
              return {
                id: subProgram[0].id,
                displayName: subProgram[0].name,
                key: `program_${itemData.id}`,
                count: Number(itemData.count) || 0,
                filters: {
                  kpiCategory: 'programs',
                  kpiFilter: `program_${itemData.id}`,
                },
              };
            });
            const totalBlessedSeekers = blessedSeekersValues.reduce(
              (sum, item) => sum + Number(item.count),
              0,
            );
            return {
              displayName: 'Blessed Seekers - Snapshot',
              key: 'blessedSeekers',
              values: blessedSeekersValues,
              count: totalBlessedSeekers,
            };

          case 'programApprovalStatus':
            const registrationStatuses = [
              {
                displayName: 'Blessed',
                key: 'blessed',
                count: 0,
                approvalStatus: 'approved',
                filters: { kpiCategory: 'registrations', kpiFilter: 'approvedSeekers' },
              },
              {
                displayName: 'Unassigned',
                key: 'yetToAssign',
                count: 0,
                approvalStatus: 'pending',
                filters: { kpiCategory: 'registrations', kpiFilter: 'newSeekersPending' },
              },
              {
                displayName: 'Hold',
                key: 'onHold',
                count: 0,
                approvalStatus: 'rejected',
                filters: { kpiCategory: 'registrations', kpiFilter: 'rejected' },
              },
              {
                displayName: 'Swap demand',
                key: 'yetToDecide',
                count: 0,
                approvalStatus: 'on_hold',
                filters: { kpiCategory: 'registrations', kpiFilter: 'onHold' },
              },
              {
                displayName: 'Cancelled',
                key: 'cancelled',
                count: 0,
                approvalStatus: 'cancelled',
                filters: { kpiCategory: 'registrations', kpiFilter: 'cancelled' },
              },
              {
                displayName: 'Swap request',
                key: 'swapRequest',
                count: 0,
                approvalStatus: 'swap_request',
                filters: { "swapRequests":["can_shift","wants_swap"] },
              }
            ];

            const values = registrationStatuses.map((item) => {
              const filterItem = resultData.find(
                (status) =>
                  status.approvalStatus.toLowerCase() === item.approvalStatus.toLowerCase(),
              );
              return {
                displayName: item.displayName,
                key: item.key,
                count: Number(filterItem?.count) || 0,
                filters: item.filters,
              };
            });
            const totalSeekers = values.reduce((sum, item) => sum + Number(item.count), 0);

           return {
             displayName: 'Registrations - Snapshot',
             key: 'programApprovalStatus',
             values: values,
             count: totalSeekers,
           };
          
          case  'detailedData':
            const detailedDataColumnDefinition  = [
                  {
                      displayName: 'HDBs/MSDs',
                      key: 'hdbsMsds'
                  },
                  {
                      displayName: '1st Timer',
                      key: 'firstTimers'
                  },
                  {
                      displayName: '1 - 10 HDBs/MSDs',
                      key: 'oneToTenHdbsMsds'
                  },
                  {
                      displayName: '>10 HDBs/MSDs',
                      key: 'greaterThanTenHdbsMsds'
                  },
                  {
                      displayName: 'Organisation',
                      key: 'organisation'
                  },
                  {
                      displayName: 'Total',
                      key: 'totalSeekers'
                  },
                  {
                      displayName: 'Male',
                      key: 'male'
                  },
                  {
                      displayName: 'Female',
                      key: 'female'
                  }

              ]
            const detailedDataValues = subProgramIds.map(subProgramId => {
              const subProgram = subPrograms.filter((sp) => sp.id === subProgramId);
              const filteredItems = resultData.filter((r) => r.programId == subProgramId);
              const item =
                filteredItems.length > 0
                  ? filteredItems[0]
                  : {
                      programId: subProgramId,
                      hdbsMsds: subProgram[0].name,
                      totalSeekers: 0,
                      firstTimers: 0,
                      oneToTenHdbsMsds: 0,
                      greaterThanTenHdbsMsds: 0,
                      organisation: 0,
                      male: 0,
                      female: 0,
                    };
              return {
                id: item.programId,
                hdbsMsds: subProgram[0].name,
                totalSeekers: Number(item.totalSeekers) || 0,
                firstTimers: Number(item.firstTimers) || 0,
                oneToTenHdbsMsds: Number(item.oneToTenHdbsMsds) || 0,
                greaterThanTenHdbsMsds: Number(item.greaterThanTenHdbsMsds) || 0,
                organisation: Number(item.organisation) || 0,
                male: Number(item.male) || 0,
                female: Number(item.female) || 0,
                filters: {
                  totalSeekers: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}` },
                  firstTimers: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    numberOfHdbs: '0',
                  },
                  oneToTenHdbsMsds: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    numberOfHdbs: '1-10',
                  },
                  greaterThanTenHdbsMsds: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    numberOfHdbs: '>10',
                  },
                  organisation: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,
                    organisation: 'yes',
                  },
                  male: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`, 
                    gender: ['Male'],
                  },
                  female: {
                    kpiCategory: 'programs',
                    kpiFilter: `program_${item.programId}`,   
                    gender: ['Female'],
                  }
                },
              };
            })
              return {
                displayName: "Blessed seekers by no. of HDBs",
                key: "detailedData",
                columnDefinition: detailedDataColumnDefinition,
                values: detailedDataValues,
                count: detailedDataValues.reduce(
                  (sum, item) => sum + Number(item.totalSeekers || 0),
                  0,
                ),
              };
          
          case 'demographics':
            const ageGenderGroupCounts = ageGroups.map((group) => {
              const genders = ['male', 'female'];
              const genderCounts = genders.map((gender) => {
                const count = resultData
                  .filter((row) => {
                    const age = Number(row.age);
                    return (
                      age >= group.min && age <= group.max && row.gender?.toLowerCase() === gender
                    );
                  })
                  .reduce((sum, row) => sum + Number(row.count), 0);
                return { gender, count };
              });
              return {
                key: group.key,
                displayName: group.displayName,
                genders: {
                  [genderCounts[0].gender.toLowerCase()]: genderCounts[0].count,
                  [genderCounts[1].gender.toLowerCase()]: genderCounts[1].count,
                },
                filters: group.filters,
              };
            });
            const demographicsValues = ageGenderGroupCounts.map((item) => {
              const subProgramFilters = subProgramId
                ? { kpiCategory: 'programs', kpiFilter: `program_${subProgramId}` }
                : {};
              return {
                displayName: item.displayName,
                key: item.key,
                values: [
                  {
                    displayName: 'Male',
                    key: 'Male',
                    label: 'Gender',
                    count: item.genders.male || 0,
                    filters: {
                      ...item.filters,
                      gender: ['Male'],
                      ...subProgramFilters,
                    },
                  },
                  {
                    displayName: 'Female',
                    key: 'Female',
                    label: 'Gender',
                    count: item.genders.female || 0,
                    filters: {
                      ...item.filters,
                      gender: ['Female'],
                      ...subProgramFilters,
                    },
                  },
                ],
              };
            });
            return {
              displayName: subProgramId ? 'Demographics': 'All Seekers - Demographics' ,
              key: 'demographics',
              values: demographicsValues,
              count: demographicsValues.reduce((sum, item) => {
                const male = item.values.find(v => v.key === 'Male');
                const female = item.values.find(v => v.key === 'Female');
                return sum + Number(male?.count || 0) + Number(female?.count || 0);
              }, 0),
            };

          case 'preferredVsBlessed':
            const preferredVsBlessedValues = resultData.map((item) => {
              return {
                id: item.programId,
                displayName: item.displayName,
                key: item.displayName.toLowerCase().replace(/\s+/g, '_'),
                values: [
                  {
                    displayName: 'Total preferences',
                    key: 'totalPreferences',
                    count: Number(item.totalPreferences) || 0,
                    filters: {
                      kpiCategory: 'programs',
                      kpiFilter: `program_${item.programId}`,
                      preferredProgram: [item.programId],
                    },
                  },
                  {
                    displayName: 'Total blessed seekers',
                    key: 'totalBlessedSeekers',
                    count: Number(item.totalBlessedSeekers) || 0,
                    filters: {
                      kpiCategory: 'programs',
                      kpiFilter: `program_${item.programId}`,
                    },
                  },
                ],
              };
            });

            return {
              displayName: 'Preference against blessings',
              key: 'preferredVsBlessed',
              values: preferredVsBlessedValues,
              count: this.calculateNestedCount(preferredVsBlessedValues),
            };

          case 'payment':
            const onlinePending = resultData.find(
              (item) => item.paymentStatus === 'online_pending',
            );
            const onlineCompleted = resultData.find(
              (item) => item.paymentStatus === 'online_completed',
            );
            const onlineFailed = resultData.find((item) => item.paymentStatus === 'online_failed');
            const offlinePending = resultData.find(
              (item) => item.paymentStatus === 'offline_pending',
            );
            const offlineCompleted = resultData.find(
              (item) => item.paymentStatus === 'offline_completed',
            );
            const offlineFailed = resultData.find(
              (item) => item.paymentStatus === 'offline_failed',
            );
            const paymentValues = [
              {
                displayName: 'Completed',
                key: 'completed',
                label: 'Payment status',
                values: [
                  {
                    displayName: 'Online',
                    key: 'online',
                    label: 'Payment mode',
                    count: onlineCompleted ? Number(onlineCompleted?.count) : 0,
                    filters : {
                      paymentStatus : ['payment_completed'],
                      paymentMode : ['online'],
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                      
                    }
                  },
                  {
                    displayName: 'Offline',
                    key: 'offline',
                    label: 'Payment mode',
                    count: offlineCompleted ? Number(offlineCompleted?.count) : 0,
                    filters : {
                      paymentStatus : ['payment_completed'],
                      paymentMode : ['offline'],
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                     
                    }
                  },
                ],
                count:
                  (onlineCompleted ? Number(onlineCompleted?.count) : 0) +
                  (offlineCompleted ? Number(offlineCompleted?.count) : 0),
              },
              {
                displayName: 'Pending',
                key: 'pending',
                label: 'Payment status',
                values: [
                  {
                    displayName: 'Online',
                    key: 'online',
                    label: 'Payment mode',
                    count: onlinePending ? Number(onlinePending?.count) : 0,
                    filters : {
                      paymentStatus : ['payment_pending'],
                      paymentMode : ['online'],
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                    }
                  },
                  {
                    displayName: 'Offline',
                    key: 'offline',
                    label: 'Payment mode',
                    count: offlinePending ? Number(offlinePending?.count) : 0,
                    filters: {
                      paymentStatus : ['payment_pending'],
                      paymentMode : ['offline'],
                      kpiCategory: "programs",
                      kpiFilter: `program_${subProgramId}`,
                     
                    }
                  },
                ],
                count:
                  (onlinePending ? Number(onlinePending?.count) : 0) +
                  (offlinePending ? Number(offlinePending?.count) : 0),
              },
              {
                displayName: 'Failed',
                key: 'failed',
                values: [
                  {
                    displayName: 'Online',
                    key: 'online',
                    label: 'Payment mode',
                    count: onlineFailed ? Number(onlineFailed?.count) : 0,
                  },
                  {
                    displayName: 'Offline',
                    key: 'offline',
                    label: 'Payment mode',
                    count: offlineFailed ? Number(offlineFailed?.count) : 0,
                  },
                ],
                count:
                  (onlineFailed ? Number(onlineFailed?.count) : 0) +
                  (offlineFailed ? Number(offlineFailed?.count) : 0),
              },
            ];

            return {
              displayName: 'Payment status',
              key: 'paymentStatus',
              label: 'Payment status',
              values: paymentValues,
              count: paymentValues.reduce((sum, item) => {
                return sum + Number(item?.count);
              }, 0),
            };
          
          case 'invoice':
            const invoiceCompleted = resultData?.invoiceStatusCounts.find(
              (item) => item.invoiceStatus === 'invoice_completed',
            );
            const invoicePending = resultData?.invoiceStatusCounts.find(
              (item) => ['invoice_pending', 'draft'].includes(item.invoiceStatus),
            );
            const programCompletedFilters = {
              kpiCategory: 'registrations',
              kpiFilter: 'approvedSeekers',
              invoiceStatus: ['invoice_completed'],
            };
            const subProgramCompletedFilters = {
              kpiCategory: 'programs',
              kpiFilter: `program_${subProgramId}`,
              invoiceStatus: ['invoice_completed'],
            };
            const programPendingFilters = {
              kpiCategory: 'registrations',
              kpiFilter: 'approvedSeekers',
              invoiceStatus: ['invoice_pending'],
            };
            const subProgramPendingFilters = {
              kpiCategory: 'programs',
              kpiFilter: `program_${subProgramId}`,
              invoiceStatus: ['invoice_pending'],
            };
            const invoiceValues = [
              {
                displayName: 'Completed',
                key: 'completed',
                label: 'Invoice status',
                count: invoiceCompleted ? Number(invoiceCompleted?.count) : 0,
                filters: subProgramId ? subProgramCompletedFilters : programCompletedFilters,
              },
              {
                displayName: 'Pending',
                key: 'pending',
                label: 'Invoice status',
                count: invoicePending ? Number(invoicePending?.count) : 0,
                filters: subProgramId ? subProgramPendingFilters : programPendingFilters,
              },
            ];

            return {
              displayName: 'Invoice generated',
              key: 'invoiceStatus',
              label: 'Invoice status',
              values: invoiceValues,
              count: resultData?.totalInvoiceCount
            };
          
          case 'travelPlan':
            const flightCount = resultData.find(
              (item) => item.travelType === TravelTypeEnum.FLIGHT,
            );
            const ownTransportCount = resultData.find(
              (item) => item.travelType === TravelTypeEnum.OWN_TRANSPORT,
            );
            const cityPickupCount = resultData.find(
              (item) => item.travelType === TravelTypeEnum.CITY_PICK_UP,
            );

            const travelValues = [
              {
                displayName: 'Flight',
                key: 'flight',
                count: flightCount ? Number(flightCount?.count) : 0,
              },
              {
                displayName: 'Own transport',
                key: 'ownTransport',
                count: ownTransportCount ? Number(ownTransportCount?.count) : 0,
              },
              {
                displayName: 'City pickup',
                key: 'cityPickup',
                count: cityPickupCount ? Number(cityPickupCount?.count) : 0,
              },
            ];
            return {
              displayName: 'Completed travel plan',
              key: 'travelPlan',
              values: travelValues,
              count: travelValues.reduce((sum, item) => sum + Number(item.count), 0),
            };
          
          case 'travelStatus':
            const travelStatusValues = [
              {
                displayName: 'Completed',
                key: 'completed',
                label: 'Travel status',
                count: resultData?.completedCount ? Number(resultData?.completedCount) : 0,
                filters: {
                  travelStatus: 'travel_completed',
                  kpiCategory: 'programs',
                  kpiFilter: `program_${subProgramId}`,
             
                },
              },
              {
                displayName: 'Pending',
                key: 'pending',
                label: 'Travel status',
                count: resultData?.pendingCount ? Number(resultData?.pendingCount) : 0,
                filters: {
                  travelStatus: 'travel_pending',
                  kpiCategory: 'programs',
                  kpiFilter: `program_${subProgramId}`,
                  
                },
              },
            ];

            return {
              displayName: 'Travel status',
              key: 'travelStatus',
              label: 'Travel status',
              values: travelStatusValues,
              count: resultData?.totalRegistrations ? Number(resultData?.totalRegistrations) : 0,
            };
          
          case 'location':
          const cities = [...MajorCities, 'Other'];
            const locationValues = cities.map((city) => {
              const cityData = resultData.find((item) => item.location === city);
              return {
                displayName: city,
                key: city,
                label: 'Location',
                count: cityData ? Number(cityData.count) : 0,
                filters: {
                  location: [city],
                },
              };
            });
            return {
              displayName: 'Location insights',
              key: 'location',
              label: 'Location',
              values: locationValues,
              count: locationValues.reduce((sum, item) => sum + Number(item.count), 0),
            };
          
          case 'goodies':
          case 'goodiesData':
            const goodiesDataColumnDefinition = [
              {
                  displayName: 'HDB/MSD',
                  key: 'hdbMsd'
              },
              {
                  displayName: 'Total Offered',
                  key: 'totalOffered'
              },
              {
                  displayName: 'Notebook',
                  key: 'noteBook'
              },
              {
                  displayName: 'Flask',
                  key: 'flask'
              },
              {
                  displayName: 'T-Shirt',
                  key: 'tShirtTotal'
              },
              {
                  displayName: 'Jacket',
                  key: 'jacketTotal'
              },
              {
                  displayName: 'Ratria Pillar (Leonia)',
                  key: 'ratriaPillarLeonia'
              },
                            {
                  displayName: 'Ratria Pillar (Other Locations)',
                  key: 'ratriaPillarOtherLocations'
              },
              // {
              //   displayName: 'Ratria Pillar Total',
              //   key: 'ratriaPillarTotal'
              // },
              {
                  displayName: 'Goodies Completed',
                  key: 'goodiesComplete'
              },
              {
                  displayName: 'Goodies Pending',
                  key: 'goodiesPending'
              }
            ]
            
            const goodiesDataValues = subProgramIds.map(subProgramId => {
              const subProgram = subPrograms.filter((sp) => sp.id === subProgramId);
              const filteredItems = resultData.filter((r) => r.programId == subProgramId);
              const item = filteredItems.length > 0
                ? filteredItems[0]
                : {
                    programId: subProgramId,
                    hdbMsd: subProgram[0].name,
                    totalOffered: 0,
                    noteBook: 0,
                    flask: 0,
                    tShirtTotal: 0,
                    jacketTotal: 0,
                    ratriaPillarOtherLocations: 0,
                    // ratriaPillarTotal: 0,
                    ratriaPillarLeonia: 0,
                    goodiesComplete: 0,
                    goodiesPending: 0,
                  };
              
              return {
                id: item.programId,
                hdbMsd: subProgram[0].name,
                totalOffered: Number(item.totalOffered) || 0,
                noteBook: Number(item.noteBook) || 0,
                flask: Number(item.flask) || 0,
                tShirtTotal: Number(item.tShirtTotal) || 0,
                jacketTotal: Number(item.jacketTotal) || 0,
                ratriaPillarOtherLocations: Number(item.ratriaPillarOtherLocations) || 0,
                // ratriaPillarTotal: Number(item.ratriaPillarTotal) || 0,
                ratriaPillarLeonia: Number(item.ratriaPillarLeonia) || 0,
                goodiesComplete: Number(item.goodiesComplete) || 0,
                goodiesPending: Number(item.goodiesPending) || 0,
                filters: {
                //   noteBook: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}`, goodiesField: 'notebook' },
                //   flask: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}`, goodiesField: 'flask' },
                //   tShirtTotal: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}`, goodiesField: 'tshirt' },
                //   jacketTotal: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}`, goodiesField: 'jacket' },
                //   ratriaPillarOtherLocations: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}`, goodiesField: 'ratria_pillar_other_location' },
                //   ratriaPillarLeonia: { kpiCategory: 'programs', kpiFilter: `program_${item.programId}`, goodiesField: 'ratria_pillar_leonia' },

                  goodiesComplete: { 
                    kpiCategory: 'programs', 
                    kpiFilter: `program_${item.programId}`,
                    view: ViewTypeEnum.GOODIES,
                    goodiesStatus: ['goodies_completed'] 
                  },
                  goodiesPending: { 
                    kpiCategory: 'programs', 
                    kpiFilter: `program_${item.programId}`,
                    view: ViewTypeEnum.GOODIES,
                    goodiesStatus: ['goodies_pending'] 
                  },
                },
              };
            });
            
            return {
              displayName: "Goodies and Ratria Pillars - Snapshot",
              key: "goodiesData",
              columnDefinition: goodiesDataColumnDefinition,
              values: goodiesDataValues,
              count: goodiesDataValues.reduce(
                (sum, item) => sum + Number(item.totalOffered || 0),
                0,
              ),
            };
          
          case 'goodiesTshirt':
            const tshirtSizesArray = await this.repository.getActiveTshirtSizes();

            const tshirtSizes = tshirtSizesArray.length > 0 ? tshirtSizesArray.map(size => size.lookupKey) : ['XS', 'S', 'M', 'L', 'XL', 'XXL', 'XXXL'];

            // Build column definitions with headers and sub-headers
            const tshirtColumnDefinition: any = [
              { 
                displayName: 'T-Shirt Size', 
                key: 'tshirtSize',
                type: 'single'
              }
            ];
            
            // Add program columns with sub-headers (Male, Female)
            subPrograms.forEach(subProgram => {
              tshirtColumnDefinition.push({
                displayName: subProgram.name, // Main header: MSD, MSD 2, etc.
                key: `program${subProgram.id}`,
                type: 'grouped',
                subColumns: [
                  { displayName: 'Male', key: `program${subProgram.id}Male` },
                  { displayName: 'Female', key: `program${subProgram.id}Female` }
                ]
              });
            });
            
            // Add Summary column
            tshirtColumnDefinition.push({ 
              displayName: 'Summary Total Count', 
              key: 'summaryTotalCount',
              type: 'single'
            });

            // Build rows for each size
            const tshirtSizeRows: any[] = tshirtSizes.map((size, index) => {
              const row: any = { tshirtSize: size, sizeOrder: index + 1 };
              let summaryTotal = 0;

              subPrograms.forEach(subProgram => {
                const subProgramData = resultData.find(r => r.programId === subProgram.id);
                const maleCount = Number(subProgramData?.[`male${size}`] || 0);
                const femaleCount = Number(subProgramData?.[`female${size}`] || 0);
                
                row[`program${subProgram.id}Male`] = maleCount;
                row[`program${subProgram.id}Female`] = femaleCount;
                summaryTotal += maleCount + femaleCount;
              });

              row.summaryTotalCount = summaryTotal;
              return row;
            });

            // Add N/A row
            /*
            const naRow: any = { tshirtSize: 'N/A', sizeOrder: 998 };
            let naSummaryTotal = 0;

            subPrograms.forEach(subProgram => {
              const subProgramData = resultData.find(r => r.programId === subProgram.id);
              const maleNA = Number(subProgramData?.maleNA || 0);
              const femaleNA = Number(subProgramData?.femaleNA || 0);
              
              naRow[`subProgram${subProgram.id}Male`] = maleNA;
              naRow[`subProgram${subProgram.id}Female`] = femaleNA;
              naSummaryTotal += maleNA + femaleNA;
            });

            naRow.summaryTotalCount = naSummaryTotal;
            tshirtSizeRows.push(naRow);

            // Add No Value row (registrations with no goodies record)
            const noValueRow: any = { tshirtSize: 'No Value', sizeOrder: 999 };
            let noValueSummaryTotal = 0;

            subPrograms.forEach(subProgram => {
              const subProgramData = resultData.find(r => r.programId === subProgram.id);
              const maleNoValue = Number(subProgramData?.maleNoValue || 0);
              const femaleNoValue = Number(subProgramData?.femaleNoValue || 0);
              
              noValueRow[`subProgram${subProgram.id}Male`] = maleNoValue;
              noValueRow[`subProgram${subProgram.id}Female`] = femaleNoValue;
              noValueSummaryTotal += maleNoValue + femaleNoValue;
            });

            noValueRow.summaryTotalCount = noValueSummaryTotal;
            tshirtSizeRows.push(noValueRow);
            */

            // Calculate grand totals
            const grandTotals: any = { tshirtSize: 'Grand Count', sizeOrder: 1000 };
            
            subPrograms.forEach(subProgram => {
              grandTotals[`program${subProgram.id}Male`] = tshirtSizeRows.reduce((sum, row) => 
                sum + (row[`program${subProgram.id}Male`] || 0), 0);
              grandTotals[`program${subProgram.id}Female`] = tshirtSizeRows.reduce((sum, row) => 
                sum + (row[`program${subProgram.id}Female`] || 0), 0);
            });

            grandTotals.summaryTotalCount = tshirtSizeRows.reduce((sum, row) => 
              sum + (row.summaryTotalCount || 0), 0);
            
            tshirtSizeRows.push(grandTotals);

            return {
              displayName: "T-Shirt Details",
              key: "goodiesTshirt",
              columnDefinition: tshirtColumnDefinition,
              values: tshirtSizeRows,
              count: grandTotals.summaryTotalCount,
            };

          case 'goodiesJacket':
            const jacketSizesArray = await this.repository.getActiveTshirtSizes();

            const jacketSizes = jacketSizesArray.length > 0 ? jacketSizesArray.map(size => size.lookupKey) : ['XS', 'S', 'M', 'L', 'XL', 'XXL', 'XXXL'];

            // Build column definitions with headers and sub-headers
            const jacketColumnDefinition: any = [
              { 
                displayName: 'Jacket Size', 
                key: 'jacketSize',
                type: 'single'
              }
            ];
            
            // Add program columns with sub-headers (Male, Female)
            subPrograms.forEach(subProgram => {
              jacketColumnDefinition.push({
                displayName: subProgram.name, // Main header: MSD, MSD 2, etc.
                key: `program${subProgram.id}`,
                type: 'grouped',
                subColumns: [
                  { displayName: 'Male', key: `program${subProgram.id}Male` },
                  { displayName: 'Female', key: `program${subProgram.id}Female` }
                ]
              });
            });
            
            // Add Summary column
            jacketColumnDefinition.push({ 
              displayName: 'Summary Total Count', 
              key: 'summaryTotalCount',
              type: 'single'
            });

            // Build rows for each size
            const jacketSizeRows: any[] = jacketSizes.map((size, index) => {
              const row: any = { jacketSize: size, sizeOrder: index + 1 };
              let summaryTotal = 0;

              subPrograms.forEach(subProgram => {
                const subProgramData = resultData.find(r => r.programId === subProgram.id);
                const maleCount = Number(subProgramData?.[`male${size}`] || 0);
                const femaleCount = Number(subProgramData?.[`female${size}`] || 0);
                
                row[`program${subProgram.id}Male`] = maleCount;
                row[`program${subProgram.id}Female`] = femaleCount;
                summaryTotal += maleCount + femaleCount;
              });

              row.summaryTotalCount = summaryTotal;
              return row;
            });

            // Add N/A row
            /*
            const jacketNaRow: any = { jacketSize: 'N/A', sizeOrder: 998 };
            let jacketNaSummaryTotal = 0;

            subPrograms.forEach(subProgram => {
              const subProgramData = resultData.find(r => r.programId === subProgram.id);
              const maleNA = Number(subProgramData?.maleNA || 0);
              const femaleNA = Number(subProgramData?.femaleNA || 0);
              
              jacketNaRow[`subProgram${subProgram.id}Male`] = maleNA;
              jacketNaRow[`subProgram${subProgram.id}Female`] = femaleNA;
              jacketNaSummaryTotal += maleNA + femaleNA;
            });

            jacketNaRow.summaryTotalCount = jacketNaSummaryTotal;
            jacketSizeRows.push(jacketNaRow);

            // Add No Value row (registrations with no goodies record)
            const jacketNoValueRow: any = { jacketSize: 'No Value', sizeOrder: 999 };
            let jacketNoValueSummaryTotal = 0;

            subPrograms.forEach(subProgram => {
              const subProgramData = resultData.find(r => r.programId === subProgram.id);
              const maleNoValue = Number(subProgramData?.maleNoValue || 0);
              const femaleNoValue = Number(subProgramData?.femaleNoValue || 0);
              
              jacketNoValueRow[`subProgram${subProgram.id}Male`] = maleNoValue;
              jacketNoValueRow[`subProgram${subProgram.id}Female`] = femaleNoValue;
              jacketNoValueSummaryTotal += maleNoValue + femaleNoValue;
            });

            jacketNoValueRow.summaryTotalCount = jacketNoValueSummaryTotal;
            jacketSizeRows.push(jacketNoValueRow);
            */

            // Calculate grand totals
            const jacketGrandTotals: any = { jacketSize: 'Grand Count', sizeOrder: 1000 };
            
            subPrograms.forEach(subProgram => {
              jacketGrandTotals[`program${subProgram.id}Male`] = jacketSizeRows.reduce((sum, row) => 
                sum + (row[`program${subProgram.id}Male`] || 0), 0);
              jacketGrandTotals[`program${subProgram.id}Female`] = jacketSizeRows.reduce((sum, row) => 
                sum + (row[`program${subProgram.id}Female`] || 0), 0);
            });

            jacketGrandTotals.summaryTotalCount = jacketSizeRows.reduce((sum, row) => 
              sum + (row.summaryTotalCount || 0), 0);
            
            jacketSizeRows.push(jacketGrandTotals);

            return {
              displayName: "Jacket Details",
              key: "goodiesJacket",
              columnDefinition: jacketColumnDefinition,
              values: jacketSizeRows,
              count: jacketGrandTotals.summaryTotalCount,
            };

          default:
            this.logger.warn(`Unknown dimension for goodiesTshirt: ${dimension}`);
        }
      });

      const settledResults = await Promise.allSettled(result);
      data = settledResults
        .filter((res) => res.status === 'fulfilled' && res.value)
        .map((res) => (res as PromiseFulfilledResult<any>).value);

      if (subProgramId) {
        const paymentStatus = data.find(d => d.key === 'paymentStatus');
        const invoiceStatus = data.find(d => d.key === 'invoiceStatus');
        const travelStatus = data.find(d => d.key === 'travelStatus');
        const programRegistrations = data.find(d => d.key === 'programRegistrations');
      
        // Get total seekers count
        const totalSeekers = programRegistrations?.values?.find(v => v.key === 'totalRegistrations')?.count || 0;
      
        // Get completed counts
        const paymentCompleted = paymentStatus?.values?.find(v => v.key === 'completed')?.count || 0;
        const invoiceCompleted = invoiceStatus?.values?.find(v => v.key === 'completed')?.count || 0;
        const travelCompleted = travelStatus?.values?.find(v => v.key === 'completed')?.count || 0;
      
        // Calculate percentages (avoid division by zero)
        const paymentPercent = totalSeekers > 0 ? Math.round((paymentCompleted / totalSeekers) * 100) : 0;
        const invoicePercent = paymentCompleted > 0 ? Math.round((invoiceCompleted / paymentCompleted) * 100) : 0;
        const travelPercent = totalSeekers > 0 ? Math.round((travelCompleted / totalSeekers) * 100) : 0;
      
        const subProgramValues = {
          displayName: 'Sub program',
          key: 'subProgramRegistrations',
          values: [
            {
              displayName: 'All',
              key: 'totalSeekers',
              count: totalSeekers,
              order: 1,
            },
            {
              displayName: `Payment Pending`,
              key: 'paymentPending',
              count: paymentCompleted,
              totalCount: totalSeekers,
              pendingCount: totalSeekers - paymentCompleted,
              percentCompleted: paymentPercent,
              order: 2,
            },
            {
              displayName: `Invoice Pending`,
              key: 'invoicePending',
              count: invoiceCompleted,
              totalCount: paymentCompleted,
              pendingCount: paymentCompleted - invoiceCompleted,
              percentCompleted: invoicePercent,
              order: 3,
            },
            {
              displayName: `Travel Pending`,
              key: 'travelPending',
              count: travelCompleted,
              totalCount: totalSeekers,
              pendingCount: totalSeekers - travelCompleted,
              percentCompleted: travelPercent,
              order: 4,
            },
          ],
        };
        data = [subProgramValues, ...data];
      }
      return data;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_DASHBOARD_DATA_GET_FAILED, error);
    }
  }

  public formatPaymentStatus(status: string, allocatedProgramId: number | null, isFreeSeat: boolean): string | null {
    switch (status) {
      case 'online_pending':
        return 'Online Pending';
      case 'offline_pending':
        return 'Offline Pending';
      case 'online_completed':
        return 'Online Completed';
      case 'offline_completed':
        return 'Offline Completed';
      case 'draft':
        return 'Draft';
      case 'save_as_draft':
        return '-';
      case 'payment_pending':
        return 'Pending';
      case 'failed':
        return 'Failed';
      case '':
        if (allocatedProgramId && !isFreeSeat) {
          return 'Pending';
        } else if (allocatedProgramId && isFreeSeat) {
          return 'No Payment';
        } else {
          return null;
        }
      default:
        return status;
    }
  }

  public formatInvoiceStatus(status: string, allocatedProgramId: number | null, isFreeSeat: boolean, paymentStatus: string | null): string {
    const isPaymentCompleted = paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED || paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED;
    switch (status) {
      case 'invoice_pending':
        return 'Pending';
      case 'invoice_completed':
        return 'Completed';
      case 'draft':
        if (isPaymentCompleted) {
          return 'Pending';
        }
        return '-';
      case 'save_as_draft':
        return '-';
      case 'invoice_failed':
        return 'Failed';
      case 'failed':
        return 'Failed';
      case '':

        if (allocatedProgramId && !isFreeSeat && isPaymentCompleted) {
          return 'Pending';
        } else if (allocatedProgramId && isFreeSeat) {
          return 'No Payment';
        } else {
          return '-';
        }
      default:
        return status;
    }
  }

  public formatTravelStatus(status: string, allocatedProgramId: number | null): string | null {
    switch (status) {
      case 'travel_pending':
        return 'Pending';
      case 'travel_completed':
        return 'Completed';
      case 'pending':
        return 'Pending';
      case 'completed':
        return 'Completed';
      case 'save_as_draft':
        return '-';
      case 'draft':
        return 'Draft';
      case '':
        if (allocatedProgramId) {
          return 'Pending';
        } else {
          return null;
        }
      default:
        return status;
    }
  }

  /**
   * Combine experiences from all sources with priority to RM-updated experiences
   * @param user - User object with relations loaded
   * @returns Formatted experiences string
   */
  private formatUserExperiences(user: any): string {
    if (!user) return '';

    const experiences: string[] = [];

    // 1. Add RM-updated experiences from UserProgramExperience (non-deleted, has createdBy or updatedBy)
    if (user.programExperiences && user.programExperiences.length > 0) {
      const rmUpdatedExperiences = user.programExperiences
        .filter(exp => exp.deletedAt === null)
        .map(exp => exp.lookupData?.lookupLabel || '')
        .filter(Boolean);
      
      experiences.push(...rmUpdatedExperiences);
    }

    // 2. Add migrated experiences from UserParticipationSummary
    if (user.participationSummary && user.participationSummary.length > 0) {
      const migratedExperiences = user.participationSummary
        .filter(summary => summary.deletedAt === null && summary.programName)
        .map(summary => summary.programName as string)
        .filter(Boolean);
      
      // Add to experiences array
      experiences.push(...migratedExperiences);
    }

    // 3. Add profile extension experiences
    if (user.profileExtension?.whichOfTheFollowingHaveYouExperienced) {
      const profileExperiences = user.profileExtension.whichOfTheFollowingHaveYouExperienced
        .split(',')
        .map((exp: string) => exp.trim())
        .filter(Boolean);
      
       experiences.push(...profileExperiences);
    }
    
    // Remove duplicates and join
    const uniqueExperiences = [...new Set(experiences)];

    // sort alphabetically for sort check for lower variance
    uniqueExperiences.sort((a, b) => a.localeCompare(b));

    return uniqueExperiences.join(', ');
  }

  /**
   * Get signed URL for report generation
   * Returns signed URL if available, otherwise returns plain URL
   * @param plainUrl - Plain S3 URL
   * @param signedUrl - Pre-generated signed URL (if available)
   * @returns Signed URL or empty string
   */
  private getSignedUrlForReport(plainUrl: string | null | undefined, signedUrl: string | null | undefined): string {
    // If no URL exists, return empty string
    if (!plainUrl) {
      return '';
    }

    if (shouldExcludeFromSignedUrl(plainUrl)) {
      return plainUrl;
    }
    // If signed URL exists, return it (it should be fresh from database)
    if (signedUrl) {
      return signedUrl;
    }

    return plainUrl;
  
  }

  async remove(id: number, user: User) {
    this.logger.log(registrationMessages.DELETE_REQUEST_RECEIVED, { id });
    try {
      const result = await this.dataSource.transaction(async (manager) => {
        return await this.repository.softDeleteRegistration(id, user.id, manager);
      });
      return result;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_DELETE_FAILED, error);
    }
  }

  /**
   * Calculate the total count from a single level of nested KPI structures.
   * Each item contains a `values` array where the actual counts reside.
   * @param items - Array of KPI items with nested `values`
   * @returns Aggregated total count of all nested values
   */
  private calculateNestedCount(items: any[]): number {
    if (!Array.isArray(items)) {
      return 0;
    }

    return items.reduce((sum, item) => {
      const nestedValues = Array.isArray(item.values)
        ? item.values.reduce(
            (nestedSum, nestedItem) =>
              nestedSum + Number(nestedItem.count || 0),
            0,
          )
        : 0;

      return sum + nestedValues;
    }, 0);
  }

  /**
   * 
   * @param string - The input string to capitalize
   * @description Capitalizes the first letter of the given string.
   * If the string is empty or not a string, it returns an empty string.
   * If the string is only whitespace, it returns an empty string.
   * @returns The string with the first letter capitalized
   */
  public capitalizeFirstLetter(string: string): string {
    if (!string || typeof string !== 'string') return '';
    const trimmed = string.trim();
    if (trimmed === '') return '';
    return string.charAt(0).toUpperCase() + string.slice(1);
  }

  /**
   * Common function to get default sort parameters based on KPI context
   * Used across list views, seat allocation, and reports for consistent sorting behavior
   * @param currentSortKey - The current sort key being used
   * @param currentSortOrder - The current sort order being used
   * @param filters - The filter object that may contain kpiCategory and kpiFilter
   * @param forceApplyDefaults - If true, skips current sort key check and always applies defaults (used by reports)
   * @returns Object with sortKey and sortOrder for default sorting
   */
  private getDefaultSortParameters(currentSortKey: string, currentSortOrder: 'ASC' | 'DESC', filters: Record<string, any>, forceApplyDefaults = false): { sortKey: string; sortOrder: 'ASC' | 'DESC' } {
    const kpiCategory = filters?.kpiCategory;
    const kpiFilter = filters?.kpiFilter;
    
    // Only apply default sorting if no specific sort key is provided or using DEFAULT_SORT_KEY (skip for reports)
    if (!forceApplyDefaults && currentSortKey && currentSortKey !== RegistrationSortKey.DEFAULT_SORT_KEY) {
      return { sortKey: currentSortKey, sortOrder: currentSortOrder }; // Return current sort key and order as-is
    }
    
    // Part 1: Check for blessed date contexts
    const shouldUseBlessedDateSort = Boolean(
      (kpiCategory === KPI_CATEGORIES.REGISTRATIONS && kpiFilter === KPI_FILTERS.APPROVED_SEEKERS) ||
      (kpiCategory === KPI_CATEGORIES.PROGRAMS && typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')) ||
      (kpiCategory === KPI_CATEGORIES.BLESSED && ([KPI_FILTERS.APPROVED_SEEKERS].includes(kpiFilter || '') || (typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')))) ||
      (typeof kpiCategory === 'string' && kpiCategory.startsWith('program_')) ||
      (kpiCategory === KPI_CATEGORIES.ALLOCATED && kpiFilter === KPI_FILTERS.BLESSED) ||
      (kpiCategory === KPI_CATEGORIES.ALLOCATED && typeof kpiFilter === 'string' && kpiFilter.startsWith('program_'))
    );
    
    if (shouldUseBlessedDateSort) {
      return {
        sortKey: RegistrationSortKey.BLESSED_DATE,
        sortOrder: SORT_ORDER.DESC
      };
    }
    
    // Part 2: Handle status-specific sorting
    if (kpiCategory === KPI_CATEGORIES.REGISTRATIONS) {
      switch (kpiFilter) {
        case KPI_FILTERS.CANCELLED:
          return {
            sortKey: RegistrationSortKey.CANCELLED_DATE,
            sortOrder: SORT_ORDER.DESC
          };
        case KPI_FILTERS.REJECTED:
          return {
            sortKey: RegistrationSortKey.HOLD_DATE,
            sortOrder: SORT_ORDER.DESC
          };
        case KPI_FILTERS.ON_HOLD:
          return {
            sortKey: RegistrationSortKey.SWAP_DEMAND_DATE,
            sortOrder: SORT_ORDER.DESC
          };
        case KPI_FILTERS.NEW_SEEKERS_PENDING:
          return {
            sortKey: RegistrationSortKey.REGISTRATION_DATE,
            sortOrder: SORT_ORDER.DESC
          };
        case KPI_FILTERS.REG_PENDING:
          return {
            sortKey: RegistrationSortKey.PENDING_DATE,
            sortOrder: SORT_ORDER.DESC
          };
      }
    }

    if (kpiCategory === KPI_CATEGORIES.BLESSED) {
      switch (kpiFilter) {
        case KPI_FILTERS.SWAP_REQUESTS:
          return {
            sortKey: RegistrationSortKey.SWAP_REQUEST_DATE,
            sortOrder: SORT_ORDER.DESC
          };
      }
    }

    if (kpiCategory === KPI_CATEGORIES.SWAP_DEMAND) {
      return {
        sortKey: RegistrationSortKey.SWAP_DEMAND_DATE,
        sortOrder: SORT_ORDER.DESC
      };
    }

    if (kpiCategory === KPI_CATEGORIES.SWAP_REQUESTS) {
      return {
        sortKey: RegistrationSortKey.SWAP_REQUEST_DATE,
        sortOrder: SORT_ORDER.DESC
      };
    }

    if (kpiCategory === KPI_CATEGORIES.UNALLOCATED) {
      return {  
        sortKey: RegistrationSortKey.REGISTRATION_DATE,
        sortOrder: SORT_ORDER.DESC
      };
    }

    if (kpiCategory === KPI_CATEGORIES.ALLOCATED) {
       switch (kpiFilter) {
         case KPI_FILTERS.REJECT:
            return {  
              sortKey: RegistrationSortKey.HOLD_DATE,
              sortOrder: SORT_ORDER.DESC
            };
          case KPI_FILTERS.HOLD:
            return {
              sortKey: RegistrationSortKey.SWAP_DEMAND_DATE,
              sortOrder: SORT_ORDER.DESC
            };
       }
    }

    if (kpiCategory === KPI_CATEGORIES.ALL) {
      switch (kpiFilter) {
        case KPI_FILTERS.SWAP_REQUESTS:
          return {
            sortKey: RegistrationSortKey.SWAP_REQUEST_DATE,
            sortOrder: SORT_ORDER.DESC
          };
        case KPI_FILTERS.SWAP_REQUEST_HYPEN:
          return {
            sortKey: RegistrationSortKey.SWAP_REQUEST_DATE,
            sortOrder: SORT_ORDER.DESC
          };
      }
    }

    if (kpiCategory === KPI_CATEGORIES.CANCELLED) {
      return {
        sortKey: RegistrationSortKey.CANCELLED_DATE,
        sortOrder: SORT_ORDER.DESC
      };
    }

    if (kpiCategory === KPI_CATEGORIES.REG_PENDING || (kpiFilter === KPI_FILTERS.REG_PENDING)) {
      return {
        sortKey: RegistrationSortKey.PENDING_DATE,
        sortOrder: SORT_ORDER.DESC
      };
    }

    // Default: registration date with ASC order (earliest registrations first)
    return {
      sortKey: RegistrationSortKey.REGISTRATION_DATE,
      sortOrder: SORT_ORDER.DESC
    };
  }

  /**
   * Determine appropriate sort parameters for reports based on KPI context
   * Reuses getDefaultSortParameters logic with forceApplyDefaults flag
   * @param filters - The filter object that may contain kpiCategory and kpiFilter
   * @returns Object with sortKey and sortOrder for the report
   */
  private getReportSortParameters(filters: Record<string, any>): { sortKey: string; sortOrder: 'ASC' | 'DESC' } {
    // Use getDefaultSortParameters with forceApplyDefaults=true to skip current sort key checks
    return this.getDefaultSortParameters('', 'ASC', filters, true);
  }

  /**
   * Add status-specific date headers to table configuration based on KPI context
   * @param transformedData - The transformed data object containing tableHeaders
   * @param kpiCategory - The KPI category
   * @param kpiFilter - The KPI filter
   * @param parentFilter - The parent filter
   */
  private addStatusDateHeaders(transformedData: any, kpiCategory?: string, kpiFilter?: string, parentFilter?: string): void {
    if (!transformedData.tableHeaders) {
      return;
    }

    // Part 1: Blessed KPIs and Queues - Add blessed date column with descending sort
    // Check for specific kpiCategory and kpiFilter combinations that should show blessed date
    const shouldShowBlessedDate = Boolean(
      (kpiCategory === KPI_CATEGORIES.REGISTRATIONS && kpiFilter === KPI_FILTERS.APPROVED_SEEKERS) ||
      (kpiCategory === KPI_CATEGORIES.PROGRAMS && typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')) ||
      (kpiCategory === KPI_CATEGORIES.BLESSED && (kpiFilter === KPI_FILTERS.APPROVED_SEEKERS || (typeof kpiFilter === 'string' && kpiFilter.startsWith('program_')))) ||
      (typeof kpiCategory === 'string' && kpiCategory.startsWith('program_'))
    );
    
    if (shouldShowBlessedDate) {
      transformedData.tableHeaders.push({
        key: 'blessedDate',
        label: 'Blessed Date',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '12'
        });
    }

    // Part 2: Specific KPIs and queues with status-specific dates
    if (kpiCategory === KPI_CATEGORIES.REGISTRATIONS) {
      switch (kpiFilter) {
        case KPI_FILTERS.CANCELLED:
          transformedData.tableHeaders.push({
            key: 'cancelledDate',
            label: 'Cancelled Date',
            sortable: true,
            filterable: false,
            type: 'string',
            order: '12'
          });
          break;
        case KPI_FILTERS.REJECTED:
          transformedData.tableHeaders.push({
            key: 'holdDate',
            label: 'Hold Date',
            sortable: true,
            filterable: false,
            type: 'string',
            order: '12'
          });
          break;
        case KPI_FILTERS.ON_HOLD:
          transformedData.tableHeaders.push({
            key: 'swapDemandDate',
            label: 'Swap Demand Date',
            sortable: true,
            filterable: false,
            type: 'string',
            order: '12'
          });
          break;
        case KPI_FILTERS.NEW_SEEKERS_PENDING:
          transformedData.tableHeaders.push({
            key: 'registrationDate',
            label: 'Registration Date',
            sortable: true,
            filterable: false,
            type: 'string',
            order: '12'
          });
          break;
      }
    }

    if (kpiCategory === KPI_CATEGORIES.BLESSED) {
      switch (kpiFilter) {
        case KPI_FILTERS.SWAP_REQUESTS:
          transformedData.tableHeaders.push({
            key: 'swapRequestDate',
            label: 'Swap Request Date',
            sortable: true,
            filterable: false,
            type: 'string',
            order: '12'
          });
          break;
      }
    }

    if (kpiCategory === KPI_CATEGORIES.SWAP_DEMAND) {
      transformedData.tableHeaders.push({
        key: 'swapDemandDate',
        label: 'Swap Demand Date',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '12'
      });
    }

    if (kpiCategory === KPI_CATEGORIES.SWAP_REQUESTS) {
      transformedData.tableHeaders.push({
        key: 'swapRequestDate',
        label: 'Swap Request Date',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '12'
      });
    }

    if (kpiCategory === KPI_CATEGORIES.REG_PENDING || (kpiFilter === KPI_FILTERS.REG_PENDING)) {
      transformedData.tableHeaders.push({
        key: 'pendingDate',
        label: 'Last Status Modified Date',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '12'
      });
    }

    // Special handling for "all" category with "all" filter - show status date time
    if (kpiCategory === KPI_CATEGORIES.ALL && kpiFilter === KPI_FILTERS.ALL) {
      transformedData.tableHeaders.push({
        key: 'registrationDate',
        label: 'Registration Date',
        sortable: true,
        filterable: false,
        type: 'string',
        order: '12'
      });
    }

    // Sort headers by order after adding new ones
    transformedData.tableHeaders.sort((a, b) => a.order.localeCompare(b.order));
  }



  /**
   * Capitalizes the first letter of every word in a string
   * @param str - The input string to capitalize
   * @returns The string with first letter of each word capitalized
   */
  public capitalizeWords(str: string): string {
    if (!str || typeof str !== 'string') return '';
    const trimmed = str.trim();
    if (trimmed === '') return '';
    return str
      .split(' ')
      .map((word) => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
      .join(' ');
  }

  /**
   * Helper method to safely extract song preference from JSON string or object
   * @param songPreferences - The song preferences data (could be JSON string or object)
   * @param key - The key to extract (song_1, song_2, song_3)
   * @returns The song preference value or empty string
   */
  private getSongPreference(songPreferences: any, key: string): string {
    if (!songPreferences) {
      return '';
    }

    try {
      // If it's already an object, use it directly
      if (typeof songPreferences === 'object' && songPreferences !== null) {
        return songPreferences[key] || '';
      }

      // If it's a string, try to parse it as JSON
      if (typeof songPreferences === 'string') {
        const parsed = JSON.parse(songPreferences);
        return parsed[key] || '';
      }

      return '';
    } catch (error) {
      // If JSON parsing fails, return empty string
      this.logger.warn(`Failed to parse song preferences: ${songPreferences}`, error);
      return '';
    }
  }

  /**
   * Store or update question answers for a registration using existing repository logic
   * @param manager - Active entity manager managing the transaction
   * @param registrationId - Program registration ID
   * @param answers - Array of question/answer mappings
   * @param userId - User performing the update
   */
  public async storeQuestionAnswers(
    manager: any,
    registrationId: number,
    answers: RegistrationAnswerDto[],
    userId: number,
  ): Promise<void> {
    const qaMappings = answers.map((a) => ({
      questionId: a.questionId,
      answerValue: a.answer,
    }));
    

    await this.repository.upsertQuestionAnswers(
      manager,
      registrationId,
      qaMappings,
      userId,
    );
    this.logger.log(
      `Stored question answers for registration ${registrationId}: ${JSON.stringify(qaMappings)}`
    );

  }

  /**
   * Get save-as-draft registrations with optional search and filters
   */
  async getSaveDraftRegistrations(
    limit: number,
    offset: number,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles: string[],
    filterRegistrationsByUserId?: number | null,
    sortKey: string = 'id',
    sortOrder: 'ASC' | 'DESC' = 'ASC',
  ) {
    try {
      const isRMUser = userRoles.includes('relational_manager');
      return await this.repository.findSaveDraftRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText,
        parsedFilters,
        isRMUser ? filterRegistrationsByUserId : undefined,
        sortKey,
        sortOrder,
      );
    } catch (error) {
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Get role-based report options for filter-config API
   * @param userRoles - Array of user roles
   * @returns Report options allowed for the user's roles
   */
  private getRoleBasedReportOptions(userRoles: string[]): any {
    const allowedReports: { code: string; label: string, description: string }[] = [];

    // Check each report against the access matrix
    Object.keys(REPORT_ACCESS_MATRIX).forEach(reportCode => {
      // Exclude drafts_report from filter-config API response
      if (reportCode === 'drafts_report') {
        return;
      }
      
      const allowedRoles = REPORT_ACCESS_MATRIX[reportCode];
      const hasAccess = userRoles.some(role => allowedRoles.includes(role));
      
      if (hasAccess && REPORT_DEFINITIONS[reportCode]) {
        allowedReports.push({
          code: reportCode,
          label: REPORT_DEFINITIONS[reportCode].label,
          description: REPORT_DEFINITIONS[reportCode].description,
        });
      }
    });

    return {
      type: 'select',
      label: 'Download Reports',
      options: allowedReports,
    };
  }

  /**
   * Check if user has access to a specific report
   * @param userRoles - Array of user roles
   * @param reportCode - Report code to check access for
   * @returns boolean indicating access permission
   */
  private hasReportAccess(userRoles: string[], reportCode: string): boolean {
    const allowedRoles = REPORT_ACCESS_MATRIX[reportCode];
    if (!allowedRoles) return false;
    
    return userRoles.some(role => allowedRoles.includes(role));
  }

  /**
   * Validate report access for API endpoint
   * @param userRoles - Array of user roles
   * @param reportCode - Report code to validate
   * @returns Promise<boolean> indicating access permission
   */
  async validateReportAccess(userRoles: string[], reportCode: string): Promise<boolean> {
    return this.hasReportAccess(userRoles, reportCode);
  }

  /**
   * Generate report based on report code and filters
   * @param reportCode - Code of the report to generate
   * @param options - Report generation options
   * @returns Promise<string> containing the S3 URL
   */
  async generateReport(reportCode: string, options: {
    programId?: number;
    programSessionId?: number;
    searchText?: string;
    filters?: any;
    userRoles: string[];
    filterRegistrationsByUserId?: number | null;
  }): Promise<string> {
    // Validate report definition exists
    const reportDef = REPORT_DEFINITIONS[reportCode];
    if (!reportDef) {
      throw new InifniBadRequestException(ERROR_CODES.INVALID_REPORT_CODE, null, null, `Invalid report code: ${reportCode}`);
    }

    // Validate user access
    if (!this.hasReportAccess(options.userRoles, reportCode)) {
      throw new InifniBadRequestException(ERROR_CODES.ACCESS_DENIED_TO_REPORT, null, null, 'Access denied to this report');
    }

    try {
      // Get data based on report type
      let data: any[] = [];
      
      switch (reportCode) {
        case 'all_seekers':
          data = await this.getSeekersReportData(options);
          break;
        case 'registration_form_report':
          data = await this.getRegistrationFormReportData(options);
          break;
        case 'swap_can_shift_report':
          data = await this.getSwapCanShiftReportData(options);
          break;
        case 'drafts_report':
          data = await this.getDraftsReportData(options);
          break;
        case 'finance_report':
          data = await this.getFinanceReportData(options);
          break;
        case 'travel_report':
          data = await this.getTravelReportData(options);
          break;
        // case 'master_list':
        //   data = await this.getMasterListReportData(options);
        //   break;
        case 'master_travel_and_goodies_report':
          data = await this.getTravelListReportData(options);
          break;
        case 'travel_onward_and_return_report':
          data = await this.getTravelOnwardAndReturnReportData(options);
          break;
        case 'goodies_and_ratria_pillar_report':
          data = await this.getGoodiesAndRatriaPillarReportData(options);
          break;
        case 'travel_onward_report':
          data = await this.getTravelOnwardReportData(options);
          break;
        case 'travel_return_report':
          data = await this.getTravelReturnReportData(options);
          break;
        case 'payment_list':
          data = await this.getPaymentListReportData(options);
          break;
        case 'birthday_list':
          data = await this.getBirthdayListReportData(options);
          break;
        case 'song_preferences_list':
          data = await this.getSongPreferencesListReportData(options);
          break;
        case 'hold_list':
          data = await this.getHoldListReportData(options);
          break;
        case 'confirmed_list':
          data = await this.getConfirmedListReportData(options);
          break;
        default:
          throw new InifniBadRequestException(ERROR_CODES.INVALID_REPORT_CODE, null, null, `Unsupported report code: ${reportCode}`);
      }

      // Check if data is empty
      if (!data || data.length === 0) {
        throw new InifniBadRequestException(
          ERROR_CODES.NO_DATA_FOUND_FOR_REPORT, 
          null, 
          null, 
          `No data found for ${reportDef.label} report with the current filters.`
        );
      }

      // Generate Excel file and upload to S3
      const filename = `${reportCode}_report_${new Date().getTime()}`;
      const excelOptions = {
        sheetName: reportDef.label,
        makeHeaderBold: true,
        autoWidth: true
      };
      
      const s3Url = await this.excelService.jsonToExcelAndUpload(
        data,
        filename,
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        excelOptions
      );
      
      return s3Url;
    } catch (error) {
      this.logger.error(`Error generating report ${reportCode}:`, error);
      throw error;
    }
  }

  /**
   * Get data for All Seekers report
   */
  private async getSeekersReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'approvals',
      'allocatedProgram',
      'user'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'city',
      'otherCityName',
      'countryName',
      'noOfHDBs',
      'lastHdbAttended',
      'registrationDate',
      'otherInfinitheismContact'
    ];

    const relationSelect: Record<string, string[]> = {
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      allocatedProgram: ['id', 'name'],
      user: ['id', 'createdAt']
    };
    
    // Determine appropriate sort parameters based on KPI context
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      undefined,
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const approval = reg.approvals?.[0];
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;

      return {
        'Name': reg.fullName,
        'Email': reg.emailAddress,
        'Mobile': reg.mobileNumber,
        'Gender': reg.gender,
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                  ? (reg?.otherCityName ?? '')
                  : (reg?.city ?? ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'User created on': (reg.user?.createdAt) ? formatDateTimeIST(reg.user?.createdAt) : '',
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Registration Status': this.reportFormatting(approval.approvalStatus, reg.allocatedProgram, 'approval'),
        'Date of blessing': blessedDate ? formatDateTimeIST(blessedDate) : '',
      };
    });
  }

  /**
   * Get data for Registration Form report
   */
  private async getRegistrationFormReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'allocatedProgram',
      'rmContactUser', 
      'paymentDetails',
      'travelPlans',
      'travelInfo',
      'preferences',
      'preferences.preferredProgram',
      'approvals',
      'goodies',
      'recommendation',
      'swapsRequests',
      'user',
      'user.profileExtension',
      'user.programExperiences',
      'user.programExperiences.lookupData',
      'user.participationSummary'
    ];

    // Define only the fields needed for this report
    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'city',
      'otherCityName',
      'countryName',
      'noOfHDBs',
      'lastHdbAttended',
      'registrationDate',
      'otherInfinitheismContact',
      'preferredRoomMate',
      'dob',
      'averageRating',
      'isFreeSeat'
    ];

    const relationSelect: Record<string, string[]> = {
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt', 'code'],
      rmContactUser: ['id', 'orgUsrName'],
      paymentDetails: ['id', 'paymentStatus', 'paymentMode', 'offlineMeta'],
      travelPlans: ['id', 'travelPlanStatus'],
      travelInfo: ['id', 'travelInfoStatus'],
      preferences: ['id', 'priorityOrder'],
      'preferences.preferredProgram': ['id', 'name'],
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      goodies: ['id', 'jacketSize', 'ratriaPillarLeonia'],
      recommendation: ['id', 'isRecommended', 'recommendationKey', 'recommendationText'],
      swapsRequests: ['id', 'type', 'status', 'createdAt'],
      user: ['id'],
      'user.profileExtension': ['id', 'hdbYear','whichOfTheFollowingHaveYouExperienced'],
      'user.programExperiences': ['id', 'deletedAt'],
      'user.programExperiences.lookupData': ['id', 'lookupLabel', 'lookupKey', 'lookupOrder'],
      'user.participationSummary': ['id', 'programName', 'subProgramName', 'deletedAt']
    };
    
    // Determine appropriate sort parameters based on KPI context
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const preferences = reg.preferences || [];
      const sortedPreferences = preferences.sort((a, b) => (a?.priorityOrder || 0) - (b?.priorityOrder || 0));
      const allChoices = sortedPreferences.map(p => p?.preferredProgram?.name).filter(Boolean).join(', ');
      const goodies = reg.goodies?.[0];
      const payment = reg.paymentDetails?.[0];
      const approval = reg.approvals?.[0];
      const recommendation = reg.recommendation?.[0];
      const swapRequest = reg.swapsRequests?.length > 0 
        ? reg.swapsRequests.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime())[0]
        : null;
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;

      return {
        'Seeker Name': reg.fullName,
        'Roommate preference': reg.preferredRoomMate || '',
        'HDB Allocation': reg.allocatedProgram?.name || '',
        'Infinitheism Contact': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'Payment Status': this.formatPaymentStatus(payment?.paymentStatus || '', reg.allocatedProgram?.id, reg.isFreeSeat) || '',
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
        'Offered': this.formatProgrammeNameWithPeriod('', reg?.allocatedProgram) || '',
        'Status': this.reportFormatting(approval.approvalStatus, reg.allocatedProgram, 'approval'),
        'Date of blessing': blessedDate ? formatDateTimeIST(blessedDate) : '',
        'Email': reg.emailAddress,
        'DOB': reg.dob ? formatDate(reg.dob) : '',
        'Mobile': reg.mobileNumber,
        'Gender': reg.gender,
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'Room-mate Preference': reg.preferredRoomMate || '',
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Choices': allChoices || 'Any HDB/MSD',
        'Experiences': reg?.user ? this.formatUserExperiences(reg.user) : '',
        'Payment method': payment?.paymentMode || '',
        'Jacket Size': goodies?.jacketSize || '',
        'Ratria Pillar in Leonia': formatBooleanValue(goodies?.ratriaPillarLeonia),
        'Registered on': reg.registrationDate ? formatDateTimeIST(reg.registrationDate) : '',
        'Swap Status': swapRequest ? formatSwapStatus(swapRequest.status) : '',
        'Wants Swap/Can Shift': swapRequest ? (swapRequest?.type === 'wants_swap' ? 'Wants swap' : 'Can shift') : '',
        'Following up': recommendation ? (recommendation?.isRecommended ? 'Yes' : 'No') : '',
        'Rating': reg?.averageRating || '',
        'Recommendation': recommendation ? recommendation?.recommendationKey : '',
        'Recommendation Comments': recommendation?.isRecommended ? recommendation?.recommendationText : '',
        'Payment From ID': payment?.offlineMeta?.razorpayPaymentId || '',
      };
    });
  }

  /**
   * Get data for Swap Can Shift report
   */
  private async getSwapCanShiftReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'allocatedProgram',
      'rmContactUser', 
      'paymentDetails',
      'travelPlans',
      'travelInfo',
      'preferences',
      'preferences.preferredProgram',
      'approvals',
      'goodies',
      'recommendation',
      'swapsRequests',
      'swapsRequests.swapRequestedProgram',
      'swapsRequests.swapRequestedProgram.program',
      'user',
      'user.profileExtension',
      'user.programExperiences',
      'user.programExperiences.lookupData',
      'user.participationSummary'
    ];

    // Define only the fields needed for this report
    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'city',
      'otherCityName',
      'countryName',
      'noOfHDBs',
      'lastHdbAttended',
      'registrationDate',
      'otherInfinitheismContact',
      'preferredRoomMate',
      'dob',
      'averageRating',
      'isFreeSeat'
    ];

    const relationSelect: Record<string, string[]> = {
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt', 'code'],
      rmContactUser: ['id', 'orgUsrName'],
      paymentDetails: ['id', 'paymentStatus', 'paymentMode', 'offlineMeta'],
      travelPlans: ['id', 'travelPlanStatus'],
      travelInfo: ['id', 'travelInfoStatus'],
      preferences: ['id', 'priorityOrder'],
      'preferences.preferredProgram': ['id', 'name'],
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      goodies: ['id', 'jacketSize', 'ratriaPillarLeonia'],
      recommendation: ['id', 'isRecommended', 'recommendationKey', 'recommendationText'],
      swapsRequests: ['id', 'type', 'status', 'createdAt', 'comment'],
      'swapsRequests.swapRequestedProgram': ['id'],
      'swapsRequests.swapRequestedProgram.program': ['id', 'name'],
      user: ['id'],
      'user.profileExtension': ['id', 'hdbYear','whichOfTheFollowingHaveYouExperienced'],
      'user.programExperiences': ['id', 'deletedAt'],
      'user.programExperiences.lookupData': ['id', 'lookupLabel', 'lookupKey', 'lookupOrder'],
      'user.participationSummary': ['id', 'programName', 'subProgramName', 'deletedAt']
    };
    
    let filters = this.extractReportFilters(options.filters || {});
    // Filter for active wants_swap and can_shift requests
    filters.swapRequests = [SwapType.WantsSwap, SwapType.CanShift];

    // Determine appropriate sort parameters based on KPI context
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      filters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const preferences = reg.preferences || [];
      const sortedPreferences = preferences.sort((a, b) => (a?.priorityOrder || 0) - (b?.priorityOrder || 0));
      const allChoices = sortedPreferences.map(p => p?.preferredProgram?.name).filter(Boolean).join(', ');
      const goodies = reg.goodies?.[0];
      const payment = reg.paymentDetails?.[0];
      const approval = reg.approvals?.[0];
      const recommendation = reg.recommendation?.[0];
      const swapRequest = reg.swapsRequests?.length > 0 
        ? reg.swapsRequests.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime())[0]
        : null;
      const swapRequestedPrograms = Array.isArray(swapRequest?.swapRequestedProgram)
        ? swapRequest.swapRequestedProgram
        .sort((a, b) => a.id - b.id)
        .map(sp => sp.program?.name)
        .join(', ')
        : '';
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;

      return {
        'Name': reg.fullName,
        'HDB Allocation': reg.allocatedProgram?.name || '',
        'Infinitheism Contact': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'Payment Status': this.formatPaymentStatus(payment?.paymentStatus || '', reg.allocatedProgram?.id, reg.isFreeSeat) || '',
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
        'Goodies Status': formatGoodiesStatus(goodies, reg.allocatedProgram?.id, approval?.approvalStatus),
        'Offered': this.formatProgrammeNameWithPeriod('', reg?.allocatedProgram) || '',
        'Swap Status': swapRequest ? formatSwapStatus(swapRequest.status) : '',
        'Wants Swap/Can Shift': swapRequest ? (swapRequest?.type === 'wants_swap' ? 'Wants swap' : 'Can shift') : '',
        'Following up': recommendation ? (recommendation?.isRecommended ? 'Yes' : 'No') : '',
        'Swap / Can shift request to': swapRequestedPrograms || '',
        'Swap / Can Shift RM Comments': swapRequest?.comment || '',
        'Swap Raised on': swapRequest ? formatDateTimeIST(swapRequest.createdAt) : '',
        'Status': this.reportFormatting(approval.approvalStatus, reg.allocatedProgram, 'approval'),
        'Date of blessing': blessedDate ? formatDateTimeIST(blessedDate) : '',
        'Email': reg.emailAddress,
        'DOB': reg.dob ? formatDate(reg.dob) : '',
        'Mobile': reg.mobileNumber,
        'Gender': reg.gender,
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'Room-mate Preference': reg.preferredRoomMate || '',
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Choices': allChoices || 'Any HDB/MSD',
        'Experiences': reg?.user ? this.formatUserExperiences(reg.user) : '',
        'Payment method': payment?.paymentMode || '',
        'Jacket Size': goodies?.jacketSize || '',
        'Ratria Pillar in Leonia': formatBooleanValue(goodies?.ratriaPillarLeonia),
        'Registered on': reg.registrationDate ? formatDateTimeIST(reg.registrationDate) : '',
        'Rating': reg?.averageRating || '',
        'Recommendation': recommendation ? recommendation?.recommendationKey : '',
        'Recommendation Comments': recommendation?.isRecommended ? recommendation?.recommendationText : '',
        'Payment From ID': payment?.offlineMeta?.razorpayPaymentId || '',
      };
    });
  }

  /**
   * Get data for Drafts report
   */
  private async getDraftsReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report  
    const requiredRelations = [
      'preferences',
      'preferences.preferredProgram',
      'rmContactUser',
      'user',
      'user.profileExtension',
      'user.programExperiences',
      'user.programExperiences.lookupData',
      'user.participationSummary'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'city',
      'otherCityName',
      'countryName',
      'registrationDate',
      'otherInfinitheismContact',
      'createdAt',
      'dob',
      'preferredRoomMate',
      'noOfHDBs',
      'lastHdbAttended',
      'registrationStatus'
    ];

    const relationSelect: Record<string, string[]> = {
      preferences: ['id', 'priorityOrder'],
      'preferences.preferredProgram': ['id', 'name'],
      rmContactUser: ['id', 'orgUsrName'],
      user: ['id'],
      'user.profileExtension': ['id', 'hdbYear', 'whichOfTheFollowingHaveYouExperienced'],
      'user.programExperiences': ['id', 'createdAt'],
      'user.programExperiences.lookupData': ['id', 'lookupLabel', 'lookupKey', 'lookupOrder'],
      'user.participationSummary': ['id', 'programName']
    };

    const reportFilters = this.extractReportFilters(options.filters || {});

    const result = await this.repository.findSaveDraftRegistrations(
      999999, // limit - get all for report
      0, // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      'id', // sortKey
      'ASC', // sortOrder
      requiredRelations, // Only fetch needed relations
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const preferences = reg.preferences || [];
      const sortedPreferences = preferences.sort((a, b) => (a?.priorityOrder || 0) - (b?.priorityOrder || 0));
      const allChoices = sortedPreferences.map(p => p?.preferredProgram?.name).filter(Boolean).join(', ');

      return {
        'Name': reg.fullName,
        'Infinitheism Contact': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'Status': this.reportFormatting(reg.registrationStatus, null, 'approval') || '',
        'Email': reg.emailAddress,
        'DOB': reg.dob ? formatDate(reg.dob.toString()) : '',
        'Mobile': reg.mobileNumber,
        'Gender': reg.gender,
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'Room-mate Preference': reg.preferredRoomMate || '',
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Choices': allChoices || 'Any HDB/MSD',
        'Experiences': reg?.user ? this.formatUserExperiences(reg.user) : '',
        'Drafted on': reg.createdAt ? formatDateTimeIST(reg.createdAt) : '',
      };
    });
  }

  /**
   * Get data for Finance report
   */
  private async getFinanceReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'paymentDetails',
      'invoiceDetails',
      'invoiceDetails.updatedBy',
      'approvals',
      'allocatedProgram',
      'rmContactUser',
      'program'
    ];

    // Define only the fields needed for this report
    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'city',
      'otherCityName',
      'countryName',
      'otherInfinitheismContact',
      'isFreeSeat'
    ];

    const relationSelect: Record<string, string[]> = {
      paymentDetails: ['id', 'paymentStatus', 'paymentMode', 'originalAmount', 'subTotal', 'tds', 'razorpayId', 'offlineMeta', 'paymentDate', 'markAsReceivedDate'],
      invoiceDetails: ['id', 'invoiceStatus', 'invoiceSequenceNumber', 'invoiceName', 'invoiceEmail', 'invoiceAddress', 'zip', 'gstNumber', 'tanNumber', 'einvoiceAckNumber', 'einvoiceAckDate', 'updatedAt'],
      'invoiceDetails.updatedBy': ['id', 'fullName', 'firstName', 'lastName'],
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt', 'code'],
      rmContactUser: ['id', 'orgUsrName'],
      program: ['id', 'name', 'code']
    };
    
    // Determine appropriate sort parameters based on KPI context
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      null, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map((reg, index) => {
      const payment = reg.paymentDetails?.[0];
      const invoice = reg.invoiceDetails?.[0];
      const approval = reg.approvals?.[0];
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;
      const city = (reg?.city?.trim().toLowerCase() === 'other') ? (reg?.otherCityName ?? '') : (reg?.city ?? '');
      
      return {
        'S.No.': index + 1,
        'Registration ID': reg.id,
        'Seeker Email': reg.emailAddress,
        'Seeker Name': reg.fullName,
        'HDB / MSD Offered': this.formatProgrammeNameWithPeriod('', reg?.allocatedProgram) || '',
        'Blessed': approval?.approvalStatus === 'approved' ? 'Yes' : 'No',
        'Mobile': reg.mobileNumber,
        'City': city,
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'Contact Person (RM)': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'Date of blessing': blessedDate ? formatDateTimeIST(blessedDate) : '',
        'Fee Amount': payment?.originalAmount,
        'GST 18%': payment?.originalAmount ? (payment?.originalAmount * 0.18) : 0,
        'Total amount': payment?.subTotal || 0,
        'TDS': payment?.tds || 0,
        'Amount Received': payment?.subTotal || 0,
        'GSTIN': invoice?.gstNumber || '',
        'TAN': invoice?.tanNumber || '',
        'Payment Status': this.formatPaymentStatus(payment?.paymentStatus || '', reg.allocatedProgram?.id, reg.isFreeSeat) || '',
        'Mode of payment': payment?.paymentMode === 'offline' ? payment?.offlineMeta?.paymentMode || '' : (payment?.paymentMode === 'online') ? 'Online' : '',
        'Order ID (Razor pay)': payment?.razorpayId || '',
        'Payment ID (Razor pay)': payment?.offlineMeta?.razorpayPaymentId || '',
        'Cheque/DD #': payment?.offlineMeta?.chequeNo || payment?.offlineMeta?.ddNumber || '',
        'Cheque/DD Date': (payment?.offlineMeta?.chequeDate || payment?.offlineMeta?.ddDate) ? formatDate(payment?.offlineMeta?.chequeDate || payment?.offlineMeta?.ddDate) : '',
        'NEFT/Bank Transfer Ref. #': payment?.offlineMeta?.utrNumber || payment?.offlineMeta?.bankTransferRefNumber || '',
        'NEFT / Bank Transfer Date': (payment?.offlineMeta?.neftDate || payment?.offlineMeta?.bankTransferDate) ? formatDate(payment?.offlineMeta?.neftDate || payment?.offlineMeta?.bankTransferDate) : '',
        'Payment link completion date': payment?.paymentDate ? formatDateTimeIST(payment?.paymentDate) : '',
        'Invoice Number': invoice?.invoiceSequenceNumber || '',
        'Invoice Date': payment?.paymentMode === PaymentModeEnum.OFFLINE ? formatDateIST(payment?.markAsReceivedDate?.toISOString() ?? '') : formatDateIST(payment?.paymentDate?.toISOString() ?? ''),
        'Invoice Name': invoice?.invoiceName,
        'Invoice Email': invoice?.invoiceEmail,
        'Invoice Address': invoice?.invoiceAddress || '',
        'Pincode': invoice?.zip || '',
        'Place of Supply': city || '',
        'GSTIN Status': invoice?.gstNumber ? 'Yes' : 'No',
        'e-Invoice Ack #': invoice?.einvoiceAckNumber || '',
        'Invoice Status': this.formatInvoiceStatus(invoice?.invoiceStatus ?? '', reg.allocatedProgram?.id, reg.isFreeSeat, payment?.paymentStatus),
        'e-Invoice Date': invoice?.einvoiceAckDate ? formatDateTimeIST(invoice?.einvoiceAckDate) : '',
        'Programe Name with period': this.formatProgrammeNameWithPeriod(reg?.program, reg?.allocatedProgram) || '',
        'Modified User': invoice?.updatedBy?.fullName ?? (invoice?.updatedBy?.firstName && invoice?.updatedBy?.lastName ? `${invoice?.updatedBy?.firstName} ${invoice?.updatedBy?.lastName}` : ''),
        'Modified Time': formatDateTimeIST(invoice?.updatedAt)
      };
    });
  }

  /**
   * Get data for Travel Report
   */
  private async getTravelReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'travelInfo',
      'travelPlans', 
      'goodies',
      'allocatedProgram',
      'approvals',
      'updatedBy'
    ];

    // Define only the fields needed for this report
    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'city',
      'otherCityName',
      'countryName',
      'noOfHDBs',
      'lastHdbAttended',
      'dob',
      'comments'
    ];

    const relationSelect: Record<string, string[]> = {
      travelInfo: ['id', 'idPictureUrl', 'idBackPictureUrl', 'internationalIdPictureUrl', 'visaCopyPictureUrl', 'passportCopyPictureUrl'],
      travelPlans: ['id', 'travelType', 'arrivalFrom', 'airlineName', 'flightNumber', 'arrivalDatetime', 'onwardAdditionalInfo', 'returnTravelType', 'departureTo', 'departureAirlineName', 'departureFlightNumber', 'departureDatetime', 'returnAdditionalInfo'],
      goodies: ['id', 'notebook', 'ratriaPillarLeonia', 'flask', 'jacketSize', 'tshirtSize'],
      allocatedProgram: ['id', 'name'],
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      updatedBy: ['id', 'fullName', 'firstName', 'lastName']
    };
    
    // Determine appropriate sort parameters based on KPI context
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      null, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const travelInfo = reg.travelInfo?.[0];
      const travelPlan = reg.travelPlans?.[0];
      const goodies = reg.goodies?.[0];
      const approval = reg.approvals?.[0];
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;
      
      return {
        'Name': reg.fullName,
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
        'HDB': reg.allocatedProgram?.name || '',
        'Email': reg.emailAddress,
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Mobile': reg.mobileNumber,
        'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
        'Gender': reg.gender,
        'Age': calculateAge(reg.dob),
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
        'Date of blessing': blessedDate ? formatDateTimeIST(blessedDate) : '',
        'Note Book': formatBooleanValue(goodies?.notebook),
        'Ratria Pillar in Leonia': formatBooleanValue(goodies?.ratriaPillarLeonia),
        'Flask': formatBooleanValue(goodies?.flask),
        'Jacket Size': goodies?.jacketSize || '',
        'T-Shirt Size': goodies?.tshirtSize || '',
        'Upload ID card (Front / Front+Back)': travelInfo?.idPictureUrl || '',
        'Upload ID card (only Back)': travelInfo?.idBackPictureUrl || '',
        'Pick-Up from': replaceUnderscoreWithSpace(travelPlan?.travelType) || '',
        'Coming From': travelPlan?.arrivalFrom || '',
        'Onward Airlines': travelPlan?.airlineName || '',
        'Onward Flight Number': travelPlan?.flightNumber || '',
        'Onward Arrival Time': travelPlan?.arrivalDatetime ? formatDateTimeIST(travelPlan.arrivalDatetime) : '',
        'Onward Additional Information': travelPlan?.onwardAdditionalInfo || '',
        'Drop to': replaceUnderscoreWithSpace(travelPlan?.returnTravelType) || '',
        'Going to': travelPlan?.departureTo || '',
        'Return Airlines': travelPlan?.departureAirlineName || '',
        'Return Flight Number': travelPlan?.departureFlightNumber || '',
        'Return Departure Time': travelPlan?.departureDatetime ? formatDateTimeIST(travelPlan.departureDatetime) : '',
        'Return Additional Information': travelPlan?.returnAdditionalInfo || '',
        'Uploaded International ID card': travelInfo?.internationalIdPictureUrl || '',
        'Upload VISA Copy': travelInfo?.visaCopyPictureUrl || '',
        'Upload Passport Copy': travelInfo?.passportCopyPictureUrl || '',
        'Modified User': reg.updatedBy?.fullName ?? (reg.updatedBy?.firstName && reg.updatedBy?.lastName ? `${reg.updatedBy?.firstName} ${reg.updatedBy?.lastName}` : ''),
        'Comments': reg.comments || '',
      };
    });
  }

  /**
   * Get data for Master List Report
   */
  private async getMasterListReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'approvals',
      'allocatedProgram',
      'preferences',
      'preferences.preferredProgram',
      'paymentDetails',
      'rmContactUser',
      'user',
      'updatedBy',
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'city',
      'otherCityName',
      'countryName',
      'dob',
      'noOfHDBs',
      'lastHdbAttended',
      'registrationDate',
      'otherInfinitheismContact',
      'registrationSeqNumber',
      'updatedAt',
      'isFreeSeat'
    ];

    const relationSelect: Record<string, string[]> = {
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      allocatedProgram: ['id', 'name'],
      preferences: ['id', 'priorityOrder'],
      'preferences.preferredProgram': ['id', 'name'],
      paymentDetails: ['id', 'paymentStatus'],
      rmContactUser: ['id', 'orgUsrName'],
      user: ['id', 'fullName'],
      updatedBy: ['id', 'fullName', 'firstName', 'lastName']
    };
    
    // Determine appropriate sort parameters based on KPI context
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const approval = reg.approvals?.[0];
      const preferences = reg.preferences || [];
      const sortedPreferences = preferences.sort((a, b) => (a?.priorityOrder || 0) - (b?.priorityOrder || 0));
      const allPreferences = sortedPreferences.map(p => p?.preferredProgram?.name).filter(Boolean).join(', ');
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;
      
      return {
        'Name': reg.fullName,
        'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
        'Age': calculateAge(reg.dob),
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'Infinitheism Contact': reg.rmContactUser.orgUsrName || '',
        'Other': (reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other') ? reg.otherInfinitheismContact : '',
        'Email': reg.emailAddress,
        'Mobile': reg.mobileNumber,
        'Registration Date': reg.registrationDate ? formatDateTimeIST(reg.registrationDate) : '',
        'HDB/MSD done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Offered': reg.allocatedProgram?.name || '',
        'All Preferences': allPreferences || 'Any HDB/MSD',
        'Registration Status': this.reportFormatting(approval.approvalStatus, reg.allocatedProgram, 'approval'),
        'Date of blessing': blessedDate ? formatDateTimeIST(blessedDate) : '',
        'Payment Status': this.formatPaymentStatus(reg.paymentDetails?.[0]?.paymentStatus || '', reg.allocatedProgram?.id, reg.isFreeSeat) || '',
        'Modified User': reg.updatedBy?.fullName ?? (reg.updatedBy?.firstName && reg.updatedBy?.lastName ? `${reg.updatedBy?.firstName} ${reg.updatedBy?.lastName}` : ''),
        'Last Modified Time': formatDateTimeIST(reg.updatedAt),
        'Registration ID': reg.id,
        'Loggedin-Seeker': reg.user?.fullName || '',
        'Auto Number': reg.registrationSeqNumber || '',
        'Gender': reg.gender,
        'Song 1': this.getSongPreference(reg.songPreferences, 'song_1'),
        'Song 2': this.getSongPreference(reg.songPreferences, 'song_2'),
        'Song 3': this.getSongPreference(reg.songPreferences, 'song_3'),
      };
    });
  }

  /**
   * Get data for Travel List Report
   */
  private async getTravelListReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'travelInfo',
      'travelPlans',
      'goodies',
      'allocatedProgram',
      'approvals',
      'rmContactUser'
    ];

    // Define only the fields needed for this report
    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'city',
      'otherCityName',
      'countryName',
      'noOfHDBs',
      'lastHdbAttended',
      'dob',
      'otherInfinitheismContact',
      'alternatePhoneNumber',
      'preferredRoomMate'
    ];

    const relationSelect: Record<string, string[]> = {
      travelInfo: ['id', 'travelInfoStatus', 'idPictureUrl', 'idPictureSignedUrl', 'idBackPictureUrl', 'idBackPictureSignedUrl', 'passportCopyPictureUrl', 'passportCopyPictureSignedUrl', 'visaCopyPictureUrl', 'visaCopyPictureSignedUrl', 'createdAt', 'updatedAt'],
      travelPlans: ['id', 'travelPlanStatus', 'travelType', 'arrivalFrom', 'otherArrival', 'onwardTerminal', 'airlineName', 'arrivalOtherAirlineName', 'flightNumber', 'arrivalDatetime', 'onwardAdditionalInfo', 'returnTravelType', 'departureTo', 'otherDeparture', 'returnTerminal', 'departureAirlineName', 'departureOtherAirlineName', 'departureFlightNumber', 'departureDatetime', 'returnAdditionalInfo', 'createdAt', 'updatedAt'],
      goodies: ['id', 'jacketSize', 'ratriaPillarLeonia', 'ratriaPillarLocation', 'ratriaPillarOtherLocation', 'flask', 'notebook', 'tshirt', 'tshirtSize', 'jacket', 'createdAt', 'updatedAt'],
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt', 'code'],
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      rmContactUser: ['id', 'orgUsrName']
    };
    
    // Determine appropriate sort parameters based on KPI context
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      null, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map((reg, index) => {
      const travelInfo = reg.travelInfo?.[0];
      const travelPlan = reg.travelPlans?.[0];
      const goodies = reg.goodies?.[0];
      const approval = reg.approvals?.[0];
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;      

      
      // Calculate program dates (start and end) instead of blessed date
      const programStartDate = reg?.allocatedProgram?.startsAt;
      const programEndDate = reg?.allocatedProgram?.endsAt;
      const programDatesString = programStartDate && programEndDate 
        ? `${reg.allocatedProgram?.name} - ${formatDateIST(programStartDate.toString())} to ${formatDateIST(programEndDate.toString())}`
        : (reg.allocatedProgram?.name || '');

      return {
        '#': index + 1,
        'Registration number/ID': reg.id,
        'Seeker Name': reg.fullName,
        'Roommate preference': reg.preferredRoomMate || '',
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
        'Goodies Status': formatGoodiesStatus(goodies, reg.allocatedProgram?.id, approval?.approvalStatus),
        'HDB/MSD': reg.allocatedProgram?.name || '',
        'HDB/MSD\'s Done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'HDB Allocation': programDatesString,
        'Blessed Date': blessedDate ? formatDateTimeIST(blessedDate) : '',
        'Email': reg.emailAddress,
        'Registered Mobile': reg.mobileNumber,
        'Gender': toTitleCase(reg.gender),
        'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
        'Age': calculateAge(reg.dob),
        'RM': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
          ? reg.otherCityName || ''
          : (reg?.city || ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName) : '',
        'Alternate Number': reg.alternatePhoneNumber || '',
        'Pick-Up From': formatTravelType(travelPlan?.travelType),
        'Coming From': formatLocation(travelPlan?.arrivalFrom, travelPlan?.otherArrival) || '',
        'Onward Terminal': toTitleCase(travelPlan?.onwardTerminal || ''),
        'Onward Airlines': toTitleCase(getAirlineFullName(travelPlan?.airlineName, travelPlan?.arrivalOtherAirlineName)),
        'Onward Flight Number': travelPlan?.flightNumber || '',
        'Onward Arrival Time': travelPlan?.arrivalDatetime ? formatDateTimeISTWithoutSec(travelPlan.arrivalDatetime) : '',
        'Onward Additional Information': travelPlan?.onwardAdditionalInfo || '',
        'Drop To': formatTravelType(travelPlan?.returnTravelType),
        'Going To': formatLocation(travelPlan?.departureTo, travelPlan?.otherDeparture) || '',
        'Return Terminal': toTitleCase(travelPlan?.returnTerminal || ''),
        'Return Airlines': toTitleCase(getAirlineFullName(travelPlan?.departureAirlineName, travelPlan?.departureOtherAirlineName)),
        'Return Flight Number': travelPlan?.departureFlightNumber || '',
        'Return Departure Time': travelPlan?.departureDatetime ? formatDateTimeISTWithoutSec(travelPlan.departureDatetime) : '',
        'Return Additional Information': travelPlan?.returnAdditionalInfo || '',
        'Ratria Pillar': (goodies?.ratriaPillarLeonia === undefined ? '' : formatRatriaPillar(goodies?.ratriaPillarLeonia, goodies?.ratriaPillarLocation, goodies?.ratriaPillarOtherLocation)),
        'Flask': formatBooleanValue(goodies?.flask),
        'Note Book': formatBooleanValue(goodies?.notebook),
        'T-Shirt Size': goodies?.tshirtSize === undefined ? '' : formatSizeSelection(goodies.tshirt, goodies.tshirtSize),
        'Jacket Size': goodies?.jacketSize === undefined ? '' : formatSizeSelection(goodies.jacket, goodies.jacketSize),
        'Added Time': getEarliestAddedDate(travelPlan, travelInfo, goodies),
        'Modified Time': getLatestModifiedDate(travelPlan, travelInfo, goodies),
        'Upload ID Card (Front / Front+Back)': this.getSignedUrlForReport(travelInfo?.idPictureUrl, travelInfo?.idPictureSignedUrl),
        'Upload ID Card (Only Back)': this.getSignedUrlForReport(travelInfo?.idBackPictureUrl, travelInfo?.idBackPictureSignedUrl),
        'Upload Passport Copy': this.getSignedUrlForReport(travelInfo?.passportCopyPictureUrl, travelInfo?.passportCopyPictureSignedUrl),
        'Upload VISA Copy': this.getSignedUrlForReport(travelInfo?.visaCopyPictureUrl, travelInfo?.visaCopyPictureSignedUrl)
      };
    });
  }

  /**
   * Get data for Travel - Onward and Return Report
   */
  private async getTravelOnwardAndReturnReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'travelInfo',
      'travelPlans',
      'allocatedProgram',
      'rmContactUser'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'dob',
      'lastHdbAttended',
      'city',
      'otherCityName',
      'alternatePhoneNumber',
      'otherInfinitheismContact',
      'preferredRoomMate'
    ];

    const relationSelect: Record<string, string[]> = {
      travelInfo: ['id', 'travelInfoStatus', 'idPictureUrl', 'idPictureSignedUrl', 'idBackPictureUrl', 'idBackPictureSignedUrl', 'passportCopyPictureUrl', 'passportCopyPictureSignedUrl', 'visaCopyPictureUrl', 'visaCopyPictureSignedUrl', 'createdAt', 'updatedAt'],
      travelPlans: ['id', 'travelPlanStatus', 'travelType', 'arrivalFrom', 'otherArrival', 'onwardTerminal', 'airlineName', 'arrivalOtherAirlineName', 'flightNumber', 'arrivalDatetime', 'onwardAdditionalInfo', 'returnTravelType', 'departureTo', 'otherDeparture', 'returnTerminal', 'departureAirlineName', 'departureOtherAirlineName', 'departureFlightNumber', 'departureDatetime', 'returnAdditionalInfo', 'createdAt', 'updatedAt'],
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt'],
      rmContactUser: ['id', 'orgUsrName']
    };
    
    // Determine appropriate sort parameters based on KPI context
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});

    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // Dynamic sort key based on context
      sortParams.sortOrder, // Dynamic sort order based on context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map((reg, index) => {
      const travelInfo = reg.travelInfo?.[0];
      const travelPlan = reg.travelPlans?.[0];
      const approval = reg.approvals?.[0];

      // Calculate program dates (start and end)
      const programStartDate = reg?.allocatedProgram?.startsAt;
      const programEndDate = reg?.allocatedProgram?.endsAt;
      const programDatesString = programStartDate && programEndDate 
        ? `${reg.allocatedProgram?.name} - ${formatDateIST(programStartDate.toString())} to ${formatDateIST(programEndDate.toString())}`
        : (reg.allocatedProgram?.name || '');

      return {
        '#': index + 1,
        'Registration number/ID': reg.id,
        'Seeker Name': reg.fullName,
        'Roommate preference': reg.preferredRoomMate || '',
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
        'HDB/MSD': reg.allocatedProgram?.name || '',
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'HDB Allocation': programDatesString,
        'Email': reg.emailAddress,
        'Registered Mobile': reg.mobileNumber,
        'Gender': toTitleCase(reg.gender),
        'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
        'Age': calculateAge(reg.dob),
        'RM': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
          ? reg.otherCityName || ''
          : (reg?.city || ''),
        'Alternate Number': reg.alternatePhoneNumber || '',
        'Pick-Up From': formatTravelType(travelPlan?.travelType),
        'Coming From': formatLocation(travelPlan?.arrivalFrom, travelPlan?.otherArrival) || '',
        'Onward Terminal': toTitleCase(travelPlan?.onwardTerminal || ''),
        'Onward Airlines': toTitleCase(getAirlineFullName(travelPlan?.airlineName, travelPlan?.arrivalOtherAirlineName)),
        'Onward Flight Number': travelPlan?.flightNumber || '',
        'Onward Arrival Time': travelPlan?.arrivalDatetime ? formatDateTimeISTWithoutSec(travelPlan.arrivalDatetime) : '',
        'Onward Additional Information': travelPlan?.onwardAdditionalInfo || '',
        'Drop To': formatTravelType(travelPlan?.returnTravelType),
        'Going To': formatLocation(travelPlan?.departureTo, travelPlan?.otherDeparture) || '',
        'Return Terminal': toTitleCase(travelPlan?.returnTerminal || ''),
        'Return Airlines': toTitleCase(getAirlineFullName(travelPlan?.departureAirlineName, travelPlan?.departureOtherAirlineName)),
        'Return Flight Number': travelPlan?.departureFlightNumber || '',
        'Return Departure Time': travelPlan?.departureDatetime ? formatDateTimeISTWithoutSec(travelPlan.departureDatetime) : '',
        'Return Additional Information': travelPlan?.returnAdditionalInfo || '',
        'Added Time': getEarliestAddedDate(travelPlan, travelInfo, null),
        'Modified Time': getLatestModifiedDate(travelPlan, travelInfo, null),
      };
    });
  }

  /**
   * Get data for Goodies and Ratria Pillar Report
   */
  private async getGoodiesAndRatriaPillarReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'goodies',
      'allocatedProgram',
      'approvals',
      'rmContactUser'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'noOfHDBs',
      'lastHdbAttended',
      'gender',
      'city',
      'otherCityName',
      'otherInfinitheismContact',
      // 'preferredRoomMate'
    ];

    const relationSelect: Record<string, string[]> = {
      goodies: ['id', 'ratriaPillarLeonia', 'ratriaPillarLocation', 'ratriaPillarOtherLocation', 'flask', 'notebook', 'tshirt', 'tshirtSize', 'jacket', 'jacketSize', 'createdAt', 'updatedAt'],
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt'],
      approvals: ['id', 'approvalStatus'],
      rmContactUser: ['id', 'orgUsrName']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map((reg, index) => {
      const goodies = reg.goodies?.[0];

      // Calculate program dates (start and end)
      const programStartDate = reg?.allocatedProgram?.startsAt;
      const programEndDate = reg?.allocatedProgram?.endsAt;
      const programDatesString = programStartDate && programEndDate 
        ? `${reg.allocatedProgram?.name} - ${formatDateIST(programStartDate.toString())} to ${formatDateIST(programEndDate.toString())}`
        : (reg.allocatedProgram?.name || '');

      return {
        '#': index + 1,
        'Registration number/ID': reg.id,
        'Seeker Name': reg.fullName,
        // 'Roommate preference': reg.preferredRoomMate || '',
        'Goodies Status': formatGoodiesStatus(goodies, reg.allocatedProgram?.id, reg.approvals?.[0]?.approvalStatus),
        'HDB/MSD done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'HDB Allocation': programDatesString,
        'Gender': toTitleCase(reg.gender),
        'RM': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
          ? reg.otherCityName || ''
          : (reg?.city || ''),
        'Ratria Pillar': (goodies?.ratriaPillarLeonia === undefined ? '' : formatRatriaPillar(goodies?.ratriaPillarLeonia, goodies?.ratriaPillarLocation, goodies?.ratriaPillarOtherLocation)),
        'Flask': formatBooleanValue(goodies?.flask),
        'Note Book': formatBooleanValue(goodies?.notebook),
        'T-Shirt Size': goodies?.tshirtSize === undefined ? '' : formatSizeSelection(goodies.tshirt, goodies.tshirtSize),
        'Jacket Size': goodies?.jacketSize === undefined ? '' : formatSizeSelection(goodies.jacket, goodies.jacketSize),
        'Added Time': goodies?.createdAt ? formatDateTimeIST(goodies.createdAt) : '',
        'Modified Time': goodies?.updatedAt ? formatDateTimeIST(goodies.updatedAt) : '',
      };
    });
  }

  /**
   * Get data for Travel - Onward Report
   */
  private async getTravelOnwardReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    const requiredRelations = [
      'travelInfo',
      'travelPlans',
      'allocatedProgram',
      'rmContactUser',
      'roomAllocations',
      'roomAllocations.programRoomInventoryMap',
      'roomAllocations.programRoomInventoryMap.room',
      'roomAllocations.programRoomInventoryMap.room.floor',
      'roomAllocations.programRoomInventoryMap.room.floor.block',
      'roomAllocations.programRoomInventoryMap.room.floor.block.venue'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'dob',
      'lastHdbAttended',
      'city',
      'otherCityName',
      'alternatePhoneNumber',
      'otherInfinitheismContact',
      'preferredRoomMate',
      'noOfHDBs'
    ];

    const relationSelect: Record<string, string[]> = {
      travelInfo: ['id', 'travelInfoStatus'],
      travelPlans: ['id', 'travelPlanStatus', 'travelType', 'arrivalFrom', 'otherArrival', 'onwardTerminal', 'airlineName', 'arrivalOtherAirlineName', 'flightNumber', 'arrivalDatetime', 'onwardAdditionalInfo', 'createdAt', 'updatedAt'],
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt'],
      rmContactUser: ['id', 'orgUsrName'],
      roomAllocations: ['id'],
      'roomAllocations.programRoomInventoryMap': ['id'],
      'roomAllocations.programRoomInventoryMap.room': ['id', 'roomNumber'],
      'roomAllocations.programRoomInventoryMap.room.floor': ['id', 'label'],
      'roomAllocations.programRoomInventoryMap.room.floor.block': ['id', 'label'],
      'roomAllocations.programRoomInventoryMap.room.floor.block.venue': ['id', 'label']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map((reg, index) => {
      const travelInfo = reg.travelInfo?.[0];
      const travelPlan = reg.travelPlans?.[0];

      // Calculate program dates (start and end)
      const programStartDate = reg?.allocatedProgram?.startsAt;
      const programEndDate = reg?.allocatedProgram?.endsAt;
      const programDatesString = programStartDate && programEndDate 
        ? `${reg.allocatedProgram?.name} - ${formatDateIST(programStartDate.toString())} to ${formatDateIST(programEndDate.toString())}`
        : (reg.allocatedProgram?.name || '');

      // Get room number from room allocation (if exists)
      // Note: This requires adding roomAllocation relation to the query
      const roomAllocation = reg?.roomAllocations?.[0]?.programRoomInventoryMap?.room;
      const roomNumber = roomAllocation?.roomNumber ?? '';
      const roomFloor = roomAllocation?.floor?.label ?? '';
      const roomBlock = roomAllocation?.floor?.block?.label ?? '';
      const roomVenue = roomAllocation?.floor?.block?.venue?.label ?? '';
      return {
        '#': index + 1,
        'Venue': roomVenue,
        'Block': roomBlock,
        'Floor': roomFloor,
        'Room #': roomNumber,
        'Registration number/ID': reg.id,
        'Seeker Name': reg.fullName,
        'Roommate preference': reg.preferredRoomMate || '',
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
        'HDB/MSD done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'HDB Allocation': programDatesString,
        'Email': reg.emailAddress,
        'Registered Mobile': reg.mobileNumber,
        'Gender': toTitleCase(reg.gender),
        'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
        'Age': calculateAge(reg.dob),
        'RM': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
          ? reg.otherCityName || ''
          : (reg?.city || ''),
        'Alternate Number': reg.alternatePhoneNumber || '',
        'Pick-Up From': formatTravelType(travelPlan?.travelType),
        'Coming From': formatLocation(travelPlan?.arrivalFrom, travelPlan?.otherArrival) || '',
        'Onward Terminal': toTitleCase(travelPlan?.onwardTerminal || ''),
        'Onward Airlines': toTitleCase(getAirlineFullName(travelPlan?.airlineName, travelPlan?.arrivalOtherAirlineName)),
        'Onward Flight Number': travelPlan?.flightNumber || '',
        'Onward Arrival Time': travelPlan?.arrivalDatetime ? formatDateTimeISTWithoutSec(travelPlan.arrivalDatetime) : '',
        'Onward Additional Information': travelPlan?.onwardAdditionalInfo || '',
        'Added Time': getEarliestAddedDate(travelPlan, travelInfo, null),
        'Modified Time': getLatestModifiedDate(travelPlan, travelInfo, null),
      };
    });
  }

  /**
   * Get data for Travel - Return Report
   */
  private async getTravelReturnReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    const requiredRelations = [
      'travelInfo',
      'travelPlans',
      'allocatedProgram',
      'rmContactUser',
      'roomAllocations',
      'roomAllocations.programRoomInventoryMap',
      'roomAllocations.programRoomInventoryMap.room',
      'roomAllocations.programRoomInventoryMap.room.floor',
      'roomAllocations.programRoomInventoryMap.room.floor.block',
      'roomAllocations.programRoomInventoryMap.room.floor.block.venue'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'dob',
      'lastHdbAttended',
      'city',
      'otherCityName',
      'alternatePhoneNumber',
      'otherInfinitheismContact',
      'preferredRoomMate'
    ];

    const relationSelect: Record<string, string[]> = {
      travelInfo: ['id', 'travelInfoStatus'],
      travelPlans: ['id', 'travelPlanStatus', 'returnTravelType', 'departureTo', 'otherDeparture', 'returnTerminal', 'departureAirlineName', 'departureOtherAirlineName', 'departureFlightNumber', 'departureDatetime', 'returnAdditionalInfo', 'createdAt', 'updatedAt'],
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt'],
      rmContactUser: ['id', 'orgUsrName'],
      roomAllocations: ['id'],
      'roomAllocations.programRoomInventoryMap': ['id'],
      'roomAllocations.programRoomInventoryMap.room': ['id', 'roomNumber'],
      'roomAllocations.programRoomInventoryMap.room.floor': ['id', 'label'],
      'roomAllocations.programRoomInventoryMap.room.floor.block': ['id', 'label'],
      'roomAllocations.programRoomInventoryMap.room.floor.block.venue': ['id', 'label']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map((reg, index) => {
      const travelInfo = reg.travelInfo?.[0];
      const travelPlan = reg.travelPlans?.[0];

      // Calculate program dates (start and end)
      const programStartDate = reg?.allocatedProgram?.startsAt;
      const programEndDate = reg?.allocatedProgram?.endsAt;
      const programDatesString = programStartDate && programEndDate 
        ? `${reg.allocatedProgram?.name} - ${formatDateIST(programStartDate.toString())} to ${formatDateIST(programEndDate.toString())}`
        : (reg.allocatedProgram?.name || '');

      // Get room number from room allocation (if exists)
      // Note: This requires adding roomAllocation relation to the query
      const roomAllocation = reg?.roomAllocations?.[0]?.programRoomInventoryMap?.room;
      const roomNumber = roomAllocation?.roomNumber ?? '';
      const roomFloor = roomAllocation?.floor?.label ?? '';
      const roomBlock = roomAllocation?.floor?.block?.label ?? '';
      const roomVenue = roomAllocation?.floor?.block?.venue?.label ?? '';

      return {
        '#': index + 1,
        'Venue': roomVenue,
        'Block': roomBlock,
        'Floor': roomFloor,
        'Room #': roomNumber,
        'Registration number/ID': reg.id,
        'Seeker Name': reg.fullName,
        'Roommate preference': reg.preferredRoomMate || '',
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'HDB Allocation': programDatesString,
        'Email': reg.emailAddress,
        'Registered Mobile': reg.mobileNumber,
        'Gender': toTitleCase(reg.gender),
        'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
        'Age': calculateAge(reg.dob),
        'RM': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
          ? reg.otherCityName || ''
          : (reg?.city || ''),
        'Alternate Number': reg.alternatePhoneNumber || '',
        'Drop To': formatTravelType(travelPlan?.returnTravelType),
        'Going To': formatLocation(travelPlan?.departureTo, travelPlan?.otherDeparture) || '',
        'Return Terminal': toTitleCase(travelPlan?.returnTerminal || ''),
        'Return Airlines': toTitleCase(getAirlineFullName(travelPlan?.departureAirlineName, travelPlan?.departureOtherAirlineName)),
        'Return Flight Number': travelPlan?.departureFlightNumber || '',
        'Return Departure Time': travelPlan?.departureDatetime ? formatDateTimeISTWithoutSec(travelPlan.departureDatetime) : '',
        'Return Additional Information': travelPlan?.returnAdditionalInfo || '',
        'Added Time': getEarliestAddedDate(travelPlan, travelInfo, null),
        'Modified Time': getLatestModifiedDate(travelPlan, travelInfo, null),
      };
    });
  }

  /**
   * Get data for Payment List Report
   */
  private async getPaymentListReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'paymentDetails',
      'approvals',
      'allocatedProgram',
      'rmContactUser',
      'program',
      'updatedBy'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'noOfHDBs',
      'lastHdbAttended',
      'otherInfinitheismContact',
      'updatedAt',
      'isFreeSeat'
    ];

    const relationSelect: Record<string, string[]> = {
      paymentDetails: ['id', 'paymentStatus', 'paymentMode', 'originalAmount', 'gstAmount', 'razorpayId', 'offlineMeta', 'webhookOnlinePaymentStatus', 'gatewayOnlinePaymentStatus', 'portalOnlinePaymentStatus'],
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      allocatedProgram: ['id', 'name', 'startsAt', 'endsAt'],
      rmContactUser: ['id', 'orgUsrName'],
      program: ['id', 'code'],
      updatedBy: ['id', 'fullName', 'firstName', 'lastName']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const payment = reg.paymentDetails?.[0];
      const approval = reg.approvals?.[0];
      const blessedDate = approval?.approvalStatus === ApprovalStatusEnum.APPROVED ? approval?.approvalDate : null;
      
      return {
        'Name': reg.fullName,
        'Email': reg.emailAddress,
        'Mobile': reg.mobileNumber,
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Contact Person': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'HDB Offered': this.formatProgrammeNameWithPeriod('', reg?.allocatedProgram) || '',
        'Date of blessing': blessedDate ? formatDateTimeIST(blessedDate) : '',
        'Pay method': payment?.paymentMode === 'offline' ? payment?.offlineMeta?.paymentMode || '' : (payment?.paymentMode === 'online') ? 'Online' : '',
        'Payment Status': this.formatPaymentStatus(payment?.paymentStatus || '', reg.allocatedProgram?.id, reg.isFreeSeat) || '',
        'RazorPay Status': payment?.webhookOnlinePaymentStatus || payment?.gatewayOnlinePaymentStatus || payment?.portalOnlinePaymentStatus ||'',
        'Investment': payment?.originalAmount || 0,
        'GST @ 18%': payment?.gstAmount || 0,
        'Batch Code': reg.program?.code || '',
        'Modified Time': reg.updatedAt ? formatDateTimeIST(reg.updatedAt) : '',
        'Modified User': reg.updatedBy?.fullName ?? (reg.updatedBy?.firstName && reg.updatedBy?.lastName ? `${reg.updatedBy?.firstName} ${reg.updatedBy?.lastName}` : ''),
        'Registration ID': reg.id,
        'RazorPay Order ID': payment?.razorpayId || '',
        'Razor pay Payment ID': payment?.offlineMeta?.razorpayPaymentId || ''
      };
    });
  }

  /**
   * Get data for Birthday List Report
   */
  private async getBirthdayListReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'allocatedProgram',
      'approvals',
      'rmContactUser'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'dob',
      'emailAddress',
      'gender',
      'mobileNumber',
      'city',
      'otherCityName',
      'countryName',
      'noOfHDBs',
      'lastHdbAttended',
      'otherInfinitheismContact'
    ];

    const relationSelect: Record<string, string[]> = {
      allocatedProgram: ['id', 'name'],
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      rmContactUser: ['id', 'orgUsrName']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => ({
      'HDB': reg.allocatedProgram?.name || '',
      'Name': reg.fullName ? reg.fullName.trim() : '',
      'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
      'Email': reg.emailAddress,
      'Gender': reg.gender,
      'Mobile': reg.mobileNumber,
      'Contact Person': reg.otherInfinitheismContact ?? (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
      'HDB/MSD\'s done': reg.noOfHDBs || 0,
      'Last HDB/MSD': reg.lastHdbAttended || '',
      'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
      'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
      'Date of blessing': reg.approvals?.[0]?.approvalStatus === ApprovalStatusEnum.APPROVED ? formatDateTimeIST(reg.approvals?.[0]?.approvalDate) : '',
    }));
  }

  /**
   * Get data for Song Preferences List Report
   */
  private async getSongPreferencesListReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'approvals',
      'rmContactUser'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'emailAddress',
      'mobileNumber',
      'gender',
      'dob',
      'city',
      'otherCityName',
      'noOfHDBs',
      'lastHdbAttended',
      'otherInfinitheismContact'
    ];

    const relationSelect: Record<string, string[]> = {
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      rmContactUser: ['id', 'orgUsrName']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      reportFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => ({
      'Name': reg.fullName,
      'Email': reg.emailAddress,
      'Mobile': reg.mobileNumber,
      'Gender': reg.gender,
      'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
      'City': reg.city.toLowerCase() === 'other' ? (reg.otherCityName || '') : (reg.city || ''),
      'HDB/MSD\'s done': reg.noOfHDBs || 0,
      'Last HDB/MSD': reg.lastHdbAttended || '',
      'Song Preference 1': this.getSongPreference(reg.songPreferences, 'song_1'),
      'Song Preference 2': this.getSongPreference(reg.songPreferences, 'song_2'),
      'Song Preference 3': this.getSongPreference(reg.songPreferences, 'song_3'),
      'Contact Person': (reg.rmContactUser &&
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
      'Date of blessing': reg.approvals?.[0]?.approvalStatus === ApprovalStatusEnum.APPROVED ? formatDateTimeIST(reg.approvals?.[0]?.approvalDate) : '',
    }));
  }

  /**
   * Get data for Hold List Report
   */
  private async getHoldListReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    // Add specific filter for registrations on hold
    const holdFilters = {
      ...reportFilters,
      approvalStatus: 'rejected' // 'rejected' status represents 'On Hold' in the system
    };
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'allocatedProgram',
      'rmContactUser',
      'travelPlans',
      'travelInfo',
      'recommendation',
      'updatedBy'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'noOfHDBs',
      'lastHdbAttended',
      'city',
      'otherCityName',
      'countryName',
      'otherInfinitheismContact',
      'averageRating',
      'mobileNumber',
      'preferredRoomMate',
      'updatedAt'
    ];

    const relationSelect: Record<string, string[]> = {
      allocatedProgram: ['id', 'name'],
      rmContactUser: ['id', 'orgUsrName'],
      travelPlans: ['id', 'travelPlanStatus'],
      travelInfo: ['id', 'travelInfoStatus'],
      recommendation: ['id', 'recommendationKey'],
      updatedBy: ['id', 'fullName', 'firstName', 'lastName']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      holdFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const travelPlan = reg.travelPlans?.[0];
      const recommendation = reg.recommendation?.[0];

      return {
        'Name': reg.fullName,
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Offered': reg.allocatedProgram?.name || '',
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'Infinitheism Contact': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'Rating': reg?.averageRating || '',
        'Recommendation': recommendation ? recommendation.recommendationKey : '',
        'Mobile': reg.mobileNumber,
        'Room-mate Name': reg.preferredRoomMate || '',
        'Last Modified Time': formatDateTimeIST(reg.updatedAt),
        'Modified User': reg.updatedBy?.fullName ?? (reg.updatedBy?.firstName && reg.updatedBy?.lastName ? `${reg.updatedBy?.firstName} ${reg.updatedBy?.lastName}` : ''),
        'Travel Status': this.formatTravelStatus(getTravelOverAllStatus(reg) || '', reg.allocatedProgram?.id) || '',
      };
    });
  }

  /**
   * Get data for Confirmed List Report
   */
  private async getConfirmedListReportData(options: any): Promise<any[]> {
    const isRMUser = options.userRoles.includes('relational_manager');
    const reportFilters = this.extractReportFilters(options.filters || {});
    const sortParams = this.getReportSortParameters(options.filters || {});
    
    // Add specific filter for confirmed/blessed registrations
    const confirmedFilters = {
      ...reportFilters,
      approvalStatus: 'approved' // 'approved' status represents confirmed/blessed registrations
    };
    
    // Define only the relations needed for this report
    const requiredRelations = [
      'approvals',
      'allocatedProgram',
      'preferences',
      'preferences.preferredProgram',
      'rmContactUser',
      'recommendation'
    ];

    const selectFields: (keyof ProgramRegistration | string)[] = [
      'id',
      'fullName',
      'dob',
      'emailAddress',
      'mobileNumber',
      'city',
      'otherCityName',
      'countryName',
      'noOfHDBs',
      'lastHdbAttended',
      'otherInfinitheismContact',
      'preferredRoomMate',
      'averageRating'
    ];

    const relationSelect: Record<string, string[]> = {
      approvals: ['id', 'approvalStatus', 'approvalDate'],
      allocatedProgram: ['id', 'name'],
      preferences: ['id', 'priorityOrder'],
      'preferences.preferredProgram': ['id', 'name'],
      rmContactUser: ['id', 'orgUsrName'],
      recommendation: ['id', 'recommendationKey']
    };
    
    const result = await this.repository.findRegistrations(
      'all', // limit - get all for report
      'all', // offset - get all
      options.programId,
      options.programSessionId,
      '', // searchText - exclude from reports, only apply filters
      confirmedFilters,
      options.userRoles,
      isRMUser ? options.filterRegistrationsByUserId : undefined,
      sortParams.sortKey, // conditional sort key based on filter context
      sortParams.sortOrder, // conditional sort order based on filter context
      requiredRelations, // Only fetch needed relations
      undefined, // communicationTemplateKey
      selectFields,
      relationSelect
    );

    return result.data.map(reg => {
      const approval = reg.approvals?.[0];
      const preferences = reg.preferences || [];
      const sortedPreferences = preferences.sort((a, b) => (a?.priorityOrder || 0) - (b?.priorityOrder || 0));
      const allChoices = sortedPreferences.map(p => p?.preferredProgram?.name).filter(Boolean).join(', ');
      const recommendation = reg.recommendation?.[0];
      
      return {
        'Name': reg.fullName,
        'HDB Offered': reg.allocatedProgram?.name || '',
        'HDB/MSD\'s done': reg.noOfHDBs || 0,
        'Last HDB/MSD': reg.lastHdbAttended || '',
        'Choices': allChoices || 'Any HDB/MSD',
        'City': (reg?.city && reg?.city.trim().toLowerCase() === 'other')
                ? (reg?.otherCityName ?? '')
                : (reg?.city ?? ''),
        'Country': reg.countryName ? replaceUnderscoreWithSpace(reg.countryName).toUpperCase() : '',
        'Infinitheism Contact': (reg.rmContactUser && 
                typeof reg.rmContactUser.orgUsrName === 'string' &&
                reg.rmContactUser.orgUsrName.trim().toLowerCase() === 'other')
              ? reg.otherInfinitheismContact || ''
              : (reg.rmContactUser ? reg.rmContactUser.orgUsrName : ''),
        'Rating': reg.averageRating || '',
        'Recommendation': recommendation ? recommendation.recommendationKey : '',
        'Mobile': reg.mobileNumber,
        'Gender': reg.gender,
        'Room-mate Name': reg.preferredRoomMate || '',
        'Blessed Date and Time': formatDateTimeIST(approval?.approvalDate),
        'Date of Birth': reg.dob ? formatDate(reg.dob) : '',
        'Registration ID': reg.id,
        'Email': reg.emailAddress,
      };
    });
  }

  private reportFormatting(status: string | null, allocatedProgramId: number | null, statusType: string) {
    if(statusType === 'payment') {
      if(status === 'online_pending' || status === 'offline_pending'){
        return 'Pending';
      } else if(status === 'online_completed' || status === 'offline_completed'){
        return 'Completed';
      } else {
        if(allocatedProgramId) {
          return 'Pending';
        }
        return status || '';
      }
    } else if(statusType === 'approval') {
      if(status === ApprovalStatusEnum.PENDING){
        return 'Unassigned';
      } else if(status === ApprovalStatusEnum.APPROVED) {
        return 'Blessed';
      } else if(status === ApprovalStatusEnum.ON_HOLD) {
        return 'Swap Demand';
      } else if(status === ApprovalStatusEnum.CANCELLED) {
        return 'Cancelled';
      } else if(status === ApprovalStatusEnum.REJECTED) {
        return 'Hold';
      } else if(status === RegistrationStatusEnum.SAVE_AS_DRAFT) {
        return 'Draft';
      } else {
        return '';
      }
    }
  }

  /**
   * Get complete three-level filter configuration for a specific program
   * Frontend will cache this configuration per program to avoid repeated API calls
   * Returns the full configuration with actual sub-program IDs and role-based report options
   */
  async getThreeLevelFilterConfiguration(programId: number, userRoles?: string[]): Promise<any> {
    this.logger.log(`Starting getThreeLevelFilterConfiguration for programId: ${programId}, userRoles: ${JSON.stringify(userRoles)}`);
    
    try {
      // Use the imported configuration instead of reading from file
      let filterConfig = JSON.parse(JSON.stringify(FILTER_HIERARCHY_CONFIG));
      this.logger.debug('Base filter configuration loaded successfully');
      
      // Get actual sub-program mappings for HDB and MSD programs
      const programMapping = await this.getProgramMappingForHierarchy(programId);
      this.logger.debug(`Program mapping retrieved:`, programMapping);
      
      // Replace dynamic program IDs with actual sub-program IDs
      const configString = JSON.stringify(filterConfig);
      const updatedConfigString = configString
        .replace(/{HDB1_ID}/g, programMapping.HDB1_ID)
        .replace(/{HDB2_ID}/g, programMapping.HDB2_ID)
        .replace(/{HDB3_ID}/g, programMapping.HDB3_ID)
        .replace(/{MSD1_ID}/g, programMapping.MSD1_ID)
        .replace(/{MSD2_ID}/g, programMapping.MSD2_ID);
      
      this.logger.debug('Placeholder replacement completed successfully');
      
      const updatedConfig = JSON.parse(updatedConfigString);
      
      // Return complete configuration including all possible side filter sets
      const completeSideFilterSets = await this.getCompleteSideFilterSets(updatedConfig, userRoles);
      this.logger.debug(`Side filter sets processed: ${Object.keys(completeSideFilterSets.baseSets || {}).length} base sets, ${Object.keys(completeSideFilterSets.contextualFilters || {}).length} contextual filters`);
      
      // Prepare base response
      const response: any = {
        programId,
        parentOptions: updatedConfig.filterHierarchy.parentOptions,
        sideFilterSets: completeSideFilterSets,
      };

      // Add role-based report options if user roles are provided
      if (userRoles && userRoles.length > 0) {
        try {
          response.reports = this.getRoleBasedReportOptions(userRoles);
          this.logger.debug(`Role-based reports added for roles: ${userRoles.join(', ')}`);
        } catch (reportError) {
          this.logger.warn('Failed to generate role-based reports, continuing without them:', reportError);
        }
      }

      response.lastUpdated = new Date().toISOString();
      
      this.logger.log(`Successfully completed getThreeLevelFilterConfiguration for programId: ${programId}. Returned ${response.parentOptions?.length || 0} parent options`);
      return response;
    } catch (error) {
      this.logger.error(`Error in getThreeLevelFilterConfiguration for programId: ${programId}: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Generate dynamic KPIs based on the selected parent filter from filter hierarchy
   * This method dynamically gets KPI options for the parent filter and generates counts
   */
  private async generateDynamicKPIsFromParentFilter(parameters: {
    programId?: number | null;
    programSessionId?: number | null;
    parentFilter: string;
    parsedFilters?: Record<string, any>;
    userRoles: string[];
    filterRegistrationsByUserId?: number | null;
  }): Promise<any> {
    const {
      programId,
      programSessionId,
      parentFilter,
      parsedFilters,
      userRoles,
      filterRegistrationsByUserId,
    } = parameters;

    this.logger.log(`Starting generateDynamicKPIsFromParentFilter with parentFilter: ${parentFilter}, programId: ${programId}, programSessionId: ${programSessionId}`);

    try {
      // Get the filter configuration with placeholders already resolved
      const filterConfig = await this.getThreeLevelFilterConfiguration(programId || 0, userRoles);
      this.logger.debug(`Filter configuration retrieved successfully with ${filterConfig.parentOptions?.length || 0} parent options`);
      
      // Find the parent option in the resolved configuration
      const parentOption = filterConfig.parentOptions.find(option => option.key === parentFilter);
      
      if (!parentOption) {
        const availableOptions = filterConfig.parentOptions.map(opt => opt.key).join(', ');
        this.logger.error(`Parent filter '${parentFilter}' not found in configuration. Available options: ${availableOptions}`);
        throw new Error(`Invalid parentFilter '${parentFilter}'. Valid options are: ${availableOptions}`);
      }

      this.logger.log(`Found parent option: ${parentOption.label} with ${parentOption.kpiOptions?.length || 0} KPI options`);

      // Build dynamic KPIs based on the parent filter's KPI options
      const kpis: any = {
        [parentFilter]: {
          categoryLabel: parentOption.value,
          items: [],
        },
      };

        // Generate counts for each KPI option in the parent filter
        for (let i = 0; i < parentOption.kpiOptions.length; i++) {
          const kpiOption = parentOption.kpiOptions[i];
          this.logger.debug(`Processing KPI option ${i + 1}/${parentOption.kpiOptions.length}: ${kpiOption.key} (${kpiOption.label})`);
          
          try {
            // Create filters for this specific KPI
            const kpiSpecificFilters = {
              kpiCategory: parentFilter,
              kpiFilter: kpiOption.kpiFilter,
              ...this.removeKPIFilters(parsedFilters || {}),
            };

            // Get count for this KPI using comprehensive parent filter logic
            let count = 0;
            let orgCount = 0;
            let communicationCategory: CommunicationCategoryEnum | null = null;
            if (parentFilter === 'all') {
              // For 'all' parent filter, handle both general metrics and program-specific filters
              if (kpiOption.kpiFilter?.startsWith('program_')) {
                // For program-specific filters in 'all' parent filter, extract actual program ID
                const allocatedProgramId = parseInt(kpiOption.kpiFilter.replace('program_', ''));
                const result = await this.repository.getTotalAllocatedRegistrationsWithOrg(
                  programId!,
                  allocatedProgramId,
                  this.removeKPIFilters(parsedFilters || {}),
                  '', // no search text for KPI counts
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
                communicationCategory = CommunicationCategoryEnum.HDB_BLESSED;
              } else {
                // Use seats metrics for general registration status filters
                const seatsMetrics = await this.repository.getSeatsMetrics(
                  programId,
                  programSessionId,
                  kpiSpecificFilters,
                  '', // No search text for KPIs
                  filterRegistrationsByUserId,
                );
                
                switch (kpiOption.kpiFilter) {
                  case 'all':
                    count = parseInt(seatsMetrics?.all ?? '0');
                    orgCount = parseInt(seatsMetrics?.allOrgCount ?? '0');
                    communicationCategory = CommunicationCategoryEnum.HDB_ALL;
                    break;
                  case 'newSeekersPending':
                    count = parseInt(seatsMetrics?.pending ?? '0');
                    orgCount = parseInt(seatsMetrics?.pendingOrgCount ?? '0');
                    communicationCategory = CommunicationCategoryEnum.HDB_SEEKER_PENDING;
                    break;
                  case 'approvedSeekers':
                    count = parseInt(seatsMetrics?.approved ?? '0');
                    orgCount = parseInt(seatsMetrics?.approvedOrgCount ?? '0');
                    communicationCategory = CommunicationCategoryEnum.HDB_BLESSED;
                    break;
                  case 'onHold':
                    count = parseInt(seatsMetrics?.onHold ?? '0');
                    orgCount = parseInt(seatsMetrics?.onHoldOrgCount ?? '0');
                    communicationCategory = CommunicationCategoryEnum.HDB_HOLD;
                    break;
                  case 'rejected':
                    count = parseInt(seatsMetrics?.rejected ?? '0');
                    orgCount = parseInt(seatsMetrics?.rejectedOrgCount ?? '0');
                    communicationCategory = CommunicationCategoryEnum.HDB_REJECT;
                    break;
                  case 'regPending':
                    count = parseInt(seatsMetrics?.registrationPendingCount ?? '0');
                    orgCount = parseInt(seatsMetrics?.registrationPendingOrgCount ?? '0');
                    communicationCategory = CommunicationCategoryEnum.HDB_HOLD;
                    break;
                  case 'cancelled':
                    count = parseInt(seatsMetrics?.cancelled ?? '0');
                    orgCount = parseInt(seatsMetrics?.cancelledOrgCount ?? '0');
                    break;
                  default:
                    count = 0;
                    orgCount = 0;
                    break;
                }
              }
            } else if (parentFilter === 'blessed') {
              // Handle main 'blessed' parent filter
              if (!programId) {
                throw new Error('Program ID is required for blessed KPI generation');
              }

              if (kpiOption.kpiFilter === 'approvedSeekers') {
                // Get total count of all blessed registrations across all sub-programs
                const result = await this.repository.getTotalBlessedRegistrationsWithOrg(
                  programId,
                  this.removeKPIFilters(parsedFilters || {}),
                  '', // no search text for KPI counts
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
                communicationCategory = CommunicationCategoryEnum.HDB_BLESSED;
                // Note: getTotalBlessedRegistrations returns only count, not orgCount
                // We may need to create a separate method for blessed org counts if needed
              } else if (kpiOption.kpiFilter?.startsWith('program_')) {
                // For program-specific blessed counts, extract the actual program ID from resolved filter
                const allocatedProgramId = parseInt(kpiOption.kpiFilter.replace('program_', ''));
                
                const result = await this.repository.getTotalAllocatedRegistrationsWithOrg(
                  programId, // main program ID from query params
                  allocatedProgramId,
                  this.removeKPIFilters(parsedFilters || {}),
                  '', // no search text for KPI counts
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
                communicationCategory = CommunicationCategoryEnum.HDB_BLESSED;
              } else if (kpiOption.kpiFilter === 'swapRequests') {
                // Handle swap requests for blessed category
                // For swapRequests in blessed category, get count for all blessed seekers with active swap requests
                const result = await this.repository.getTotalSwapRequestsWithOrgCount(
                  programId,
                  null, // no specific allocated program for general swap requests
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
              } else {
                count = 0;
                orgCount = 0;
              }
            } else if (parentFilter.endsWith('Blessed')) {
              // Handle specific program blessed categories (hdb1Blessed, hdb2Blessed, etc.)
              if (!programId) {
                throw new Error('Program ID is required for program-specific blessed KPI generation');
              }

              // Extract program ID from kpiCategory (e.g., program_791 -> 791)
              if (kpiOption.kpiCategory?.startsWith('program_')) {
                const allocatedProgramId = parseInt(kpiOption.kpiCategory.replace('program_', ''));
                communicationCategory = CommunicationCategoryEnum.HDB_BLESSED;
                if (kpiOption.kpiFilter?.startsWith('program_')) {
                  // For program-specific blessed counts within specific program blessed category
                  count = await this.repository.getTotalAllocatedRegistrations(
                    programId,
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                } else if (kpiOption.kpiFilter === 'blessed') {
                  // For the default blessed filter in specific program blessed category
                  count = await this.repository.getTotalAllocatedRegistrations(
                    programId,
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                } else if (kpiOption.kpiFilter === 'paymentPending') {
                  const blessedPaymentMetrics = await this.repository.getBlessedPaymentsMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedPaymentMetrics?.paymentPending ?? '0');
                } else if (kpiOption.kpiFilter === 'paymentComplete') {
                  const blessedPaymentMetrics = await this.repository.getBlessedPaymentsMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedPaymentMetrics?.paymentCompleted ?? '0');
                } else if (kpiOption.kpiFilter === 'invoicePending') {
                  const blessedInvoiceMetrics = await this.repository.getBlessedInvoicesMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedInvoiceMetrics?.invoicePending ?? '0');
                } else if (kpiOption.kpiFilter === 'invoiceComplete') {
                  const blessedInvoiceMetrics = await this.repository.getBlessedInvoicesMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedInvoiceMetrics?.invoiceCompleted ?? '0');
                } else if (kpiOption.kpiFilter === 'travelPending') {
                  const blessedTravelMetrics = await this.repository.getBlessedTravelMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedTravelMetrics?.travelPending ?? '0');
                } else if (kpiOption.kpiFilter === 'travelComplete') {
                  const blessedTravelMetrics = await this.repository.getBlessedTravelMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedTravelMetrics?.travelCompleted ?? '0');
                } else {
                  count = 0;
                }
              } else {
                // Fallback to old logic for backwards compatibility
                // Extract program type from parent filter (e.g., hdb1Blessed -> HDB1)
                const programType = parentFilter.replace('Blessed', '').toUpperCase();
                const programMapping = await this.getProgramMappingForHierarchy(programId);
                const allocatedProgramId = programMapping[`${programType}_ID`] || null;

                if (!allocatedProgramId) {
                  this.logger.warn(`No allocated program ID found for ${programType} in program ${programId}`);
                  count = 0;
                } else {
                  // Use the same logic as above with the allocated program ID
                  if (kpiOption.kpiFilter?.startsWith('program_')) {
                    count = await this.repository.getTotalAllocatedRegistrations(
                      programId,
                      allocatedProgramId,
                      this.removeKPIFilters(parsedFilters || {}),
                      '',
                      filterRegistrationsByUserId,
                    );
                  } else {
                    count = 0;
                  }
                }
              }
            } else if (parentFilter === 'swapRequests') {
              // Handle swap requests parent filter
              if (!programId) {
                throw new Error('Program ID is required for swap requests KPI generation');
              }

              if (kpiOption.kpiFilter === 'swapRequests') {
                // General swap requests count
                count = await this.repository.getTotalSwapRequests(
                  programId,
                  null, // no specific allocated program for general swap requests
                  filterRegistrationsByUserId,
                );
              } else if (kpiOption.kpiFilter?.startsWith('swapRequests_program_')) {
                // Program-specific swap requests
                const allocatedProgramId = parseInt(kpiOption.kpiFilter.replace('swapRequests_program_', ''));
                count = await this.repository.getTotalSwapRequests(
                  programId,
                  allocatedProgramId,
                  filterRegistrationsByUserId,
                );
              } else {
                count = 0;
              }
            } else if (parentFilter === 'swapDemand') {
              // Handle swap demand parent filter
              this.logger.log('Processing swap demand KPI option:', kpiOption);
              if (!programId) {
                this.logger.warn('Program ID is required for swap demand KPI generation');
                continue;
              }

              if (kpiOption.kpiFilter === 'onHold') {
                // General swap demand count (users on hold who want to swap)
                const result = await this.repository.getTotalOnHoldRegistrationsWithSwapPreference(
                  programId,
                  null,
                  this.removeKPIFilters(parsedFilters || {}),
                  '', // no search text for KPI counts
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
                communicationCategory = CommunicationCategoryEnum.HDB_HOLD;
           
              } else if (kpiOption.kpiFilter?.startsWith('onHold_program_')) {
                // Program-specific swap demand count
                const prefferedProgramId = parseInt(kpiOption.kpiFilter.replace('onHold_program_', ''));
                this.logger.log(`Retrieving on-hold registrations with swap demand preference for programId: ${programId}, prefferedProgramId: ${prefferedProgramId}`);
                // Get count of users on hold who want to swap to this specific program
                const result = await this.repository.getTotalOnHoldRegistrationsWithSwapPreference(
                  programId,
                  prefferedProgramId,
                  this.removeKPIFilters(parsedFilters || {}),
                  '', // no search text for KPI counts
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
              } else {
                count = 0;
              }
            } else if (parentFilter === 'regPending') {
              // Handle registration pending parent filter
              this.logger.log('Processing registration pending KPI option:', kpiOption);
              if (!programId) {
                this.logger.warn('Program ID is required for registration pending KPI generation');
                continue;
              }
              if (kpiOption.kpiFilter === 'regPending') {
                // General registration pending count (all ON_HOLD/REJECTED with any preference/swap demand)
                const result = await this.repository.getTotalPendingRegistrationsWithPreference(
                  programId,
                  null,
                  this.removeKPIFilters(parsedFilters || {}),
                  '',
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
                communicationCategory = CommunicationCategoryEnum.HDB_HOLD;
              } else if (kpiOption.kpiFilter?.startsWith('regPending_program_')) {
                // Program-specific registration pending count
                const pendingProgramId = parseInt(kpiOption.kpiFilter.replace('regPending_program_', ''));
                this.logger.log(`Retrieving pending registrations for programId: ${programId}, pendingProgramId: ${pendingProgramId}`);
                const result = await this.repository.getTotalPendingRegistrationsWithPreference(
                  programId,
                  pendingProgramId,
                  this.removeKPIFilters(parsedFilters || {}),
                  '',
                  filterRegistrationsByUserId,
                );
                count = parseInt(result?.totalCount ?? '0');
                orgCount = parseInt(result?.orgCount ?? '0');
              }
            } else {
              // For any other parent filters, use general approach
              if (!programId) {
                throw new Error('Program ID is required for dynamic KPI generation');
              }

              // Check if this is a program-specific category (program_{ID})
              if (kpiOption.kpiCategory?.startsWith('program_')) {
                const allocatedProgramId = parseInt(kpiOption.kpiCategory.replace('program_', ''));
                
                if (kpiOption.kpiFilter?.startsWith('program_')) {
                  // For program-specific registrations count
                  count = await this.repository.getTotalAllocatedRegistrations(
                    programId,
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                } else if (kpiOption.kpiFilter === 'paymentPending') {
                  const blessedPaymentMetrics = await this.repository.getBlessedPaymentsMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedPaymentMetrics?.paymentPending ?? '0');
                } else if (kpiOption.kpiFilter === 'paymentComplete') {
                  const blessedPaymentMetrics = await this.repository.getBlessedPaymentsMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedPaymentMetrics?.paymentCompleted ?? '0');
                } else if (kpiOption.kpiFilter === 'invoicePending') {
                  const blessedInvoiceMetrics = await this.repository.getBlessedInvoicesMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedInvoiceMetrics?.invoicePending ?? '0');
                } else if (kpiOption.kpiFilter === 'invoiceComplete') {
                  const blessedInvoiceMetrics = await this.repository.getBlessedInvoicesMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedInvoiceMetrics?.invoiceCompleted ?? '0');
                } else if (kpiOption.kpiFilter === 'travelPending') {
                  const blessedTravelMetrics = await this.repository.getBlessedTravelMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedTravelMetrics?.travelPending ?? '0');
                } else if (kpiOption.kpiFilter === 'travelComplete') {
                  const blessedTravelMetrics = await this.repository.getBlessedTravelMetrics(
                    programId,
                    null, // programSessionId
                    allocatedProgramId,
                    this.removeKPIFilters(parsedFilters || {}),
                    '', // no search text for KPI counts
                    filterRegistrationsByUserId,
                  );
                  count = parseInt(blessedTravelMetrics?.travelCompleted ?? '0');
                } else {
                  count = 0;
                }
              } else {
                // For general filters that don't use program-specific logic
                const categoryToUse = kpiOption.kpiCategory || parentFilter;
                const filterCondition = this.getKPIFilterCondition(categoryToUse, kpiOption.kpiFilter || kpiOption.value);
                
                if (filterCondition) {
                  const finalFilters = {
                    ...this.removeKPIFilters(parsedFilters || {}),
                    ...filterCondition,
                  };
                  
                  count = await this.repository.getTotalRegistrations(
                    programId!,
                    finalFilters,
                    '', // no search text for KPI counts
                  );
                } else {
                  count = 0;
                }
              }
            }
            
          
          kpis[parentFilter].items.push({
            label: kpiOption.label,
            value: kpiOption.value,
            count: count,
            kpiCategory: kpiOption.kpiCategory,
            kpiFilter: kpiOption.kpiFilter,
            order: i + 1,
            orgCount: orgCount || 0, // Include orgCount only if defined
            communicationCategory: communicationCategory || null,
            category: parentFilter,
            categoryLabel: parentFilter,
          });

        } catch (error) {
          this.logger.error(`Error generating count for KPI option ${kpiOption.key} (${kpiOption.label}) in parentFilter ${parentFilter}: ${error.message}`, error.stack);

          // Add zero count on error (without sideFilters for registration-list-view endpoint)
          kpis[parentFilter].items.push({
            label: kpiOption.label,
            value: kpiOption.value,
            count: 0,
            kpiCategory: kpiOption.kpiCategory,
            kpiFilter: kpiOption.kpiFilter,
            order: i + 1,
            orgCount: 0,
            communicationCategory: null,
            category: parentFilter,
            categoryLabel: parentFilter,
          });
        }
      }

      this.logger.log(`Successfully generated ${kpis[parentFilter].items.length} KPI items for parentFilter: ${parentFilter}`);
      return kpis;
    } catch (error) {
      this.logger.error(`Error in generateDynamicKPIsFromParentFilter for parentFilter: ${parentFilter}, programId: ${programId}`, error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Get contextual side filters based on parent and KPI selections
   */
  private getContextualSideFilters(config: any, parentFilter?: string, kpiFilter?: string): any {
    // Handle both config structures - flattened (from getThreeLevelFilterConfiguration) or nested (from raw config)
    const sideConfig = config.sideFilterSets || config.filterHierarchy?.sideFilterSets;
    
    if (!sideConfig) {
      this.logger.error('sideFilterSets not found in config structure');
      return {
        availableFilters: [],
        filterSetKeys: []
      };
    }
    
    const baseSets = sideConfig.baseSets;
    const contextualFilters = sideConfig.contextualFilters;
    
    if (!baseSets || !contextualFilters) {
      this.logger.error('baseSets or contextualFilters not found in sideFilterSets');
      return {
        availableFilters: [],
        filterSetKeys: []
      };
    }
    
    // If no parent or KPI filter is selected, return all common filters
    if (!parentFilter || !kpiFilter) {
      return {
        availableFilters: baseSets.common || [],
        filterSetKeys: ['common']
      };
    }
    
    // Create a context key based on parent and KPI filters
    let contextKey = parentFilter;
    
    // For blessed programs with specific allocations
    if (parentFilter === 'blessed' && ['hdb1', 'hdb2', 'hdb3', 'msd1', 'msd2'].includes(kpiFilter)) {
      contextKey = `${kpiFilter}Blessed`;
    }
    
    // Check if we have specific contextual filters for this combination
    const contextFilters = contextualFilters[contextKey] && contextualFilters[contextKey][kpiFilter];
    
    if (contextFilters) {
      // Build the available filters from the specified sets
      const availableFilters: any[] = [];
      const filterSetKeys = contextFilters;
      
      contextFilters.forEach(setKey => {
        if (baseSets[setKey]) {
          availableFilters.push(...baseSets[setKey]);
        }
      });
      
      return {
        availableFilters,
        filterSetKeys,
        contextKey,
        appliedContext: `${parentFilter} > ${kpiFilter}`
      };
    }
    
    // Fallback to parent based filtering
    const parentFilters = contextualFilters[parentFilter] && contextualFilters[parentFilter][kpiFilter];
    
    if (parentFilters) {
      const availableFilters: any[] = [];
      const filterSetKeys = parentFilters;
      
      parentFilters.forEach(setKey => {
        if (baseSets[setKey]) {
          availableFilters.push(...baseSets[setKey]);
        }
      });
      
      return {
        availableFilters,
        filterSetKeys,
        contextKey: parentFilter,
        appliedContext: `${parentFilter} > ${kpiFilter}`
      };
    }
    
    // Final fallback to common filters
    return {
      availableFilters: baseSets.common,
      filterSetKeys: ['common'],
      contextKey: 'default',
      appliedContext: 'default'
    };
  }

  /**
   * Get actual program mapping for filter hierarchy based on sub-programs
   * Fetches real sub-program IDs instead of using dummy values
   */
  private async getProgramMappingForHierarchy(programId: number): Promise<any> {
    this.logger.debug(`Starting getProgramMappingForHierarchy for programId: ${programId}`);
    
    try {
      // Get all sub-programs for this main program
      const subPrograms = await this.repository.getSubPrograms(programId);
      
      this.logger.log(`Found ${subPrograms.length} sub-programs for program ${programId}:`, 
        subPrograms.map(sp => `${sp.name} (${sp.code}): ${sp.id}`));
      
      // Create mapping based on program codes
      const mapping = {
        HDB1_ID: programId.toString(), // Default fallback as string for replacement
        HDB2_ID: programId.toString(), // Default fallback as string for replacement  
        HDB3_ID: programId.toString(), // Default fallback as string for replacement
        MSD1_ID: programId.toString(), // Default fallback as string for replacement
        MSD2_ID: programId.toString(), // Default fallback as string for replacement
        currentProgramId: programId
      };
      
      // Map actual sub-program IDs based on their codes or names
      subPrograms.forEach(subProgram => {
        const code = subProgram.code?.toLowerCase() || '';
        const name = subProgram.name?.toLowerCase() || '';
        
        this.logger.log(`Checking sub-program: ${subProgram.name} (code: ${subProgram.code})`);
        
        // Check for HDB programs with more flexible matching
        if (code.includes('hdb1') || code.includes('hdb 1') || 
            name.includes('hdb1') || name.includes('hdb 1') ||
            name.includes('hdb-1') || code.includes('hdb-1')) {
          mapping.HDB1_ID = subProgram.id.toString();
          this.logger.log(`✓ Mapped HDB1_ID to program ${subProgram.id} (${subProgram.name})`);
        } else if (code.includes('hdb2') || code.includes('hdb 2') || 
                   name.includes('hdb2') || name.includes('hdb 2') ||
                   name.includes('hdb-2') || code.includes('hdb-2')) {
          mapping.HDB2_ID = subProgram.id.toString();
          this.logger.log(`✓ Mapped HDB2_ID to program ${subProgram.id} (${subProgram.name})`);
        } else if (code.includes('hdb3') || code.includes('hdb 3') || 
                   name.includes('hdb3') || name.includes('hdb 3') ||
                   name.includes('hdb-3') || code.includes('hdb-3')) {
          mapping.HDB3_ID = subProgram.id.toString();
          this.logger.log(`✓ Mapped HDB3_ID to program ${subProgram.id} (${subProgram.name})`);
        }
        
        // Check for MSD programs with more flexible matching
        else if (code.includes('msd1') || code.includes('msd 1') || 
                 name.includes('msd1') || name.includes('msd 1') ||
                 name.includes('msd-1') || code.includes('msd-1')) {
          mapping.MSD1_ID = subProgram.id.toString();
          this.logger.log(`✓ Mapped MSD1_ID to program ${subProgram.id} (${subProgram.name})`);
        } else if (code.includes('msd2') || code.includes('msd 2') || 
                   name.includes('msd2') || name.includes('msd 2') ||
                   name.includes('msd-2') || code.includes('msd-2')) {
          mapping.MSD2_ID = subProgram.id.toString();
          this.logger.log(`✓ Mapped MSD2_ID to program ${subProgram.id} (${subProgram.name})`);
        } else {
          this.logger.warn(`Could not map sub-program: ${subProgram.name} (code: ${subProgram.code})`);
        }
      });
      
      this.logger.log(`Final program mapping for program ${programId}:`, mapping);
      return mapping;
    } catch (error) {
      this.logger.error('Error getting program mapping:', error);
      // Return default mapping with current programId if any error occurs
      const defaultMapping = {
        HDB1_ID: programId.toString(),
        HDB2_ID: programId.toString(),
        HDB3_ID: programId.toString(),
        MSD1_ID: programId.toString(),
        MSD2_ID: programId.toString(),
        currentProgramId: programId
      };
      
      this.logger.warn(`Returning default mapping for programId ${programId} due to error:`, defaultMapping);
      return defaultMapping;
    }
  }

  /**
   * Get complete side filter sets for frontend caching
   * Returns all possible side filter configurations with dynamic data
   */
  private async getCompleteSideFilterSets(config: any, userRoles?: string[]): Promise<any> {
    const sideConfig = config.filterHierarchy.sideFilterSets;
    let baseSets = { ...sideConfig.baseSets };
    const contextualFilters = sideConfig.contextualFilters;
    
    // Dynamically populate RM Contact filter options (only for OM, Coordinator, Super Admin, and Mahatria roles)

    const hasAllowedRole = userRoles && userRoles.some(role => allowedRolesForRMFilter.includes(role));
    if (baseSets.common && hasAllowedRole) {
      try {
        const rmFilters = { roleKey: ROLE_KEYS.RELATIONAL_MANAGER };
        const rmUsers = await this.userService.getAllUsersWithoutPagination( '', rmFilters);
        const rmOptions = rmUsers.map(user => ({
          label: user.orgUsrName,
          value: user.id.toString()
        }));

        // Find and update the RM Contact filter
        const rmContactFilter = baseSets.common.find(filter => filter.key === 'rmContact');
        if (rmContactFilter) {
          rmContactFilter.options = rmOptions;
        } else {
          this.logger.log('RM Contact filter not found in baseSets.common');
        }
      } catch (error) {
        this.logger.error('Failed to fetch RM users for filter configuration', error);
        // Keep empty options array as fallback
      }
    } else {
      // Remove RM Contact filter if user does not have allowed roles
      baseSets.common = baseSets.common.filter(filter => filter.key !== 'rmContact');
      this.logger.log('Removed RM Contact filter due to insufficient user roles');
    }

    // allowedRolesForExperienceTags 
    const hasExperienceTagRole = userRoles && userRoles.some(role => allowedRolesForExperienceTags.includes(role));
    if (baseSets.common && hasExperienceTagRole) {
      try {
        const tags = await this.getExperienceTagFilterOptions();
        // Find and update the Experience Tags filter
        const expTagFilter = baseSets.common.find(filter => filter.key === 'experienceTags');
        if (expTagFilter) {
          expTagFilter.options = tags;
        } else {
          this.logger.log('Experience Tags filter not found in baseSets.common');
        }
      } catch (error) {
        this.logger.error('Failed to fetch Experience Tags for filter configuration', error);
        // Keep empty options array as fallback
      }
    } else {
      // Remove Experience Tags filter if user does not have allowed roles
      baseSets.common = baseSets.common.filter(filter => filter.key !== 'experienceTags');
      this.logger.log('Removed Experience Tags filter due to insufficient user roles');
    }

    this.logger.log('Final baseSets.common filters:', baseSets.common?.map(f => ({ key: f.key, label: f.label, optionsCount: f.options?.length || 0 })));
    
    return {
      baseSets,
      contextualFilters,
      // Provide helper method for frontend to get contextual filters
      getContextualFilters: (parentFilter: string, kpiFilter: string) => {
        return this.getContextualSideFilters(config, parentFilter, kpiFilter);
      }
    };
  }

  /**
   * NEW: Optimized registration list view with dynamic KPIs driven by parentFilter
   * Separate from old getRegistrationsListView to avoid breaking existing functionality
   */
  async newGetRegistrationsListView(
    limit: number,
    offset: number,
    programId: number | null,
    programSessionId: number | null,
    searchText: string,
    parsedFilters: Record<string, any>,
    userRoles: string[],
    filterRegistrationsByUserId?: number | null,
    downloadType?: string,
    sortKey: string = 'sortKey',
    sortOrder: 'ASC' | 'DESC' = 'ASC',
    communicationTemplateKey?: CommunicationTemplatesKeysEnum,
    view?: string,
  ) {
    this.logger.log('Getting NEW registrations list view with dynamic KPIs', {
      limit,
      offset,
      programId,
      programSessionId,
      searchText,
      parsedFilters,
      userRoles,
      downloadType,
      sortKey,
      sortOrder,
      view,
    });
  
    try {
      const isRMUser = userRoles.includes('relational_manager');
      
      // Use common default sort function for consistent sorting behavior
      const defaultSort = this.getDefaultSortParameters(sortKey, sortOrder, parsedFilters);
      const finalSortKey = defaultSort.sortKey;
      const finalSortOrder = defaultSort.sortOrder;

      // Handle download case - return all data without pagination with optimized relations
      if (downloadType) {
        const registrationData = await this.repository.findRegistrations(
          'all',
          'all',
          programId,
          programSessionId,
          downloadType === 'filtered' ? searchText : '',
          downloadType === 'filtered' ? this.extractDataFilters(parsedFilters) : {},
          userRoles,
          isRMUser ? filterRegistrationsByUserId : undefined,
          finalSortKey, // sortKey
          finalSortOrder, // sortOrder
          REGISTRATION_LIST_REQUIRED_RELATIONS, // requiredRelations - optimized for list view
          communicationTemplateKey, // communicationTemplateKey
          REGISTRATION_LIST_SELECT_FIELDS, // selectFields - only needed fields
          REGISTRATION_LIST_RELATION_SELECT // relationSelect - only needed relation fields
        );
        
        return {
          data: registrationData.data,
          pagination: {
            totalPages: 1,
            pageNumber: 1,
            pageSize: registrationData.data.length,
            totalRecords: registrationData.data.length,
            numberOfRecords: registrationData.data.length,
          },
        };
      }
      if (communicationTemplateKey) {
        const dataFilters = this.extractDataFilters(parsedFilters);
        const result = await this.repository.findRegistrations(
          'all',
          'all',
          programId,
          programSessionId,
          searchText, // ✅ Search applies to data table
          dataFilters,
          userRoles,
          isRMUser ? filterRegistrationsByUserId : undefined,
          finalSortKey,
          finalSortOrder,
          [],
          communicationTemplateKey,
        );
        
        return {
          data: result.data,
          pagination: {
            totalPages: 1,
            pageNumber: 1,
            pageSize: result.data.length,
            totalRecords: result.data.length,
            numberOfRecords: result.data.length,
          },
        };
      }
  
      // ✅ STRICT: Validate parentFilter is required and exists in hierarchy
      const parentFilter = parsedFilters?.parentFilter;
      
      if (!parentFilter) {
        // Return 400 Bad Request for missing parentFilter
        const validationError = new Error('parentFilter is required in filters');
        (validationError as any).statusCode = 400;
        throw validationError;
      }
      
      // ✅ NEW: Validate parentFilter against filter hierarchy configuration
      try {
        const filterConfig = await this.getThreeLevelFilterConfiguration(programId || 0, userRoles);
        const parentOption = filterConfig.parentOptions.find(option => option.key === parentFilter);
        
        if (!parentOption) {
          const validOptions = filterConfig.parentOptions.map(opt => opt.key).join(', ');
          // Return 400 Bad Request for invalid parentFilter
          const validationError = new Error(`Invalid parentFilter '${parentFilter}'. Valid options are: ${validOptions}`);
          (validationError as any).statusCode = 400;
          throw validationError;
        }
        
      } catch (configError) {
        this.logger.error('Error accessing filter-config service:', configError);
        // Return 503 Service Unavailable for filter-config issues
        const serviceError = new Error('Filter configuration service unavailable. Please try again later.');
        (serviceError as any).statusCode = 503;
        throw serviceError;
      }
      
      // ✅ FIXED: Separate filters for KPIs and data
      const kpiFilters = this.extractKPIFilters(parsedFilters);
      const dataFilters = this.extractDataFilters(parsedFilters);

      // Get registration data with optimized relations and field selection
      const result = await this.repository.findRegistrations(
        limit,
        offset,
        programId,
        programSessionId,
        searchText, // ✅ Search applies to data table
        dataFilters,
        userRoles,
        isRMUser ? filterRegistrationsByUserId : undefined,
        finalSortKey,
        finalSortOrder,
        REGISTRATION_LIST_REQUIRED_RELATIONS, // requiredRelations - optimized for list view
        undefined, // communicationTemplateKey
        REGISTRATION_LIST_SELECT_FIELDS, // selectFields - only needed fields
        REGISTRATION_LIST_RELATION_SELECT // relationSelect - only needed relation fields
      );
  
      // ✅ NEW: Generate dynamic KPIs based on parentFilter from filter hierarchy
      const dynamicKpis = await this.generateDynamicKPIsFromParentFilter({
        programId,
        programSessionId,
        parentFilter,
        parsedFilters: this.removeKPIFilters(parsedFilters), // Remove KPI filters from KPI generation
        userRoles,
        filterRegistrationsByUserId: isRMUser ? filterRegistrationsByUserId : undefined,
      });
  
      const finalResult = {
        data: result.data,
        pagination: result.pagination,
        kpis: {
          dynamic: dynamicKpis,
        },
      };

      return finalResult;
    } catch (error) {
      this.logger.error('Error in newGetRegistrationsListView:', error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }

  registrationEditLink(programId:number, userId:number | undefined): string {
    let url = `${process.env.SEEKER_FE_BASE_URL}${SEEKER_FE_REG_PATH}${programId}${SEEKER_FE_EDIT_REGISTRATION_PATH}`;
    if (userId) {
      url += `&userId=${userId}`;
    }
    return url;
  }
  async getAllRegistrationsWithoutPagination(
    programId: number | null, 
    programSessionId: number | null, 
    searchText?: string, 
    parsedFilters?: Record<string, any>, 
    userRoles?: string[], 
    communicationTemplateKey?: CommunicationTemplatesKeysEnum) {
      try {
        if (programId && communicationTemplateKey === CommunicationTemplatesKeysEnum.FIRST_TIMER) {
          const filters = parsedFilters ? this.extractDataFilters(parsedFilters) : {}
          const registrations = await this.repository.findUniqueRegistrationsForPrograms(programId, filters?.allocatedProgramId ?? undefined);
          return {
            data: Array.isArray(registrations) ? registrations.map((item) => item) : [],
            pagination: {
              totalPages: 1,
              pageNumber: 1,
              pageSize: Array.isArray(registrations) ? registrations.length : 0,
              totalRecords: Array.isArray(registrations) ? registrations.length : 0,
              numberOfRecords: Array.isArray(registrations) ? registrations.length : 0,
            },
          };
        } else {
          const registrationData = await this.repository.findRegistrations(
            'all',
            'all',
            programId,
            programSessionId,
            searchText ? searchText : '',
            parsedFilters ? this.extractDataFilters(parsedFilters) : {},
            userRoles,
          );
          
          return {
            data: registrationData.data,
            pagination: {
              totalPages: 1,
              pageNumber: 1,
              pageSize: registrationData.data.length,
              totalRecords: registrationData.data.length,
              numberOfRecords: registrationData.data.length,
            },
          };
        }
      } catch (error) {
        console.error('Error fetching all registrations:', error);
        handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
      } 
  }
  async getUniqueRegistrationsForPrograms(
    programId: number, 
    allocatedProgramId?: number | null,
    parsedFilters?: Record<string, any>,
    offset: number = 0,
    limit: number = 100,
  ) {
      try {
        const filters = parsedFilters ? this.extractDataFilters(parsedFilters) : {};
        const registrations = await this.repository.findUniqueRegistrationsForPrograms(
          programId,
          allocatedProgramId ?? filters?.allocatedProgramId ?? undefined,
          limit,
          offset
        );

        return {
          data: Array.isArray(registrations) ? registrations.map((item) => item) : [],
          pagination: {
            totalPages: Math.ceil(registrations.length / limit),
            pageNumber: Math.floor(offset / limit) + 1,
            pageSize: limit,
            totalRecords: registrations.length,
            numberOfRecords: Array.isArray(registrations) ? registrations.length : 0,
          },
        };
      } catch (error) {
        console.error('Error fetching unique registrations:', error);
        handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
      }
  }
  async sendCommunicationToRegistrations(data: any[], templateType: CommunicationTypeEnum, templateKey: CommunicationTemplatesKeysEnum) {

    try{
      if (!data || (Array.isArray(data) && data.length === 0)) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATIONS_NOT_FOUND_TO_SEND_COMMUNICATION,
          null,
          'No data found for communication'
        );
      } else {
        const batchSize = 100; // Define the batch size
        for (let i = 0; i < data.length; i += batchSize) {
          const batch = data.slice(i, i + batchSize);
          await this.communicationService.bulkCommunicationData(batch, templateType, templateKey);
        }
      }
      return { message: 'Communication sent successfully' };
    } catch (error) {
      this.logger.error(`Error sending communication: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_SEND_FAILED, error);
    }
  }

  private extractUserProfileUpdates(answers: RegistrationAnswerDto[],user: User,countryCode?: string): Partial<UpdateUserDto> {
    const updateData: Partial<UpdateUserDto> = {};
    const mobileNumberTypeAnswer = answers.find((a) => a.bindingKey === 'phoneNumberType');
    for (const ans of answers) {
      const key = ans?.bindingKey;
      if (key && USER_PROFILE_BINDING_KEYS.includes(key)) {
        switch (key) {
          case 'name':
            updateData.legalFullName = ans.answer;
            break;
          case 'city':
            updateData.address = ans.answer?.toLowerCase() === 'other' ? 'Other' : ans.answer;
            break;
          case 'otherCityName':
            updateData.otherAddress =
              updateData.address?.toLowerCase() === 'other' ? ans.answer : '';
            break;
          case 'gender':
            updateData.gender = ans.answer as GenderEnum;
            break;
          case 'profileUrl':
            updateData.profileUrl = ans.answer;
            break;
          case 'dob':
            updateData.dob = ans.answer;
            break;
          case 'countryName':
            updateData.countryName = ans.answer;
            break;
          case 'email':
            if (ans.answer) {
              const incomingEmail = String(ans.answer).trim().toLowerCase();
              const existingEmail = user.email?.trim().toLowerCase();
              // Only update if incoming email is not empty and different from existing
              if (incomingEmail && incomingEmail !== '' && incomingEmail !== existingEmail) {
                updateData.email = incomingEmail;
              }
            }
            break;
          case 'mobileNumber':
            if (ans.answer && mobileNumberTypeAnswer?.answer !== phoneNumberType.differentNumber ) {
              const fullNumber = String(ans.answer).trim();
              const existingNumber = `${user.countryCode ?? ''}${user.phoneNumber ?? ''}`;
              const cc = countryCode ?? user.countryCode ?? '';
              // Only update if fullNumber is not empty and different from existing
              if (fullNumber && fullNumber !== '' && fullNumber !== existingNumber) {
                let phoneWithoutCc = fullNumber;
                if (cc && phoneWithoutCc.startsWith(cc)) {
                  phoneWithoutCc = phoneWithoutCc.slice(cc.length);
                }
                // Ensure phoneWithoutCc is not empty after processing
                if (phoneWithoutCc && phoneWithoutCc !== '' && cc && cc !== '') {
                  updateData.countryCode = cc;
                  updateData.phoneNumber = phoneWithoutCc;
                }
              }
            }
            break;
        }
      }
    }
  
    return updateData;
  }

  private extractUserProfileExtensionUpdates(answers: RegistrationAnswerDto[]): Partial<UpdateUserProfileExtensionDto> {
    const updateData: Partial<UpdateUserProfileExtensionDto> = {};

    for (const ans of answers) {
      const key = ans?.bindingKey;
      if (key && USER_PROFILE_EXTENSION_BINDING_KEYS.includes(key)) {
        switch (key) {
          case 'pictureUrl':
            updateData.idFrontBackOnlyFromDownloadLink = ans.answer;
            break;
          case 'idBack':
            updateData.idBackOnlyDownloadLink = ans.answer;
            break;
          case 'invoiceEmail':
            updateData.emailToWhichInvoiceToBeSent = ans.answer;
            break;
          case 'invoiceName':
            updateData.nameInWhichInvoiceToBeIssued = ans.answer;
            break;
          case 'pro_forma_invoice_name':
            updateData.nameInWhichInvoiceToBeIssued = ans.answer;
            break;
          case 'invoiceAddress':
            updateData.invoiceAddress = ans.answer;
            break;
          case 'proFormaBillingAddress':
            updateData.invoiceAddress = ans.answer;
            break;
          case 'tanNumber':
            updateData.tan = ans.answer;
            break;
          case 'proFormaIsGstRegistered':
            updateData.areYouRegisteredUnderGst = ans.answer.toLowerCase().trim() === 'yes' || ans.answer === true;
            break;
          case 'proFormaGstNumber':
            updateData.gstin = ans.answer;
            break;
          case 'gstNumber':
            updateData.gstin = ans.answer;
            break;
          case 'isGstRegistered':
            updateData.areYouRegisteredUnderGst = ans.answer.toLowerCase().trim() === 'yes' || ans.answer === true;
            break;
          case 'QK_LAST_HDB':
            updateData.whenWasYourLastHdbMsd = ans.answer;
            break;
          case 'QK_HDB_ASSOCIATION_SINCE':
            updateData.sinceWhenHaveYouBeenAssociatedWithThisPath = ans.answer;
            break;
          case 'no_of_hdbs':
            updateData.howManyHdbMsdHaveYouDone = parseInt(ans.answer) || 0;
            break;
          // case 'whichOfTheFollowingHaveYouExperienced':
          //   updateData.whichOfTheFollowingHaveYouExperienced = ans.answer;
          //   break;
          case 'QK_HDB_SONG_1':
            updateData.songPreference1 = ans.answer;
            break;
          case 'QK_HDB_SONG_2':
            updateData.songPreference2 = ans.answer;
            break;
          case 'rmContact':
            // Handle infinitheism contact ID and name
            updateData.infinitheismContactId = parseInt(ans.answer) || undefined;
            // Note: The name should be handled separately through another binding or lookup
            break;
          case 'otherInfinitheismContact':
            updateData.contactPerson = ans.answer;
            break;
          case 'zip':
            updateData.pincode = ans.answer;
            break;
          case 'proFormaZip':
            updateData.pincode = ans.answer;
        }
      }
    }

    return updateData;
  }

  async sendRefundCommunication(registrationId: number) {
    try {
      const registration = await this.repository.findRegistrationById(registrationId);
      if (!registration) {
        throw new Error('Registration not found');
      }
      const financeUsers = await this.userRepository.getUsersByRoleKey(
        ROLE_KEYS.FINANCE_MANAGER,
      );
        
      if (!financeUsers || financeUsers.length === 0) {
        throw new Error('Finance users not found');
      }

      const to = financeUsers.map(user => {
        const fmName = (user.firstName && user.lastName) ? (user.firstName + ' ' + user.lastName) : user.fullName;
        return {
          emailAddress: user.email,
          name: fmName,
          mergeInfo: {
            reg_name: registration.fullName,
            fm_name: (user.firstName && user.lastName) ? (user.firstName + ' ' + user.lastName) : user.fullName,
            reg_id: registration.id,
            reg_status: registration.registrationStatus == RegistrationStatusEnum.REJECTED ? 'Hold' : registration.registrationStatus,
          }
        };
      });
      const emailData: SendBulkEmailDto = {
        templateKey: process.env.ZEPTO_REGISTRATION_REFUND_EMAIL_TEMPLATE_ID,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name: registration.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        to: to,
        attachments: [],
        subject: '',
      };

      await this.communicationService.sendBulkEmail(emailData)
      return { message: 'Refund communication sent successfully' };
    } catch (error) {
      this.logger.error(`Error sending refund communication: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_SEND_FAILED, error);
    }
  }

  async sendRegistrationCancelCommunication(registrationId: number) {
    try {
      const registration = await this.repository.findRegistrationById(registrationId);
      if (!registration) {
        throw new Error('Registration not found');
      }
      const mergeInfo = {
        reg_name: registration.fullName || 'Infinitheist',
        hdb_msd: registration?.program?.name,
      };
      const templateKey = process.env.ZEPTO_CANCEL_EMAIL_TEMPLATE_ID;
      
      const emailData: SendSingleEmailDto = {
        templateKey: templateKey,
        from: {
          address: zeptoEmailCreadentials.ZEPTO_EMAIL,
          name:
            registration.program.emailSenderName ||
            zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
        },
        to: {
          emailAddress: registration.emailAddress,
          name: registration.fullName,
        },
        mergeInfo: mergeInfo,
        attachments: [],
        subject: '',
        trackinfo: {
          registrationId: registration.id,
        },
      };

      await this.communicationService.sendSingleEmail(emailData)
      return { message: 'Refund communication sent successfully' };
    } catch (error) {
      this.logger.error(`Error sending refund communication: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_SEND_FAILED, error);
    }
  }

  async updateParentalConsentForm(id: number, updateParentalConsentDto: UpdateParentalConsentDto, user: User) {
    try {
      const registration = await this.repository.findRegistrationById(id);
      if (!registration) {
        throw new Error('Registration not found');
      }

      // Update the parental consent form field (assuming the field name is parentalConsentForm)
      registration.parentalFormPdfUrl = updateParentalConsentDto.parentalFormUrl;
      registration.parentalFormStatus = updateParentalConsentDto.parentalFormStatus;
      registration.updatedBy = user;

      await this.repository.updateRegistration(id, registration, user.id);

      return { message: 'Parental consent form updated successfully' };
    } catch (error) {
      this.logger.error(`Error updating parental consent form: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_SEND_FAILED, error);
    }
  }

  async uploadBufferToAWS(
    pdfBuffer: Buffer,
    fileName: string,
  ): Promise<{ url: string }> {

    try {

      const key = `${fileName} || uuidv4()}.pdf`;
      const uploadedInvoice = await this.awsS3Service.uploadToS3(key, pdfBuffer, 'application/pdf');
      this.logger.log(`PDF generated and uploaded to S3 at key: ${key}`);
      // const s3Url = this.awsS3Service.getS3Url(key);
      // const s3Url = await this.awsS3Service.getSignedUrl(key);
      return { url: uploadedInvoice };
    } catch (error) {
      this.logger.error('Error uploading PDF to S3:', error);
      throw error;
    }
  }

  /**
   * Get basic registrations with minimal fields
   * For relational_manager role, filter by their rmContact assignments
   * @param userRoles - Array of user role objects
   * @param userId - Current user ID
   * @param programId - Program ID to filter registrations
   * @returns Array of basic registration data with comprehensive user and registration details
   */
  async getBasicRegistrations(userRoles: any[], userId: number, programId: number, searchText: string = ''): Promise<{
    registrationId: number;
    userId: number;
    userName: string;
    userLegalName: string;
    userFirstName: string;
    userLastName: string;
    userPhoneNumber: string;
    userEmail: string;
    userGender: string;
    userDob: number;
    registrationName: string;
    registrationSeqNumber: string | null;
    registrationPhoneNumber: string;
    registrationEmailAddress: string;
    registrationGender: string;
    registrationDob: number; 
  }[]> {
    try {
      // Check if user is relational_manager
      const isRelationalManager = userRoles.some(role => 
        (typeof role === 'string' ? role : role.name) === 'relational_manager'
      );

      // If relational_manager, filter by rmContact, otherwise get all
      const rmContactId = isRelationalManager ? userId : undefined;
      const registrations = await this.repository.getBasicRegistrations(rmContactId, programId, searchText);
      return registrations.map(registration => ({
        registrationId: registration.id,
        userId: registration.userId,
        //user data
        userName: registration.userName ?? '',
        userLegalName: registration.userLegalName ?? '',
        userFirstName: registration.userFirstName ?? '',
        userLastName: registration.userLastName ?? '',
        userPhoneNumber: registration.userCountryCode && registration.userPhoneNumber ? `${registration.userCountryCode}${registration.userPhoneNumber}` : '',
        userEmail: registration.userEmailAddress ?? '',
        userGender: registration.userGender ?? '',
        userDob: registration.userDob ? calculateAge(registration.userDob) : 0,
        //registration data
        registrationName: registration.registrationName ?? '',
        registrationSeqNumber: registration.registrationSeqNumber ?? null,
        registrationPhoneNumber: registration.registrationPhoneNumber ?? '',
        registrationEmailAddress: registration.registrationEmailAddress ?? '',
        registrationGender: registration.registrationGender ?? '',
        registrationDob: registration.registrationDob ? calculateAge(registration.registrationDob) : 0
      }));
    } catch (error) {
      this.logger.error('Error getting basic registrations:', error);
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_GET_FAILED, error);
    }
  }

  /**
   * Clear all registrations for a specific program (environment-specific)
   * This is a dangerous operation and should only be enabled in specific environments
   * @param programId - The program ID to clear registrations for
   * @param user - The user performing the operation
   * @returns Result of the clear operation
   */
  async clearAllRegistrationsForProgram(programId: number, user: User): Promise<{
    message: string;
    clearedCount: number;
    programId: number;
    environment: string;
  }> {
    this.logger.log(replaceStringPlaceholders(CLEAR_REGISTRATION_MESSAGES.CLEAR_ALL_REQUESTED, [programId.toString(), user.id.toString()]));
    try {
      // Environment check - only allow in specific environments
      const currentEnv = this.configService.get<string>('ENV') || 'production';
      const allowedEnvironments = [ENVIRONMENTS.DEVELOPMENT, ENVIRONMENTS.QA];
      
      if (!allowedEnvironments.includes(currentEnv)) {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_CLEAR_NOT_ALLOWED,
          null,
          null,
          currentEnv
        );
      }

      // Additional safety check - require explicit environment variable
      const clearEnabled = this.configService.get<string>('ENABLE_CLEAR_REGISTRATIONS') || 'false';
      if (clearEnabled !== 'true') {
        throw new InifniBadRequestException(
          ERROR_CODES.REGISTRATION_CLEAR_NOT_ENABLED,
          null,
          null,
          currentEnv);
      }

      // Validate program exists
      const program = await this.repository.findProgramWithType(programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString()
        );
      }

      // Perform the clear operation using transaction
      const result = await this.dataSource.transaction(async (manager) => {
        return await this.repository.clearAllRegistrationsForProgram(programId, user.id, manager);
      });

      this.logger.log(replaceStringPlaceholders(CLEAR_REGISTRATION_MESSAGES.SUCCESSFULLY_CLEARED, [result.clearedCount.toString(), programId.toString()]));

      return {
        message: `Successfully cleared ${result.clearedCount} registrations for program ${program.name}`,
        clearedCount: result.clearedCount,
        programId,
        environment: currentEnv,
      };
    } catch (error) {
      this.logger.error(replaceStringPlaceholders(CLEAR_REGISTRATION_MESSAGES.ERROR_CLEARING, [programId.toString()]), error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_CLEAR_FAILED, error);
    }
  }

  /**
   * Update signed URLs for registrations
   * Always regenerates signed URLs since they expire after 7 days (AWS S3 limit)
   * Updates signed URLs in registration_travel_info and registration_travel_plan tables
   * @param programId - The program ID to update signed URLs for
   * @param registrationId - Optional registration ID. If provided, updates only this registration. Otherwise updates all registrations of the program.
   * @param user - The user performing the operation
   * @returns Summary of the update operation
   */
  async updateSignedUrlsForRegistrations(
    programId: number,
    registrationId: number | undefined,
    user: User
  ): Promise<{
    message: string;
    programId: number;
    registrationId?: number;
    updatedCount: {
      registration: number;
      travelInfo: number;
      travelPlan: number;
    };
    details: {
      registration: string[];
      travelInfo: string[];
      travelPlan: string[];
    };
  }> {
    this.logger.log(`Update signed URLs requested for program ${programId}${registrationId ? ` and registration ${registrationId}` : ' (all registrations)'}`);
    
    try {
      // Validate program exists
      const program = await this.repository.findProgramWithType(programId);
      if (!program) {
        throw new InifniNotFoundException(
          ERROR_CODES.PROGRAM_NOTFOUND,
          null,
          null,
          programId.toString()
        );
      }

      // If registrationId is provided, validate it exists and belongs to the program
      if (registrationId) {
        const registration = await this.dataSource
          .getRepository(ProgramRegistration)
          .findOne({ 
            where: { id: registrationId },
            select: ['id', 'programId']
          });
        
        if (!registration) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
            null,
            null,
            registrationId.toString()
          );
        }
        if (registration.programId !== programId) {
          throw new InifniBadRequestException(
            ERROR_CODES.PQ_VALIDATION_FAILED,
            null,
            null,
            `Registration ${registrationId} does not belong to program ${programId}`
          );
        }
      }

      const updatedCount = {
        travelInfo: 0,
        travelPlan: 0,
        registration: 0
      };
      const details = {
        travelInfo: [] as string[],
        travelPlan: [] as string[],
        registration: [] as string[]
      };

      // Perform the update using transaction
      await this.dataSource.transaction(async (manager) => {
        // Build query to get registrations
        const registrationQuery = manager
          .createQueryBuilder(ProgramRegistration, 'reg')
          .leftJoinAndSelect('reg.travelInfo', 'travelInfo')
          .leftJoinAndSelect('reg.travelPlans', 'travelPlan')
          .where('reg.program_id = :programId', { programId })
          .andWhere('reg.deleted_at IS NULL');

        if (registrationId) {
          registrationQuery.andWhere('reg.id = :registrationId', { registrationId });
        }

        const registrations = await registrationQuery.getMany();

        this.logger.log(`Found ${registrations.length} registrations to process`);

        // Process each registration
        for (const registration of registrations) {
          // Update profile URL and video URL signed URLs in registration
          let registrationUpdated = false;
          const registrationUpdatedFields: string[] = [];

          // Regenerate profile picture signed URL
          if (registration.profileUrl && !shouldExcludeFromSignedUrl(registration.profileUrl)) {
            try {
              registration.profileSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(registration.profileUrl);
              registrationUpdatedFields.push('profileSignedUrl');
              registrationUpdated = true;
            } catch (error) {
              this.logger.error(`Failed to generate signed URL for profileUrl in registration ${registration.id}:`, error);
            }
          }

          // Regenerate video signed URL
          if (registration.videoUrl && !shouldExcludeFromSignedUrl(registration.videoUrl)) {
            try {
              registration.videoSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(registration.videoUrl);
              registrationUpdatedFields.push('videoSignedUrl');
              registrationUpdated = true;
            } catch (error) {
              this.logger.error(`Failed to generate signed URL for videoUrl in registration ${registration.id}:`, error);
            }
          }

          if (registrationUpdated) {
            registration.updatedBy = { id: user.id } as User;
            await manager.save(ProgramRegistration, registration);
            updatedCount.registration++;
            details.registration.push(`Registration ${registration.id}: ${registrationUpdatedFields.join(', ')}`);
            this.logger.log(`Regenerated signed URLs for registration ${registration.id}: ${registrationUpdatedFields.join(', ')}`);
          }

          // Update travel info signed URLs
          if (registration.travelInfo && registration.travelInfo.length > 0) {
            for (const travelInfo of registration.travelInfo) {
              let updated = false;
              const updatedFields: string[] = [];

              // Regenerate ID picture signed URL (always regenerate since they expire after 7 days)
              if (travelInfo.idPictureUrl && !shouldExcludeFromSignedUrl(travelInfo.idPictureUrl)) {
                try {
                  travelInfo.idPictureSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(travelInfo.idPictureUrl);
                  updatedFields.push('idPictureSignedUrl');
                  updated = true;
                } catch (error) {
                  this.logger.error(`Failed to generate signed URL for idPictureUrl in travelInfo ${travelInfo.id}:`, error);
                }
              }

              // Regenerate ID back picture signed URL
              if (travelInfo.idBackPictureUrl && !shouldExcludeFromSignedUrl(travelInfo.idBackPictureUrl)) {
                try {
                  travelInfo.idBackPictureSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(travelInfo.idBackPictureUrl);
                  updatedFields.push('idBackPictureSignedUrl');
                  updated = true;
                } catch (error) {
                  this.logger.error(`Failed to generate signed URL for idBackPictureUrl in travelInfo ${travelInfo.id}:`, error);
                }
              }

              // Regenerate passport picture signed URL
              if (travelInfo.passportCopyPictureUrl && !shouldExcludeFromSignedUrl(travelInfo.passportCopyPictureUrl)) {
                try {
                  travelInfo.passportCopyPictureSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(travelInfo.passportCopyPictureUrl);
                  updatedFields.push('passportCopyPictureSignedUrl');
                  updated = true;
                } catch (error) {
                  this.logger.error(`Failed to generate signed URL for passportCopyPictureUrl in travelInfo ${travelInfo.id}:`, error);
                }
              }

              // Regenerate visa picture signed URL
              if (travelInfo.visaCopyPictureUrl && !shouldExcludeFromSignedUrl(travelInfo.visaCopyPictureUrl)) {
                try {
                  travelInfo.visaCopyPictureSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(travelInfo.visaCopyPictureUrl);
                  updatedFields.push('visaCopyPictureSignedUrl');
                  updated = true;
                } catch (error) {
                  this.logger.error(`Failed to generate signed URL for visaCopyPictureUrl in travelInfo ${travelInfo.id}:`, error);
                }
              }

              if (updated) {
                travelInfo.updatedBy = { id: user.id } as User;
                await manager.save(travelInfo);
                updatedCount.travelInfo++;
                details.travelInfo.push(`Registration ${registration.id} - TravelInfo ${travelInfo.id}: ${updatedFields.join(', ')}`);
                this.logger.log(`Regenerated signed URLs for travelInfo ${travelInfo.id}: ${updatedFields.join(', ')}`);
              }
            }
          }

          // Update travel plan signed URLs
          if (registration.travelPlans && registration.travelPlans.length > 0) {
            for (const travelPlan of registration.travelPlans) {
              let updated = false;
              const updatedFields: string[] = [];

              // Regenerate onward journey ticket signed URL (always regenerate since they expire after 7 days)
              if (travelPlan.onwardJourneyTicketUrl && !shouldExcludeFromSignedUrl(travelPlan.onwardJourneyTicketUrl)) {
                try {
                  travelPlan.onwardJourneyTicketSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(travelPlan.onwardJourneyTicketUrl);
                  updatedFields.push('onwardJourneyTicketSignedUrl');
                  updated = true;
                } catch (error) {
                  this.logger.error(`Failed to generate signed URL for onwardJourneyTicketUrl in travelPlan ${travelPlan.id}:`, error);
                }
              }

              // Regenerate return journey ticket signed URL
              if (travelPlan.returnJourneyTicketUrl && !shouldExcludeFromSignedUrl(travelPlan.returnJourneyTicketUrl)) {
                try {
                  travelPlan.returnJourneyTicketSignedUrl = await this.awsS3Service.generateLongLivedSignedUrl(travelPlan.returnJourneyTicketUrl);
                  updatedFields.push('returnJourneyTicketSignedUrl');
                  updated = true;
                } catch (error) {
                  this.logger.error(`Failed to generate signed URL for returnJourneyTicketUrl in travelPlan ${travelPlan.id}:`, error);
                }
              }

              if (updated) {
                travelPlan.updatedBy = { id: user.id } as User;
                await manager.save(travelPlan);
                updatedCount.travelPlan++;
                details.travelPlan.push(`Registration ${registration.id} - TravelPlan ${travelPlan.id}: ${updatedFields.join(', ')}`);
                this.logger.log(`Regenerated signed URLs for travelPlan ${travelPlan.id}: ${updatedFields.join(', ')}`);
              }
            }
          }
        }
      });

      const message = registrationId
        ? `Successfully updated signed URLs for registration ${registrationId}`
        : `Successfully updated signed URLs for all registrations in program ${programId}`;

      this.logger.log(`${message}. Registration: ${updatedCount.registration}, TravelInfo: ${updatedCount.travelInfo}, TravelPlan: ${updatedCount.travelPlan}`);

      return {
        message,
        programId,
        ...(registrationId && { registrationId }),
        updatedCount,
        details
      };
    } catch (error) {
      this.logger.error(`Error updating signed URLs for program ${programId}:`, error);
      handleKnownErrors(ERROR_CODES.REGISTRATION_GET_FAILED, error);
    }
  }
  /**
   * Validates that preference program start dates are still in the future.
   * Prevents users from selecting programs that have already started as preferences.
   *
   * @param preferenceProgramIds - Array of preference program IDs to validate
   * @throws InifniBadRequestException if any preference program has already started
   * @returns Promise<void>
   */
  private async validatePreferenceStartDates(
    preferenceProgramIds: number[],
  ): Promise<void> {
    this.logger.debug('Validating preference start dates', {
      preferenceProgramIds,
    });

    await this.programRegistrationService.validateTargetProgramStartDates(
      preferenceProgramIds,
    );
  }
}

