import { Injectable } from '@nestjs/common';
import { DataSource, EntityManager } from 'typeorm';
import { User, Address } from 'src/common/entities';
import { CreateProgramDto } from './dto/create-program.dto';
import { UpdateProgramDto } from './dto/update-program.dto';
import { UpdateProgramStatusDto } from './dto/update-program-status.dto';
import { ProgramRepository } from './program.repository';
import { ProgramSessionRepository } from '../program-session/program-session.repository';
import { ProgramTypeRepository } from '../program-type/program-type.repository';
import { ProgramValidationService } from './program-validation.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { programConstMessages } from 'src/common/constants/strings-constants';
import { CreateProgramSessionDto } from '../program-session/dto/create-program-session.dto';
import { v4 as uuidv4 } from 'uuid';
import { FeatureFlagService } from '../feature-flag/feature-flag.service';

@Injectable()
export class ProgramService {
  constructor(
    private readonly programRepository: ProgramRepository,
    private readonly programSessionRepository: ProgramSessionRepository,
    private readonly programTypeRepository: ProgramTypeRepository,
    private readonly programValidationService: ProgramValidationService,
    private readonly featureFlagService: FeatureFlagService,
    private readonly logger: AppLoggerService,
    private readonly dataSource: DataSource,
  ) {}

  /**
   * Creates a new program and its sessions as a transaction.
   * Handles inheritance from program type and grouped programs.
   */
  async create(createDto: CreateProgramDto) {
    this.logger.log(programConstMessages.CREATING_PROGRAM(createDto));

    return await this.dataSource.transaction(async (manager: EntityManager) => {
      try {
        // Validate the creation request
        await this.programValidationService.validateCreateProgram(createDto, manager);

        // Get program type to inherit properties and check if it supports grouping
        const programType = await this.programTypeRepository.findOneById(createDto.typeId);
        if (!programType) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_TYPE_NOTFOUND,
            null,
            null,
            createDto.typeId.toString(),
          );
        }

        // Check if this is a grouped program request
        if (createDto.groupedPrograms && createDto.groupedPrograms.length > 0) {
          return await this.createGroupedPrograms(createDto, programType, manager);
        } else {
          // Create single program
          const program = await this.createSingleProgram(createDto, programType, manager);
          return program;
        }
      } catch (error) {
        this.logger.error(ERROR_CODES.PROGRAM_SAVE_FAILED, error?.stack, { error });
        handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
      }
    });
  }

  /**
   * Creates grouped programs with primary (parent) and child programs
   */
  private async createGroupedPrograms(
    createDto: CreateProgramDto,
    programType: any,
    manager: EntityManager,
  ) {
    try {
      const groupId = uuidv4();
      const results: any[] = [];

    // Create and save venue address if provided for primary program
    let savedVenueAddress: Address | undefined = undefined;
    if (createDto.venueAddress) {
      const addressEntity = new Address({
        ...createDto.venueAddress,
        createdAt: new Date(),
        updatedAt: new Date(),
        createdBy: createDto.createdBy.toString(),
        updatedBy: createDto.updatedBy.toString(),
      });
      savedVenueAddress = await manager.save(Address, addressEntity);
    }

    // Create primary (parent) program
    const primaryProgramData = this.inheritFromProgramType(createDto, programType);
    primaryProgramData.isPrimaryProgram = true;
    primaryProgramData.primaryProgramId = null; // Primary program has no parent
    primaryProgramData.groupId = groupId;
    primaryProgramData.groupDisplayOrder = 1;
    
    // Set the saved address ID for primary program
    if (savedVenueAddress) {
      primaryProgramData.venueAddressId = savedVenueAddress.id;
      primaryProgramData.venueAddress = savedVenueAddress;
    }

    const primaryProgram = await this.programRepository.createProgram(primaryProgramData, manager);

    // Create sessions for primary program with address
    await this.createProgramSessions(primaryProgram, primaryProgramData, savedVenueAddress, manager);
    results.push(primaryProgram);

    // Create child programs (grouped programs)
    if (createDto.groupedPrograms && createDto.groupedPrograms.length > 0) {
      for (let i = 0; i < createDto.groupedPrograms.length; i++) {
        const groupedProgramData = { ...createDto.groupedPrograms[i] };

        // Inherit from program type and override with grouped program data
        const inheritedData = this.inheritFromProgramType(groupedProgramData, programType);

        // Set hierarchy properties
        inheritedData.isPrimaryProgram = false;
        inheritedData.primaryProgramId = primaryProgram.id; // Reference to parent
        inheritedData.groupId = groupId; // Same group ID as parent
        inheritedData.groupDisplayOrder = groupedProgramData.groupDisplayOrder || i + 2;
        inheritedData.typeId = createDto.typeId;
        inheritedData.createdBy = createDto.createdBy;
        inheritedData.updatedBy = createDto.updatedBy;

        // Handle individual invoice address for grouped program if provided
        let groupedVenueAddress: Address | undefined = undefined;
        if (groupedProgramData.venueAddress) {
          const groupedAddressEntity = new Address({
            ...groupedProgramData.venueAddress,
            createdAt: new Date(),
            updatedAt: new Date(),
            createdBy: createDto.createdBy.toString(),
            updatedBy: createDto.updatedBy.toString(),
          });
          groupedVenueAddress = await manager.save(Address, groupedAddressEntity);
          inheritedData.venueAddressId = groupedVenueAddress.id;
          inheritedData.venueAddress = groupedVenueAddress;
        } else if (savedVenueAddress) {
          // Inherit primary program's address if no specific address provided
          inheritedData.venueAddressId = savedVenueAddress.id;
          inheritedData.venueAddress = savedVenueAddress;
        }

        const groupedProgram = await this.programRepository.createProgram(inheritedData, manager);

        // Create sessions for grouped program with appropriate address
        const sessionAddress = groupedVenueAddress || savedVenueAddress;
        await this.createProgramSessions(groupedProgram, inheritedData, sessionAddress, manager);
        results.push(groupedProgram);
      }
    }

    return {
      primaryProgram: results[0],
      groupedPrograms: results.slice(1),
      totalCreated: results.length,
    };
    } catch (error) {
      this.logger.log('Error in createGroupedPrograms', { error: error.message, createDto });
      handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
    }
  }

  /**
   * Creates a single program (non-grouped)
   */
  private async createSingleProgram(
    createDto: CreateProgramDto,
    programType: any,
    manager: EntityManager,
  ) {
    try {
      // Create and save venue address if provided
      let savedVenueAddress: Address | undefined = undefined;
    if (createDto.venueAddress) {
      const addressEntity = new Address({
        ...createDto.venueAddress,
        createdAt: new Date(),
        updatedAt: new Date(),
        createdBy: createDto.createdBy.toString(),
        updatedBy: createDto.updatedBy.toString(),
      });
      savedVenueAddress = await manager.save(Address, addressEntity);
    }

    // Inherit properties from program type
    const programData = this.inheritFromProgramType(createDto, programType);

    // Set grouped program properties to false/null for single programs
    programData.isPrimaryProgram = false;
    programData.primaryProgramId = null;
    programData.groupId = null;
    programData.groupDisplayOrder = null;
    programData.isGroupedProgram = false;

    // Set the saved address ID
    if (savedVenueAddress) {
      programData.venueAddressId = savedVenueAddress.id;
      programData.venueAddress = savedVenueAddress;
    }

    const program = await this.programRepository.createProgram(programData, manager);

    // Create sessions with address
    await this.createProgramSessions(program, programData, savedVenueAddress, manager);
    return program;
    } catch (error) {
      this.logger.log('Error in createSingleProgram', { error: error.message, createDto });
      handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
    }
  }

  /**
   * Updates an existing program and handles grouped programs
   */
  async update(id: number, updateDto: UpdateProgramDto) {
    this.logger.log(programConstMessages.UPDATING_PROGRAM(id));

    return await this.dataSource.transaction(async (manager: EntityManager) => {
      try {
        // Validate the update request
        await this.programValidationService.validateUpdateProgram(id, updateDto, manager);

        const existingProgram = await this.programRepository.findOneById(id, manager);
        if (!existingProgram) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            id.toString(),
          );
        }

        // Get program type for inheritance
        const programType = await this.programTypeRepository.findOneById(existingProgram.typeId);

        // Check if this is a grouped program update
        if (existingProgram.isPrimaryProgram && existingProgram.groupId) {
          return await this.updateGroupedPrograms(
            id,
            updateDto,
            programType,
            existingProgram.groupId,
            manager,
          );
        } else {
          // Single program update
          return await this.updateSingleProgram(id, updateDto, programType, manager);
        }
      } catch (error) {
        this.logger.error(ERROR_CODES.PROGRAM_SAVE_FAILED, error?.stack, { error, id });
        handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
      }
    });
  }

/**
 * Updates grouped programs - intelligently updates existing programs, creates new ones, and explicitly deletes specified ones
 */
private async updateGroupedPrograms(
  primaryProgramId: number,
  updateDto: UpdateProgramDto,
  programType: any,
  groupId: string,
  manager: EntityManager,
) {
  try {
    // Handle venue address for primary program update
    let savedVenueAddress: Address | undefined = undefined;
  if (updateDto.venueAddress) {
    const addressEntity = new Address({
      ...updateDto.venueAddress,
      createdAt: new Date(),
      updatedAt: new Date(),
      createdBy: updateDto.updatedBy.toString(),
      updatedBy: updateDto.updatedBy.toString(),
    });
    savedVenueAddress = await manager.save(Address, addressEntity);
  }

  // Update primary program while preserving group properties
  const inheritedData = this.inheritFromProgramType(updateDto, programType);

  // Preserve group hierarchy
  inheritedData.isPrimaryProgram = true;
  inheritedData.groupId = groupId; // Preserve existing groupId
  inheritedData.primaryProgramId = null;
  inheritedData.groupDisplayOrder = 1;

  // Set address for primary program
  if (savedVenueAddress) {
    inheritedData.venueAddressId = savedVenueAddress.id;
    inheritedData.venueAddress = savedVenueAddress;
  }

  // Update primary program
  const updatedPrimaryProgram = await this.programRepository.updateProgram(
    primaryProgramId,
    inheritedData,
    manager,
  );

  // Delete existing sessions for primary program and recreate them
  await this.deleteRelatedSessions(primaryProgramId, { id: updateDto.updatedBy } as User, manager);
  await this.createProgramSessions(updatedPrimaryProgram, inheritedData, savedVenueAddress, manager);

  // Handle explicit program deletions first
  if (updateDto.deletedProgramIds && updateDto.deletedProgramIds.length > 0) {
    this.logger.log(`Explicitly deleting programs: ${updateDto.deletedProgramIds.join(', ')}`);
    
    for (const programId of updateDto.deletedProgramIds) {
      // Delete sessions first
      await this.deleteRelatedSessions(programId, { id: updateDto.updatedBy } as User, manager);
      // Then soft delete the program
      await this.programRepository.softDeleteProgram(programId, { id: updateDto.updatedBy } as User, manager);
    }
  }

  // Handle grouped programs intelligently
  const results = [updatedPrimaryProgram];
  
  if (updateDto.groupedPrograms && updateDto.groupedPrograms.length > 0) {
    // Get existing child programs for comparison (excluding those explicitly marked for deletion)
    const existingChildPrograms = await this.programRepository.findGroupedPrograms(groupId, manager);
    const existingChildIds = existingChildPrograms
      .filter(p => !p.isPrimaryProgram)
      .map(p => p.id);

    // Filter out explicitly deleted programs
    const deletedIds = updateDto.deletedProgramIds || [];
    const availableChildIds = existingChildIds.filter(id => !deletedIds.includes(id));

    // Collect IDs from the update request
    const requestedIds = updateDto.groupedPrograms
      .filter(gp => gp.id)
      .map(gp => gp.id);

    // Find programs to delete (existing programs not in the request and not explicitly deleted)
    // Only delete programs that exist, are not being updated, and are not explicitly deleted
    const programsToDelete = availableChildIds.filter(id => 
      !requestedIds.includes(id) && !deletedIds.includes(id)
    );
    
    // Soft delete programs that are no longer needed (and not explicitly deleted)
    for (const programId of programsToDelete) {
      this.logger.log(`Auto-deleting unreferenced program: ${programId}`);
      await this.deleteRelatedSessions(programId, { id: updateDto.updatedBy } as User, manager);
      await this.programRepository.softDeleteProgram(programId, { id: updateDto.updatedBy } as User, manager);
    }

    // Process each grouped program
    for (let i = 0; i < updateDto.groupedPrograms.length; i++) {
      const groupedProgramData = { ...updateDto.groupedPrograms[i] };
      const inheritedGroupData = this.inheritFromProgramType(groupedProgramData, programType);

      // Set hierarchy properties
      inheritedGroupData.isPrimaryProgram = false;
      inheritedGroupData.primaryProgramId = updatedPrimaryProgram.id;
      inheritedGroupData.groupId = groupId;
      inheritedGroupData.groupDisplayOrder = groupedProgramData.groupDisplayOrder || i + 2;
      inheritedGroupData.typeId = updatedPrimaryProgram.typeId;
      inheritedGroupData.updatedBy = updateDto.updatedBy;

      // Handle individual venue address for grouped program
      let groupedVenueAddress: Address | undefined = undefined;
      if (groupedProgramData.venueAddress) {
        const groupedAddressEntity = new Address({
          ...groupedProgramData.venueAddress,
          createdAt: new Date(),
          updatedAt: new Date(),
          createdBy: updateDto.updatedBy.toString(),
          updatedBy: updateDto.updatedBy.toString(),
        });
        groupedVenueAddress = await manager.save(Address, groupedAddressEntity);
        inheritedGroupData.venueAddressId = groupedVenueAddress.id;
        inheritedGroupData.venueAddress = groupedVenueAddress;
      } else if (savedVenueAddress) {
        // Inherit primary program's address if no specific address provided
        inheritedGroupData.venueAddressId = savedVenueAddress.id;
        inheritedGroupData.venueAddress = savedVenueAddress;
      }

      let groupedProgram;

      if (groupedProgramData.id) {
        // Update existing program (only if not marked for deletion)
        if (!deletedIds.includes(groupedProgramData.id)) {
          this.logger.log(`Updating existing grouped program with ID: ${groupedProgramData.id}`);
          
          // Delete existing sessions for this program before updating
          await this.deleteRelatedSessions(groupedProgramData.id, { id: updateDto.updatedBy } as User, manager);
          
          groupedProgram = await this.programRepository.updateProgram(
            groupedProgramData.id,
            inheritedGroupData,
            manager,
          );
        } else {
          // Skip programs that are marked for deletion
          this.logger.log(`Skipping update for program ${groupedProgramData.id} as it's marked for deletion`);
          continue;
        }
      } else {
        // Create new program
        this.logger.log(`Creating new grouped program: ${groupedProgramData.name}`);
        inheritedGroupData.createdBy = updateDto.updatedBy;
        
        groupedProgram = await this.programRepository.createProgram(
          inheritedGroupData,
          manager,
        );
      }

      // Create sessions for the grouped program (whether updated or created)
      const sessionAddress = groupedVenueAddress || savedVenueAddress;
      await this.createProgramSessions(groupedProgram, inheritedGroupData, sessionAddress, manager);
      results.push(groupedProgram);
    }
  } else {
    // If no grouped programs provided, delete all existing child programs (except explicitly deleted ones)
    const existingChildPrograms = await this.programRepository.findGroupedPrograms(groupId, manager);
    const childPrograms = existingChildPrograms.filter(p => !p.isPrimaryProgram);
    const deletedIds = updateDto.deletedProgramIds || [];
    
    // Only delete programs that are not explicitly deleted (to avoid double deletion)
    const programsToDelete = childPrograms.filter(program => !deletedIds.includes(program.id));
    
    for (const childProgram of programsToDelete) {
      this.logger.log(`Auto-deleting all child programs: ${childProgram.id}`);
      await this.deleteRelatedSessions(childProgram.id, { id: updateDto.updatedBy } as User, manager);
      await this.programRepository.softDeleteProgram(childProgram.id, { id: updateDto.updatedBy } as User, manager);
    }
  }

  // Calculate total operations for response
  const totalDeleted = (updateDto.deletedProgramIds || []).length;
  const totalUpdated = results.length;

  this.logger.log(`Update summary: ${totalUpdated} programs updated/created, ${totalDeleted} programs explicitly deleted`);

  return {
    primaryProgram: results[0],
    groupedPrograms: results.slice(1),
    totalUpdated: results.length,
    totalDeleted,
    deletedProgramIds: updateDto.deletedProgramIds || [],
  };
  } catch (error) {
    this.logger.log('Error in updateGroupedPrograms', { error: error.message, primaryProgramId, updateDto });
    handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
  }
}

  /**
   * Updates a single program
   */
  private async updateSingleProgram(
    id: number,
    updateDto: UpdateProgramDto,
    programType: any,
    manager: EntityManager,
  ) {
    try {
      // Handle venue address for single program update
      let savedVenueAddress: Address | undefined = undefined;
    if (updateDto.venueAddress) {
      const addressEntity = new Address({
        ...updateDto.venueAddress,
        createdAt: new Date(),
        updatedAt: new Date(),
        createdBy: updateDto.updatedBy.toString(),
        updatedBy: updateDto.updatedBy.toString(),
      });
      savedVenueAddress = await manager.save(Address, addressEntity);
    }

    const inheritedData = this.inheritFromProgramType(updateDto, programType);
    
    // Set address for single program
    if (savedVenueAddress) {
      inheritedData.venueAddressId = savedVenueAddress.id;
      inheritedData.venueAddress = savedVenueAddress;
    }

    const updatedProgram = await this.programRepository.updateProgram(id, inheritedData, manager);

    // Remove existing sessions and create new ones
    await this.deleteRelatedSessions(id, { id: updateDto.updatedBy } as User, manager);
    await this.createProgramSessions(updatedProgram, inheritedData, savedVenueAddress, manager);

    return updatedProgram;
    } catch (error) {
      this.logger.log('Error in updateSingleProgram', { error: error.message, id, updateDto });
      handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
    }
  }

  /**
   * Deletes a program by its ID and handles grouped programs
   */
  async remove(id: number, user: User) {
    this.logger.log(programConstMessages.REMOVING_PROGRAM(id));

    return await this.dataSource.transaction(async (manager: EntityManager) => {
      try {
        // Validate deletion
        await this.programValidationService.validateDeleteProgram(id, manager);

        const program = await this.programRepository.findOneById(id, manager);
        if (!program) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            id.toString(),
          );
        }

        // Check deletion safety
        const { canDelete, reasons } = await this.programValidationService.validateDeletionSafety(
          id,
          manager,
        );
        if (!canDelete) {
          throw new Error(`Cannot delete program: ${reasons.join(', ')}`);
        }

        // If this is a primary program, delete all child programs in the group
        if (program.isPrimaryProgram && program.groupId) {
          // Delete sessions for all programs in the group before removing them
          await this.deleteSessionsForGroup(program.groupId, user, manager);

          // Delete all child programs in the group
          await this.programRepository.softDeleteChildPrograms(program.groupId, manager);
        } else {
          // For single programs or child programs, just delete their sessions
          await this.deleteRelatedSessions(id, user, manager);
        }

        // Delete the program itself
        await this.programRepository.softDeleteProgram(id, user, manager);

        this.logger.log(`Successfully deleted program ${id} and related data`);
      } catch (error) {
        this.logger.error(ERROR_CODES.PROGRAM_DELETE_FAILED, error?.stack, { error, id });
        handleKnownErrors(ERROR_CODES.PROGRAM_DELETE_FAILED, error);
      }
    });
  }

  /**
   * Removes only child programs in a group (preserves parent)
   */
  private async removeChildPrograms(groupId: string, manager: EntityManager) {
    try {
      await this.programRepository.softDeleteChildPrograms(groupId, manager);
    } catch (error) {
      this.logger.error(`Error removing child programs for group ${groupId}:`, error?.stack, { error, groupId });
      handleKnownErrors(ERROR_CODES.PROGRAM_DELETE_FAILED, error);
    }
  }

  /**
   * Deletes sessions for all programs in a group
   */
  private async deleteSessionsForGroup(groupId: string, user: User, manager: EntityManager) {
    try {
      // Find all programs in the group using the updated method
      const groupPrograms = await this.programRepository.findAllGroupPrograms(groupId, manager);

      // Delete sessions for each program
      for (const program of groupPrograms) {
        await this.deleteRelatedSessions(program.id, user, manager);
      }
    } catch (error) {
      this.logger.error(`Error deleting sessions for group ${groupId}:`, error);
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_DELETE_FAILED, error);
    }
  }

  /**
   * Creates sessions for a program based on configuration
   * Updated to properly handle invoice address mapping
   */
  private async createProgramSessions(
    program: any,
    programData: any,
    venueAddress: Address | undefined,
    manager: EntityManager
  ) {
    try {
      if (!programData.noOfSession || programData.noOfSession === 0) {
      // Create single session
      const session = await this.createProgramSession(program, programData, venueAddress, manager);
      program.sessions = [session];
    } else if (programData.programSessions?.length > 0) {
      // Create multiple sessions
      const sessions = await this.createMultipleSessions(program, programData, venueAddress, manager);
      program.sessions = sessions;
    } else {
      // Create default session if no sessions provided
      const session = await this.createProgramSession(program, programData, venueAddress, manager);
      program.sessions = [session];
    }
    } catch (error) {
      this.logger.log('Error in createProgramSessions', { error: error.message, program: program.id });
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
    }
  }

  /**
   * Creates a session using program data with proper invoice address mapping
   */
  private async createProgramSession(
    program: any,
    programData: any,
    venueAddress: Address | undefined,
    manager: EntityManager
  ) {
    try {
      const sessionDto: CreateProgramSessionDto = {
        programId: program.id,
        name: program.name,
        code: program.code,
        displayOrder: 1,
        startsAt: program.startsAt || programData.startsAt,
        endsAt: program.endsAt || programData.endsAt,
        venue: program.venue,
        helplineNumber: program.helplineNumber,
        emailSenderName: program.emailSenderName,
        venueNameInEmails: program.venueNameInEmails,
        launchDate: program.launchDate,
        modeOfOperation: program.modeOfOperation,
        meetingLink: program.meta?.meetingLink,
        meetingId: program.meta?.meetingId,
        meetingPassword: program.meta?.meetingPassword,
        totalSeats: program.totalSeats,
        waitlistTriggerCount: program.waitlistTriggerCount,
        availableSeats: program.availableSeats,
        reservedSeats: program.filledSeats,
        status: this.mapProgramStatusToSession(program.status || 'active'),
        isActive: program.isActive,
        checkinAt: program.checkinAt || programData.checkinAt,
        checkoutAt: program.checkoutAt || programData.checkoutAt,
        bannerImageUrl: program.bannerImageUrl,
        bannerAnimationUrl: program.bannerAnimationUrl,
        description: program.description,
        registrationStartsAt: program.registrationStartsAt || programData.registrationStartsAt,
        registrationEndsAt: program.registrationEndsAt || programData.registrationEndsAt,
        checkinEndsAt: program.checkinEndsAt || programData.checkinEndsAt,
        checkoutEndsAt: program.checkoutEndsAt || programData.checkoutEndsAt,
        logoUrl: program.logoUrl,
        meta: program.meta,
        cgst: program.cgst,
        igst: program.igst,
        sgst: program.sgst,
        tdsPercent: program.tdsPercent,
        gstNumber: program.gstNumber,
        basePrice: program.basePrice,
        gstPercentage: program.gstPercentage,
        tdsApplicability: program.tdsApplicability,
        invoiceSenderPan: program.invoiceSenderPan,
        invoiceSenderName: program.invoiceSenderName,
        invoiceSenderAddress: program.invoiceSenderAddress,
        venueAddress: venueAddress ? {
          addr1: venueAddress?.addr1,
          addr2: venueAddress?.addr2,
          landmark: venueAddress?.landmark,
          city: venueAddress?.city,
          state: venueAddress?.state,
          country: venueAddress?.country,
          pincode: venueAddress?.pincode,
          lat: venueAddress?.lat,
          long: venueAddress?.long,
          type: venueAddress?.type,
        } : undefined,
        invoiceSenderCin: program.invoiceSenderCin,
        programFee: program.programFee,
        currency: program.currency,
        limitedSeats: program.limitedSeats,
        createdBy: programData.createdBy,
        updatedBy: programData.updatedBy,
      };

      return await this.programSessionRepository.createSession(sessionDto, manager);
    } catch (error) {
      this.logger.error(`Error creating program session for program ${program.id}:`, error?.stack, { error, programId: program.id });
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
    }
  }

  /**
   * Creates multiple sessions for a program with proper invoice address mapping
   */
  private async createMultipleSessions(
    program: any,
    programData: any,
    venueAddress: Address | undefined,
    manager: EntityManager
  ) {
    try {
      const sessions: any[] = [];
      for (let i = 0; i < programData.programSessions.length; i++) {
        const sessionDto = {
          ...programData.programSessions[i],
          programId: program.id,
          displayOrder: programData.programSessions[i].displayOrder || i + 1,
          createdBy: programData.createdBy,
          updatedBy: programData.updatedBy,
          helplineNumber: program.helplineNumber,
          emailSenderName: program.emailSenderName,
          venueNameInEmails: program.venueNameInEmails,
          launchDate: program.launchDate,
          invoiceSenderAddress: program.invoiceSenderAddress,
          venueAddress: venueAddress ? {
            addr1: venueAddress.addr1,
            addr2: venueAddress.addr2,
            landmark: venueAddress.landmark,
            city: venueAddress.city,
            state: venueAddress.state,
            country: venueAddress.country,
            pincode: venueAddress.pincode,
            lat: venueAddress.lat,
            long: venueAddress.long,
            type: venueAddress.type,
          } : undefined,
        };

        const session = await this.programSessionRepository.createSession(sessionDto, manager);
        sessions.push(session);
      }
      return sessions;
    } catch (error) {
      this.logger.error(`Error creating multiple sessions for program ${program.id}:`, error?.stack, { error, programId: program.id });
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
    }
  }

  /**
   * Inherits properties from program type and overrides with provided data
   */
  private inheritFromProgramType(createDto: any, programType: any): any {
    return {
      // Inherited from program type (can be overridden)
      modeOfOperation: createDto.modeOfOperation || programType.modeOfOperation,
      onlineType: createDto.onlineType || programType.onlineType,
      maxSessionDurationDays:
        createDto.maxSessionDurationDays || programType.maxSessionDurationDays,
      hasMultipleSessions:
        createDto.hasMultipleSessions !== undefined
          ? createDto.hasMultipleSessions
          : programType.hasMultipleSessions,
      frequency: createDto.frequency || programType.frequency,
      defaultStartTime: createDto.startTime || programType.defaultStartTime,
      defaultEndTime: createDto.endTime || programType.defaultEndTime,
      defaultDuration: createDto.duration || programType.defaultDuration,
      requiresResidence:
        createDto.requiresResidence !== undefined
          ? createDto.requiresResidence
          : programType.requiresResidence,
      involvesTravel:
        createDto.isTravelInvolved !== undefined
          ? createDto.isTravelInvolved
          : programType.involvesTravel,
      hasCheckinCheckout:
        createDto.hasCheckinCheckout !== undefined
          ? createDto.hasCheckinCheckout
          : programType.hasCheckinCheckout,
      requiresPayment:
        createDto.requiresPayment !== undefined
          ? createDto.requiresPayment
          : programType.requiresPayment,
      requiresAttendanceAllSessions:
        createDto.requiresAttendanceAllSessions !== undefined
          ? createDto.requiresAttendanceAllSessions
          : programType.requiresAttendanceAllSessions,
      allowsMinors:
        createDto.allowsMinors !== undefined ? createDto.allowsMinors : programType.allowsMinors,
      allowsProxyRegistration:
        createDto.allowsProxyRegistration !== undefined
          ? createDto.allowsProxyRegistration
          : programType.allowsProxyRegistration,
      requiresApproval:
        createDto.requiresApproval !== undefined
          ? createDto.requiresApproval
          : programType.requiresApproval,
      registrationLevel: createDto.registrationLevel || programType.registrationLevel,
      noOfSession:
        createDto.noOfSession !== undefined ? createDto.noOfSession : programType.noOfSession,
      waitlistApplicable:
        createDto.waitlistApplicable !== undefined
          ? createDto.waitlistApplicable
          : programType.waitlistApplicable,
      maxCapacity:
        createDto.maxCapacity !== undefined ? createDto.maxCapacity : programType.maxCapacity,
      venue: createDto.venue || programType.venue,
      limitedSeats:
        createDto.limitedSeats !== undefined ? createDto.limitedSeats : programType.limitedSeats,

      // Always from createDto (required fields)
      typeId: createDto.typeId,
      name: createDto.name,
      code: createDto.code,
      description: createDto.description,
      startsAt: createDto.startsAt,
      endsAt: createDto.endsAt,
      duration: createDto.duration,
      totalSeats: createDto.totalSeats,
      waitlistTriggerCount: createDto.waitlistTriggerCount,
      availableSeats: createDto.availableSeats,
      meta: createDto.meta,
      registrationStartsAt: createDto.registrationStartsAt,
      registrationEndsAt: createDto.registrationEndsAt,
      basePrice: createDto.basePrice,
      gstPercentage: createDto.gstPercentage,
      cgst: createDto.cgst,
      sgst: createDto.sgst,
      igst: createDto.igst,
      tdsApplicability: createDto.tdsApplicability,
      invoiceSenderPan: createDto.invoiceSenderPan,
      invoiceSenderName: createDto.invoiceSenderName,
      invoiceSenderCin: createDto.invoiceSenderCin,
      invoiceSenderAddress: createDto.invoiceSenderAddress,
      tdsPercent: createDto.tdsPercent,
      gstNumber: createDto.gstNumber,
      currency: createDto.currency,
      helplineNumber: createDto.helplineNumber,
      emailSenderName: createDto.emailSenderName,
      venueNameInEmails: createDto.venueNameInEmails,
      launchDate: createDto.launchDate,
      status: createDto.status,
      isActive: createDto.isActive !== undefined ? createDto.isActive : true,
      program: createDto.program,
      programFee: createDto.programFee,
      checkinAt: createDto.checkinAt,
      checkoutAt: createDto.checkoutAt,
      checkinEndsAt: createDto.checkinEndsAt,
      checkoutEndsAt: createDto.checkoutEndsAt,
      seekerCanShareExperience: createDto.seekerCanShareExperience || false,
      totalBedCount: createDto.totalBedCount || 0,
      logoUrl: createDto.logoUrl,
      bannerImageUrl: createDto.bannerImageUrl,
      subProgramType: createDto.subProgramType,
      bannerAnimationUrl: createDto.bannerAnimationUrl,
      programSessions: createDto.programSessions,
      createdBy: createDto.createdBy,
      updatedBy: createDto.updatedBy,
      childMaxAge: createDto?.childMaxAge,
      elderMinAge: createDto?.elderMinAge,
    };
  }

  /**
   * Retrieves all programs with grouped program handling
   */
  async findAll(
    limit: number,
    offset: number,
    searchText: string,
    typeId: number[],
    parsedFilters: Record<string, any>,
  ) {
    this.logger.log(programConstMessages.FINDING_ALL_PROGRAMS(limit, offset, searchText));
    try {
      const result = await this.programRepository.findAllPrograms(
        limit,
        offset,
        searchText,
        typeId,
        parsedFilters,
      );

       const orderedPrograms = this.reorderProgramsByType(result.data);
      // Group the programs if they are grouped
      const groupedResult = this.groupProgramsInResponse(orderedPrograms);

      const featureFlags = await this.featureFlagService.getActiveFlags();
      // these are array of objects need to map like key value pair 
      const featureFlagsMap = featureFlags.reduce((acc, flag) => {
        acc[flag.key] = flag.flag;
        return acc;
      }, {});

      return {
        ...result,
        data: groupedResult,
        featureFlags: featureFlagsMap,
      };
    } catch (error) {
      this.logger.error(ERROR_CODES.PROGRAM_GET_FAILED, error?.stack, { error });
      handleKnownErrors(ERROR_CODES.PROGRAM_GET_FAILED, error);
    }
  }

  /**
   * Groups programs in response - shows primary program with grouped programs
   */
  private groupProgramsInResponse(programs: any[]) {
    const grouped: any[] = [];
    const processedGroupIds = new Set();

    for (const program of programs) {
      if (program.groupId && !processedGroupIds.has(program.groupId)) {
        // This is a grouped program
        if (program.isPrimaryProgram) {
          const groupedPrograms = programs
            .filter((p) => p.groupId === program.groupId && !p.isPrimaryProgram)
            .sort((a, b) => (a.groupDisplayOrder || 0) - (b.groupDisplayOrder || 0));

          grouped.push({
            ...program,
            groupedPrograms,
          });

          processedGroupIds.add(program.groupId);
        }
      } else if (!program.groupId) {
        // Regular single program
        grouped.push(program);
      }
    }

    return grouped;
  }

  /**
   * Retrieves a single program by its ID with grouped program handling
   */
  async findOne(id: number) {
    this.logger.log(programConstMessages.FINDING_PROGRAM_BY_ID(id));
    try {
      const program = await this.programRepository.findOneById(id);
      if (!program) {
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_NOTFOUND, null, null, id.toString());
      }

      // If this is a primary program, include grouped programs
      if (program.isPrimaryProgram && program.groupId) {
        const groupedPrograms = await this.programRepository.findGroupedPrograms(program.groupId);
        return {
          ...program,
          groupedPrograms: groupedPrograms
            .filter((p) => !p.isPrimaryProgram)
            .sort((a, b) => (a.groupDisplayOrder || 0) - (b.groupDisplayOrder || 0)),
        };
      }

      return program;
    } catch (error) {
      this.logger.error(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error?.stack, { error, id });
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Updates only the status of a program and its sessions
   */
  async updateStatus(id: number, statusDto: UpdateProgramStatusDto) {
    this.logger.log(`Updating program status for ${id}`);
    return await this.dataSource.transaction(async (manager: EntityManager) => {
      try {
        const program = await this.programRepository.findOneById(id, manager);
        if (!program) {
          throw new InifniNotFoundException(
            ERROR_CODES.PROGRAM_NOTFOUND,
            null,
            null,
            id.toString(),
          );
        }

        const programIds = [program.id];

        if (program.isPrimaryProgram && program.groupId) {
          const grouped = await this.programRepository.findGroupedPrograms(program.groupId, manager);
          programIds.push(...grouped.filter((g) => !g.isPrimaryProgram).map((g) => g.id));
        }

        for (const programId of programIds) {
          await this.programRepository.updateById(
            programId,
            { status: statusDto.status, updatedBy: statusDto.updatedBy },
            manager,
          );

          const sessionStatus = this.mapProgramStatusToSession(statusDto.status);
          await this.programSessionRepository.updateSessionsStatusByProgramId(
            programId,
            sessionStatus,
            statusDto.updatedBy,
            manager,
          );
        }

        return await this.programRepository.findOneById(id, manager);
      } catch (error) {
        this.logger.error(ERROR_CODES.PROGRAM_SAVE_FAILED, error?.stack, { error, id });
        handleKnownErrors(ERROR_CODES.PROGRAM_SAVE_FAILED, error);
      }
    });
  }

  /**
   * Soft deletes all program sessions related to a program
   */
  private async deleteRelatedSessions(programId: number, user: User, manager?: EntityManager) {
    try {
      await this.programSessionRepository.softDeleteSessionsByProgramId(programId, user, manager);
    } catch (error) {
      this.logger.error(ERROR_CODES.PROGRAM_SESSION_DELETE_FAILED, error?.stack, {
        error,
        programId,
      });
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_DELETE_FAILED, error);
    }
  }

  /**
   * Maps program status to allowed session status values
   */
  private mapProgramStatusToSession(
    programStatus: string,
  ): 'active' | 'completed' | 'cancelled' | 'scheduled' | 'postponed' {
    const statusMap: Record<
      string,
      'active' | 'completed' | 'cancelled' | 'scheduled' | 'postponed'
    > = {
      active: 'active',
      completed: 'completed',
      cancelled: 'cancelled',
      registration_open: 'scheduled',
      registration_closed: 'scheduled',
      draft: 'scheduled',
    };
    return statusMap[programStatus] || 'active';
  }

    /**
   * Reorders programs based on program type key priority
   */
  private reorderProgramsByType(programs: any[]) {
    const priority = ['PT_HDBMSD', 'PT_ENTRAINMENT', 'PT_INFINIPATH'];
    return [...programs].sort((a, b) => {
      const indexA = priority.indexOf(a?.type?.key);
      const indexB = priority.indexOf(b?.type?.key);
      const valA = indexA === -1 ? priority.length : indexA;
      const valB = indexB === -1 ? priority.length : indexB;
      return valA - valB;
    });
  }

  async findByIds(ids: number[]) {
    try {
      return await this.programRepository.findByIds(ids);
    } catch (error) {
      this.logger.error(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error?.stack, { error, ids });
      handleKnownErrors(ERROR_CODES.PROGRAM_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Retrieves only completed programs with basic information (id and name).
   * @returns Array of completed programs with only id and name fields.
   */
  async getCompletedPrograms(): Promise<{ id: number; name: string }[]> {
    try {
      this.logger.log('Fetching completed programs');
      const completedPrograms = await this.programRepository.getCompletedPrograms();
      this.logger.log(`Found ${completedPrograms.length} completed programs`);
      return completedPrograms;
    } catch (error) {
      this.logger.error('Error fetching completed programs', error?.stack, { error });
      handleKnownErrors(ERROR_CODES.PROGRAM_GET_FAILED, error);
    }
  }
}