import { Injectable } from '@nestjs/common';
import { User } from 'src/common/entities';
import { CreateProgramSessionDto } from './dto/create-program-session.dto';
import { UpdateProgramSessionDto } from './dto/update-program-session.dto';
import { ProgramSessionRepository } from './program-session.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { programSessionConstMessages } from 'src/common/constants/strings-constants';
import { DataSource } from 'typeorm';
import { InjectDataSource } from '@nestjs/typeorm';

@Injectable()
export class ProgramSessionService {
  constructor(
    private readonly programSessionRepository: ProgramSessionRepository,
    private readonly logger: AppLoggerService,
    @InjectDataSource()
    private readonly dataSource: DataSource,
  ) {}

  /**
   * Creates a new program session.
   * Validates the existence of the program and creator user.
   * Ensures the session is unique (name and number within program).
   * @param createDto - Data transfer object containing session details.
   * @returns The created session.
   * @throws NotFoundException if the program or user does not exist.
   * @throws BadRequestException if the session already exists or data is invalid.
   */
  async create(createDto: CreateProgramSessionDto) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(programSessionConstMessages.CREATING_SESSION(createDto));
      try {
        return await this.programSessionRepository.createSession(createDto, manager);
      } catch (error) {
        this.logger.error('Error creating program session', error?.stack, { error, createDto });
        handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
      }
    });
  }

  /**
   * Retrieves all program sessions with optional pagination and search functionality.
   * @param limit - Number of records per page.
   * @param offset - Offset for pagination.
   * @param searchText - Optional search text to filter sessions by name.
   * @param programId - Optional program ID to filter sessions by program.
   * @param parsedFilters - Optional additional filters.
   * @returns A paginated list of sessions and metadata.
   */
  async findAll(limit: number, offset: number, searchText: string, programId: number[], parsedFilters: Record<string, any>) {
    this.logger.log(programSessionConstMessages.FINDING_ALL_SESSIONS(limit, offset, searchText));
    try {
      return await this.programSessionRepository.findAllSessions(limit, offset, searchText, programId, parsedFilters);
    } catch (error) {
      this.logger.error('Error finding all program sessions', error?.stack, { error, limit, offset, searchText, programId });
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_GET_FAILED, error);
    }
  }

  /**
   * Retrieves a single program session by its ID.
   * @param id - ID of the session to retrieve.
   * @returns The session data if found.
   * @throws NotFoundException if the session does not exist.
   */
  async findOne(id: number) {
    this.logger.log(programSessionConstMessages.FINDING_SESSION_BY_ID(id));
    try {
      const session = await this.programSessionRepository.findOneById(id);
      if (!session) {
        this.logger.error(programSessionConstMessages.SESSION_NOT_FOUND_ID(id));
        throw new InifniNotFoundException(ERROR_CODES.PROGRAM_SESSION_NOTFOUND, null, null, id.toString());
      }
      return session;
    } catch (error) {
      this.logger.error('Error finding program session by ID', error?.stack, { error, id });
      handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_FIND_BY_ID_FAILED, error);
    }
  }

  /**
   * Updates an existing program session by its ID.
   * Validates the existence of the program and updater user.
   * Ensures the updated session is unique (name and number within program).
   * @param id - ID of the session to update.
   * @param updateDto - Data transfer object containing updated session details.
   * @returns The updated session.
   * @throws NotFoundException if the session, program, or user does not exist.
   * @throws BadRequestException if a duplicate session exists or data is invalid.
   */
  async update(id: number, updateDto: UpdateProgramSessionDto) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(programSessionConstMessages.UPDATING_SESSION(id));
      try {
        return await this.programSessionRepository.updateSession(id, updateDto, manager);
      } catch (error) {
        this.logger.error('Error updating program session', error?.stack, { error, id, updateDto });
        handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_SAVE_FAILED, error);
      }
    });
  }

  /**
   * Deletes a program session by its ID (soft delete).
   * @param id - ID of the session to delete.
   * @param user - User performing the deletion.
   * @returns void
   * @throws NotFoundException if the session does not exist.
   * @throws BadRequestException if the deletion fails.
   */
  async remove(id: number, user: User) {
    return await this.dataSource.transaction(async (manager) => {
      this.logger.log(programSessionConstMessages.REMOVING_SESSION(id));
      try {
        return await this.programSessionRepository.softDeleteSession(id, user, manager);
      } catch (error) {
        this.logger.error('Error deleting program session', error?.stack, { error, id, userId: user?.id });
        handleKnownErrors(ERROR_CODES.PROGRAM_SESSION_DELETE_FAILED, error);
      }
    });
  }
}