import { 
  Controller, 
  Get, 
  Post, 
  Body, 
  Param, 
  Put, 
  Delete, 
  HttpStatus,
  ParseIntPipe,
  ValidationPipe,
  UsePipes,
  HttpCode,
  Query,
  Res,
  UseGuards,
  Req,
} from '@nestjs/common';
import { Response } from 'express';
import { ProgramSessionService } from './program-session.service';
import { CreateProgramSessionDto } from './dto/create-program-session.dto';
import { UpdateProgramSessionDto } from './dto/update-program-session.dto'; 
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiQuery, ApiBearerAuth, ApiSecurity } from '@nestjs/swagger';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { programSessionConstMessages, paginationConstants, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('program-session')
@Controller('program-session')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class ProgramSessionController {
  constructor(
    private readonly programSessionService: ProgramSessionService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Creates a new program session.
   * Validates the input using a validation pipe and ensures the session is created successfully.
   * @param createProgramSessionDto - Data transfer object containing session details.
   * @returns A success message and the created session data.
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programSessionConstMessages.CREATE_NEW_SESSION })
  @ApiBody({ type: CreateProgramSessionDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(@Body() createProgramSessionDto: CreateProgramSessionDto, @Req() req: any, @Res() res: Response) {
    this.logger.log(programSessionConstMessages.CREATE_SESSION_REQUEST_RECEIVED, createProgramSessionDto);
    try {
      const userId = req.user?.id || createProgramSessionDto.createdBy;
      createProgramSessionDto.createdBy = userId;
      createProgramSessionDto.updatedBy = userId;
      const data = await this.programSessionService.create(createProgramSessionDto);
      const respData = {
        id: data.id,
        name: data.name,
        code: data.code,
        programId: data.programId,
        displayOrder: data.displayOrder,
        status: data.status,
      };
      this.logger.log(programSessionConstMessages.SESSION_CREATED, respData);
      await this.responseService.success(res, programSessionConstMessages.SESSION_CREATED, respData, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves all program sessions with optional pagination and search functionality.
   * @param limit - Number of records per page (default: 10).
   * @param offset - Offset for pagination (default: 0).
   * @param searchText - Optional search text to filter sessions by name.
   * @returns A paginated list of sessions and metadata.
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programSessionConstMessages.GET_ALL_SESSIONS })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: paginationConstants.SEARCH_TEXT })
  @ApiQuery({ name: 'programId', type: String, required: false, description: 'Program ID to filter sessions' })
  @ApiQuery({ name: 'filters', type: String, required: false, description: paginationConstants.FILTERS })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Query('programId') programId: number[] = [],
    @Query('filters') filters: string = '',
    @Res() res: Response,
  ) {
    this.logger.log(programSessionConstMessages.FIND_ALL_SESSIONS_REQUEST_RECEIVED, {
      limit,
      offset,
      searchText,
      programId,
      filters,
    });
    try {
      // Parse the filters string into an object
      let parsedFilters;
      if (filters) {
        parsedFilters = decodeURIComponent(filters);
        parsedFilters = JSON.parse(parsedFilters);
      }
      let parsedProgramId;
      if (programId && programId.length > 0) {  
        parsedProgramId = JSON.parse(programId.toString());
        if (!Array.isArray(parsedProgramId)) {
          return this.errorHandler.badRequest(res, 'Program ID must be an array');
        }
      }
      const result = await this.programSessionService.findAll(limit, offset, searchText, parsedProgramId, parsedFilters);
      this.logger.log(programSessionConstMessages.SESSIONS_RETRIEVED, result);
      await this.responseService.success(res, programSessionConstMessages.SESSIONS_RETRIEVED, result);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Retrieves a single program session by its ID.
   * @param id - ID of the session to retrieve.
   * @returns The session data if found.
   * validates the ID using a validation pipe.
   */
  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programSessionConstMessages.GET_SESSION_BY_ID })
  @ApiParam({ name: 'id', description: programSessionConstMessages.SESSION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(programSessionConstMessages.FIND_ONE_SESSION_REQUEST_RECEIVED, { id });
    try {
      const data = await this.programSessionService.findOne(id);
      this.logger.log(programSessionConstMessages.SESSION_RETRIEVED, data);
      await this.responseService.success(res, programSessionConstMessages.SESSION_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Updates an existing program session by its ID.
   * @param id - ID of the session to update.
   * @param updateProgramSessionDto - Data transfer object containing updated session details.
   * @returns A success message and the updated session data.
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programSessionConstMessages.UPDATE_SESSION_BY_ID })
  @ApiParam({ name: 'id', description: programSessionConstMessages.SESSION_ID, type: Number })
  @ApiBody({ type: UpdateProgramSessionDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number, 
    @Body() updateProgramSessionDto: UpdateProgramSessionDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(programSessionConstMessages.UPDATE_SESSION_REQUEST_RECEIVED, { id, updateProgramSessionDto });
    try {
      const userId = req.user?.id || updateProgramSessionDto.updatedBy;
      updateProgramSessionDto.updatedBy = userId;
      const data = await this.programSessionService.update(id, updateProgramSessionDto);
      const respData = {
        id: data.id,
        name: data.name,
        code: data.code,
        programId: data.programId,
        displayOrder: data.displayOrder,
        status: data.status,
        program: {
          id: data.program.id,
          name: data.program.name,
        },
      };
      this.logger.log(programSessionConstMessages.SESSION_UPDATED, respData);
      await this.responseService.success(res, programSessionConstMessages.SESSION_UPDATED, respData);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Deletes a program session by its ID.
   * @param id - ID of the session to delete.
   * @returns A success message indicating the session was deleted.
   */
  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programSessionConstMessages.DELETE_SESSION_BY_ID })
  @ApiParam({ name: 'id', description: programSessionConstMessages.SESSION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any, @Res() res: Response) {
    this.logger.log(programSessionConstMessages.DELETE_SESSION_REQUEST_RECEIVED, { id });
    try {
      const user = req.user;
      if (!user) {
        return this.errorHandler.unauthorized(res);
      }
      await this.programSessionService.remove(id, user);
      this.logger.log(programSessionConstMessages.SESSION_DELETED, { id });
      await this.responseService.success(res, programSessionConstMessages.SESSION_DELETED, undefined, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res, error);
    }
  }
}