import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, EntityManager } from 'typeorm';
import { ProgramRegistrationSwap } from 'src/common/entities/program-registration-swap.entity';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { SwapRequestStatus } from 'src/common/enum/swap-request-status-enum';
import { SwapType } from 'src/common/enum/swap-type-enum';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';
import { RegistrationStatusEnum } from 'src/common/enum/registration-status.enum';
import { ApprovalStatusEnum } from 'src/common/enum/approval-status.enum';
import { SwapRequirementEnum } from 'src/common/enum/swap-requirement.enum';

@Injectable()
export class ProgramRegistrationSwapRepository {
  constructor(
    @InjectRepository(ProgramRegistrationSwap)
    private readonly repo: Repository<ProgramRegistrationSwap>,
  ) { }

  /**
   * Finds a ProgramRegistrationSwap entity by its ID.
   * @param id The ID of the ProgramRegistrationSwap entity to retrieve.
   * @returns The ProgramRegistrationSwap entity if found.
   */
  async findOneById(id: number) {
    try {
      const record = await this.repo.findOneBy({ id });
      if (!record) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
            id.toString()
          )
        );
      }
      return record;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Finds a ProgramRegistrationSwap entity by a specific field and value.
   * @param field The field to search by.
   * @param value The value to search for.
   * @returns The ProgramRegistrationSwap entity if found.
   */ 
  async findByField(field : keyof ProgramRegistrationSwap, value: any) {
    try {
      const record = await this.repo.findOneBy({ [field]: value });
      if (!record) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
          )
        );
      }
      return record;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Finds a ProgramRegistrationSwap entity by a specific condition.
   * @param where The condition to search for.
   * @returns The ProgramRegistrationSwap entity if found.
   */
  async findOneByWhere(
    where: Partial<Omit<ProgramRegistrationSwap, 'requestedPrograms'>>,
  ): Promise<ProgramRegistrationSwap | null> {
    try {
      return await this.repo.findOne({ where });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Finds all ProgramRegistrationSwap entities with pagination.
   * @param limit The maximum number of entities to return.
   * @param offset The number of entities to skip.
   * @returns An array of ProgramRegistrationSwap entities.
   */
  async createAndSave(data: Partial<ProgramRegistrationSwap>) {
    try {
      const entity = this.repo.create(data);
      if (data.swapRequirement) {
        entity.swapRequirement = data.swapRequirement;
      }
      const savedEntity = await this.repo.save(entity);

      // Set audit fields after creation when ID is available
      if (savedEntity && savedEntity.id) {
        await this.repo.update(savedEntity.id, {
          auditRefId: savedEntity.id,
          // Use programRegistrationId as parentRefId since this entity is related to a specific registration
          parentRefId: savedEntity.programRegistrationId,
          type: data.type,
          swapRequirement: data.swapRequirement || SwapRequirementEnum.SWAP_REQUEST,
        });
        savedEntity.auditRefId = savedEntity.id;
        savedEntity.parentRefId = savedEntity.programRegistrationId;
        savedEntity.type = data.type as any;
        savedEntity.swapRequirement = data.swapRequirement || SwapRequirementEnum.SWAP_REQUEST;
      }

      return savedEntity;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_REGISTRATION_SAVE_FAILED, error);
    }
  }

  /**
   * Updates a ProgramRegistrationSwap entity and saves it to the database.
   * @param id The ID of the entity to update.
   * @param data The new data to update the entity with.
   * @returns The updated entity or null if not found.
   */
  async updateAndSave(
    id: number,
    data: Partial<ProgramRegistrationSwap>,
    manager?: EntityManager
  ) {
    try {
      const repo = manager ? manager.getRepository(ProgramRegistrationSwap) : this.repo;
      const entity = await repo.findOne({ where: { id } });
      if (!entity) {
        handleKnownErrors(
          ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
          new InifniNotFoundException(
            ERROR_CODES.PROGRAM_SWAP_REQUEST_NOT_FOUND,
            null,
            null,
            id.toString()
          )
        );
      }
      
      // Ensure audit fields are properly set
      const updateData = {
        ...data,
        auditRefId: data.auditRefId || id,
        // Use programRegistrationId from entity if available
        parentRefId: data.parentRefId || entity.programRegistrationId,
      };
      
      Object.assign(entity, updateData);
      if (updateData.swapRequirement) {
        entity.swapRequirement = updateData.swapRequirement;
      }
      return await repo.save(entity);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_UPDATE_FAILED, error);
    }
  }

  /**
   * Retrieves all ProgramRegistrationSwap records.
   * @returns An array of ProgramRegistrationSwap entities.
   */
  async findAll(where?: Partial<Omit<ProgramRegistrationSwap, 'requestedPrograms'>>, limit?: number, offset?: number): Promise<ProgramRegistrationSwap[]> {
    try {
      return await this.repo.find({ where, relations: ['requestedPrograms', 'allocatedProgram','programRegistration'], take: limit, skip: offset });
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Gets the count of swap requests grouped by a specific field.
   * @param field The field to group by, currently supports 'gender' and 'requestedPrograms'.
   * @param programId The program ID to filter by for gender field.
   * @returns An array of objects containing the count and the field value.
   */
  async getCountsByGroupOf(field: 'gender' | 'requestedPrograms', programId: number): Promise<{ count: string; field: string | number }[]> {
    try {
      if (field === 'gender') {
        const query = this.repo.createQueryBuilder('swap')
          .select('COUNT(swap.id)', 'count')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .addSelect('registration.gender', 'field')
          .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
          .andWhere('swap.type = :type', { type: SwapType.WantsSwap })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED])
          .andWhere('approval.approval_status NOT IN (:...excludeApprovalStatuses)')
          .setParameter('excludeApprovalStatuses', [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD])
          .groupBy('registration.gender');
        
        return await query.getRawMany();
      } else if (field === 'requestedPrograms') {
        // Create subquery for ranked programs
        const rankedProgramsSubquery = this.repo.createQueryBuilder()
          .select('srp.swap_request_id', 'swap_request_id')
          .addSelect('srp.program_id', 'program_id')
          .addSelect('ROW_NUMBER() OVER (PARTITION BY srp.swap_request_id ORDER BY srp.id ASC)', 'rn')
          .from('hdb_swap_requested_program', 'srp');

        const query = this.repo.createQueryBuilder('swap')
          .select('COUNT(swap.id)', 'count')
          .addSelect('ranked_programs.program_id', 'field')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .leftJoin(
            `(${rankedProgramsSubquery.getQuery()})`,
            'ranked_programs',
            'ranked_programs.swap_request_id = swap.id AND ranked_programs.rn = 1'
          )
          .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
          .andWhere('swap.type = :type', { type: SwapType.WantsSwap })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED])
          .andWhere('approval.approval_status NOT IN (:...excludeApprovalStatuses)')
          .setParameter('excludeApprovalStatuses', [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD])
          .groupBy('ranked_programs.program_id');

        return await query.getRawMany();
      } else {
        // For other fields, we can directly group by the field in swap
        const query = this.repo.createQueryBuilder('swap')
          .select('COUNT(swap.id)', 'count')
          .addSelect(`swap.${field}`, 'field')
          .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
          .andWhere('swap.type = :type', { type: SwapType.WantsSwap })
          .groupBy(`swap.${field}`);
        
        return await query.getRawMany();
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }
  
  /**
   * Gets the count of swap requests grouped by a specific field for organization users.
   * @param field The field to group by, currently supports 'gender' and 'requestedPrograms'.
   * @param programId The program ID to filter by for gender field.
   * @returns An array of objects containing the count and the field value.
   */
  async getOrgCountsByGroupOf(field: 'gender' | 'requestedPrograms', programId: number): Promise<{ count: string; field: string | number }[]> {
    try {
      if (field === 'gender') {
        const query = this.repo.createQueryBuilder('swap')
          .select('COUNT(swap.id)', 'count')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.user', 'user')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .addSelect('registration.gender', 'field')
          .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
          .andWhere('swap.type = :type', { type: SwapType.WantsSwap })
          .andWhere('user.userType = :org', { org: UserTypeEnum.ORG })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED])
          .andWhere('approval.approval_status NOT IN (:...excludeApprovalStatuses)')
          .setParameter('excludeApprovalStatuses', [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD])
          .groupBy('registration.gender');

        return await query.getRawMany();
      } else if (field === 'requestedPrograms') {
        // Create subquery for ranked programs
        const rankedProgramsSubquery = this.repo.createQueryBuilder()
          .select('srp.swap_request_id', 'swap_request_id')
          .addSelect('srp.program_id', 'program_id')
          .addSelect('ROW_NUMBER() OVER (PARTITION BY srp.swap_request_id ORDER BY srp.id ASC)', 'rn')
          .from('hdb_swap_requested_program', 'srp');

        const query = this.repo.createQueryBuilder('swap')
          .select('COUNT(swap.id)', 'count')
          .addSelect('ranked_programs.program_id', 'field')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.user', 'user')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .leftJoin(
            `(${rankedProgramsSubquery.getQuery()})`,
            'ranked_programs',
            'ranked_programs.swap_request_id = swap.id AND ranked_programs.rn = 1'
          )
          .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
          .andWhere('swap.type = :type', { type: SwapType.WantsSwap })
          .andWhere('user.userType = :org', { org: UserTypeEnum.ORG })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED])
          .andWhere('approval.approval_status NOT IN (:...excludeApprovalStatuses)')
          .setParameter('excludeApprovalStatuses', [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD])
          .groupBy('ranked_programs.program_id');

        return await query.getRawMany();
      } else {
        // For other fields, we can directly group by the field in swap
        const query = this.repo.createQueryBuilder('swap')
          .select('COUNT(swap.id)', 'count')
          .addSelect(`swap.${field}`, 'field')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.user', 'user')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .where('swap.status = :status', { status: SwapRequestStatus.ACTIVE })
          .andWhere('swap.type = :type', { type: SwapType.WantsSwap })
          .andWhere('user.userType = :org', { org: UserTypeEnum.ORG })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [RegistrationStatusEnum.SAVE_AS_DRAFT, RegistrationStatusEnum.CANCELLED, RegistrationStatusEnum.REJECTED])
          .andWhere('approval.approval_status NOT IN (:...excludeApprovalStatuses)')
          .setParameter('excludeApprovalStatuses', [ApprovalStatusEnum.CANCELLED, ApprovalStatusEnum.REJECTED, ApprovalStatusEnum.ON_HOLD])
          .groupBy(`swap.${field}`);

        return await query.getRawMany();
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
    }
  }

  /**
   * Gets the count of swap demands grouped by a specific field.
   * Gets only the latest swap demand per registration (highest ID).
   * @param field The field to group by, currently supports 'gender' and 'requestedPrograms'.
   * @param programId The program ID to filter by.
   * @returns An array of objects containing the count and the field value.
   */
  async getSwapDemandCountsByGroupOf(
    field: 'gender' | 'requestedPrograms',
    programId: number,
  ): Promise<{ count: string; field: string | number }[]> {
    try {
      if (field === 'gender') {
        // Use DISTINCT ON to get the latest swap demand per registration
        const latestSwapsSubquery = this.repo
          .createQueryBuilder('s')
          .select('DISTINCT ON (s.program_registration_id) s.id', 'id')
          .addSelect('s.program_registration_id', 'program_registration_id')
          .where('s.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('s.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .orderBy('s.program_registration_id', 'ASC')
          .addOrderBy('s.id', 'DESC');

        const query = this.repo
          .createQueryBuilder('swap')
          .select('COUNT(DISTINCT swap.program_registration_id)', 'count')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .innerJoin(
            `(${latestSwapsSubquery.getQuery()})`,
            'latest',
            'latest.id = swap.id',
          )
          .addSelect('registration.gender', 'field')
          .where('swap.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('swap.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [
            RegistrationStatusEnum.SAVE_AS_DRAFT,
            RegistrationStatusEnum.CANCELLED,
            RegistrationStatusEnum.REJECTED,
          ])
          .andWhere('approval.approval_status = :approvalStatus', {
            approvalStatus: ApprovalStatusEnum.ON_HOLD,
          })
          .groupBy('registration.gender');

        return await query.getRawMany();
      } else if (field === 'requestedPrograms') {
        // Use DISTINCT ON to get the latest swap demand per registration
        const latestSwapsSubquery = this.repo
          .createQueryBuilder('s')
          .select('DISTINCT ON (s.program_registration_id) s.id', 'id')
          .addSelect('s.program_registration_id', 'program_registration_id')
          .where('s.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('s.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .orderBy('s.program_registration_id', 'ASC')
          .addOrderBy('s.id', 'DESC');

        // Subquery for ranked programs - get the first (priority) program
        const rankedProgramsSubquery = this.repo
          .createQueryBuilder()
          .select('srp.swap_request_id', 'swap_request_id')
          .addSelect('srp.program_id', 'program_id')
          .addSelect(
            'ROW_NUMBER() OVER (PARTITION BY srp.swap_request_id ORDER BY srp.id ASC)',
            'rn',
          )
          .from('hdb_swap_requested_program', 'srp');

        const query = this.repo
          .createQueryBuilder('swap')
          .select('COUNT(DISTINCT swap.program_registration_id)', 'count')
          .addSelect('ranked_programs.program_id', 'field')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .innerJoin(
            `(${latestSwapsSubquery.getQuery()})`,
            'latest',
            'latest.id = swap.id',
          )
          .leftJoin(
            `(${rankedProgramsSubquery.getQuery()})`,
            'ranked_programs',
            'ranked_programs.swap_request_id = swap.id AND ranked_programs.rn = 1',
          )
          .where('swap.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('swap.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [
            RegistrationStatusEnum.SAVE_AS_DRAFT,
            RegistrationStatusEnum.CANCELLED,
            RegistrationStatusEnum.REJECTED,
          ])
          .andWhere('approval.approval_status = :approvalStatus', {
            approvalStatus: ApprovalStatusEnum.ON_HOLD,
          })
          .groupBy('ranked_programs.program_id');

        return await query.getRawMany();
      } else {
        return [];
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
      return [];
    }
  }

  /**
   * Gets the count of swap demands grouped by a specific field for organization users.
   * Gets only the latest swap demand per registration (highest ID).
   * @param field The field to group by, currently supports 'gender' and 'requestedPrograms'.
   * @param programId The program ID to filter by.
   * @returns An array of objects containing the count and the field value.
   */
  async getSwapDemandOrgCountsByGroupOf(
    field: 'gender' | 'requestedPrograms',
    programId: number,
  ): Promise<{ count: string; field: string | number }[]> {
    try {
      if (field === 'gender') {
        // Use DISTINCT ON to get the latest swap demand per registration
        const latestSwapsSubquery = this.repo
          .createQueryBuilder('s')
          .select('DISTINCT ON (s.program_registration_id) s.id', 'id')
          .addSelect('s.program_registration_id', 'program_registration_id')
          .where('s.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('s.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .orderBy('s.program_registration_id', 'ASC')
          .addOrderBy('s.id', 'DESC');

        const query = this.repo
          .createQueryBuilder('swap')
          .select('COUNT(DISTINCT swap.program_registration_id)', 'count')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.user', 'user')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .innerJoin(
            `(${latestSwapsSubquery.getQuery()})`,
            'latest',
            'latest.id = swap.id',
          )
          .addSelect('registration.gender', 'field')
          .where('swap.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('swap.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .andWhere('user.userType = :org', { org: UserTypeEnum.ORG })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [
            RegistrationStatusEnum.SAVE_AS_DRAFT,
            RegistrationStatusEnum.CANCELLED,
            RegistrationStatusEnum.REJECTED,
          ])
          .andWhere('approval.approval_status = :approvalStatus', {
            approvalStatus: ApprovalStatusEnum.ON_HOLD,
          })
          .groupBy('registration.gender');

        return await query.getRawMany();
      } else if (field === 'requestedPrograms') {
        // Use DISTINCT ON to get the latest swap demand per registration
        const latestSwapsSubquery = this.repo
          .createQueryBuilder('s')
          .select('DISTINCT ON (s.program_registration_id) s.id', 'id')
          .addSelect('s.program_registration_id', 'program_registration_id')
          .where('s.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('s.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .orderBy('s.program_registration_id', 'ASC')
          .addOrderBy('s.id', 'DESC');

        // Subquery for ranked programs - get the first (priority) program
        const rankedProgramsSubquery = this.repo
          .createQueryBuilder()
          .select('srp.swap_request_id', 'swap_request_id')
          .addSelect('srp.program_id', 'program_id')
          .addSelect(
            'ROW_NUMBER() OVER (PARTITION BY srp.swap_request_id ORDER BY srp.id ASC)',
            'rn',
          )
          .from('hdb_swap_requested_program', 'srp');

        const query = this.repo
          .createQueryBuilder('swap')
          .select('COUNT(DISTINCT swap.program_registration_id)', 'count')
          .addSelect('ranked_programs.program_id', 'field')
          .leftJoin('swap.programRegistration', 'registration')
          .leftJoin('registration.user', 'user')
          .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
          .innerJoin(
            `(${latestSwapsSubquery.getQuery()})`,
            'latest',
            'latest.id = swap.id',
          )
          .leftJoin(
            `(${rankedProgramsSubquery.getQuery()})`,
            'ranked_programs',
            'ranked_programs.swap_request_id = swap.id AND ranked_programs.rn = 1',
          )
          .where('swap.status = :status', { status: SwapRequestStatus.ON_HOLD })
          .andWhere('swap.swap_requirement = :requirement', {
            requirement: SwapRequirementEnum.SWAP_DEMAND,
          })
          .andWhere('user.userType = :org', { org: UserTypeEnum.ORG })
          .andWhere('registration.program_id = :programId', { programId })
          .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
          .setParameter('excludeStatuses', [
            RegistrationStatusEnum.SAVE_AS_DRAFT,
            RegistrationStatusEnum.CANCELLED,
            RegistrationStatusEnum.REJECTED,
          ])
          .andWhere('approval.approval_status = :approvalStatus', {
            approvalStatus: ApprovalStatusEnum.ON_HOLD,
          })
          .groupBy('ranked_programs.program_id');

        return await query.getRawMany();
      } else {
        return [];
      }
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
      return [];
    }
  }

  /**
   * Gets the total count of swap demands for a program.
   * Gets only the latest swap demand per registration (highest ID).
   * @param programId The program ID to filter by.
   * @param allocatedProgramId Optional allocated program ID to filter by.
   * @returns The total count of swap demands.
   */
  async getTotalSwapDemandCount(
    programId: number,
    allocatedProgramId?: number | null,
  ): Promise<number> {
    try {
      // Use DISTINCT ON to get the latest swap demand per registration
      const latestSwapsSubquery = this.repo
        .createQueryBuilder('s')
        .select('DISTINCT ON (s.program_registration_id) s.id', 'id')
        .addSelect('s.program_registration_id', 'program_registration_id')
        .where('s.status = :status', { status: SwapRequestStatus.ON_HOLD })
        .andWhere('s.swap_requirement = :requirement', {
          requirement: SwapRequirementEnum.SWAP_DEMAND,
        })
        .orderBy('s.program_registration_id', 'ASC')
        .addOrderBy('s.id', 'DESC');

      let query = this.repo
        .createQueryBuilder('swap')
        .leftJoin('swap.programRegistration', 'registration')
        .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
        .innerJoin(
          `(${latestSwapsSubquery.getQuery()})`,
          'latest',
          'latest.id = swap.id',
        )
        .where('swap.status = :status', { status: SwapRequestStatus.ON_HOLD })
        .andWhere('swap.swap_requirement = :requirement', {
          requirement: SwapRequirementEnum.SWAP_DEMAND,
        })
        .andWhere('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .setParameter('excludeStatuses', [
          RegistrationStatusEnum.SAVE_AS_DRAFT,
          RegistrationStatusEnum.CANCELLED,
          RegistrationStatusEnum.REJECTED,
        ])
        .andWhere('approval.approval_status = :approvalStatus', {
          approvalStatus: ApprovalStatusEnum.ON_HOLD,
        })
        .andWhere('registration.deleted_at IS NULL');

      if (allocatedProgramId) {
        query = query.andWhere('registration.allocated_program_id = :allocatedProgramId', {
          allocatedProgramId,
        });
      }

      return await query.getCount();
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
      return 0;
    }
  }

  /**
   * Gets the total count of swap demands with organization user count.
   * Gets only the latest swap demand per registration (highest ID).
   * @param programId The program ID to filter by.
   * @param allocatedProgramId Optional allocated program ID to filter by.
   * @returns Object containing totalCount and orgCount.
   */
  async getTotalSwapDemandCountWithOrg(
    programId: number,
    allocatedProgramId?: number | null,
  ): Promise<{ totalCount: string; orgCount: string }> {
    try {
      // Use DISTINCT ON to get the latest swap demand per registration
      const latestSwapsSubquery = this.repo
        .createQueryBuilder('s')
        .select('DISTINCT ON (s.program_registration_id) s.id', 'id')
        .addSelect('s.program_registration_id', 'program_registration_id')
        .where('s.status = :status', { status: SwapRequestStatus.ON_HOLD })
        .andWhere('s.swap_requirement = :requirement', {
          requirement: SwapRequirementEnum.SWAP_DEMAND,
        })
        .orderBy('s.program_registration_id', 'ASC')
        .addOrderBy('s.id', 'DESC');

      let query = this.repo
        .createQueryBuilder('swap')
        .leftJoin('swap.programRegistration', 'registration')
        .leftJoin('registration.user', 'user')
        .leftJoin('registration.approvals', 'approval', 'approval.deleted_at IS NULL')
        .innerJoin(
          `(${latestSwapsSubquery.getQuery()})`,
          'latest',
          'latest.id = swap.id',
        )
        .select([
          `COUNT(DISTINCT swap.program_registration_id) AS "totalCount"`,
          `COUNT(DISTINCT CASE WHEN user.userType = '${UserTypeEnum.ORG}' THEN swap.program_registration_id END) AS "orgCount"`,
        ])
        .where('swap.status = :status', { status: SwapRequestStatus.ON_HOLD })
        .andWhere('swap.swap_requirement = :requirement', {
          requirement: SwapRequirementEnum.SWAP_DEMAND,
        })
        .andWhere('registration.program_id = :programId', { programId })
        .andWhere('registration.registration_status NOT IN (:...excludeStatuses)')
        .setParameter('excludeStatuses', [
          RegistrationStatusEnum.SAVE_AS_DRAFT,
          RegistrationStatusEnum.CANCELLED,
          RegistrationStatusEnum.REJECTED,
        ])
        .andWhere('approval.approval_status = :approvalStatus', {
          approvalStatus: ApprovalStatusEnum.ON_HOLD,
        })
        .andWhere('registration.deleted_at IS NULL');

      if (allocatedProgramId) {
        query = query.andWhere('registration.allocated_program_id = :allocatedProgramId', {
          allocatedProgramId,
        });
      }

      const result = await query.getRawOne();
      return result ?? { totalCount: '0', orgCount: '0' };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.PROGRAM_SWAP_REQUEST_GET_FAILED, error);
      return { totalCount: '0', orgCount: '0' };
    }
  }
}
