import {
  Controller, Get, Post, Body, Param, Put, Delete, HttpStatus, ParseIntPipe,
  ValidationPipe, UsePipes, HttpCode, Query, Res, UseGuards, Req
} from '@nestjs/common';
import { Response } from 'express';
import { ProgramQuestionService } from './program-question.service';
import { CreateProgramQuestionDto } from './dto/create-program-question.dto';
import { UpdateProgramQuestionDto } from './dto/update-program-question.dto';
import { UpdateProgramQuestionBulkDto } from './dto/update-program-question.dto'; 
import { DeleteProgramQuestionDto } from './dto/delete-program-question.dto';
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiQuery, ApiBearerAuth, ApiSecurity } from '@nestjs/swagger';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { ResponseService } from 'src/common/response-handling/response-handler';
import { SWAGGER_API_RESPONSE, paginationConstants, programQuestionConstMessages, userConstMessages } from 'src/common/constants/strings-constants';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { AppLoggerService } from 'src/common/services/logger.service';
import { RegistrationLevelEnum } from 'src/common/enum/program.enums';
import { extractAndValidateUser, extractUserId } from 'src/common/utils/program-question.util';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('program-question')
@Controller('program-question')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
export class ProgramQuestionController {

  constructor(
    private readonly service: ProgramQuestionService,
    private readonly responseService: ResponseService,
    private readonly logger: AppLoggerService 
  ) {}

  // ==================== CREATE OPERATIONS ====================

  /**
   * Create a new program-question mapping with sections and questions.
   * Supports creating custom questions with options and linking existing questions.
   * @param dto - Data transfer object containing programId, sections with questions, registrationLevel, and optional programSessionId.
   * @param req - Request object containing user info.
   * @param res - Express response object.
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programQuestionConstMessages.CREATE_NEW_MAPPING })
  @ApiBody({ type: CreateProgramQuestionDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async createProgramQuestionMapping(
    @Body() dto: CreateProgramQuestionDto, 
    @Req() req: any, 
    @Res() res: Response
  ) {
    this.logger.log(programQuestionConstMessages.CREATE_PRO_MAP_REQUEST_RECEIVED, dto);
    try {
      const user = extractAndValidateUser(req);
      const data = await this.service.create(dto, user);
      await this.responseService.success(res, programQuestionConstMessages.MAPPING_CREATED, data, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  // ==================== READ OPERATIONS ====================

  /**
   * Get all program-question mappings with pagination and filtering options.
   * @param limit - Number of records to return (default: 10).
   * @param offset - Number of records to skip (default: 0).
   * @param programId - Optional filter by program ID.
   * @param programSessionId - Optional filter by program session ID.
   * @param registrationLevel - Optional filter by registration level (program/session).
   * @param res - Express response object.
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programQuestionConstMessages.FINDING_ALL_MAPPINGS })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'programId', type: Number, required: false, description: programQuestionConstMessages.PROGRAM_ID })
  @ApiQuery({ name: 'programSessionId', type: Number, required: false, description: 'Filter by Program Session ID' })
  @ApiQuery({ name: 'registrationLevel', enum: RegistrationLevelEnum, required: false, description: 'Filter by Registration Level (program/session)' })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAllProgramQuestionMappings(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('programId') programId: number | null = null,
    @Query('programSessionId') programSessionId: number | null = null,
    @Query('registrationLevel') registrationLevel: RegistrationLevelEnum | null = null,
    @Res() res: Response
  ) {
    this.logger.log(programQuestionConstMessages.FIND_ALL_MAPPINGS_REQUEST_RECEIVED, { 
      limit, 
      offset, 
      programId, 
      programSessionId, 
      registrationLevel 
    });
    try {
      const data = await this.service.findAll(limit, offset, programId, programSessionId, registrationLevel);
      await this.responseService.success(res, programQuestionConstMessages.MAPPINGS_RETRIEVED, data);
    } catch (error) {
       handleControllerError(res, error);
    }
  }

  /**
   * Get a specific program-question mapping by its ID.
   * @param id - Mapping ID to retrieve.
   * @param res - Express response object.
   */
  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programQuestionConstMessages.GET_MAPPING_BY_ID })
  @ApiParam({ name: 'id', description: programQuestionConstMessages.MAPPING_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOneProgramQuestionMapping(
    @Param('id', ParseIntPipe) id: number, 
    @Res() res: Response
  ) {
    this.logger.log(programQuestionConstMessages.FIND_ONE_MAPPING_REQUEST_RECEIVED, { id });
    try {
      const data = await this.service.findOne(id);
      await this.responseService.success(res, programQuestionConstMessages.MAPPING_RETRIEVED, data);
    } catch (error) {
       handleControllerError(res, error);
    }
  }

  // ==================== UPDATE OPERATIONS ====================

  /**
   * Update a single program-question mapping by its ID.
   * Allows updating program reference, question reference, display order, and registration level.
   * @param id - Mapping ID to update.
   * @param dto - Data transfer object containing update data.
   * @param req - Request object containing user info.
   * @param res - Express response object.
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: programQuestionConstMessages.UPDATE_MAPPING_BY_ID })
  @ApiParam({ name: 'id', description: programQuestionConstMessages.MAPPING_ID, type: Number })
  @ApiBody({ type: UpdateProgramQuestionDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async updateProgramQuestionMapping(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateProgramQuestionDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(programQuestionConstMessages.UPDATE_MAPPING_REQUEST_RECEIVED, { id, dto });
    try {
      const userId = extractUserId(req);
      const data = await this.service.update(id, dto, userId);
      await this.responseService.success(res, programQuestionConstMessages.MAPPING_UPDATED, data);
    } catch (error) {
       handleControllerError(res, error);
    }
  }

  /**
   * Bulk update operation for program questions.
   * Supports creating new sections/questions, updating existing ones, and deleting sections/questions.
   * This endpoint handles complex operations including:
   * - Creating new sections and custom questions
   * - Adding existing questions to sections
   * - Deleting entire sections with their questions
   * - Deleting individual questions from program/session
   * @param dto - Data transfer object containing all update operations.
   * @param req - Request object containing user info.
   * @param res - Express response object.
   */
  @Put()
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ 
    summary: 'Bulk update program questions with sections',
    description: 'Comprehensive update operation that can create, update, and delete sections and questions in a single transaction'
  })
  @ApiBody({ type: UpdateProgramQuestionBulkDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async bulkUpdateProgramQuestions(
    @Body() dto: UpdateProgramQuestionBulkDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log('BULK_UPDATE_REQUEST_RECEIVED', dto);
    try {
      const user = extractAndValidateUser(req);
      const data = await this.service.bulkUpdate(dto, user);
      await this.responseService.success(res, 'Bulk update operation completed successfully', data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  // ==================== DELETE OPERATIONS ====================

  /**
   * Bulk delete program-question mappings by sections or individual questions.
   * Supports deleting entire sections (which removes all questions in those sections)
   * or deleting specific questions from the program/session scope.
   * @param dto - Data transfer object containing sections or question IDs to delete.
   * @param req - Request object containing user info.
   * @param res - Express response object.
   */
  @Delete('')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ 
    summary: 'Delete program-question mappings by sections or questions',
    description: 'Bulk delete operation that can remove entire sections or individual questions'
  })
  @ApiBody({ type: DeleteProgramQuestionDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async bulkDeleteProgramQuestionMappings(
    @Body() dto: DeleteProgramQuestionDto, 
    @Req() req: any, 
    @Res() res: Response
  ) {
    this.logger.log(programQuestionConstMessages.BULK_DELETE_REQUEST_RECEIVED, dto);
    try {
      const user = extractAndValidateUser(req);
      const data = await this.service.delete(dto, user.id);
      await this.responseService.success(res, 'Delete operation completed successfully', data);
    } catch (error) {
      handleControllerError(res, error);
    }
  }

  /**
   * Delete a single program-question mapping by its ID.
   * This performs a soft delete on the specific mapping.
   * @param id - Mapping ID to delete.
   * @param req - Request object containing user info.
   * @param res - Express response object.
   */
  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: programQuestionConstMessages.DELETE_MAPPING_BY_ID })
  @ApiParam({ name: 'id', description: programQuestionConstMessages.MAPPING_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async removeProgramQuestionMapping(
    @Param('id', ParseIntPipe) id: number, 
    @Req() req: any, 
    @Res() res: Response
  ) {
    this.logger.log(programQuestionConstMessages.DELETE_MAPPING_REQUEST_RECEIVED, { id });
    try {
      const userId = extractUserId(req);
      await this.service.remove(id, userId);
      await this.responseService.success(res, programQuestionConstMessages.MAPPING_DELETED);
    } catch (error) {
     handleControllerError(res, error);
    }
  }

}