import { ApiProperty } from '@nestjs/swagger';
import { IsEnum, IsOptional, IsString, IsDateString, IsObject, ValidateIf } from 'class-validator';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';

export class UpdatePaymentDto {
  @ApiProperty({
    description: 'Updated payment status',
    enum: PaymentStatusEnum,
    example: PaymentStatusEnum.OFFLINE_COMPLETED
  })
  @IsEnum(PaymentStatusEnum)
  paymentStatus: PaymentStatusEnum;

  @ApiProperty({
    description: 'Payment confirmation date',
    example: '2024-01-15T10:30:00Z',
    required: false
  })
  @IsOptional()
  @IsDateString()
  paymentDate?: string;

  @ApiProperty({
    description: 'Mark as received date confirmation date (Required when payment status is ONLINE_COMPLETED or OFFLINE_COMPLETED)',
    example: '2024-01-15T10:30:00Z',
    required: false
  })
  @ValidateIf((o) => o.paymentStatus === PaymentStatusEnum.ONLINE_COMPLETED || o.paymentStatus === PaymentStatusEnum.OFFLINE_COMPLETED)
  @IsDateString()
  markAsReceivedDate: string;

  @ApiProperty({
    description: 'Payment reference ID or transaction ID',
    example: 'TXN123456789',
    required: false
  })
  @IsOptional()
  @IsString()
  paymentReference?: string;

  @ApiProperty({
    description: 'Additional payment metadata',
    example: { bankName: 'HDFC Bank', chequeNumber: '123456' },
    required: false
  })
  @IsOptional()
  @IsObject()
  paymentMeta?: Record<string, any>;

  @ApiProperty({
    description: 'Admin remarks for payment verification',
    example: 'Payment verified through bank statement',
    required: false
  })
  @IsOptional()
  @IsString()
  adminRemarks?: string;
}