import { ApiProperty } from '@nestjs/swagger';
import { IsEnum, IsOptional, IsObject, IsString, MaxLength, IsArray, Matches, IsPositive } from 'class-validator';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import { RegistrationAnswerDto } from 'src/registration/dto/create-registration.dto';

export class InitiatePaymentDto {
  @ApiProperty({
    description: 'Payment mode - online or offline',
    enum: PaymentModeEnum,
    example: PaymentModeEnum.ONLINE
  })
  @IsEnum([PaymentModeEnum.ONLINE, PaymentModeEnum.OFFLINE], {
    message: `Payment mode must be one of the following values: ${[PaymentModeEnum.ONLINE, PaymentModeEnum.OFFLINE].join(', ')}`
  })
  paymentMode: PaymentModeEnum;

  @ApiProperty({
    description: 'Billing name for invoice',
    example: 'John Doe'
  })
  @IsString()
  @MaxLength(255)
  invoiceName: string;

  @ApiProperty({
    description: 'Billing address for invoice',
    example: '123 Main St, City, State, ZIP'
  })
  @IsString()
  @MaxLength(100 , {
    message: 'Invoice address cannot exceed 100 characters'
  })
  invoiceAddress: string;

  @ApiProperty({
    description: 'PAN number for tax purposes',
    example: 'ABCDE1234F',
    required: false
  })
  @IsOptional()
  @IsString()
  @MaxLength(10)
  panNumber?: string;

  @ApiProperty({
    description: 'TAN number for TDS',
    example: 'ABCD12345E',
    required: false
  })
  @IsOptional()
  @IsString()
  @Matches(/^[A-Za-z]{4}[0-9]{5}[A-Za-z]{1}$/, {
    message: 'Invalid TAN number format',
  })
  tanNumber?: string;

  @ApiProperty({
    description: 'TDS amount for tax deduction',
    type: Number,
    example: 1000,
    required: false
  })
  @IsOptional()
  @IsPositive({
    message: 'TDS amount must be a positive number greater than zero'
  })
  tdsAmount?: number;

  @ApiProperty({
    description: 'Email for invoice delivery',
    example: 'john.doe@example.com'
  })
  @IsString()
  invoiceEmail: string;

  @ApiProperty({
    description: 'gst number for tax purposes',
    example: '27ABCDE1234F4',
    required: false
  })
  @IsString()
  @IsOptional()
  @MaxLength(50)
  @Matches(/^[0-9]{2}[A-Z]{5}[0-9]{4}[A-Z]{1}[1-9A-Z]{1}Z[0-9A-Z]{1}$/, {
    message: 'Invalid GST number format',
  })
  gstNumber?: string;

  @ApiProperty({
    description: 'zip code for billing address',
    example: '123456',
  })
  @IsString()
  @IsOptional()
  @MaxLength(10)
  zip?: string;

  @ApiProperty({
    description: 'Offline payment metadata (for offline payments)',
    example: { paymentMethod: "Back Transfer", bankName: 'HDFC Bank', transactionId: 'TXN123456' },
    required: false
  })
  @IsOptional()
  @IsObject()
  offlinePaymentMeta?: Record<string, any>;

  @ApiProperty({
    description: 'Payment metadata for additional information',
    example: [ {key: 'value'} ],
    required: false
  })
  @IsOptional()
  @IsArray()
  paymentMeta?: [RegistrationAnswerDto];

  @ApiProperty({
    description: 'Flag to indicate if the user is an admin (for internal use)',
    example: true,
    required: false
  })
  @IsOptional()
  isSeeker: boolean = true;
}