import { 
  Controller, 
  Get, 
  Post, 
  Body, 
  Param, 
  Put, 
  Delete, 
  HttpStatus,
  ParseIntPipe,
  ValidationPipe,
  UsePipes,
  HttpCode,
  Query,
  Res,
  UseGuards,
  Req,
} from '@nestjs/common';
import { Response } from 'express';
import { OptionService } from './option.service';
import { CreateOptionDto } from './dto/create-option.dto';
import { UpdateOptionDto } from './dto/update-option.dto'; 
import { ApiTags, ApiOperation, ApiResponse, ApiParam, ApiBody, ApiQuery, ApiBearerAuth, ApiSecurity } from '@nestjs/swagger';
import { ResponseService } from 'src/common/response-handling/response-handler';
import ErrorHandler from 'src/common/response-handling/error-handling';
import { optionConstMessages, paginationConstants, SWAGGER_API_RESPONSE } from 'src/common/constants/strings-constants';
import { RolesGuard } from 'src/common/guards/roles.guard';
import { Roles } from 'src/common/decorators/roles.decorator';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleControllerError } from 'src/common/utils/controller-response-handling';
import { CombinedAuthGuard } from 'src/auth/combined-auth.guard';

@ApiTags('option')
@Controller('option')
@UseGuards(CombinedAuthGuard, RolesGuard)
@Roles('admin')
@ApiBearerAuth('Authorization')
@ApiSecurity('userIdAuth')
@ApiSecurity('activeRoleAuth')
export class OptionController {
  constructor(
    private readonly optionService: OptionService,
    private readonly responseService: ResponseService,
    private readonly errorHandler: ErrorHandler,
    private readonly logger: AppLoggerService
  ) {}

  /**
   * Creates a new option.
   * Validates the input using a validation pipe and ensures the option is created successfully.
   * @param createOptionDto - Data transfer object containing option details.
   * @returns A success message and the created option data.
   */
  @Post()
  @HttpCode(HttpStatus.CREATED)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: optionConstMessages.CREATE_NEW_OPTION })
  @ApiBody({ type: CreateOptionDto })
  @ApiResponse({ status: HttpStatus.CREATED, description: SWAGGER_API_RESPONSE.CREATED })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.CONFLICT, description: SWAGGER_API_RESPONSE.CONFLICT })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async create(@Body() createOptionDto: CreateOptionDto,@Req() req : any ,@Res() res: Response) {
    this.logger.log(optionConstMessages.CREATE_OPTION_REQUEST_RECEIVED,createOptionDto);
    try {
      const userId = req.user?.id || createOptionDto.createdBy;
      createOptionDto.createdBy = userId;
      createOptionDto.updatedBy = userId;
      const data = await this.optionService.create(createOptionDto);
      const respData = {
        id: data.id,
        name: data.name,
        type: data.type,
        categoryId: data.categoryId,
      };
      this.logger.log(optionConstMessages.OPTION_CREATED, respData);
      await this.responseService.success(res, optionConstMessages.OPTION_CREATED, respData, HttpStatus.CREATED);
    } catch (error) {
      handleControllerError(res,error);
    }
  }

  /**
   * Retrieves all options with optional pagination and search functionality.
   * @param limit - Number of records per page (default: 10).
   * @param offset - Offset for pagination (default: 0).
   * @param searchText - Optional search text to filter options by name.
   * @returns A paginated list of options and metadata.
   */
  @Get()
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: optionConstMessages.GET_ALL_OPTIONS_SEARCH })
  @ApiQuery({ name: 'limit', type: Number, required: false, description: paginationConstants.LIMIT })
  @ApiQuery({ name: 'offset', type: Number, required: false, description: paginationConstants.OFFSET })
  @ApiQuery({ name: 'searchText', type: String, required: false, description: paginationConstants.SEARCH_TEXT })
  @ApiQuery({ name: 'categoryId', type: String, required: false, description: paginationConstants.CATEGORY_ID })
  @ApiQuery({ name: 'filters', type: String, required: false, description: paginationConstants.FILTERS })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findAll(
    @Query('limit') limit: number = 10,
    @Query('offset') offset: number = 0,
    @Query('searchText') searchText: string = '',
    @Query('categoryId') categoryId: number[] = [], // Default to null if not provided
    @Query('filters') filters: string = '',
    @Res() res: Response,
  ) {
    this.logger.log(optionConstMessages.FIND_ALL_OPTIONS_REQUEST_RECEIVED, {
      limit,
      offset,
      searchText,
      categoryId,
      filters,
    });
    try {
      // Parse the filters string into an object
    let parsedFilters;
    if (filters) {
      parsedFilters = decodeURIComponent(filters); // Decode the filters string
      parsedFilters = JSON.parse(parsedFilters); // Parse the JSON string into an object
    }
    let parsedCategoryId;
    if (categoryId && categoryId.length > 0) {  
      parsedCategoryId = JSON.parse(categoryId.toString());
      if (!Array.isArray(parsedCategoryId)) {
        return this.errorHandler.badRequest(res, optionConstMessages.CATEGORY_MUST_BE_ARRAY);
      }
    }
      const result = await this.optionService.findAll(limit, offset, searchText, parsedCategoryId, parsedFilters);
      this.logger.log(optionConstMessages.OPTIONS_RETRIEVED, result);
      await this.responseService.success(res, optionConstMessages.OPTIONS_RETRIEVED, result);
    } catch (error) {
      handleControllerError(res,error);
    }
  }

  /**
   * Retrieves a single option by its ID.
   * @param id - ID of the option to retrieve.
   * @returns The option data if found.
   * validates the ID using a validation pipe.
   */
  @Get(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: optionConstMessages.GET_OPTION_BY_ID })
  @ApiParam({ name: 'id', description: optionConstMessages.OPTION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async findOne(@Param('id', ParseIntPipe) id: number, @Res() res: Response) {
    this.logger.log(optionConstMessages.FIND_ONE_OPTION_REQUEST_RECEIVED, {id});
    try {
      const data = await this.optionService.findOne(id);
      this.logger.log(optionConstMessages.OPTION_RETRIEVED, data);
      await this.responseService.success(res, optionConstMessages.OPTIONS_RETRIEVED, data);
    } catch (error) {
      handleControllerError(res,error);
    }
  }

  /**
   * Updates an existing option by its ID.
   * @param id - ID of the option to update.
   * @param updateOptionDto - Data transfer object containing updated option details.
   * @returns A success message and the updated option data.
   */
  @Put(':id')
  @HttpCode(HttpStatus.OK)
  @UsePipes(new ValidationPipe({ transform: true, whitelist: true }))
  @ApiOperation({ summary: optionConstMessages.UPDATE_OPTION_ID })
  @ApiParam({ name: 'id', description: optionConstMessages.OPTION_ID, type: Number })
  @ApiBody({ type: UpdateOptionDto })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.BAD_REQUEST, description: SWAGGER_API_RESPONSE.BAD_REQUEST })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async update(
    @Param('id', ParseIntPipe) id: number, 
    @Body() updateOptionDto: UpdateOptionDto,
    @Req() req: any,
    @Res() res: Response,
  ) {
    this.logger.log(optionConstMessages.UPDATE_OPTION_REQUEST_RECEIVED, { id, updateOptionDto });
    try {
      const userId = req.user?.id || updateOptionDto.updatedBy;
      updateOptionDto.updatedBy = userId;
      const data = await this.optionService.update(id, updateOptionDto);
      const respData = {
        id: data.id,
        name: data.name,
        type: data.type,
        categoryId: data.categoryId,
        category: {
          id: data.category.id,
          name: data.category.name,
        },
      };
      this.logger.log(optionConstMessages.OPTION_UPDATED, respData);
      await this.responseService.success(res, optionConstMessages.OPTION_UPDATED, respData);
    } catch (error) {
      handleControllerError(res,error);
    }
  }

  /**
   * Deletes an option by its ID.
   * @param id - ID of the option to delete.
   * @returns A success message indicating the option was deleted.
   */
  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  @ApiOperation({ summary: optionConstMessages.DELETE_OPTION_ID })
  @ApiParam({ name: 'id', description: optionConstMessages.OPTION_ID, type: Number })
  @ApiResponse({ status: HttpStatus.OK, description: SWAGGER_API_RESPONSE.OK })
  @ApiResponse({ status: HttpStatus.NOT_FOUND, description: SWAGGER_API_RESPONSE.NOT_FOUND })
  @ApiResponse({ status: HttpStatus.INTERNAL_SERVER_ERROR, description: SWAGGER_API_RESPONSE.INTERNAL_SERVER_ERROR })
  async remove(@Param('id', ParseIntPipe) id: number, @Req() req: any,@Res() res: Response) {
    this.logger.log(optionConstMessages.DELETE_OPTION_REQUEST_RECEIVED, { id });
    try {
      const user = req.user;
      if (!user) {
        return this.errorHandler.unauthorized(res);
      }
      await this.optionService.remove(id, user);
      this.logger.log(optionConstMessages.OPTION_DELETED, { id });
      await this.responseService.success(res, optionConstMessages.OPTION_DELETED, undefined, HttpStatus.OK);
    } catch (error) {
      handleControllerError(res,error);
    }
  }
}