import { Injectable } from '@nestjs/common';
import { MessagesRepository } from './messages.repository';
import { CreateMessageDto } from './dto/create-message.dto';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { UserService } from 'src/user/user.service';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';
import { User } from 'src/common/entities';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { messageConstMessages } from 'src/common/constants/strings-constants';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { CommunicationService } from 'src/communication/communication.service';

@Injectable()
export class MessagesService {
  constructor(
    private readonly repo: MessagesRepository,
    private readonly userService: UserService,
    private readonly logger: AppLoggerService,
    private communicationService: CommunicationService,
  ) {}

  async create(dto: CreateMessageDto, userId: number) {
    try {
      const user = await this.userService.getUserById(userId);
      if (user.userType !== UserTypeEnum.ORG) {
        this.logger.error(messageConstMessages.INVALID_SENDER_TYPE);
        throw new InifniBadRequestException(ERROR_CODES.MESSAGE_INVALID_SENDER);
      }
      const receiver = await this.userService.getUserById(dto.receiverId);
      if (receiver.userType !== UserTypeEnum.ORG) {
        this.logger.error(messageConstMessages.INVALID_RECEIVER_TYPE);
        throw new InifniBadRequestException(ERROR_CODES.MESSAGE_INVALID_RECEIVER);
      }
      const result = await this.repo.createMessage(dto, userId);

      // Uncomment to enable message notifications
      // try {
      //   await this.sendMessageNotification(
      //     receiver.fullName,
      //     receiver.countryCode + receiver.phoneNumber,
      //     user.fullName,
      //   );
      // } catch (error) {
      //   this.logger.log(messageConstMessages.MESSAGE_NOTIFICATION_FAILED, { error });
      // }
      return result;
    } catch (error) {
      handleKnownErrors(ERROR_CODES.MESSAGE_SAVE_FAILED, error);
    }
  }

  async findAll(
    programId: number,
    userId: number,
    seekerId?: number,
    messengerId?: number,
    limit = 20,
    offset = 0,
    sort: 'ASC' | 'DESC' = 'DESC',
  ) {
    try {
      const result = await this.repo.findMessages(
        programId,
        userId,
        seekerId,
        messengerId,
        limit,
        offset,
        sort,
      );
      const totalPages = Math.ceil(result.total / limit);
      return {
        data: result.data,
        pagination: {
          totalPages,
          pageNumber: Math.floor(offset / limit) + 1,
          pageSize: limit,
          totalRecords: result.total,
          numberOfRecords: result.data.length,
        },
      };
    } catch (error) {
      handleKnownErrors(ERROR_CODES.MESSAGE_GET_FAILED, error);
    }
  }

  async findSenders(searchText = ''): Promise<User[]> {
    try {
      return await this.repo.findSenders(searchText);
    } catch (error) {
      handleKnownErrors(ERROR_CODES.MESSAGE_GET_FAILED, error);
    }
  }
  async sendMessageNotification(name: string, mobileNumber: string, senderName: string) {
    try {
      // Logic to send message notification
      const messageData: SendTemplateMessageDto = {
        whatsappNumber: mobileNumber.replace('+', ''), // Remove + prefix
        templateName: process.env.WATI_ADMINS_MESSAGE_NOTIFICATION_TEMPLATE_ID || '',
        broadcastName: process.env.WATI_ADMINS_MESSAGE_NOTIFICATION_TEMPLATE_ID || '',
        parameters: [
          { name: 'admin_name', value: name },
          { name: 'sender_admin_name', value: senderName },
        ],
      };
      await this.communicationService.sendTemplateMessage(messageData);
    } catch (error) {
      this.logger.log(messageConstMessages.MESSAGE_NOTIFICATION_FAILED, { error });
    }
  }
}
