import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { Message, Program, User } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { CreateMessageDto } from './dto/create-message.dto';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { ROLE_KEYS } from 'src/common/constants/strings-constants';
import { UserTypeEnum } from 'src/common/enum/user-type.enum';

@Injectable()
export class MessagesRepository {
  constructor(
    @InjectRepository(Message)
    private readonly repo: Repository<Message>,
    @InjectRepository(User)
    private readonly userRepo: Repository<User>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async createMessage(dto: CreateMessageDto, senderId: number): Promise<Message> {
    this.logger.log(
      `Creating message from senderId ${senderId} to receiverId ${dto.receiverId} for programId ${dto.programId}`,
    );
    try {
      const entity = this.repo.create({
        program: { id: dto.programId } as Program,
        seeker: { id: dto.seekerId } as User,
        sender: { id: senderId } as User,
        receiver: { id: dto.receiverId } as User,
        content: dto.content,
        createdBy: { id: senderId } as User,
      });
      const saved = await this.commonDataService.save(this.repo, entity);
      this.logger.log(`Message created with id ${saved.id}`);
      return saved;
    } catch (error) {
      this.logger.error('Error creating message', error);
      handleKnownErrors(ERROR_CODES.MESSAGE_SAVE_FAILED, error);
    }
  }

  async findMessages(
    programId: number,
    userId: number,
    seekerId?: number,
    messengerId?: number,
    limit = 20,
    offset = 0,
    sort: 'ASC' | 'DESC' = 'DESC',
  ): Promise<{ data: Message[]; total: number }> {
    this.logger.log(
      `Finding messages for programId ${programId}, userId ${userId}, seekerId ${seekerId}, messengerId ${messengerId}`,
    );
    try {
      const qb = this.repo
        .createQueryBuilder('message')
        .select([
          'message.id',
          'message.programId',
          'message.seekerId',
          'message.senderId',
          'message.receiverId',
          'message.content',
          'message.createdAt',
        ])
        .leftJoin('message.sender', 'sender')
        .addSelect(['sender.id', 'sender.fullName', 'sender.profileUrl', 'sender.legalFullName'])
        .leftJoin('message.receiver', 'receiver')
        .addSelect(['receiver.id', 'receiver.fullName', 'receiver.profileUrl', 'receiver.legalFullName'])
        .leftJoin('message.seeker', 'seeker')
        .addSelect(['seeker.id', 'seeker.fullName', 'seeker.profileUrl', 'seeker.legalFullName'])
        .leftJoin(
          'hdb_program_registration',
          'registration',
          'registration.program_id = message.program_id AND registration.user_id = message.seeker_id AND registration.deleted_at IS NULL',
        )
        .addSelect('registration.id', 'registrationId')
        .where('message.programId = :programId', { programId })
        .andWhere('registration.id IS NOT NULL');

      if (seekerId) {
        qb.andWhere('message.seekerId = :seekerId', { seekerId });
      }

      if (messengerId) {
        qb.andWhere(
          '( (message.senderId = :userId AND message.receiverId = :messengerId) OR (message.senderId = :messengerId AND message.receiverId = :userId) )',
          { userId, messengerId },
        );
      } else if (seekerId) {
        qb.andWhere(
          new Brackets((qb1) => {
            qb1.where('message.senderId = :userId').orWhere('message.receiverId = :userId');
          }),
          { userId },
        );
      } else {
        qb.andWhere('message.receiverId = :userId', { userId });
      }

      qb.orderBy('message.createdAt', sort).take(limit).skip(offset);

      const [entities, total] = await qb.getManyAndCount();
      const rawData = await qb.getRawMany();
      // Create a map from message id to registrationId from rawData
      const registrationIdMap = new Map<number, number | null>();
      rawData.forEach((row: any) => {
        registrationIdMap.set(Number(row.message_id || row.message_id), row.registrationId ? Number(row.registrationId) : null);
      });

      const data = entities.map((msg) => ({
        ...msg,
        registrationId: registrationIdMap.get(msg.id) ?? null,
      }));
      this.logger.log(`Found ${data.length} messages (total: ${total})`);
      return { data, total };
    } catch (error) {
      this.logger.error('Error finding messages', error);
      handleKnownErrors(ERROR_CODES.MESSAGE_GET_FAILED, error);
    }
  }

  async findSenders(searchText = ''): Promise<User[]> {
    this.logger.log(`Finding senders with searchText: "${searchText}"`);
    try {
      const qb = this.userRepo
        .createQueryBuilder('user')
        .select(['user.id', 'user.fullName', 'user.profileUrl'])
        .leftJoin('user.userRoleMaps', 'urm')
        .leftJoin('urm.role', 'role')
        .where('role.roleKey != :viewer', { viewer: ROLE_KEYS.VIEWER })
        .andWhere('user.userType = :org', { org: UserTypeEnum.ORG });

      if (searchText) {
        qb.andWhere('user.fullName ILIKE :search', { search: `%${searchText}%` });
      }

      const users = await qb.getMany();
      this.logger.log(`Found ${users.length} senders`);
      return users;
    } catch (error) {
      this.logger.error('Error finding senders', error);
      handleKnownErrors(ERROR_CODES.MESSAGE_GET_FAILED, error);
    }
  }
}
