import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, DataSource } from 'typeorm';
import { RegistrationInvoiceDetail, RegistrationPaymentDetail, ProgramRegistration} from 'src/common/entities';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { InvoiceStatusEnum } from 'src/common/enum/invoice-status.enum';
import { InvoiceTypeEnum } from 'src/common/enum/invoice-type.enum';
import InifniBadRequestException from 'src/common/exceptions/infini-badrequest-exception';
import { amountInWordsIndian, areGSTsFromSameState, deductDaysFromDate, formatDate, formatDateIST, formatDateTime, generateNextSequence, getAcademicYearFromStartDate } from 'src/common/utils/common.util';
import { eInvoiceConstants, stateGstCodes, zeptoEmailCreadentials } from 'src/common/constants/strings-constants';
import { CommunicationService } from 'src/communication/communication.service';
import { SendTemplateMessageDto } from 'src/communication/dto/whatsapp-communication.dto';
import { SendSingleEmailDto } from 'src/communication/dto/email-communication.dto';
import { INVOICE } from 'src/common/templates/templates';
import { InvoiceTemplateDto } from 'src/common/templates/dtos/templates.dto';
import axios from 'axios';
import { v4 as uuidv4 } from 'uuid';
import { EInvoiceResponse } from './dto/create-invoice.dto';
import { oAuth } from 'src/common/utils/common.util';
import { AwsS3Service } from 'src/common/services/awsS3.service';
import { UserRepository } from 'src/user/user.repository';
import chromium from "@sparticuz/chromium-min";
import { InvoiceEinvoiceHistoryService } from './services/invoice-einvoice-history.service';
import { InvoiceRepository } from './invoice.repository';
import { PaymentStatusEnum } from 'src/common/enum/payment-status.enum';
import { PaymentModeEnum } from 'src/common/enum/payment-mode.enum';
import { SubProgramTypeEnum } from 'src/common/enum/sub-program-type.enum';
import { RegistrationApprovalService } from 'src/registration-approval/registration-approval.service';

const puppeteer = require('puppeteer-core');
// const chromium = require('@sparticuz/chromium-min');


@Injectable()
export class InvoiceService {
  constructor(
    @InjectRepository(RegistrationInvoiceDetail)
    private readonly invoiceRepo: Repository<RegistrationInvoiceDetail>,
    @InjectRepository(RegistrationPaymentDetail)
    private readonly paymentRepo: Repository<RegistrationPaymentDetail>,
    @InjectRepository(ProgramRegistration)
    private readonly registrationRepo: Repository<ProgramRegistration>,
    private readonly dataSource: DataSource,
    private readonly logger: AppLoggerService,
    private readonly communicationService: CommunicationService,
    private readonly awsS3Service: AwsS3Service, 
    private readonly userRepository: UserRepository,
    private readonly einvoiceHistoryService: InvoiceEinvoiceHistoryService,
    private readonly invoiceRepository: InvoiceRepository,
    private readonly registrationApprovalService: RegistrationApprovalService,
  ) {}

  async generateAndSendInvoice(
    registrationId: number,
    userId?: number,
    isSendInvoice: boolean = true,
  ) {
    this.logger.log('Generating and sending invoice', { registrationId });

    try {
      // Get invoice and related data
      
      const invoiceData = await this.getInvoiceGenerationData(registrationId);
      // Generate PDF (implement based on your requirements)
      if (!invoiceData) {
        handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, new Error('Invoice not found'));
      }
      if (
        invoiceData.payment?.paymentStatus !== PaymentStatusEnum.ONLINE_COMPLETED &&
        invoiceData.payment?.paymentStatus !== PaymentStatusEnum.OFFLINE_COMPLETED
      ) {
        handleKnownErrors(ERROR_CODES.PAYMENT_INCOMPLETE, new Error('Payment not completed'));
      }
      // Generate invoice sequence number if not exists
      if (!invoiceData.invoice.invoiceSequenceNumber) {
        if (!process.env.ONLINE_INVOICE_CODE || !process.env.OFFLINE_INVOICE_CODE) {
          handleKnownErrors(
            ERROR_CODES.INVOICE_CODE_NOT_FOUND,
            new Error('Invoice code not found'),
          );
        }
        const invoiceType =
          invoiceData.payment.paymentMode === PaymentModeEnum.ONLINE
            ? process.env.ONLINE_INVOICE_CODE
            : process.env.OFFLINE_INVOICE_CODE;
        const sequenceNumber = await this.generateInvoiceSequenceNumber(
          null,
          invoiceData.registration,
          invoiceType,
        );
        if (!sequenceNumber) {
          handleKnownErrors(
            ERROR_CODES.INVOICE_SEQUENCE_NUMBER_NOT_FOUND,
            new Error('Invoice sequence number not found'),
          );
        }
        await this.updateInvoiceSequenceNumber(invoiceData.invoice.id, sequenceNumber);
        invoiceData.invoice.invoiceSequenceNumber = sequenceNumber;
      } else {
        this.logger.log('Invoice sequence number already exists', {
          invoiceSequenceNumber: invoiceData.invoice.invoiceSequenceNumber,
        });
      }
      let invoicePdfUrl
      if (invoiceData.invoice.invoicePdfUrl) {
        invoicePdfUrl = invoiceData.invoice.invoicePdfUrl;
      } else {
        let updatedInvoiceData = invoiceData
        if (invoiceData.invoice.gstNumber && invoiceData.invoice.isGstRegistered && !invoiceData.invoice.einvoiceAckNumber) {
          if (process.env.ENABLE_EINVOICE === 'true') {
            const eInvoice = await this.generateEInvoice(registrationId);
            this.logger.log('E-invoice generated successfully', { eInvoice });
          } else {
            this.logger.warn(
              'E-invoice generation is disabled. Generating regular invoice PDF instead.',
            );
          }
          updatedInvoiceData = await this.getInvoiceGenerationData(registrationId);
        }
        const pdfBuffer = await this.generateInvoicePDF(updatedInvoiceData);
        invoicePdfUrl = pdfBuffer.url;
        await this.einvoiceHistoryService.createEinvoiceHistory({
          invoiceId: invoiceData.invoice.id,
          invoiceType: invoiceData.invoice.invoiceType,
          zip: invoiceData.invoice.zip,
          gstNumber: invoiceData.invoice.gstNumber,
          panNumber: invoiceData.invoice.panNumber,
          invoiceSequenceNumber: invoiceData.invoice.invoiceSequenceNumber,
          invoiceIssuedDate: invoiceData.invoice.invoiceIssuedDate,
          einvoiceQrLink: invoiceData.invoice.einvoiceQrLink,
          einvoiceAckDate: invoiceData.invoice.einvoiceAckDate,
          einvoiceInvRefNum: invoiceData.invoice.einvoiceInvRefNum,
          einvoiceIsCancellable: invoiceData.invoice.einvoiceIsCancellable,
          einvoiceStatusFormatted: invoiceData.invoice.einvoiceStatusFormatted,
          einvoiceFormattedStatus: invoiceData.invoice.einvoiceFormattedStatus,
          einvoiceAckNumber: invoiceData.invoice.einvoiceAckNumber,
          einvoiceStatus: invoiceData.invoice.einvoiceStatus,
          einvoicePdfUrl: invoicePdfUrl,
          createdBy: userId,
        });
        // Update invoice status
        await this.updateInvoiceStatus(
          invoiceData.invoice.id,
          InvoiceStatusEnum.INVOICE_COMPLETED,
          invoicePdfUrl,
          userId,
        );
      }
      
      this.logger.log('PDF generated successfully',  invoicePdfUrl );
      const response = await axios.get(invoicePdfUrl, {
        responseType: 'arraybuffer',
        headers: {
          referer: process.env.BACKEND_URL,
        },
      });
      this.logger.log('PDF buffer retrieved successfully', { length: response.data.byteLength });
      const fileData = Buffer.from(response.data, 'binary').toString('base64');


      // Send email with invoice (implement based on your email service)
      if (isSendInvoice) {
        try {
          await this.sendInvoiceEmail(invoiceData, fileData);
        } catch (error) {
          this.logger.error('Failed to send invoice email', '', { error });
        }
      }

      this.logger.log('Invoice generated and sent successfully', { 
        registrationId, 
        invoiceId: invoiceData.invoice.id,
        sequenceNumber: invoiceData.invoice.invoiceSequenceNumber
      });

      return {
        invoiceId: invoiceData.invoice.id,
        sequenceNumber: invoiceData.invoice.invoiceSequenceNumber,
        status: InvoiceStatusEnum.INVOICE_COMPLETED,
      };

    } catch (error) {
      this.logger.error('Invoice generation failed', '', { registrationId, error: error.message });
      handleKnownErrors(ERROR_CODES.INVOICE_GENERATION_FAILED, error);
    }
  }

  async getInvoiceDetails(registrationId: number) {
    try {
      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program', 'registration.allocatedProgram', 'registration.allocatedSession', 'registration.user', 'registration.paymentDetails' ],
      });

      if (!invoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      return invoice;
    } catch (error) {
      this.logger.error('Error getting invoice details', '', { registrationId, error });
      throw error;
    }
  }
  async getRegistrationDetails(registrationId: number) {
    try {
      const registration = await this.registrationRepo.findOne({
        where: { id: registrationId },
        relations: [
          'program', 
          'program.type', 
          'program.venueAddress',
          'programSession', 
          'user',
          'createdBy',
          'allocatedProgram',
          'allocatedSession',
        ],
      });

      if (!registration) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          `Registration ${registrationId} not found`
        );
      }

      return registration;
    } catch (error) {
      this.logger.error('Error getting registration details', '', { registrationId, error });
      throw error;
    }
  }

  async updateInvoiceStatus(invoiceId: number, status: InvoiceStatusEnum, invoicePdfUrl?: string, userId?: number) {
    return await this.dataSource.transaction(async (manager) => {
      const invoiceRepo = manager.getRepository(RegistrationInvoiceDetail);

      // Get existing invoice to retrieve its details
      const existingInvoice = await invoiceRepo.findOne({ where: { id: invoiceId } });
      if (!existingInvoice) {
        this.logger.error('Invoice not found for update', invoiceId.toString());
      }

      const updateData: any = {
        invoiceStatus: status,
        auditRefId: invoiceId,
        parentRefId: existingInvoice?.registrationId,
      };

      if (userId) {
        updateData.updatedBy = { id: userId } as any;
      }

      if (status === InvoiceStatusEnum.INVOICE_COMPLETED) {
        updateData.invoiceIssuedDate = new Date();
      }

      if (invoicePdfUrl) {
        updateData.invoicePdfUrl = invoicePdfUrl;
      }

      await invoiceRepo.update(invoiceId, updateData);

      this.logger.log('Invoice status updated', { invoiceId, status });
    });
  }

  async regenerateInvoice(registrationId: number, userId: number) {
    this.logger.log('Regenerating invoice', { registrationId });

    try {
      // Check if invoice exists
      const existingInvoice = await this.invoiceRepo.findOne({
        where: { registrationId },
      });

      if (!existingInvoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      // Reset invoice status and regenerate
      await this.updateInvoiceStatus(existingInvoice.id, InvoiceStatusEnum.DRAFT, undefined, userId);
      
      // Generate new invoice
      const result = await this.generateAndSendInvoice(registrationId);

      return result;

    } catch (error) {
      this.logger.error('Invoice regeneration failed', '', { registrationId, error: error.message });
      throw error;
    }
  }

  private async getInvoiceGenerationData(registrationId: number) {
    try {
      // Get invoice details
      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program', 'registration.allocatedProgram', 'registration.allocatedSession'],
      });

      if (!invoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      // Get payment details
      const payment = await this.paymentRepo.findOne({
        where: { registrationId },
      });

      // Get registration details
      const registration = await this.registrationRepo.findOne({
        where: { id: registrationId },
        relations: [
          'program', 
          'program.type', 
          'program.venueAddress',
          'programSession', 
          'user',
          'createdBy',
          'allocatedProgram',
          'allocatedSession',
        ],
      });

      if (!registration) {
        throw new InifniBadRequestException(
          ERROR_CODES.PROGRAM_REGISTRATION_NOTFOUND,
          null,
          null,
          `Registration ${registrationId} not found`
        );
      }

      return {
        invoice,
        payment,
        registration,
        program: registration.program,
        session: registration.programSession,
        allocatedProgram: registration.allocatedProgram,
        allocatedSession: registration.allocatedSession,
        user: registration.user,
      };
    } catch (error) {
      this.logger.error('Error getting invoice generation data', '', { registrationId, error });
      throw error;
    }
  }

  async generateInvoiceSequenceNumber(
    queryRunner: any,
    registration: ProgramRegistration,
    invoiceType: string
  ): Promise<string | undefined> {
    const startsAt = registration.program?.startsAt;
    const yearSeq = getAcademicYearFromStartDate(startsAt);
    const programCode = registration.allocatedProgram?.code || registration.program.code;

    const pattern = `${programCode}${yearSeq}/${invoiceType}`;

    // Check if we need to create our own QueryRunner
    const shouldReleaseRunner = !queryRunner;
    const qrRunner = queryRunner ?? this.dataSource.createQueryRunner();

    try {
      if (shouldReleaseRunner) {
        await qrRunner.connect();
      }
      // Fetch the latest invoice matching the pattern
      const latestInvoice = await qrRunner.manager
        .createQueryBuilder(RegistrationInvoiceDetail, 'invoice')
        .andWhere('invoice.invoiceSequenceNumber IS NOT NULL')
        .andWhere('invoice.invoiceSequenceNumber LIKE :pattern', { pattern: `${pattern}%` })
        .orderBy('invoice.invoiceSequenceNumber', 'DESC')
        .getOne();

      if (latestInvoice) {
        // Generate the next sequence number based on the latest invoice
        return generateNextSequence(pattern, latestInvoice.invoiceSequenceNumber);
      } else {
        // If no previous invoice, start with 001
        if (!process.env.INVOICE_SEQUENCE_START) {
          handleKnownErrors(
            ERROR_CODES.INVOICE_SEQUENCE_START_NOT_FOUND,
            new Error('Invoice sequence start not found'),
          );
        }
        return `${pattern}${process.env.INVOICE_SEQUENCE_START}`;
      }
    } catch (error) {
      this.logger.error('Error generating invoice sequence number', '', { error });
      handleKnownErrors(ERROR_CODES.INVOICE_SEQUENCE_GENERATION_FAILED, error);
    } finally {
      // Only release if we created the QueryRunner ourselves
      if (shouldReleaseRunner) {
        await qrRunner.release();
      }
    }
  }

  private async updateInvoiceSequenceNumber(invoiceId: number, sequenceNumber: string) {
    try {
      // Get existing invoice to retrieve its details for audit
      const existingInvoice = await this.invoiceRepo.findOne({ where: { id: invoiceId } });
      if (!existingInvoice) {
        this.logger.error('Invoice not found for update', invoiceId.toString());
        handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, new Error('Invoice not found'));
      }
      await this.invoiceRepo.update(invoiceId, {
        invoiceSequenceNumber: sequenceNumber,
        auditRefId: invoiceId,
        parentRefId: existingInvoice?.registrationId,
      });
    } catch (error) {
      this.logger.error('Error updating invoice sequence number', '', {
        invoiceId,
        sequenceNumber,
        error,
      });
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_UPDATE_FAILED, error);
      throw error;
    }
  }

  private async generateInvoicePDF(invoiceData: any): Promise<{ url: string; buffer: Buffer }> {
    // Implement PDF generation based on your requirements
    // This is a placeholder implementation
    
    this.logger.log('Generating invoice PDF', { 
      invoiceId: invoiceData.invoice.id,
      registrationId: invoiceData.registration.id
    });

    try {
      // You can use libraries like:
      // - puppeteer for HTML to PDF
      // const puppeteer = require('puppeteer');

      // Launch Puppeteer
      // const browser = await puppeteer.launch()
      const browser = await puppeteer.launch({
        args: [
          ...(Array.isArray(chromium.args) ? chromium.args : []),
          '--no-sandbox',
          '--disable-setuid-sandbox',
          '--disable-features=AudioServiceOutOfProcess',
          '--disable-gpu',
          '--disable-software-rasterize',
          '--disable-features=AudioServiceOutOfProcessKillAtHang',
          '--single-process',
          '--disable-software-rasterizer',
        ],
        defaultViewport: null,
        executablePath: await chromium.executablePath('https://github.com/Sparticuz/chromium/releases/download/v119.0.2/chromium-v119.0.2-pack.tar'),
        headless: true,
      });
      const page = await browser.newPage();
      // Set the HTML content
      this.logger.log('Generating invoice template data',invoiceData );
      const totalAmount = Math.round(
        Number(invoiceData?.payment?.originalAmount) + Number(invoiceData?.payment?.gstAmount) || 0,
      );
      const currentDate =
        invoiceData.payment.paymentMode === PaymentModeEnum.OFFLINE
          ? formatDateIST(invoiceData?.payment?.markAsReceivedDate?.toISOString() ?? '')
          : formatDateIST(invoiceData?.payment?.paymentDate?.toISOString() ?? '');
      const invoiceTemplateData: InvoiceTemplateDto = {
        companyAddress: invoiceData.program?.invoiceSenderAddress || 'Infinitheism',
        companyGST: invoiceData.program?.gstNumber,
        companyCIN: invoiceData.program?.invoiceSenderCin || '',
        invoiceNumber: invoiceData.invoice.invoiceSequenceNumber || '',
        currentDate: currentDate
          ? formatDateIST(currentDate)
          : formatDateIST(new Date().toISOString()),
        placeOfSupply: invoiceData?.invoice?.gstNumber ? stateGstCodes.find(code => code.code === invoiceData.invoice?.gstNumber.substring(0, 2))?.stateName || null : null,
        seekerName: invoiceData.invoice.invoiceName,
        seekerAddress: invoiceData.invoice.invoiceAddress,
        seekerGstin: invoiceData.invoice?.gstNumber || '',
        programName:
          invoiceData.allocatedProgram?.subProgramType === SubProgramTypeEnum.PST_MSD
            ? 'My Sacred Days - MSD'
            : invoiceData.allocatedProgram?.subProgramType === SubProgramTypeEnum.PST_HDB
              ? 'Higher Deeper Beyond - HDB'
              : invoiceData.program?.name || '',
        reportingTime: formatDateIST(
          String(invoiceData?.allocatedProgram?.startsAt?.toISOString() ?? ''),
        ),
        checkOutTime: formatDateIST(
          String(invoiceData?.allocatedProgram?.endsAt?.toISOString() ?? ''),
        ),
        amount: String(Math.round(Number(invoiceData.payment?.originalAmount))) || '',
        igstCheck: (invoiceData.invoice?.gstNumber && invoiceData.program?.gstNumber) ? !areGSTsFromSameState(invoiceData.program?.gstNumber, invoiceData.invoice?.gstNumber) : false,
        igst: `${Math.round(Number(invoiceData.registration.allocatedProgram?.basePrice) * (Number(invoiceData.program.igst) / 100)) || 0}`,
        cgst: `${Math.round(Number(invoiceData.registration.allocatedProgram?.basePrice) * (Number(invoiceData.program.cgst) / 100)) || 0}`,
        sgst: `${Math.round(Number(invoiceData.registration.allocatedProgram?.basePrice) * (Number(invoiceData.program.sgst) / 100)) || 0}`,
        igstPercentage: (Number(invoiceData.program?.igst)) || 0,
        cgstPercentage: (Number(invoiceData.program?.cgst)) || 0,
        sgstPercentage: (Number(invoiceData.program?.sgst)) || 0,
        totalAmount: totalAmount,
        amountInWords: amountInWordsIndian(Number(totalAmount) || 0),
        qr: invoiceData?.invoice?.einvoiceQrLink || '', // 'https://delta-node-test.s3.ap-south-1.amazonaws.com/qr.png', // sample file, // Placeholder QR code URL
        irpDate: invoiceData?.invoice?.einvoiceAckDate
          ? formatDateTime(String(invoiceData?.invoice?.einvoiceAckDate ?? ''))
          : '', //formatDate(new Date().toISOString()) || '',
        acknowledgeNumber: invoiceData?.invoice?.einvoiceAckNumber || '', // '1234567890123456789',
        irn: invoiceData?.invoice?.einvoiceInvRefNum || '',
        companyPAN: invoiceData.program?.invoiceSenderPan || '',
      };
      await page.setContent(INVOICE(invoiceTemplateData));


      
      // For now, return a placeholder buffer
      const pdfBuffer = await page.pdf({
        format: 'A4',
        printBackground: true,
      });      
      await browser.close();

      // const bucketName = process.env.AWS_S3_BUCKET_NAME || '';
      const key = `invoices/${invoiceData.program.code}/${invoiceData.invoice.invoiceSequenceNumber || uuidv4()}.pdf`;
      const uploadedInvoice = await this.awsS3Service.uploadToS3(key, pdfBuffer, 'application/pdf');
      // const s3Url = this.awsS3Service.getS3Url(key);
      // const s3Url = await this.awsS3Service.getSignedUrl(key);
      return {url: uploadedInvoice, buffer: pdfBuffer};
      

    } catch (error) {
      this.logger.error('PDF generation failed', '', { invoiceId: invoiceData.invoice.id, error });
      throw error;
    }
  }

  private async sendInvoiceEmail(invoiceData: any, pdfBuffer: string) {
    // Implement email sending based on your email service
    
    this.logger.log('Sending invoice email', { 
      invoiceId: invoiceData.invoice.id,
      email: invoiceData.invoice.invoiceEmail
    });

    try {
      // You can use your existing email service here
      // Example:
      /*
      await this.emailService.sendEmail({
        to: invoiceData.invoice.invoiceEmail,
        subject: `Invoice ${invoiceData.invoice.invoiceSequenceNumber} - ${invoiceData.program.name}`,
        template: 'invoice',
        context: {
          invoiceNumber: invoiceData.invoice.invoiceSequenceNumber,
          customerName: invoiceData.invoice.invoiceName,
          programName: invoiceData.program.name,
          amount: invoiceData.payment.subTotal,
        },
        attachments: [
          {
            filename: `invoice-${invoiceData.invoice.invoiceSequenceNumber}.pdf`,
            content: pdfBuffer,
            contentType: 'application/pdf',
          }
        ]
      });
      */
     const getFinanceUsers = await this.userRepository.getUsersByRoleKey('ROLE_FINANCE_MANAGER');
     const ccEmails: { emailAddress: string; name: string }[] = [];
     let bccEmails: { emailAddress: string; name: string }[] = [];
    //  if (process.env.HDB_BCC_EMAIL) {
    //    bccEmails.push({ emailAddress: process.env.HDB_BCC_EMAIL, name: process.env.HDB_BCC_EMAIL_NAME || 'HDB' });
    //  }
     if (getFinanceUsers && getFinanceUsers.length > 0) {
      //  ccEmails = getFinanceUsers.map((user) => ({
      //    emailAddress: user.email,
      //    name: user.fullName || 'Finance Manager',
      //  }));
       bccEmails = bccEmails.concat(
         getFinanceUsers.map((user) => ({
           emailAddress: user.email,
           name: user.fullName || 'Finance Manager',
         }))
       );
     }
      const email = invoiceData.invoice.invoiceEmail;
      if (email) {
        // Prepare merge info for email
        const lastDate = formatDateIST(deductDaysFromDate(new Date(invoiceData?.allocatedProgram?.startsAt?.toISOString() ?? ""), 10).toISOString());
        const totalAmount =
        Number(invoiceData?.payment?.originalAmount) + Number(invoiceData?.payment?.gstAmount) || 0;
        const mergeInfo = {
          hdb_msd_no: invoiceData?.allocatedProgram?.name || '',
          hdb_msd_date: `${formatDateIST(invoiceData?.allocatedProgram?.startsAt.toISOString() ?? '')} to ${formatDateIST(invoiceData?.allocatedProgram?.endsAt.toISOString() ?? '')}`,
          hdb_msd_amount: totalAmount || 0,
          reg_name: invoiceData?.registration?.fullName || 'Infinitheist',
          hdb_msd: invoiceData?.allocatedProgram?.name || '',
          last_date: lastDate, // formatDate(invoiceData?.allocatedProgram?.registrationEndsAt?.toISOString())
        };
        const startsAt = invoiceData?.program?.startsAt;
        const startYear = startsAt ? startsAt.getFullYear().toString() : '';
        const nextYear = startsAt ? (startsAt.getFullYear() + 1).toString().slice(-2) : '';
        const programName = invoiceData?.program?.type?.key === 'PT_HDBMSD' ? 'HDB/MSD' : invoiceData.program.code;
        const attachments = {
          filename: `${programName} ${startYear}-${nextYear} Invoice - ${invoiceData?.invoice?.invoiceSequenceNumber ?? "invoice"}.pdf`, //`invoice-${invoiceData.invoice.invoiceSequenceNumber || invoiceData.invoice.id}.pdf`,
          content: pdfBuffer,
          contentType: 'application/pdf',
        };
        const emailData: SendSingleEmailDto = {
          templateKey: process.env.ZEPTO_INVOICE_EMAIL_TEMPLATE_ID,
          from: {
            address: zeptoEmailCreadentials.ZEPTO_EMAIL,
            name: invoiceData.program.emailSenderName || zeptoEmailCreadentials.ZEPTO_EMAIL_NAME,
          },
          to: {
            emailAddress: email,
            name: email || 'Infinitheist',
          },
          cc: ccEmails,
          bcc: bccEmails,
          mergeInfo: mergeInfo,
          attachments: [{
            name: attachments.filename,
            content: attachments.content,
            mime_type: attachments.contentType,
          }],
          subject: 'Payment Confirmation - Infinitheism',
          trackinfo: {
            registrationId: invoiceData.registrationId,
            // createdBy: invoiceData.user.id,
            // updatedBy: invoiceData.user.id,
          },
        };

        try {
          await this.communicationService.sendSingleEmail(emailData);
          this.logger.log('Invoice email sent successfully', { emailData });
        } catch (error) {
          this.logger.error('Error sending invoice email', '', { emailData, error });
        }

        const messageData: SendTemplateMessageDto = {
          whatsappNumber: invoiceData.registration.mobileNumber.slice(0) || '',
          templateName: process.env.WATI_INVOICE_TEMPLATE_ID || '',
          broadcastName: process.env.WATI_INVOICE_TEMPLATE_ID || '',
          parameters: [
            { name: 'reg_name', value: invoiceData?.registration?.fullName },
            { name: 'hbd_msd_variable', value: invoiceData?.allocatedProgram?.name },
            {
              name: 's_date',
              value: formatDateIST(invoiceData?.allocatedProgram?.startsAt?.toISOString()),
            },
            {
              name: 'e_date',
              value: formatDateIST(invoiceData?.allocatedProgram?.endsAt?.toISOString()),
            },
          ],
          trackinfo: {
            registrationId: invoiceData.invoice.registrationId,
            // createdBy: invoiceData.user.id,
            // updatedBy: invoiceData.user.id,
          }
        };

        try {
          await this.communicationService.sendTemplateMessage(messageData);
          this.logger.log('WhatsApp message sent successfully', { messageData });
        } catch (error) {
          this.logger.error('Error sending WhatsApp message', '', { messageData, error });
        }




      } else {
        this.logger.warn('No invoice email provided for payment confirmation', {
          invoiceData,
          invoiceEmail: invoiceData.invoice?.invoiceEmail,
        });

      }

    } catch (error) {
      this.logger.error('Invoice email sending failed', '', { 
        invoiceId: invoiceData.invoice.id, 
        email: invoiceData.invoice.invoiceEmail,
        error 
      });
      throw error;
    }
  }

  async getInvoicesByRegistration(registrationId: number) {
    try {
      const invoices = await this.invoiceRepo.find({
        where: { registrationId },
        order: { createdAt: 'DESC' },
      });

      return invoices;
    } catch (error) {
      this.logger.error('Error getting invoices by registration', '', { registrationId, error });
      throw error;
    }
  }

  async downloadInvoice(invoiceId: number): Promise<{ buffer: Buffer; filename: string }> {
    try {
      const invoice = await this.invoiceRepo.findOne({
        where: { id: invoiceId },
        relations: ['registration'],
      });

      if (!invoice) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `Invoice ${invoiceId} not found`
        );
      }

      // Get full invoice data for PDF generation
      const invoiceData = await this.getInvoiceGenerationData(invoice.registrationId);
      
      // Generate PDF
      const pdfBuffer = await this.generateInvoicePDF(invoiceData);
      
      const filename = `invoice-${invoice.invoiceSequenceNumber || invoice.id}.pdf`;
      
      return { buffer: pdfBuffer.buffer, filename };

    } catch (error) {
      this.logger.error('Invoice download failed', '', { invoiceId, error });
      throw error;
    }
  }
  async generateEInvoice(registrationId: number): Promise<EInvoiceResponse | null> {
    this.logger.log('Generating e-invoice', { registrationId });

    try {
      // Generate e-invoice data
      const eInvoiceData = await this.generateEInvoiceData(registrationId);
      
      if (!eInvoiceData) {
        throw new InifniBadRequestException(
          ERROR_CODES.EINVOICE_GENERATION_FAILED,
          null,
          null,
          `Failed to generate e-invoice for registration ${registrationId}`
        );
      }
      const zohoEinvoiceAuth = await oAuth();

      const eInvoiceInfo = {
        method: 'post',
        url: process.env.ZOHO_E_INVOICE_URL,
        headers: {
          'Content-Type': 'application/json',
          Authorization: `Zoho-oauthtoken ${zohoEinvoiceAuth.access_token}`,
          'X-com-zoho-invoice-organizationid': 782126836,
        },
        data: JSON.stringify(eInvoiceData),
      };
      this.logger.log('E-invoice data prepared for API', { 
        registrationId, 
        eInvoiceData, 
        eInvoiceInfo
      });
      const response = await axios(eInvoiceInfo);
      // const einvoiceBody = {
      //   einvoiceQrLink: response.data.invoice.einvoice_details.qr_link,
      //   einvoiceAckNumber: response.data.invoice.einvoice_details.ack_number,
      //   einvoiceAckDate: response.data.invoice.einvoice_details.ack_date,
      //   einvoiceInvRefNum: response.data.invoice.einvoice_details.inv_ref_num,
      //   einvoiceIsCancellable: response.data.invoice.einvoice_details.is_cancellable,
      //   einvoiceStatusFormatted: response.data.invoice.einvoice_details.status_formatted,
      //   einvoiceStatus: response.data.invoice.einvoice_details.status,
      //   einvoiceFormattedStatus: response.data.invoice.einvoice_details.formatted_status,
      //   updatedAt: new Date(),
      // };

      // this.logger.log("einvoice generated data", {einvoiceBody})

      // Update the invoice with e-invoice details
      // Get existing invoice to retrieve its details for audit
      const existingInvoice = await this.invoiceRepo.findOne({ where: { registrationId } });
      if (!existingInvoice) {
        this.logger.error('Invoice not found for update', registrationId.toString());
      }

      await this.invoiceRepo.update(
        { registrationId },
        {
          einvoiceQrLink: response.data.invoice.einvoice_details.qr_link || null,
          einvoiceAckNumber: response.data.invoice.einvoice_details.ack_number || null,
          einvoiceAckDate: response.data.invoice.einvoice_details.ack_date || null,
          einvoiceInvRefNum: response.data.invoice.einvoice_details.inv_ref_num || null,
          einvoiceIsCancellable: response.data.invoice.einvoice_details.is_cancellable || null,
          einvoiceStatusFormatted: response.data.invoice.einvoice_details.status_formatted || null,
          einvoiceStatus: response.data.invoice.einvoice_details.status || null,
          einvoiceFormattedStatus: response.data.invoice.einvoice_details.formatted_status || null,
          updatedAt: new Date(),
          auditRefId: existingInvoice?.id,
          parentRefId: registrationId,
        }
      );
      // await this.invoiceRepo.update(
      //   { registrationId },
      //   {
      //     einvoiceQrLink: 'https://delta-node-test.s3.ap-south-1.amazonaws.com/qr.png',
      //     einvoiceAckNumber: '1234567890123456789',
      //     einvoiceAckDate: new Date().toISOString(),
      //     einvoiceInvRefNum: '23f498ee41441ecad30f72ba5b9907506c3df70a17b0e0dff46b76a78640',
      //     updatedAt: new Date(),
      //   }
      // );
      // Here you can implement the logic to send the e-invoice to IRP or save it as needed
      // For now, we will just return the generated data
      return eInvoiceData;

    } catch (error) {
      this.logger.error('E-invoice generation failed', '', {
        registrationId,
        errordesc: {
          message: error?.message,
          stack: error?.stack,
          name: error?.name,
          code: error?.code,
          status: error?.response?.status,
          response: error?.response?.data,
        },
        error: error,
      });
      handleKnownErrors(ERROR_CODES.INVOICE_GENERATION_FAILED, error);
      // return null;
    }
  }
  async generateEInvoiceData(registrationId: number): Promise<EInvoiceResponse | null> {
    try {
      this.logger.log('Generating e-invoice data', { registrationId });

      // Get invoice data with all related information
      const invoiceData = await this.getInvoiceDetails(registrationId);

      
      if (!invoiceData) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }
      // if (invoiceData.gstNumber && invoiceData.isGstRegistered) {
      //   return null; // No e-invoice for GST registered users
      // }

      // Determine tax name based on GST comparison
      const taxName = this.determineTaxName(
        invoiceData.gstNumber,
        invoiceData?.registration?.program?.gstNumber || ''
      );

      this.logger.log('Tax name determined', { 
        taxName, 
        billingGst: invoiceData?.gstNumber,
        programGst: invoiceData?.registration?.program?.gstNumber 
      });

     
      // Process address information
      const addressInfo = this.processAddressInformation(invoiceData);

      const startsAt = formatDateIST(
        new Date(invoiceData?.registration?.allocatedProgram?.startsAt?.toISOString() ?? ''),
      );
      const endsAt = formatDateIST(
        new Date(invoiceData?.registration?.allocatedProgram?.endsAt?.toISOString() ?? ''),
      );
      const programName =
        invoiceData?.registration?.allocatedProgram?.subProgramType === SubProgramTypeEnum.PST_MSD
          ? 'My Sacred Days - MSD'
          : invoiceData?.registration?.allocatedProgram?.subProgramType === SubProgramTypeEnum.PST_HDB
            ? 'Higher Deeper Beyond - HDB'
            : invoiceData?.registration?.allocatedProgram?.name;

      const description = `Fee for '${programName}' program from ${startsAt} to ${endsAt} SAC: 999293 (Fee includes accommodation and food)`;

      this.logger.log('Program description', { description });
      const date =
        invoiceData?.registration?.paymentDetails?.[0]?.paymentMode === PaymentModeEnum.OFFLINE
          ? invoiceData?.registration?.paymentDetails?.[0]?.markAsReceivedDate
          : invoiceData?.registration?.paymentDetails?.[0]?.paymentDate;
      // Build e-invoice response
      const eInvoiceData: EInvoiceResponse = {
        contact: {
          contact_name: invoiceData?.invoiceName,
          company_name: eInvoiceConstants?.COMPANY_NAME,
          currency_code: eInvoiceConstants?.CURRENCY.INDIA,
          billing_address: {
            attention: eInvoiceConstants?.COMPANY_NAME,
            address: addressInfo?.address_1,
            street2: addressInfo?.address_2,
            state_code: addressInfo?.stateCode,
            city: addressInfo?.city,
            state: addressInfo?.state,
            zip: Number(addressInfo.zip ?? ''),
            country: addressInfo?.country,
            phone: Number(addressInfo?.phoneNumber),
          },
          gst_no: invoiceData?.gstNumber ?? '',
          gst_treatment: eInvoiceConstants.GST_TREATMENT,
        },
        invoice: {
          place_of_supply: addressInfo?.placeOfSupplyCode,
          invoice_number: invoiceData?.invoiceSequenceNumber,
          date: date
            ? new Date(date).toISOString().split('T')[0]
            : new Date().toISOString().split('T')[0],
          discount: 0,
          is_discount_before_tax: true,
          discount_type: 'item_level',
          shipping_charge: 0,
          line_items: [
            {
              description: description,
              rate: invoiceData?.registration?.allocatedProgram?.basePrice || 0,
              quantity: 1,
              unit: eInvoiceConstants?.UNITS,
              product_type: eInvoiceConstants?.SERVICE,
              hsn_or_sac: eInvoiceConstants?.HSN_CODE,//Number(invoiceData.hsn_or_sac),
              tax_name: taxName,
            },
          ],
          adjustment: 0,
          adjustment_description: '',
          notes: '',
          terms: '',
          subject_content: '',
          seller_gstin: invoiceData?.registration?.program?.gstNumber || '',
          is_inclusive_tax: false,
          tax_rounding: '',
          shipping_charge_tax_name: taxName,
          shipping_charge_sac_code: eInvoiceConstants.SHIPPING_CHARGE_SEC_CODE,
          is_reverse_charge_applied: false,
          is_customer_liable_for_tax: false,
          is_export_with_payment: false,
        },
      };

      // this.logger.log('E-invoice data generated successfully', { 
      //   registrationId,
      //   invoiceNumber: eInvoiceData.invoice.invoice_number,
      //   contactName: eInvoiceData.contact.contact_name
      // });

      return eInvoiceData;

    } catch (error) {
      this.logger.error('Failed to generate e-invoice data', '', { 
        registrationId, 
        error: error.message 
      });
      return null;
    }
  }
  private determineTaxName(billingGst: string, programGst: string): string {
    if (!billingGst || !programGst) {
      return eInvoiceConstants.SHIPPING_CHANRGE_TAX; // Default to intrastate
    }

    const billingStateCode = billingGst.substring(0, 2);
    const programStateCode = programGst.substring(0, 2);

    this.logger.log('GST code comparison', { 
      billingStateCode, 
      programStateCode,
      isInterstate: billingStateCode !== programStateCode
    });

    // If state codes are the same, it's intrastate (CGST + SGST)
    // If state codes are different, it's interstate (IGST)
    return billingStateCode === programStateCode 
      ? eInvoiceConstants.SHIPPING_CHANRGE_TAX 
      : eInvoiceConstants.SHIPPING_INTERSTATE_CHARGE_TAX;
  }
 /**
   * Processes and formats address information
   */
 private processAddressInformation(invoiceData: RegistrationInvoiceDetail) {
  let address_1 = 'No billing address found';
  let address_2 = '';
  let city = '';
  let stateCode = '';
  let state = '';
  const country = 'India';
  let phoneNumber = 0;
  let zip = '';
  let placeOfSupplyCode = '';


  // try {
  //   const registration = await this.paymentRepo.findOne({
  //     where: { registrationId: invoiceData.registrationId }
  //   });
    
    if (invoiceData) {
      address_1 = invoiceData?.invoiceAddress;
      address_2 = '';
      city =
        invoiceData?.registration?.city === 'Other'
          ? invoiceData?.registration?.otherCityName
          : invoiceData?.registration?.city;

      // Find state information from GST code
      if (invoiceData?.gstNumber) {
        const gstStateCode = invoiceData?.gstNumber?.substring(0, 2);
        const stateInfo = stateGstCodes.find(code => code.code === gstStateCode);
        
        if (stateInfo) {
          stateCode = stateInfo.state;
          state = stateInfo.stateName;
          placeOfSupplyCode = stateInfo.state;
        }
      }
  
      // Process phone number
      if (invoiceData?.registration?.mobileNumber) {
        const cleanPhone = invoiceData?.registration?.mobileNumber.replace('+91', '').trim();
        phoneNumber = cleanPhone ? Number(cleanPhone) : 0;
      }
  
      // Process ZIP code
      if (invoiceData.zip) {
        zip = invoiceData.zip ?? '';
      }
    }
  
    const addressInfo = {
      address_1,
      address_2,
      city,
      stateCode,
      state,
      country,
      phoneNumber,
      zip,
      placeOfSupplyCode
    };
  
    this.logger.log('Address information processed', { addressInfo });
  
    return addressInfo;
  // }catch (error) {
  //   this.logger.error('Error fetching registration data for address processing', error);
  //  return {
  //     address_1: '',
  //     city: '',
  //     stateCode: '',
  //     state: '',
  //     country: 'India',
  //     phoneNumber: 0,
  //     zip: 0,
  //     placeOfSupplyCode: '',
  //   };
  // }
  

  
}
 async getInvoiceByRegistrationId(registrationId: number): Promise<object | null> {
    try {
      let invoice = await this.invoiceRepository.findByRegistrationId(registrationId);

      if (!invoice?.invoicePdfUrl) {
        this.logger.log('No invoice found for registration so generating the ', { registrationId });
        await this.generateAndSendInvoice(registrationId, undefined, false);
        invoice = await this.invoiceRepository.findByRegistrationId(registrationId);
        if (!invoice || !invoice.invoicePdfUrl) {
          handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, null);
        }
      }

      const startsAt = invoice?.registration?.program?.startsAt;
      const startYear = startsAt ? startsAt.getFullYear().toString() : '';
      const nextYear = startsAt ? (startsAt.getFullYear() + 1).toString().slice(-2) : '';
      const programName =
        invoice?.registration?.program?.type?.key === 'PT_HDBMSD'
          ? 'HDB/MSD'
          : invoice?.registration?.program?.code;
      const filename = `${programName} ${startYear}-${nextYear} Invoice - ${invoice?.invoiceSequenceNumber ?? 'invoice'}.pdf`; //`invoice-${invoiceData.invoice.invoiceSequenceNumber || invoiceData.invoice.id}.pdf`,

      return { invoicePdfUrl: invoice.invoicePdfUrl, fileName: filename };
    } catch (error) {
      this.logger.error('Error getting invoice by registration ID', '', { registrationId, error });
      handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, error);
    }
  }
  async downloadProformaInvoice(registrationId: number): Promise<object | null> {
    this.logger.log('Downloading proforma invoice', { registrationId });

    try {
      // Get invoice data
      const invoiceData = await this.getRegistrationDetails(registrationId) // this.getInvoiceDetails(registrationId);
      
      if (!invoiceData) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }
      let proformaInvoiceUrl = invoiceData.proFormaInvoicePdfUrl;
      // If proforma invoice URL doesn't exist, generate it
      if (!proformaInvoiceUrl) {
        const generatedUrl = await this.registrationApprovalService.generateProFormaInvoicePDF(
          invoiceData,
          invoiceData.proFormaInvoiceSeqNumber,
        );
        await this.registrationRepo.update(
          { id: registrationId },
          { proFormaInvoicePdfUrl: generatedUrl.url },
        );
        proformaInvoiceUrl = generatedUrl.url;
      }

      this.logger.log('Proforma invoice download successfully', { registrationId });

      return { proFormaInvoice: proformaInvoiceUrl };

    } catch (error) {
      this.logger.error('Proforma invoice download failed', '', { registrationId, error: error.message });
      handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, error);
    }
  }
  async updateInvoice(registrationId: number, updateData: Partial<RegistrationInvoiceDetail>) {
    this.logger.log('Updating invoice', { registrationId, updateData });

    try {
      // Get existing invoice to retrieve its details for audit
      const existingInvoice = await this.invoiceRepo.findOne({ where: { registrationId } });
      if (!existingInvoice) {
        this.logger.error('Invoice not found for update', registrationId.toString());
      }

      // Add audit fields to updateData
      const auditableUpdateData = {
        ...updateData,
        auditRefId: existingInvoice?.id,
        parentRefId: registrationId,
      };

      const result = await this.invoiceRepo.update({ registrationId }, auditableUpdateData);

      if (!result.affected) {
        throw new InifniBadRequestException(
          ERROR_CODES.INVOICE_NOTFOUND,
          null,
          null,
          `No invoice found for registration ${registrationId}`
        );
      }

      this.logger.log('Invoice updated successfully', { registrationId });
    } catch (error) {
      this.logger.error('Error updating invoice', '', { registrationId, error });
      handleKnownErrors(ERROR_CODES.REGISTRATION_INVOICE_DETAIL_SAVE_FAILED, error);
    }
  }

 async getInvoiceByRegistration(registrationId: number): Promise<RegistrationInvoiceDetail | null> {
    this.logger.log('Fetching invoice by registration ID', { registrationId });

    try {
      const invoice = await this.invoiceRepo.findOne({
        where: { registrationId },
        relations: ['registration', 'registration.program'],
      });

      if (!invoice) {
        this.logger.warn('No invoice found for registration', { registrationId });
        return null;
      }

      return invoice;
    } catch (error) {
      this.logger.error('Error fetching invoice by registration ID', '', { registrationId, error });
      handleKnownErrors(ERROR_CODES.INVOICE_NOTFOUND, error);
    }
  }

}