import { Injectable, InternalServerErrorException } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { alwaysIncludeArray } from './../common/utils/fieldPool';
import { systemPrompt, userPrompt } from './../common/utils/prompts';
import { QuestionRepository } from 'src/question/question.repository';
import { FormSectionRepository } from 'src/form-section/form-section.repository';
import { ProgramQuestionRepository } from 'src/program-question/program-question.repository';
import { InifniNotFoundException } from 'src/common/exceptions/infini-notfound-exception';
import { UserRepository } from 'src/user/user.repository';
import { FormSection, Option, ProgramQuestion, Question } from 'src/common/entities';
import { InjectRepository } from '@nestjs/typeorm';
import { In, Repository, DataSource } from 'typeorm';
import { InjectDataSource } from '@nestjs/typeorm';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { OptionRepository } from 'src/option/option.repository';
import { QuestionOptionRepository } from 'src/question-option/question-option.repository';
import { AppLoggerService } from 'src/common/services/logger.service';
import { formSchemaConstMessages } from '../common/constants/strings-constants';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';

@Injectable()
export class FormSchemaService {
  private openai: any;
  constructor(
    private configService: ConfigService,
    @InjectDataSource()
    private readonly dataSource: DataSource,
    private readonly questionService: QuestionRepository,
    private readonly formSectionRepo: FormSectionRepository,
    private readonly programQuestionRepo: ProgramQuestionRepository,
    private readonly userRepo: UserRepository,
    @InjectRepository(FormSection)
    private readonly formSectionRepository: Repository<FormSection>,
    @InjectRepository(Option)
    private readonly optionRepo: Repository<Option>,
    private readonly optionRepository: OptionRepository,
    private readonly questionOptionRepo: QuestionOptionRepository,
    @InjectRepository(ProgramQuestion)
    private readonly programQuestionRepository: Repository<ProgramQuestion>,
    private readonly logger: AppLoggerService,
  ) {
    // const config = openAIConfig(this.configService);
    // this.openai = new OpenAI({ apiKey: config.apiKey });
  }

  async generateSchemaFromMetadata(programMetadata: any) {
    try {
      const questions = await this.questionService.findAllNoLimit('published');
      const fieldPoolArray = questions.map((q) => {
        const options =
          Array.isArray(q.questionOptionMaps) && q.questionOptionMaps.length > 0
            ? q.questionOptionMaps
                .filter((opt) => opt.option)
                .map((opt) => ({
                  id: opt.option.id,
                  label: opt.option.name,
                  type: opt.option.type,
                }))
            : undefined;

        return {
          id: q.id,
          fieldName: q.label,
          type: q.type,
          ...(options ? { options } : {}),
        };
      });

      const formSectionData = await this.formSectionRepo.findAllNoLimit();
      const formSectionArray = formSectionData.map((section) => ({
        id: section.id,
        label: section.name,
      }));
      const messages = [
        {
          role: 'system',
          content: systemPrompt({ fieldPoolArray, alwaysIncludeArray, formSectionArray }),
        },
        { role: 'user', content: userPrompt(programMetadata) },
      ];

      const completion = await this.openai.chat.completions.create({
        model: 'gpt-4o',
        messages,
        temperature: 0.2,
      });

      let responseContent = completion.choices[0].message.content.trim();
      responseContent = responseContent.replace(/```json|```javascript|```/g, '').trim();

      const firstBracket = responseContent.indexOf('[');
      const lastBracket = responseContent.lastIndexOf(']');
      if (firstBracket !== -1 && lastBracket !== -1) {
        responseContent = responseContent.substring(firstBracket, lastBracket + 1);
      }

      let schema: any;
      try {
        schema = JSON.parse(responseContent);
      } catch (parseErr) {
        throw new InternalServerErrorException('Failed to parse OpenAI response as JSON');
      }

      const existing = Array.isArray(schema)
        ? schema.filter((f) => f.typeSource === 'existing')
        : [];
      const suggested = Array.isArray(schema)
        ? schema.filter((f) => f.typeSource === 'suggested')
        : [];

      const processed = await this.processFormSchema(
        existing,
        suggested,
        programMetadata.programId,
      );

      this.logger.log('ai generated form schema', {
        existing,
        suggested,
      });

      return { existing, suggested };
    } catch (error) {
      this.logger.log('Error in generateSchemaFromMetadata', { error: error.message, programMetadata });
      handleKnownErrors(ERROR_CODES.FORM_SCHEMA_GENERATION_FAILED, error);
    }
  }

  async processFormSchema(existingSchema: any[], suggestedSchema: any[], programId: number) {
    try {
      return await this.dataSource.transaction(async (manager) => {
        this.logger.log('Starting form schema processing transaction');

      const user = await this.userRepo.getUserByField('isAiUser', true);
      if (!user) {
        throw new InifniNotFoundException(
          ERROR_CODES.USER_NOTFOUND,
          null,
          null,
          'AI user not found',
        );
      }
      const userId = user.id;
      this.logger.log(formSchemaConstMessages.PROCESSING_FORM_SCHEMA, { userId, programId });

      const allQuestions = [...existingSchema, ...suggestedSchema];
      const existingIds = allQuestions.filter((q) => q && q.id != null).map((q) => q.id);
      const existingQuestionWithoutId = allQuestions.filter((q) => !q.id);
      
      this.logger.log(formSchemaConstMessages.ANALYZING_QUESTIONS, {
        totalQuestions: allQuestions.length,
        existingQuestionsCount: existingIds.length,
        newQuestionsCount: existingQuestionWithoutId.length,
        existingIds,
      });

      // Get transaction repositories
      const transactionalQuestionRepo = manager.getRepository(Question);
      const transactionalFormSectionRepo = manager.getRepository(FormSection);
      const transactionalOptionRepo = manager.getRepository(Option);
      const transactionalProgramQuestionRepo = manager.getRepository(ProgramQuestion);

      const existingQuestions = await Promise.all(
        existingIds.map((id) => this.questionService.findOneById(id)),
      );
      
      this.logger.log(formSchemaConstMessages.RETRIEVED_EXISTING_QUESTIONS, {
        count: existingQuestions.length,
      });

      if (existingQuestions.some((q) => !q)) {
        this.logger.error(formSchemaConstMessages.MISSING_QUESTIONS_DB, existingIds.toString());
        throw new InifniNotFoundException(
          'One or more existing questions not found in the database',
        );
      }

      // Process questions that don't have IDs (new questions)
      for (const q of allQuestions) {
        if (!q.id) {
          this.logger.log(formSchemaConstMessages.PROCESSING_NEW_QUESTION, {
            label: q.label,
            type: q.type,
            sectionId: q.section,
          });

          const formSectionId = q.section ? q.section : null;
          const foundFormSection = await transactionalFormSectionRepo.findOne({
            where: { id: formSectionId },
          });
          
          this.logger.log(formSchemaConstMessages.FORM_SECTION_LOOKUP, {
            sectionId: formSectionId,
            found: !!foundFormSection,
          });

          if (!foundFormSection) {
            this.logger.error('Form section not found', formSectionId.toString());
            throw new InifniNotFoundException(
              ERROR_CODES.FORM_SECTION_NOTFOUND,
              null,
              null,
              formSectionId.toString(),
            );
          }

          const questionEntity = new Question({
            label: q.label,
            type: q.type,
            config: {
              required: q.required,
            },
            status: 'published',
            createdBy: user,
            updatedBy: user,
            formSection: foundFormSection,
          });
          
          this.logger.log(formSchemaConstMessages.CREATED_QUESTION_ENTITY, {
            questionData: questionEntity,
          });

          // Save using transactional repository
          const createdQuestion = await transactionalQuestionRepo.save(questionEntity);
          this.logger.log(formSchemaConstMessages.QUESTION_SAVED, {
            questionId: createdQuestion.id,
          });

          // Process options within transaction
          if (q.options && Array.isArray(q.options)) {
            this.logger.log(formSchemaConstMessages.PROCESSING_QUESTION_OPTIONS, {
              questionId: createdQuestion.id,
              optionCount: q.options.length,
            });

            const createdOptions: Option[] = [];

            for (const opt of q.options) {
              this.logger.log(formSchemaConstMessages.PROCESSING_OPTION, {
                label: opt.label,
                type: opt.type,
              });

              const existingOption = await transactionalOptionRepo.findOne({
                where: { name: opt.label, type: opt.type },
              });

              if (existingOption) {
                if (existingOption.status !== 'published') {
                  existingOption.status = 'published';
                  existingOption.updatedBy = user;
                  await transactionalOptionRepo.save(existingOption);
                  this.logger.log(formSchemaConstMessages.UPDATED_EXISTING_OPTION, {
                    optionId: existingOption.id,
                  });
                }
                createdOptions.push(existingOption);
              } else {
                // Create new option within transaction
                const newOption = transactionalOptionRepo.create({
                  name: opt.label,
                  type: opt.dataType || opt.type,
                  status: 'published',
                  createdBy: user,
                  updatedBy: user,
                });
                const createdOption = await transactionalOptionRepo.save(newOption);
                createdOptions.push(createdOption);
                this.logger.log(formSchemaConstMessages.CREATED_NEW_OPTION, {
                  optionId: createdOption.id,
                });
              }
            }

            // Create question-option mappings within transaction
            // Note: You'll need to adapt this based on your QuestionOptionMap entity structure
            for (const option of createdOptions) {
              const questionOptionMap = manager.create('QuestionOptionMap', {
                question: createdQuestion,
                option: option,
                createdBy: user,
                updatedBy: user,
              });
              await manager.save('QuestionOptionMap', questionOptionMap);
            }

            this.logger.log(formSchemaConstMessages.MAPPED_OPTIONS_TO_QUESTION, {
              questionId: createdQuestion.id,
              optionCount: createdOptions.length,
            });
          }
          q.id = createdQuestion.id;
        }
      }

      // Handle program-question mappings within transaction
      const existingMappings = await transactionalProgramQuestionRepo.find({
        where: {
          program: { id: programId },
          question: { id: In(allQuestions.map((q) => q.id)) },
        },
        relations: ['question'],
      });
      
      this.logger.log(formSchemaConstMessages.FOUND_PROGRAM_MAPPINGS, {
        programId,
        mappingCount: existingMappings.length,
      });

      const existingQuestionIds = existingMappings.map((mapping) => mapping.question.id);
      const newQuestionIds = allQuestions
        .map((q) => q.id)
        .filter((id) => !existingQuestionIds.includes(id));

      if (newQuestionIds.length > 0) {
        // Create program-question mappings within transaction
        for (const questionId of newQuestionIds) {
          const programQuestionMap = transactionalProgramQuestionRepo.create({
            program: { id: programId },
            question: { id: questionId },
            createdBy: user,
            updatedBy: user,
          });
          await transactionalProgramQuestionRepo.save(programQuestionMap);
        }

        this.logger.log(formSchemaConstMessages.CREATED_NEW_MAPPINGS, {
          programId,
          newQuestionCount: newQuestionIds.length,
          questionIds: newQuestionIds,
        });
      } else {
        this.logger.log(formSchemaConstMessages.NO_NEW_QUESTIONS, { programId });
      }
      
      this.logger.log(formSchemaConstMessages.FORM_SCHEMA_COMPLETED, {
        programId,
        totalQuestionsProcessed: allQuestions.length,
        questionIds: allQuestions.map((q) => q.id),
      });

      this.logger.log('Form schema processing transaction completed successfully');
    });
    } catch (error) {
      this.logger.log('Error in processFormSchema', { error: error.message, programId });
      handleKnownErrors(ERROR_CODES.FORM_SCHEMA_PROCESSING_FAILED, error);
    }
  }
}