import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository, IsNull } from 'typeorm';
import { FeatureFlag } from 'src/common/entities';
import { CommonDataService } from 'src/common/services/commonData.service';
import { AppLoggerService } from 'src/common/services/logger.service';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { featureFlagMessages } from 'src/common/constants/strings-constants';
import { CommonStatus } from 'src/common/enum/common-status.enum';

@Injectable()
export class FeatureFlagRepository {
  constructor(
    @InjectRepository(FeatureFlag)
    private readonly featureFlagRepo: Repository<FeatureFlag>,
    private readonly commonDataService: CommonDataService,
    private readonly logger: AppLoggerService,
  ) {}

  async findActiveFlags(): Promise<FeatureFlag[]> {
    this.logger.log(featureFlagMessages.FINDING_ACTIVE_FLAGS);
    try {
      return await this.commonDataService.get(
        this.featureFlagRepo,
        ['id', 'key', 'label', 'flag'],
        { status: CommonStatus.ACTIVE, deletedAt: IsNull() },
        'all',
        'all',
        { id: 'ASC' },
      );
    } catch (error) {
      this.logger.error(featureFlagMessages.ERROR_FETCHING_FLAGS, error);
      handleKnownErrors(ERROR_CODES.FEATURE_FLAG_GET_FAILED, error);
    }
  }
  async findFlagByKey(featureKey: string): Promise<FeatureFlag | null> {
    this.logger.log(`Finding feature flag by key: ${featureKey}`);
    try {
      return await this.featureFlagRepo.findOne({
        where: { key: featureKey, status: CommonStatus.ACTIVE, deletedAt: IsNull() },
      });
    } catch (error) {
      this.logger.error(`Error fetching feature flag with key: ${featureKey}`, error);
      handleKnownErrors(ERROR_CODES.FEATURE_FLAG_GET_FAILED, error);
    }
  }
}
