import { Injectable, Logger } from '@nestjs/common';
import { CommunicationTemplatesRepository } from '../repositories/communication-templates.repository';
import { CommunicationTemplates } from 'src/common/entities/communication-templates.entity';
import { handleKnownErrors } from 'src/common/utils/handle-error.util';
import { ERROR_CODES } from 'src/common/constants/error-string-constants';
import { CommunicationTemplatesKeysEnum } from 'src/common/enum/communication-template-keys.enum';
import { CommunicationTypeEnum } from 'src/common/enum/communication-type.enum';
import { COMMUNICATION_TEMPLATE_NAMES_BY_KEYS } from 'src/common/constants/constants';

@Injectable()
export class CommunicationTemplatesService {
  private readonly logger = new Logger(CommunicationTemplatesService.name);

  constructor(private readonly templatesRepo: CommunicationTemplatesRepository) {}


  /**
   * Get all communication templates with pagination
   */
  async getAllTemplates(limit: number, offset: number): Promise<any> {
    try {
      this.logger.log(`Fetching all communication templates with limit: ${limit}, offset: ${offset}`);
      return await this.templatesRepo.findAll(limit, offset);
    } catch (error) {
      this.logger.error(`Failed to fetch communication templates: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_GET_FAILED, error);
    }
  }

  /**
   * Get a single communication template by ID
   */
  async getTemplateById(id: number): Promise<CommunicationTemplates> {
    try {
      this.logger.log(`Fetching communication template with ID: ${id}`);
      return await this.templatesRepo.findOne(id);
    } catch (error) {
      this.logger.error(`Failed to fetch communication template by ID: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_FIND_BY_ID_FAILED, error);
    }
  }
  async getTemplateByKey(templateKey: CommunicationTemplatesKeysEnum): Promise<CommunicationTemplates> {
    try {
      this.logger.log(`Fetching communication template with key: ${templateKey}`);
      return await this.templatesRepo.findByTemplateKey(templateKey);
    } catch (error) {
      this.logger.error(`Failed to fetch communication template by key: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_FIND_BY_KEY_FAILED, error);
    }
  }

  async getTemplateByKeyAndType( templateKey: CommunicationTemplatesKeysEnum, templateType: CommunicationTypeEnum): Promise<CommunicationTemplates> {
    try {
      this.logger.log(`Fetching communication template with key: ${templateKey} and type: ${templateType}`);
      const template = await this.templatesRepo.findByTemplateKeyAndType(templateKey, templateType);
      if (template && template.templateType === templateType) {
        return template;
      }
      throw new Error(`Template with key ${templateKey} and type ${templateType} not found`);
    } catch (error) {
      this.logger.error(`Failed to fetch communication template by key and type: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_FIND_BY_KEY_AND_TYPE_FAILED, error);
    }
  }
  async getAllTemplatesBy(data: any, limit: number, offset: number) {
    try {
      const responseData = await this.templatesRepo.findAllBy(data, limit, offset);
      const final = responseData.data.map((item) => {
        
        const { templateContent, templateId, ...rest } = item;
        const updatedItem = {
          ...rest,
          templateName: COMMUNICATION_TEMPLATE_NAMES_BY_KEYS[item.templateKey],
        };
        return updatedItem;
      });

      const finalResponseData = {
        data: final,
        pagination: responseData.pagination,
      }

      return finalResponseData;
    } catch (error) {
      this.logger.error(`Failed to fetch communication templates by data: ${error.message}`, error.stack);
      handleKnownErrors(ERROR_CODES.COMMUNICATION_TEMPLATE_GET_FAILED, error);
    }

  }

}